﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "SoundEngineMME.h"
#include <assert.h>

namespace nw {
namespace internal {
namespace winext {

CSoundEngineMME* CSoundEngineMME::sInstance = NULL;

CSoundEngineMME::CSoundEngineMME() :
    m_hThread( NULL ),
    m_hWaveOut( NULL ),
    m_DeviceID( WAVE_MAPPER ),
    m_nWaveBufferNum( 0 )
{
    HANDLE heap = GetProcessHeap();
    assert( heap );

    OSVERSIONINFOW versionInfo;
    versionInfo.dwOSVersionInfoSize = sizeof( OSVERSIONINFOW );

    if( ::GetVersionEx( &versionInfo ) && versionInfo.dwMajorVersion >= 6 && versionInfo.dwMinorVersion == 0 ){
        m_nWaveBufferNum = WAVE_BUF_NUM_FOR_VISTA;
    }
    else{
        m_nWaveBufferNum = WAVE_BUF_NUM;
    }

    for( unsigned int i = 0; i < m_nWaveBufferNum ; i++ )
    {
        m_WaveBuffer[i] = (LPSTR)HeapAlloc( heap, HEAP_ZERO_MEMORY, WAVE_BUF_SIZE );
        assert( m_WaveBuffer[i] );
        m_WaveHdr[i] = (LPWAVEHDR)HeapAlloc( heap, HEAP_ZERO_MEMORY, sizeof(WAVEHDR) );
        assert( m_WaveHdr[i] );
    }

    m_WaveFormat.wFormatTag = WAVE_FORMAT_PCM;
    m_WaveFormat.nChannels = CHANNEL_COUNT;
    m_WaveFormat.nSamplesPerSec = SAMPLE_RATE;
    m_WaveFormat.wBitsPerSample = 16;
    m_WaveFormat.nBlockAlign = 4;
    m_WaveFormat.nAvgBytesPerSec = m_WaveFormat.nSamplesPerSec * m_WaveFormat.nChannels * m_WaveFormat.wBitsPerSample / 8;
    m_WaveFormat.cbSize = 0;
}

CSoundEngineMME::~CSoundEngineMME(void)
{
    HANDLE heap = GetProcessHeap();
    assert( heap );

    for( unsigned int i = 0; i < m_nWaveBufferNum ; i++ )
    {
        HeapFree( heap, 0, m_WaveHdr[i] );
        HeapFree( heap, 0, m_WaveBuffer[i] );
    }
}

BOOL CSoundEngineMME::IsActive() const
{
    if ( m_hThread == NULL ) return FALSE;

    if ( ::WaitForSingleObject( m_hThread, 0 ) != WAIT_OBJECT_0 ) return TRUE;

    return FALSE;
}

BOOL CSoundEngineMME::OpenStream( UINT devideID )
{
    CloseStream();

    m_DeviceID = devideID;

    m_hThread = CreateThread( NULL, 0, (LPTHREAD_START_ROUTINE)ThreadProcFunc, this, 0, &m_ThreadID );
    if ( m_hThread == NULL )
    {
        return FALSE;
    }

    if ( ! SetupWaveOut() )
    {
        CloseStream();
        return FALSE;
    }

    return TRUE;
}

void CSoundEngineMME::CloseStream()
{
    if ( m_hThread )
    {
        if ( ::PostThreadMessage( m_ThreadID, WM_QUIT, 0, 0 ) ) {
            ::WaitForSingleObject( m_hThread, INFINITE );
        }
        ::CloseHandle( m_hThread );
        m_hThread = NULL;
    }
}

DWORD CSoundEngineMME::ThreadProcFunc( LPVOID pParam )
{
    CSoundEngineMME* engine = reinterpret_cast<CSoundEngineMME*>( pParam );
    return engine->ThreadProc();
}

BOOL CSoundEngineMME::SetupWaveOut()
{
    MMRESULT result = waveOutOpen( &m_hWaveOut, m_DeviceID, &m_WaveFormat, m_ThreadID, 0, CALLBACK_THREAD );
    if ( result != MMSYSERR_NOERROR ) {
        return FALSE;
    }

    result = waveOutPause( m_hWaveOut );
    if ( result != MMSYSERR_NOERROR ) {
        return FALSE;
    }

    for( unsigned int i = 0; i < m_nWaveBufferNum ; i++ )
    {
        m_WaveHdr[i]->lpData = m_WaveBuffer[i];
        m_WaveHdr[i]->dwBufferLength = WAVE_BUF_SIZE;
        m_WaveHdr[i]->dwUser = 0;
        m_WaveHdr[i]->dwFlags = 0;
        m_WaveHdr[i]->dwLoops = 0;

        WaveOutCallback( CHANNEL_COUNT, (signed short*)( m_WaveHdr[i]->lpData), WAVE_BUF_SAMPLES, SAMPLE_RATE );

        MMRESULT result = waveOutPrepareHeader( m_hWaveOut, m_WaveHdr[i], sizeof(WAVEHDR) );
        if ( result != MMSYSERR_NOERROR ) {
            return FALSE;
        }
        result = waveOutWrite( m_hWaveOut, m_WaveHdr[i], sizeof(WAVEHDR) );
        if ( result != MMSYSERR_NOERROR ) {
            return FALSE;
        }
    }

    result = waveOutRestart( m_hWaveOut );
    if ( result != MMSYSERR_NOERROR ) {
        return FALSE;
    }

    return TRUE;
}

void CSoundEngineMME::ShutdownWaveOut()
{
    if ( m_hWaveOut )
    {
        MMRESULT result = waveOutReset( m_hWaveOut );
        if ( result != MMSYSERR_NOERROR ) {
            assert( FALSE );
        }

        for( unsigned int i = 0; i < m_nWaveBufferNum ; i++ )
        {
            if (!  m_WaveHdr[i])							continue;
            if (! (m_WaveHdr[i]->dwFlags & WHDR_PREPARED))	continue;
            if (! (m_WaveHdr[i]->dwFlags & WHDR_DONE))		continue;

            result = waveOutUnprepareHeader(m_hWaveOut, m_WaveHdr[i], sizeof(WAVEHDR));
            if (result != MMSYSERR_NOERROR)
            {
                assert( FALSE );
            }
        }

        result = waveOutClose( m_hWaveOut );
        if ( result != MMSYSERR_NOERROR ) {
            assert( FALSE );
        }

        m_hWaveOut = NULL;
    }
}

DWORD CSoundEngineMME::ThreadProc()
{
    UINT retCode = MMSYSERR_NOERROR;

    try {
        MMRESULT result;

        ::SetThreadPriority(
            m_hThread,
#ifndef _DEBUG
            THREAD_PRIORITY_TIME_CRITICAL
#else
            THREAD_PRIORITY_HIGHEST
#endif
        );

        MSG msg;
        while( ::GetMessage( &msg, NULL, 0, 0 ) )
        {
            switch( msg.message ) {
            case MM_WOM_OPEN:
//                TRACE0("WOM_OPEN\n");
                break;
            case MM_WOM_CLOSE:
//                TRACE0("WOM_CLOSE\n");
                break;
            case MM_WOM_DONE:
                result = OnWaveOutDone( (LPWAVEHDR)msg.lParam );
                if ( result != MMSYSERR_NOERROR ) {
                    assert( FALSE );
                    throw result;
                }
                break;
            default:
                break;
            }
        }
    }
    catch ( MMRESULT result ) {
        retCode = result;
    }
    catch ( ... ) {
        retCode = 1;
    }

    ShutdownWaveOut();

    return retCode;
}

MMRESULT CSoundEngineMME::OnWaveOutDone( LPWAVEHDR waveHdr )
{
    MMRESULT result = waveOutUnprepareHeader( m_hWaveOut, waveHdr, sizeof(WAVEHDR) );
    if ( result != MMSYSERR_NOERROR ) {
        assert( FALSE );
        return result;
    }

    WaveOutCallback( CHANNEL_COUNT, (signed short*)( waveHdr->lpData), WAVE_BUF_SAMPLES, SAMPLE_RATE );

    result = waveOutPrepareHeader( m_hWaveOut, waveHdr, sizeof(WAVEHDR) );
    if ( result != MMSYSERR_NOERROR ) {
        assert( FALSE );
        return result;
    }
    result = waveOutWrite( m_hWaveOut, waveHdr, sizeof(WAVEHDR) );
    if ( result != MMSYSERR_NOERROR ) {
        assert( FALSE );
        return result;
    }

    return MMSYSERR_NOERROR;
}

} // namespace winext
} // namespace internal
} // namespace nw
