﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "SoundEngineCoreAudio.h"
#include "ai_dma.h"
#include <assert.h>
#include <sys/time.h>
#include <errno.h>
#include <stdio.h>

// #define ENABLE_MEASURE_RENDERING_TIME

namespace  {

#if defined(ENABLE_MEASURE_RENDERING_TIME)
static s64 GetCurrentTimeMs()
{
    int ret;
    timeval tv;

    ret = gettimeofday(&tv, NULL);

    return static_cast<s64>(tv.tv_sec) * 1000 + static_cas<s64>(tv.tv_usec) / 1000;
}
#endif

} // namespace

namespace nw {
namespace internal {
namespace winext {

CSoundEngineCoreAudio::CSoundEngineCoreAudio()
    : m_IsActive(false), m_RequestExitThread(false), m_WaveOutProc(NULL)
{

}

CSoundEngineCoreAudio::~CSoundEngineCoreAudio()
{

}

CSoundEngineCoreAudio* CSoundEngineCoreAudio::sInstance;
u8 CSoundEngineCoreAudio::sInstanceMemory[sizeof(CSoundEngineCoreAudio)];

bool CSoundEngineCoreAudio::IsActive() const
{
    return m_IsActive;
}

bool CSoundEngineCoreAudio::OpenStream(unsigned int devideID)
{
    int ret;

    AIDMA_Init(SAMPLE_RATE);

    m_RequestExitThread = false;


    // すでに名前付きセマフォがあった時には削除した上で、名前付きセマフォを作成
    const char* semaphoreName = "/jp.co.nintendo.nw.snd.sem";
    sem_unlink(semaphoreName);
    m_pRenderRequestSemaphore = sem_open(semaphoreName, O_CREAT | O_EXCL, 0644, 0);

    assert(m_pRenderRequestSemaphore != SEM_FAILED);

    ret = pthread_create(&m_Thread, NULL, threadProcFunc, NULL);
    if(ret < 0)
    {
        return false;
    }

    m_IsActive = true;

    return true;
}

void* CSoundEngineCoreAudio::threadProcFunc(void*)
{
    CSoundEngineCoreAudio* engine = reinterpret_cast<CSoundEngineCoreAudio*>(CSoundEngineCoreAudio::Instance());
    return engine->ThreadProc(NULL);
}


void CSoundEngineCoreAudio::RenderAudioFrames(s16 *dst, s32 numFrames)
{
#if defined(ENABLE_MEASURE_RENDERING_TIME)
    const s64 beginTimeMs = GetCurrentTimeMs();
    const s32 originalNumFrames = numFrames;
#endif
    for(; numFrames > 0; --numFrames)
    {
        WaveOutCallback(CHANNEL_COUNT, dst, WAVE_BUF_SAMPLES, SAMPLE_RATE);
        dst += (WAVE_BUF_SAMPLES * CHANNEL_COUNT);
    }
#if defined(ENABLE_MEASURE_RENDERING_TIME)
    printf("[%s:%05d] renderd[ms] (%d), elapsed[ms] (%lld)\n", __func__, __LINE__, 3 * originalNumFrames, GetCurrentTimeMs() - beginTimeMs);
#endif
}

void* CSoundEngineCoreAudio::ThreadProc(void*)
{
    const int numBuffers = BUFFER_COUNT;
    const int numCh = CHANNEL_COUNT;
    // 1 AudioFrame = 3 ms
    const s32 numRenderFrames = 32;
    const s32 bufferSize = WAVE_BUF_SAMPLES * numRenderFrames * numCh * sizeof(s16);

    static s16 buffers[bufferSize * numBuffers];

    s16* currentBuffer = buffers;
    u32 numBufferEnqueued = 0;

    // レンダラスレッドの処理ループ
    while(!m_RequestExitThread)
    {
        // バッファキューが空いた通知を待機
        // 備考）バッファリング中はリクエストを待たない
        if(numBufferEnqueued >= numBuffers && !WaitForRequestRender())
        {
            continue;
        }

        // レンダリング (ステレオ)
        RenderAudioFrames(currentBuffer, numRenderFrames);

        // OpenSL ES へ転送
        AIDMA_ExecuteDMA(reinterpret_cast<u8*>(currentBuffer), bufferSize);
        ++numBufferEnqueued;

        currentBuffer = buffers + WAVE_BUF_SAMPLES * numRenderFrames * numCh * (numBufferEnqueued % numBuffers);
    }


    return NULL;
}

void CSoundEngineCoreAudio::CloseStream()
{
    int ret;

    m_RequestExitThread = true;

    ret = pthread_join(m_Thread, NULL);
    assert(ret >= 0);

    AIDMA_Quit();

    ret = sem_close(m_pRenderRequestSemaphore);
    assert(ret >= 0);

    m_IsActive = false;
}

// レンダリングを要求する
void CSoundEngineCoreAudio::RequestRender()
{
    int ret;
    ret = sem_post(m_pRenderRequestSemaphore);
    assert(ret >= 0);
}

// レンダリングを要求されるまで待つ
bool CSoundEngineCoreAudio::WaitForRequestRender()
{
    int ret;
    timeval tv;

    ret = gettimeofday(&tv, NULL);

    // TODO: ios に timedwait がない！
    ret = sem_wait(m_pRenderRequestSemaphore);
    assert(ret >= 0 || errno == ETIMEDOUT);

    return ret >= 0;
}

} // namespace winext
} // namespace internal
} // namespace nw

