﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "AudioBufferQueue.h"

namespace nw {
namespace internal {
namespace winext {

AudioBufferQueue::AudioBufferQueue()
  : m_EnqueuedBuffersCount(0), m_DequeuedBuffersCount(0)
{

}

AudioBufferQueue::~AudioBufferQueue()
{

}

AudioBuffer* AudioBufferQueue::Enqueue(u8* address, u32 size)
{
    std::lock_guard<std::mutex> lockGuard(m_Mutex);
    AudioBuffer *pEnqueuedAudioBuffer = nullptr;

    // サイズが上限を超えていないか確認
    if((m_EnqueuedBuffersCount + 1) - m_DequeuedBuffersCount <= BuffersCount)
    {
        m_AudioBufferQueue[m_EnqueuedBuffersCount % BuffersCount].mData  = address;
        m_AudioBufferQueue[m_EnqueuedBuffersCount % BuffersCount].mDataByteSize = size;
        m_AudioBufferQueue[m_EnqueuedBuffersCount % BuffersCount].mNumberChannels = 2;
        pEnqueuedAudioBuffer = &m_AudioBufferQueue[m_EnqueuedBuffersCount % BuffersCount];
        ++m_EnqueuedBuffersCount;
    }

    return pEnqueuedAudioBuffer;
}

AudioBuffer* AudioBufferQueue::Dequeue()
{
    std::lock_guard<std::mutex> lockGuard(m_Mutex);
    AudioBuffer *pDequeuedAudioBuffer = nullptr;

    // Dequeue できるバッファがあるか確認
    if(m_EnqueuedBuffersCount > m_DequeuedBuffersCount)
    {
        ++m_DequeuedBuffersCount;

        // Dequeue されたバッファの個数が一周した時、値を 0 に巻き戻し、
        // Enqueue されたバッファの個数もそれに合わせて減算する
        if(m_DequeuedBuffersCount == BuffersCount)
        {
            m_DequeuedBuffersCount = 0;
            m_EnqueuedBuffersCount -= BuffersCount;
        }

        pDequeuedAudioBuffer = &m_AudioBufferQueue[m_DequeuedBuffersCount];
    }

    return pDequeuedAudioBuffer;
}

AudioBuffer* AudioBufferQueue::Peek()
{
    std::lock_guard<std::mutex> lockGuard(m_Mutex);
    AudioBuffer* pAudioBuffer = nullptr;

    if(m_EnqueuedBuffersCount > m_DequeuedBuffersCount)
    {
        pAudioBuffer = &m_AudioBufferQueue[m_DequeuedBuffersCount];
    }

    return pAudioBuffer;
}

uint32_t AudioBufferQueue::GetRemainBuffersCount()
{
    std::lock_guard<std::mutex> lockGuard(m_Mutex);

    return m_EnqueuedBuffersCount - m_DequeuedBuffersCount;
}

} // namespace nw::snd::internal
} // namespace nw::snd
} // namespace nw

