﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <cstdarg>
#include <wctype.h>
#include <nw/ut/ut_String.h>
#if 0
#include <nw/ut/ut_FormatParser16.h>
#endif

namespace nw {
namespace ut {

const char16 HIGH_SURROGATE_START = 0xD800;
const char16 LOW_SURROGATE_START = 0xDC00;
const char16 HIGH_SURROGATE_END = 0xDBFF;
const char16 LOW_SURROGATE_END = 0xDFFF;

//--------------------------------------------------------------------------
size_t
wcs16towcs32(char32* dst, const char16* src, std::size_t dstCount, char32 replacement)
{
    const char16* srcCurrent = src;
    size_t copyCount = 0;
    while (*srcCurrent != 0)
    {
        if ((dstCount - 1) <= copyCount)
        {
            // 出力バッファのサイズを超える場合は変換を中断します。
            copyCount = (dstCount - 1);
            break;
        }

        char16 charCurrent = *srcCurrent++;
        if ((charCurrent >= HIGH_SURROGATE_START) && (charCurrent <= HIGH_SURROGATE_END))
        {
            if (*srcCurrent != 0)
            {
                char16 charNext = *srcCurrent++;
                if ((charNext >= LOW_SURROGATE_START) && (charNext <= LOW_SURROGATE_END))
                {
                    // 4バイトで表現される文字の処理
                    dst[copyCount++] = static_cast<char32>(
                        ((charCurrent - HIGH_SURROGATE_START) << 10) +
                        (charNext - LOW_SURROGATE_START) +
                        0x0010000);
                }
                else
                {
                    // 不正な文字
                    if (replacement)
                    {
                        dst[copyCount++] = replacement;
                    }
                }
            }
        }
        else if ((charCurrent >= LOW_SURROGATE_START) && (charCurrent <= LOW_SURROGATE_END))
        {
            // 不正な文字
            if (replacement)
            {
                dst[copyCount++] = replacement;
            }
        }
        else
        {
            // UTF-16 からそのままキャストできる範囲
            dst[copyCount++] = static_cast<char32>(charCurrent);
        }
    }
    dst[copyCount] = 0;

    return copyCount;
}

//--------------------------------------------------------------------------
size_t
wcs32towcs16(char16* dst, const char32* src, std::size_t dstCount, char16 replacement)
{
    const char32* srcCurrent = src;
    size_t copyCount = 0;
    while (*srcCurrent != 0)
    {
        if ((dstCount - 1) <= copyCount)
        {
            // 出力バッファのサイズを超える場合は変換を中断します。
            copyCount = (dstCount - 1);
            break;
        }

        char32 charCurrent = *srcCurrent++;
        if (charCurrent < 0xFFFF)
        {
            if ((charCurrent >= HIGH_SURROGATE_START) && (charCurrent <= LOW_SURROGATE_END))
            {
                // 不正な文字
                if (replacement)
                {
                    dst[copyCount++] = replacement;
                }
            }
            else
            {
                // UTF-32 からそのままキャストできる範囲
                dst[copyCount++] = static_cast<char16>(charCurrent);
            }
        }
        else if (charCurrent > 0x0010FFFF)
        {
            // 不正な文字
            if (replacement)
            {
                dst[copyCount++] = replacement;
            }
        }
        else
        {
            if ((dstCount - 2) <= copyCount)
            {
                // 出力バッファのサイズを超える場合は変換を中断します。
                break;
            }

            // 4バイトで表現される文字の処理
            charCurrent -= 0x0010000;
            dst[copyCount++] = static_cast<char16>((charCurrent >> 10)     + HIGH_SURROGATE_START);
            dst[copyCount++] = static_cast<char16>((charCurrent & 0x3FFUL) + LOW_SURROGATE_START);
        }
    }
    dst[copyCount] = 0;

    return copyCount;
}

//--------------------------------------------------------------------------
size_t
mbstowcs16(char16* dst, const char* src, std::size_t dstCount, char16 replacement)
{
    static const int UTF8TrailingBytes[] =
    {
        0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
        0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
        0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
        1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 2, 2, 2, 2, 2, 2, 2, 2, 3, 3, 3, 3, 4, 4, 5, 5
    };

    static const char32 UTF8Offsets[] =
    {
        0x00000000, 0x00003080, 0x000E2080, 0x03C82080, 0xFA082080, 0x82082080
    };

    const char* srcCurrent = src;
    size_t copyCount = 0;
    while (*srcCurrent != 0)
    {
        if (dst && (dstCount - 1) <= copyCount)
        {
            // 出力バッファのサイズを超える場合は変換を中断します。
            break;
        }

        int trailingBytes = UTF8TrailingBytes[static_cast<unsigned char>(*srcCurrent) >> 1];
        char32 charCurrent = 0;

        // 複数バイトで形成された UTF-8 の文字を UTF-32 に変換
        switch (trailingBytes)
        {
        case 5 : charCurrent += static_cast<unsigned char>(*srcCurrent); srcCurrent++; charCurrent <<= 6;  // don't break;
        case 4 : if (*srcCurrent != 0) { charCurrent += static_cast<unsigned char>(*srcCurrent); srcCurrent++; charCurrent <<= 6; }    // don't break;
        case 3 : if (*srcCurrent != 0) { charCurrent += static_cast<unsigned char>(*srcCurrent); srcCurrent++; charCurrent <<= 6; }    // don't break;
        case 2 : if (*srcCurrent != 0) { charCurrent += static_cast<unsigned char>(*srcCurrent); srcCurrent++; charCurrent <<= 6; }    // don't break;
        case 1 : if (*srcCurrent != 0) { charCurrent += static_cast<unsigned char>(*srcCurrent); srcCurrent++; charCurrent <<= 6; }    // don't break;
        case 0 : if (*srcCurrent != 0) { charCurrent += static_cast<unsigned char>(*srcCurrent); srcCurrent++; }
        }
        charCurrent -= UTF8Offsets[trailingBytes];

        // UTF-16 へ変換
        if (charCurrent < 0xFFFF)
        {
            if ((charCurrent >= HIGH_SURROGATE_START) && (charCurrent <= LOW_SURROGATE_END))
            {
                // 不正な文字
                if (replacement)
                {
                    if ( dst ) { dst[copyCount] = replacement; }

                    ++copyCount;
                }
            }
            else
            {
                // そのままキャストできる範囲
                if ( dst ) { dst[copyCount] = static_cast<char16>(charCurrent); }

                ++copyCount;
            }
        }
        else if (charCurrent > 0x0010FFFF)
        {
            // 不正な文字
            if (replacement)
            {
                if ( dst ) { dst[copyCount] = replacement; }

                ++copyCount;
            }
        }
        else
        {
            if (dst && (dstCount - 2) <= copyCount)
            {
                // 出力バッファのサイズを超える場合は変換を中断します。
                break;
            }

            // 4バイトで表現される文字の処理
            charCurrent -= 0x0010000;

            if ( dst ) { dst[copyCount] = static_cast<char16>((charCurrent >> 10)     + HIGH_SURROGATE_START); }
            ++copyCount;
            if ( dst ) { dst[copyCount] = static_cast<char16>((charCurrent & 0x3FFUL) + LOW_SURROGATE_START); }
            ++copyCount;
        }
    }

    if ( dst ) { dst[copyCount] = 0; }

    return copyCount;
}

//--------------------------------------------------------------------------
size_t
wcs16tombs(char* dst, const char16* src, std::size_t dstCount, char replacement)
{
    static const u8 UTF8FirstBytes[] =
    {
        0x00, 0x00, 0xC0, 0xE0, 0xF0, 0xF8, 0xFC
    };

    const char16* srcCurrent = src;
    size_t copyCount = 0;
    while (*srcCurrent != 0)
    {
        if ( dst && (dstCount - 1) <= copyCount)
        {
            // 出力バッファのサイズを超える場合は変換を中断します。
            copyCount = (dstCount - 1);
            break;
        }

        char32 charCurrent = *srcCurrent++;

        if ((charCurrent >= HIGH_SURROGATE_START) && (charCurrent <= HIGH_SURROGATE_END))
        {
            if (*srcCurrent != 0)
            {
                char32 charNext = *srcCurrent++;
                if ((charNext >= LOW_SURROGATE_START) && (charNext <= LOW_SURROGATE_END))
                {
                    charCurrent = static_cast<char32>(((charCurrent - HIGH_SURROGATE_START) << 10) + (charNext - LOW_SURROGATE_START) + 0x0010000);
                }
            }
            else
            {
                // 不正な文字
                if (replacement)
                {
                    if ( dst ) { dst[copyCount] = replacement; }

                    ++copyCount;
                }
            }
        }

        if (charCurrent > 0x0010FFFF)
        {
            // 不正な文字
            if (replacement)
            {
                if ( dst ) { dst[copyCount] = replacement; }

                ++copyCount;
            }
        }
        else
        {
            // 書き込むバイト数を取得
            int bytesToWrite = 1;
            if (charCurrent <  0x80)
            {
                bytesToWrite = 1;
            }
            else if (charCurrent <  0x800)
            {
                bytesToWrite = 2;
            }
            else if (charCurrent <  0x10000)
            {
                bytesToWrite = 3;
            }
            else if (charCurrent <= 0x0010FFFF)
            {
                bytesToWrite = 4;
            }

            if (dst && dstCount <= (copyCount + bytesToWrite))
            {
                // 出力バッファのサイズを超える場合は変換を中断します。
                if ((dstCount - 1) < copyCount)
                {
                    copyCount = (dstCount - 1);
                }
                break;
            }

            u8 bytes[4];
            switch (bytesToWrite)
            {
            case 4 : bytes[3] = static_cast<u8>((charCurrent | 0x80) & 0xBF); charCurrent >>= 6;    // don't break;
            case 3 : bytes[2] = static_cast<u8>((charCurrent | 0x80) & 0xBF); charCurrent >>= 6;    // don't break;
            case 2 : bytes[1] = static_cast<u8>((charCurrent | 0x80) & 0xBF); charCurrent >>= 6;    // don't break;
            case 1 : bytes[0] = static_cast<u8> (charCurrent | UTF8FirstBytes[bytesToWrite]);
            }

            const u8* currentByte = bytes;
            switch (bytesToWrite)
            {
            case 4 :
                if ( dst ) { dst[copyCount] = *currentByte++; }
                ++copyCount;
                // don't break;
            case 3 :
                if ( dst ) { dst[copyCount] = *currentByte++; }
                ++copyCount;
                // don't break;
            case 2 :
                if ( dst ) { dst[copyCount] = *currentByte++; }
                ++copyCount;
                // don't break;
            case 1 :
                if ( dst ) { dst[copyCount] = *currentByte++; }
                ++copyCount;
                // don't break;
            }
        }
    }

    if ( dst ) { dst[copyCount] = 0; }

    return copyCount;
}

//--------------------------------------------------------------------------
size_t
mbslen(const char* src, std::size_t bufferSize)
{
    static const int UTF8TrailingBytes[] =
    {
        0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
        0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
        0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
        1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 2, 2, 2, 2, 2, 2, 2, 2, 3, 3, 3, 3, 4, 4, 5, 5
    };

    const char* srcCurrent = src;

    size_t count = 0;

    while ( count < bufferSize )
    {
        int trailingBytes = UTF8TrailingBytes[static_cast<unsigned char>(*srcCurrent) >> 1] + 1;

        if ( count + trailingBytes > bufferSize )
        {
            break;
        }

        srcCurrent += trailingBytes;
        count += trailingBytes;
    }

    return count;
}

//--------------------------------------------------------------------------
size_t
strncat(char* dst, std::size_t dstCount, const char* src, std::size_t srcCount)
{
#ifdef NW_COMPILER_MSVC
#pragma warning(push)
#pragma warning(disable : 4996)
#endif
    size_t length  = std::strlen(dst);
    if (dstCount <= length + srcCount)
    {
        srcCount = dstCount - length - 1;
    }

    std::strncat( dst, src, srcCount );

    dst[dstCount-1] = '\0';

    return srcCount;

#ifdef NW_COMPILER_MSVC
#pragma warning(pop)
#endif
}

//--------------------------------------------------------------------------
#if 0
// MEMO: 32 ビットへ変換しない独自実装を追加。ただし、まだ検証が済むまで旧バージョンを利用します。
#ifndef NW_COMPILER_MSVC
int
vsnw16printf(char16* buffer, std::size_t bufferSize, std::size_t strCount, const char16* format, va_list vargs)
{
    if (bufferSize <= 0)
    {
        return -1;
    }

    detail::OutBuffer16 outBuffer(buffer, bufferSize);
    detail::FormatParser16 parser(vargs);

    u32 format_idx = 0;

    //  本来は文字定数をL'*'とするのがいいのでしょうが、
    //  Cafeでうまくいかないらしいので'*'形式(char)で書いています。
    //  定数なのでコンパイラがうまくやってくれるはず

    while ( true )
    {
        char16 format_letter = format[format_idx];
        format_idx++;

        if (format_letter == '\0' || format_idx > strCount)
        {
            // すべて終了 -> 出力した文字数を返す
            return outBuffer.getLetterNum();
        }

        if (format_letter == '%')
        {
            // '%' -> 解析して引数の値を出力
            u32 parsed_letter_num = parser.execute(&outBuffer, &format[format_idx]);
            format_idx += parsed_letter_num;
        }
        else
        {
            // 普通の文字 -> そのまま出力
            outBuffer.append(format_letter);
        }

        if (outBuffer.isOverflow())
        {
            // dstのサイズが足りない
            return -1;
        }
    }
}
#endif
#endif

//--------------------------------------------------------------------------
int
stricmp(const char *string1, const char *string2)
{
#if defined(NW_COMPILER_MSVC)
    return ::_stricmp(string1, string2);
#elif defined(NW_COMPILER_CWCC)
    return ::stricmp(string1, string2);
#elif defined(NW_COMPILER_GHS) || defined(NW_COMPILER_CLANG)
    return ::strcasecmp(string1, string2);
#else
    return std::strcasecmp(string1, string2);
#endif
}

//--------------------------------------------------------------------------
int
wcs16icmp(const char16 *string1, const char16 *string2)
{
#if defined(NW_COMPILER_MSVC)
    return ::_wcsicmp(string1, string2);
#elif defined(NW_COMPILER_CWCC)
    return ::wcsicmp(string1, string2);
#elif defined(NW_COMPILER_GHS) || defined(NW_COMPILER_CLANG)
    while (::towlower(static_cast<wchar_t>(*string1)) == ::towlower(static_cast<wchar_t>(*string2)))
    {
        if (*string1 == 0) { return 0; }
        ++string1;
        ++string2;
    }
    return ::towlower(static_cast<wchar_t>(*string1)) - ::towlower(static_cast<wchar_t>(*string2));
#else
    return std::wcscasecmp(string1, string2);
#endif
}

//--------------------------------------------------------------------------
static char32
tolower_(const char32 c)
{
    if ('A' <= c && c <= 'Z') return c + 'a' - 'A';
    return c;
}

//--------------------------------------------------------------------------
int
wcs32icmp(const char32 *string1, const char32 *string2)
{
    while (tolower_(*string1) == tolower_(*string2))
    {
        if (*string1 == 0) { return 0; }
        ++string1;
        ++string2;
    }

    return tolower_(*string1) - tolower_(*string2);
}

//--------------------------------------------------------------------------
size_t
wcs16len(const char16* src)
{
    std::size_t length = 0;

    while (*src != 0) { ++src; ++length; }

    return length;
}

}   /* namespace ut */
}   /* namespace nw */
