﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nw/types.h>

#if defined(NW_MCS_ENABLE)

#include <nw/mcs/mcs_Pcio2CommDevice.h>
#include <nw/mcs/mcs_Common.h>
#include <nw/mcs/mcs_SimpleRingBuffer.h>
#include <nw/ut/ut_ScopedLock.h>
#include <nw/ut/ut_Inlines.h>

#if defined( NW_PLATFORM_WIN32 ) || defined(NW_USE_NINTENDO_SDK)
// TODO: NintendoSdk 対応後、このコメントを削除してください。

#include <winsock.h>
#include <windows.h>
#include <time.h>

#endif


namespace nw
{
namespace mcs
{
namespace internal
{

namespace
{

enum ChunkID
{
    ChunkID_RegistName = 1, // 未使用
    ChunkID_Reboot,
    ChunkID_InitBuffer,
    ChunkID_Ack,
    ChunkID_Disconnect,
    ChunkID_MessageToServer,
    ChunkID_MessageToApplication,
    ChunkID_ServerTime,
    ChunkID_CheckAlive,
    ChunkID_Unknown = 0
};

enum ResultCode
{
    ProtocolError = 1,
    AlreadyRegisted,
    Success = 0
};

struct ChunkHeader
{
    u32 chunkID;
    u32 chunkBytes;
};

struct RegistNameChunk
{
    u32 chunkID;
    u32 chunkBytes;
    u32 nameLen;
};

struct MessageChunk
{
    u32 chunkID;
    u32 chunkBytes;
    u32 channel;
    u32 msgBytes;
};

}

class PCIO2CommDeviceImpl : public PCIO2CommDevice
{
    static const int MCS_PORT = 4000;

    enum {
        ChunkHeaderBytes = 4 + 4, // id + size
        MessageHeaderBytes = 4 + 4, // channel + size
        WritableBytes = 32 * 1024 - ChunkHeaderBytes - MessageHeaderBytes,
        ReadBuffBytes = 1024,
        EnumEnd_ = 0
    };

    //! MCSサーバとのネゴシエーションの状態を表します。
    //! PCIO2CommDeviceImpl::m_IsOpen が true の場合にのみ意味を持ちます。
    enum NegoPhase
    {
        //! ソケットが接続されていない状態です。
        NEGOPHASE_NOATTACH,

        //! ソケットを接続しようとしている状態です。
        NEGOPHASE_CONNECTING,

        //! ソケットが接続された状態です。
        NEGOPHASE_ATTACHED,

        //! 再起動メッセージを送信した状態です。
        NEGOPHASE_REBOOT,

        //! 通信が確立した状態です。
        NEGOPHASE_CONNECT,

        //! Open関数成功後にエラーが発生した状態です。
        //! ソケットが閉じられた場合は含みません。
        //! Close()でリセットされます。
        NEGOPHASE_ERROR
    };

    //! チャンクの読み出しの進行状態を表します。
    //! PCIO2CommDeviceImpl::m_NegoPhase が NEGOPHASE_REBOOT または NEGOPHASE_CONNECT の
    //! 場合に意味を持ちます。
    enum ChunkPhase
    {
        //! チャンクヘッダーの読み込み中です。
        CHUNKPHASE_HEADER,

        //! チャンクの残り部分を読み飛ばし中です。
        CHUNKPHASE_IGNORE,

        //! メッセージヘッダーの読み込み中です。
        CHUNKPHASE_MESSAGE_HEADER,

        //! メッセージボディの読み込み中です。
        CHUNKPHASE_MESSAGE_BODY,

        //! サーバ時間の読み込み中です。
        CHUNKPHASE_SERVERTIME,
    };

    typedef nw::ut::LinkList<ChannelInfo, offsetof(ChannelInfo,link)> ChannelInfoList;

public:

    //---------------------------------------------------------------------------
    //! @brief コンストラクタです。
    //---------------------------------------------------------------------------
    PCIO2CommDeviceImpl::PCIO2CommDeviceImpl(
        const DeviceInfo& deviceInfo)
        : m_IsStartup(false)
        , m_IsOpen(false)
        , m_ServerTime(0)
    {
        (void)deviceInfo;
        int err = WSAStartup(MAKEWORD(1,1), &m_WsaData);
        if (err != 0)
        {
            NW_LOG("WSAStartup() failed. (%d)\n", err);
            return;
        }

        m_IsStartup = true;
    }

    //---------------------------------------------------------------------------
    //! @brief デストラクタです。
    //---------------------------------------------------------------------------
    virtual ~PCIO2CommDeviceImpl()
    {
        if (m_IsStartup)
        {
            WSACleanup();
        }
    }

    //---------------------------------------------------------------------------
    //! @brief データ受信用のバッファを登録します。
    //!
    //! @details
    //! 受信用バッファが受信データで一杯になり、新規に受信したデータを格納する
    //! だけの空き容量が無い場合は、その受信データは捨てられます。
    //! 従って、通信で使用するデータ量に合わせてバッファサイズを十分な大きさに
    //! 設定する必要があります。
    //!
    //! @param[in]   channel    データ送受信の通信相手を一意に識別するための値。
    //! @param[in]   buf        登録する受信用バッファ。
    //! @param[in]   bufSize    登録する受信用バッファのサイズ。
    //!
    //! @return なし。
    //---------------------------------------------------------------------------
    virtual void RegisterBuffer(
        ChannelType channel,
        void* buf,
        u32 bufSize);

    //---------------------------------------------------------------------------
    //! @brief 受信用バッファの登録を解除します。
    //!
    //! @param[in]   channel   データ送受信の通信相手を一意に識別するための値。
    //!
    //! @return      なし。
    //---------------------------------------------------------------------------
    virtual void* UnregisterBuffer(ChannelType channel);

    //---------------------------------------------------------------------------
    //! @brief      データ受信用に登録されたバッファを取得します。
    //!
    //! @return     Mcs_RegisterBuffer で登録されたバッファです。
    //!             登録されたバッファが存在しない場合は NULL を返します。
    //---------------------------------------------------------------------------
    virtual void* GetRegisteredBuffer(ChannelType channel);

    //---------------------------------------------------------------------------
    //! @brief 通信デバイスをオープンします。
    //!
    //! @return 関数が成功した場合 MCS_ERROR_SUCCESS を返します。
    //! 失敗した場合エラーコードを返します。
    //---------------------------------------------------------------------------
    virtual u32 Open();

    //---------------------------------------------------------------------------
    //! @brief 通信デバイスをクローズします。
    //---------------------------------------------------------------------------
    virtual void Close();

    //---------------------------------------------------------------------------
    //! @brief メインループ内でこの関数を呼び出してください。
    //!
    //! @return 関数が成功した場合0を返します。
    //! 失敗した場合エラーコード(0以外の値)を返します。
    //---------------------------------------------------------------------------
    virtual u32 Polling();

    //---------------------------------------------------------------------------
    //! @brief       Readメソッドで、ブロックすることなく読み込めるデータのサイズを
    //!              取得します。
    //!
    //! @param[in]   channel   データ送受信の通信相手を一意に識別するための値。
    //!
    //! @return      読み込み可能なデータのサイズを返します。
    //---------------------------------------------------------------------------
    virtual u32 GetReadableBytes(ChannelType channel);

    //---------------------------------------------------------------------------
    //! @brief       データを読み取ります。読み取ったデータは受信バッファから
    //!              削除しません。
    //!
    //! @details
    //! size で指定したサイズが、その時点で受信バッファに格納されている
    //! データサイズより大きい場合は、受信バッファに格納されている
    //! サイズだけ読み込みます。
    //!
    //! @param[in]   channel   データ送受信の通信相手を一意に識別するための値。
    //! @param[in]   data      読み込むデータを格納するバッファへのポインタ。
    //! @param[in]   size      読み込むデータを格納するバッファのサイズ。
    //!
    //! @return      読み取ったバイト数を返します。
    //---------------------------------------------------------------------------
    virtual u32 Peek(
        ChannelType channel,
        void* data,
        u32 size);

    //---------------------------------------------------------------------------
    //! @brief       データの受信を行います。
    //!
    //! @details
    //! sizeで指定したサイズ全てを受信するまで制御を返しません。
    //! 従って、その時点で受信バッファに格納されているデータサイズ
    //! より大きいサイズを読み込む場合、関数呼び出しはブロックされます。
    //!
    //! @param[in]   channel   データ受信の識別用の値。ユーザー任意に決められます。
    //! @param[in]   data      読み込むデータを格納するバッファへのポインタ。
    //! @param[in]   size      読み込むデータを格納するバッファのサイズ。
    //!
    //! @return      関数が成功した場合0を返します。
    //!               失敗した場合エラーコード(0以外の値)を返します。
    //---------------------------------------------------------------------------
    virtual u32 Read(
        ChannelType channel,
        void* data,
        u32 size);


    //---------------------------------------------------------------------------
    //! @brief       一度に送信可能なサイズを取得します。
    //!
    //! @param[in]   channel          現時点では使用しません。
    //! @param[out]  pWritableBytes   送信可能なバイト数を格納する変数へのポインタ。
    //!
    //! @return      関数が成功した場合0を返します。
    //!               失敗した場合エラーコード(0以外の値)を返します。
    //---------------------------------------------------------------------------
    virtual u32 GetWritableBytes(
        ChannelType channel,
        u32* pWritableBytes);

    //---------------------------------------------------------------------------
    //! @brief       データの送信を行います。
    //!
    //! @details
    //! sizeで指定したサイズ全てを送信するまで制御を返しません。
    //! GetWritableBytesメソッドが返すサイズよりも大きいサイズの
    //! データを送信する場合、関数呼び出しはブロックされます。
    //!
    //! @param[in]   channel   データ送受信の通信相手を一意に識別するための値。
    //! @param[in]   data      送信するデータを格納するバッファへのポインタ。
    //! @param[in]   size      送信するデータのサイズ。
    //!
    //! @return      関数が成功した場合0を返します。
    //!               失敗した場合エラーコード(0以外の値)を返します。
    //---------------------------------------------------------------------------
    virtual u32 Write(
        ChannelType channel,
        const void* data,
        u32 size);

    //---------------------------------------------------------------------------
    //! @brief       mcsサーバとの接続状態を示す値を取得します。
    //!
    //! @return      mcsサーバと接続していたらtrue、接続していない場合はfalseを
    //!               返します。
    //---------------------------------------------------------------------------
    virtual bool IsServerConnect();

    //---------------------------------------------------------------------------
    //! @brief       mcsサーバの時間を取得します。
    //!
    //! @return      2000/01/01 を基準とする msec です。
    //---------------------------------------------------------------------------
    virtual s64 GetServerTime();

private:
    //---------------------------------------------------------------------------
    //! @brief Mcsサーバとソケットの接続を試みます。
    //!
    //! @details
    //! 接続に成功すると ATTACHED 状態に遷移します。
    //! 接続できな買った場合は NOATTACH 状態のままです。
    //!
    //! @return
    //!
    //---------------------------------------------------------------------------
    u32 Connect();

    //---------------------------------------------------------------------------
    //! @brief Mcsサーバとソケット接続を切断します。
    //!
    //! @details
    //! NOATTACH 状態に遷移します。
    //---------------------------------------------------------------------------
    void Disconnect();

    //---------------------------------------------------------------------------
    //! @brief Mcsサーバとのネゴシエートを行います。
    //!
    //! @return
    //! 正常に処理が行われた場合には MCS_ERROR_SUCCESS を返します。
    //! 処理中に通信エラーが生じた場合は MCS_ERROR_COMERROR を返します。
    //---------------------------------------------------------------------------
    u32 Negotiate();

    //---------------------------------------------------------------------------
    //! @brief 指定サイズをソケットからバッファに読み込みます。
    //!
    //! @details
    //! 返り値が 0 の場合、処理すべきデータが無いのでポーリングを終了してください。
    //!
    //! ソケットが切断された場合には NEGOPHASE_NOATTACH に遷移します。
    //! その他のエラー発生時には NEGOPHASE_ERROR に遷移します。
    //!
    //! @return 読み込んだバイト数を返します。エラーが発生した場合は負の値を返します。
    //---------------------------------------------------------------------------
    u32 ReadToBuff(int size);

    //---------------------------------------------------------------------------
    //! @brief ソケットでメッセージを送信します。
    //!
    //! @details
    //! データがすべて送信できるまで Polling() を繰り返します。
    //!
    //! ソケットが切断された場合には NEGOPHASE_NOATTACH に遷移します。
    //! その他のエラー発生時には NEGOPHASE_ERROR に遷移します。
    //!
    //! @param[in] data データの格納されたメモリアドレスです。
    //! @param[in] size データのバイト数です。
    //!
    //! @return MCSのエラーコードを返します。
    //---------------------------------------------------------------------------
    u32 Write(const void* data, u32 size);

    //---------------------------------------------------------------------------
    //! @brief ソケットでデータを送信します。
    //!
    //! @details
    //! データがすべて送信できるまでブロックします。
    //!
    //! ソケットが切断された場合には NEGOPHASE_NOATTACH に遷移します。
    //! その他のエラー発生時には NEGOPHASE_ERROR に遷移します。
    //!
    //! @param[in] data データの格納されたメモリアドレスです。
    //! @param[in] size データのバイト数です。
    //!
    //! @return MCSのエラーコードを返します。
    //---------------------------------------------------------------------------
    u32 Send(const void* data, u32 size);

    //---------------------------------------------------------------------------
    //! @brief 通信を切断し、エラー状態にします。
    //---------------------------------------------------------------------------
    void Abort();

    //---------------------------------------------------------------------------
    //! @brief       エラーが発生しているかどうかの状態を取得します。
    //!
    //! @return      エラーが発生していたらtrue、そうでない場合はfalseを
    //!               返します。
    //---------------------------------------------------------------------------
    bool IsError()
    {
        return m_NegoPhase == NEGOPHASE_ERROR;
    }

    //---------------------------------------------------------------------------
    //! @brief       チャンネルを指定して、ChannelInfoへのポインタを取得します。
    //!
    //! @param[in]   channel   データ送受信の通信相手を一意に識別するための値。
    //!
    //! @return      指定したチャンネルに対するChannelInfoが存在すれば、
    //!               その変数へのポインタを返します。
    //!               見つからないときは0を返します。
    //---------------------------------------------------------------------------
    ChannelInfo* GetChannelInfo(ChannelType channel);

    //---------------------------------------------------------------------------
    //! @brief チャンネル毎に登録されている受信バッファの内容をクリアします。
    //---------------------------------------------------------------------------
    void InitRegisteredBuffer();

    //---------------------------------------------------------------------------
    //! @brief       送信のために一時的に待ちます。
    //!
    //! @return      関数が成功した場合0を返します。
    //!               失敗した場合エラーコード(0以外の値)を返します。
    //---------------------------------------------------------------------------
    u32 WaitSendData();

    //---------------------------------------------------------------------------
    //! @brief       受信のために一時的に待ちます。
    //!
    //! @return      関数が成功した場合0を返します。
    //!               失敗した場合エラーコード(0以外の値)を返します。
    //---------------------------------------------------------------------------
    u32 WaitRecvData();

    //---------------------------------------------------------------------------
    //! @brief       mcsサーバの時間を設定します。
    //!
    //! @param[in]   time mcsサーバから受信した時間です。
    //---------------------------------------------------------------------------
    void SetServerTime(s64 time);

    WSADATA m_WsaData;
    SOCKET m_Socket;

    u32 m_NegoPhase;
    u32 m_ChunkPhase;
    u32 m_ChunkID;
    u32 m_ChunkBodyBytes;
    u32 m_Channel;
    u32 m_MessageBytes;
    u32 m_ReadBytes;
    u32 m_IgnoreBytes;
    ChannelInfo* m_pChannelInfo;
    ChannelInfoList m_ChannelInfoList;
    s64 m_ServerTime;
    bool m_IsStartup;
    bool m_IsOpen;
    u8 m_ReadBuf[ReadBuffBytes];
};

//---------------------------------------------------------------------------
/* virtual */
void
PCIO2CommDeviceImpl::RegisterBuffer(
    ChannelType channel,
    void* buf,
    u32 bufSize)
{
    // 既に同じチャンネルを使用しているものが無いこと。
    NW_ASSERT(NULL == this->GetChannelInfo(channel));

    NW_ASSERT_NOT_NULL(buf);

    u32 start = reinterpret_cast<u32>(buf);
    u32 end = start + bufSize;

    start = RoundUp(start, 4);
    end = RoundDown(end, 4);

    NW_ASSERT(start < end);
    NW_ASSERT(end - start >= sizeof(ChannelInfo) + sizeof(SimpleRingBufferHeader) + sizeof(u32) + 4 + 4);

    ChannelInfo* pInfo = new(reinterpret_cast<void*>(start))ChannelInfo();

    pInfo->channel  = channel;
    const u32 ringBufStart = start + sizeof(ChannelInfo);
    pInfo->recvBuf.Init(reinterpret_cast<void*>(ringBufStart), end - ringBufStart);

    m_ChannelInfoList.push_back(pInfo);

    pInfo->allocatedBuffer = buf;
}

//---------------------------------------------------------------------------
/* virtual */ void*
PCIO2CommDeviceImpl::UnregisterBuffer(ChannelType channel)
{
    ChannelInfo* pInfo = this->GetChannelInfo(channel);
    NW_ASSERT(pInfo);

    if (pInfo)
    {
        m_ChannelInfoList.erase(pInfo);

        if (pInfo == m_pChannelInfo)
        {
            m_pChannelInfo = NULL;
        }

        return pInfo->allocatedBuffer;
    }

    return NULL;
}

//---------------------------------------------------------------------------
void*
PCIO2CommDeviceImpl::GetRegisteredBuffer(ChannelType channel)
{
    ChannelInfo* pInfo = this->GetChannelInfo(channel);

    return (pInfo) ? pInfo->allocatedBuffer : NULL;
}

//---------------------------------------------------------------------------
/* virtual */ u32
PCIO2CommDeviceImpl::Open()
{
    if (m_IsOpen)
    {
        return MCS_ERROR_SUCCESS;
    }

    m_IsOpen = true;
    m_NegoPhase = NEGOPHASE_NOATTACH;

    return this->Negotiate();
}

//---------------------------------------------------------------------------
/* virtual */ void
PCIO2CommDeviceImpl::Close()
{
    if (! m_IsOpen)
    {
        return;
    }

    if (m_NegoPhase != NEGOPHASE_NOATTACH)
    {
        this->Disconnect();
    }

    m_IsOpen = false;

    this->SetServerTime(0);
}

//---------------------------------------------------------------------------
/* private */ u32
PCIO2CommDeviceImpl::Negotiate()
{
    if (this->IsError())
    {
        return MCS_ERROR_COMERROR;
    }

    if (m_NegoPhase == NEGOPHASE_NOATTACH || m_NegoPhase == NEGOPHASE_CONNECTING)
    {
        if (MCS_ERROR_SUCCESS == this->Connect())
        {
            m_NegoPhase = NEGOPHASE_ATTACHED;
        }
    }

    if (m_NegoPhase == NEGOPHASE_ATTACHED)
    {
        ChunkHeader rebootChunk = { htonl(ChunkID_Reboot), 0 };
        if (MCS_ERROR_SUCCESS == this->Send(&rebootChunk, sizeof(rebootChunk)))
        {
            m_NegoPhase = NEGOPHASE_REBOOT;
            m_ChunkPhase = CHUNKPHASE_HEADER;
            m_ReadBytes = 0;
        }
    }

    return (m_NegoPhase == NEGOPHASE_ERROR)? MCS_ERROR_COMERROR : MCS_ERROR_SUCCESS;
}

//---------------------------------------------------------------------------
/* virtual */ u32
PCIO2CommDeviceImpl::Polling()
{
    // オープンしていないときは何もしない。
    if (! m_IsOpen)
    {
        return MCS_ERROR_SUCCESS;
    }

    // エラーが起きた場合は、Openしなおすまで何もしない。
    if (this->IsError())
    {
        return MCS_ERROR_COMERROR;
    }

    u32 errorCode;
    if ((errorCode = this->Negotiate()) != MCS_ERROR_SUCCESS)
    {
        return errorCode;
    }

    while (m_NegoPhase == NEGOPHASE_REBOOT || m_NegoPhase == NEGOPHASE_CONNECT)
    {
        switch (m_ChunkPhase)
        {
        case CHUNKPHASE_HEADER:
            // チャンクのヘッダを読み込み ChunkID に応じて分岐します。
            {
                int res = this->ReadToBuff(ChunkHeaderBytes);
                if (res < 0)
                {
                    goto Error;
                }
                else if (res == 0)
                {
                    // 処理すべきデータがないのでポーリング終了。
                    return MCS_ERROR_SUCCESS;
                }

                // ヘッダを全部読み終わった場合。
                if (m_ReadBytes == ChunkHeaderBytes)
                {
                    u32* pChunk = reinterpret_cast<u32*>(m_ReadBuf);

                    m_ChunkID = ntohl(pChunk[0]);
                    m_ChunkBodyBytes = ntohl(pChunk[1]);
                    m_ReadBytes = 0;

                    switch (m_ChunkID)
                    {
                    case ChunkID_MessageToApplication:
                        // 通信が確立するまで、届いたメッセージは無視する。
                        if (m_NegoPhase != NEGOPHASE_CONNECT)
                        {
                            m_IgnoreBytes = m_ChunkBodyBytes;
                            m_ChunkPhase = CHUNKPHASE_IGNORE;
                            break;
                        }

                        m_ChunkPhase = CHUNKPHASE_MESSAGE_HEADER;
                        break;

                    case ChunkID_RegistName:
                        m_IgnoreBytes = m_ChunkBodyBytes;
                        m_ChunkPhase = CHUNKPHASE_IGNORE;
                        break;

                    case ChunkID_InitBuffer:
                        {
                            ChunkHeader ackChunk = { htonl(ChunkID_Ack), 0 };
                            if (MCS_ERROR_SUCCESS == this->Send(&ackChunk, sizeof(ackChunk)))
                            {
                                this->InitRegisteredBuffer();
                                m_NegoPhase = NEGOPHASE_CONNECT;
                            }
                        }
                        break;

                    case ChunkID_Reboot:
                    case ChunkID_Disconnect:
                        m_NegoPhase = NEGOPHASE_ATTACHED;
                        this->Negotiate();
                        break;

                    case ChunkID_ServerTime:
                        m_ChunkPhase = CHUNKPHASE_SERVERTIME;
                        break;

                    case ChunkID_CheckAlive:
                        {
                            ChunkHeader chunk = { htonl(ChunkID_CheckAlive), 0 };
                            this->Send(&chunk, sizeof(chunk));
                            m_IgnoreBytes = m_ChunkBodyBytes;
                            m_ChunkPhase = CHUNKPHASE_IGNORE;
                        }
                        break;

                    default:
                        NW_LOG("Unknown chunk id. (%d)\n", m_ChunkID);
                        m_NegoPhase = NEGOPHASE_ATTACHED;
                        this->Negotiate();
                        break;
                    }
                }
            }
            break;

        case CHUNKPHASE_IGNORE:
            // m_IgnoreBytes に指定されたバイト数のデータを読み飛ばします。
            // 読み飛ばしが終わったら CHUNKPHASE_HEADER に戻ります。
            {
                if (m_IgnoreBytes > 0)
                {
                    int readBytes = nw::ut::Min((u32)ReadBuffBytes, m_IgnoreBytes);
                    int res = this->ReadToBuff(readBytes);
                    if (res < 0)
                    {
                        goto Error;
                    }
                    else if (res == 0)
                    {
                        return MCS_ERROR_SUCCESS;
                    }

                    m_IgnoreBytes -= m_ReadBytes;
                    m_ReadBytes = 0;
                }

                if (m_IgnoreBytes == 0)
                {
                    m_ChunkPhase = CHUNKPHASE_HEADER;
                }
            }
            break;

        case CHUNKPHASE_MESSAGE_HEADER:
            // MCSサーバからアプリへのメッセージのヘッダ部分を受信します。
            // ヘッダの受信が完了したら CHUNKPHASE_MESSAGE_BODY に遷移します。
            {
                // ヘッダが読み切れるまで recv を繰り返し。
                int res = this->ReadToBuff(MessageHeaderBytes);
                if (res < 0)
                {
                    goto Error;
                }
                else if (res == 0)
                {
                    return MCS_ERROR_SUCCESS;
                }

                // ヘッダを全部読み終わった場合。
                if (m_ReadBytes == MessageHeaderBytes)
                {
                    u32* pChunk = reinterpret_cast<u32*>(m_ReadBuf);

                    m_Channel = ntohl(pChunk[0]);
                    m_MessageBytes = ntohl(pChunk[1]);

                    m_ReadBytes = 0;

                    m_pChannelInfo = GetChannelInfo(static_cast<ChannelType>(m_Channel));

                    if (m_pChannelInfo == NULL)
                    {
                        // チャンネルが登録されていないので無視します。
                        m_ChunkPhase = CHUNKPHASE_IGNORE;
                        m_IgnoreBytes = m_ChunkBodyBytes - MessageHeaderBytes;
                    }
                    else
                    {
                        m_ChunkPhase = CHUNKPHASE_MESSAGE_BODY;
                        // チャンクのパディング部分のサイズを求めておきます。
                        m_IgnoreBytes = m_ChunkBodyBytes - MessageHeaderBytes - m_MessageBytes;
                    }
                }
            }
            break;

        case CHUNKPHASE_MESSAGE_BODY:
            // メッセージのボディを受信し、チャンネル毎の受信バッファに振り分けます。
            {
                // 受信中にチャンネルの登録が抹消された場合。
                if (m_pChannelInfo == NULL)
                {
                    // まだ読み取っていないメッセージ部分とチャンクのパディング領域を
                    // 読み飛ばします。
                    m_IgnoreBytes += m_MessageBytes;
                    m_ChunkPhase = CHUNKPHASE_IGNORE;
                    break;
                }

                u32 readBytes = nw::ut::Min((u32)ReadBuffBytes, m_MessageBytes);
                readBytes = nw::ut::Min(readBytes, m_pChannelInfo->recvBuf.GetWritableBytes());

                // チャンネルの読み込みバッファがいっぱい。
                if (readBytes == 0)
                {
                    return MCS_ERROR_SUCCESS;
                }

                int res = this->ReadToBuff(readBytes);
                if (res < 0)
                {
                    goto Error;
                }
                else if (res == 0)
                {
                    return MCS_ERROR_SUCCESS;
                }

                m_MessageBytes -= m_ReadBytes;

                m_pChannelInfo->recvBuf.Write(m_ReadBuf, m_ReadBytes);

                m_ReadBytes = 0;

                if (m_MessageBytes == 0)
                {
                    if (m_IgnoreBytes > 0)
                    {
                        m_ChunkPhase = CHUNKPHASE_IGNORE;
                    }
                    else
                    {
                        m_ChunkPhase = CHUNKPHASE_HEADER;
                    }
                }
            }
            break;

        case CHUNKPHASE_SERVERTIME:
            // MCSサーバの時間を受信します。
            // 受信が完了したら CHUNKPHASE_HEADER に遷移します。
            {
                int res = this->ReadToBuff(sizeof(s64));
                if (res < 0)
                {
                    goto Error;
                }
                else if (res == 0)
                {
                    return MCS_ERROR_SUCCESS;
                }

                // ヘッダを全部読み終わった場合。
                if (m_ReadBytes == sizeof(s64))
                {
                    s64* pChunk = reinterpret_cast<s64*>(m_ReadBuf);
                    m_ReadBytes = 0;

                    s64 msec = internal::NetToHost(pChunk[0]);
                    this->SetServerTime(msec);

                    m_ChunkPhase = CHUNKPHASE_HEADER;
                }
            }
            break;
        default:
            NW_FATAL_ERROR("Unexpected m_ChunkPhase :%d\n", m_ChunkPhase);
            break;
        }
    }

    return MCS_ERROR_SUCCESS;

Error:
    if (this->IsError())
    {
        return MCS_ERROR_COMERROR;
    }
    else
    {
        return MCS_ERROR_NOTCONNECT;
    }
}

//---------------------------------------------------------------------------
/* virtual */ u32
PCIO2CommDeviceImpl::GetReadableBytes(ChannelType channel)
{
    NW_ASSERT(m_IsOpen);

    ChannelInfo *const pInfo = this->GetChannelInfo(channel);

    NW_ASSERT_NOT_NULL(pInfo);

    return pInfo->recvBuf.GetReadableBytes();
}

//---------------------------------------------------------------------------
/* virtual */ u32
PCIO2CommDeviceImpl::Peek(
    ChannelType channel,
    void* data,
    u32 size)
{
    NW_ASSERT(m_IsOpen);

    ChannelInfo *const pInfo = this->GetChannelInfo(channel);

    NW_ASSERT_NOT_NULL(pInfo);

    SimpleDataRange dataRange;

    pInfo->recvBuf.BeginRead(&dataRange);
    return pInfo->recvBuf.ContinueRead(&dataRange, data, size);
}

//---------------------------------------------------------------------------
/* virtual */ u32
PCIO2CommDeviceImpl::Read(
    ChannelType channel,
    void* data,
    u32 size)
{
    NW_ASSERT(m_IsOpen);

    ChannelInfo *const pInfo = this->GetChannelInfo(channel);

    NW_ASSERT_NOT_NULL(pInfo);

    u8 *const pDst = static_cast<u8*>(data);
    int retryCnt = 0;
    u32 offset = 0;

    while (offset < size)
    {
        if (pInfo->recvBuf.GetReadableBytes() == 0)
        {
            if (this->IsError())
            {
                return MCS_ERROR_COMERROR;
            }

            if (! this->IsServerConnect())
            {
                return MCS_ERROR_NOTCONNECT;
            }

            if (u32 errorCode = this->WaitRecvData()) // 読み込みできるようになるまで待つ
            {
                return errorCode;
            }
        }
        else
        {
            u32 readByte = GetMin(size - offset, pInfo->recvBuf.GetReadableBytes());
            readByte = pInfo->recvBuf.Read(pDst + offset, readByte);
            offset += readByte;
            retryCnt = 0;
        }
    }

    return MCS_ERROR_SUCCESS;
}

//---------------------------------------------------------------------------
/* virtual */ u32
PCIO2CommDeviceImpl::GetWritableBytes(
    ChannelType channel,
    u32* pWritableBytes)
{
    NW_ASSERT(m_IsOpen);

    NW_UNUSED_VARIABLE(channel);

    if (this->IsError())
    {
        return MCS_ERROR_COMERROR;
    }

    if (! this->IsServerConnect())
    {
        return MCS_ERROR_NOTCONNECT;
    }

    if (pWritableBytes)
    {
        *pWritableBytes = WritableBytes;
    }

    return MCS_ERROR_SUCCESS;
}

//---------------------------------------------------------------------------
/* virtual */ u32
PCIO2CommDeviceImpl::Write(
    ChannelType channel,
    const void* data,
    u32 size)
{
    NW_ASSERT(m_IsOpen);

    if (this->IsError())
    {
        return MCS_ERROR_COMERROR;
    }

    if (! this->IsServerConnect())
    {
        return MCS_ERROR_NOTCONNECT;
    }

    MessageChunk chunk = { htonl(ChunkID_MessageToServer) };
    // TODO: パディングの計算
    chunk.chunkBytes = htonl(sizeof(MessageChunk) - sizeof(ChunkHeader) + size);
    chunk.channel = htonl(channel);
    chunk.msgBytes = htonl(size);

    // ヘッダの送信
    u32 res = this->Write(&chunk, sizeof(chunk));
    if (res != MCS_ERROR_SUCCESS)
    {
        return res;
    }

    // メッセージボディの送信
    res = this->Write(data, size);

    // TODO: パディングの送信。

    return res;
}

//---------------------------------------------------------------------------
/* virtual */ bool
PCIO2CommDeviceImpl::IsServerConnect()
{
    return m_NegoPhase == NEGOPHASE_CONNECT;
}

//---------------------------------------------------------------------------
/* virtual */ s64
PCIO2CommDeviceImpl::GetServerTime()
{
    SYSTEMTIME systemTime;
    FILETIME   fileTime;

    GetLocalTime( &systemTime );
    SystemTimeToFileTime( &systemTime, &fileTime );

    SYSTEMTIME baseSystemTime;
    FILETIME   baseFileTime;

    baseSystemTime.wYear = 2000;
    baseSystemTime.wMonth = 1;
    baseSystemTime.wDay = 1;
    baseSystemTime.wHour = 0;
    baseSystemTime.wMinute = 0;
    baseSystemTime.wSecond = 0;
    baseSystemTime.wMilliseconds = 0;

    SystemTimeToFileTime( &baseSystemTime, &baseFileTime );

    s64 time = s64(fileTime.dwLowDateTime) + (s64(fileTime.dwHighDateTime) << 32);
    s64 baseTime = s64(baseFileTime.dwLowDateTime) + (s64(baseFileTime.dwHighDateTime) << 32);

    // FILETIME は 100 nsec 単位なので、msec に変換
    return (time - baseTime) / 10000;
}

//---------------------------------------------------------------------------
/* private */ void
PCIO2CommDeviceImpl::SetServerTime(s64 time)
{
    m_ServerTime = time;
}

//---------------------------------------------------------------------------
/* private */ u32
PCIO2CommDeviceImpl::Connect()
{
    if (!m_IsStartup)
    {
        return MCS_ERROR_COMERROR;
    }

    if (m_NegoPhase == NEGOPHASE_NOATTACH)
    {
        m_Socket = socket(AF_UNSPEC, SOCK_STREAM, IPPROTO_TCP);
        if (m_Socket == INVALID_SOCKET)
        {
            int errCode = WSAGetLastError();

            NW_LOG("socket() failed. (%d)\n", errCode);

            return MCS_ERROR_COMERROR;
        }

        // ノンブロッキングモードに設定。
        {
            unsigned long arg = 1;
            if (SOCKET_ERROR == ioctlsocket(m_Socket, FIONBIO, &arg))
            {
                NW_LOG("ioctlsocket() failed. (%d)\n", WSAGetLastError());
                goto SocketError;
            }
        }
    }

    struct sockaddr_in addr;
    addr.sin_family = AF_INET;
    addr.sin_port = htons(MCS_PORT);
    addr.sin_addr.S_un.S_addr = inet_addr("127.0.0.1");

    if (m_NegoPhase == NEGOPHASE_NOATTACH)
    {
        if (SOCKET_ERROR == connect(m_Socket, (struct sockaddr *)&addr, sizeof(addr)))
        {
            int errCode = WSAGetLastError();

            switch (errCode)
            {
            case WSAEWOULDBLOCK: // 最初のconnect()時。
                m_NegoPhase = NEGOPHASE_CONNECTING;
                return MCS_ERROR_COMERROR;

            case WSAEISCONN: // 接続が確立したとき。
                break;

            case WSAECONNREFUSED:
                goto SocketError;

            default:
                NW_LOG("connect() failed. (%d)\n", errCode);
                goto SocketError;
            }
        }
    }
    else
    {
        NW_ASSERT(m_NegoPhase == NEGOPHASE_CONNECTING);

        struct timeval timeout = { 0, 0 }; // non-blocking
        fd_set wfds, efds;
        FD_ZERO(&wfds);
        FD_ZERO(&efds);
#if defined(_WIN32)
#pragma warning(push) // winsock.h 用の警告抑制
#pragma warning(disable:4127) // conditional expression is constant
#endif
        FD_SET(m_Socket, &wfds);
        FD_SET(m_Socket, &efds);
#if defined(_WIN32)
#pragma warning(pop)
#endif

        select(0, NULL, &wfds, &efds, &timeout);

        if (FD_ISSET(m_Socket, &efds))
        {
            goto SocketError;
        }

        if (!FD_ISSET(m_Socket, &wfds))
        {
            return MCS_ERROR_COMERROR;
        }
    }

    return MCS_ERROR_SUCCESS;

SocketError:
    closesocket(m_Socket);
    m_NegoPhase = NEGOPHASE_NOATTACH;
    return MCS_ERROR_COMERROR;
}

//---------------------------------------------------------------------------
/* private */ void
PCIO2CommDeviceImpl::Disconnect()
{
    shutdown(m_Socket, 2);
    closesocket(m_Socket);
}

//---------------------------------------------------------------------------
/* private */ u32
PCIO2CommDeviceImpl::ReadToBuff(int size)
{
    int res = recv(m_Socket, (char*)m_ReadBuf + m_ReadBytes, size - m_ReadBytes, 0);

    if (res == SOCKET_ERROR)
    {
        int errCode = WSAGetLastError();
        switch (errCode)
        {
        case WSAEWOULDBLOCK:
            return 0;

        case WSAENETRESET: // Keep-alive activity detecting a failure.
        case WSAECONNABORTED: // Data transmission time-out, or protocol error.
        case WSAECONNRESET: // Connection reset by peer.
            // 接続が切れた場合。
            this->Disconnect();
            m_NegoPhase = NEGOPHASE_NOATTACH;
            return 0;

        default: // 予期しないエラー。
            NW_LOG("send() failed. (%d)\n", errCode);
            m_NegoPhase = NEGOPHASE_ERROR;
            return res;
        }
    }

    // ソケットはクローズされた。
    if (res == 0)
    {
        this->Disconnect();
        m_NegoPhase = NEGOPHASE_NOATTACH;
        return 0;
    }

    if (res > 0)
    {
        m_ReadBytes += res;
    }

    return res;
}

//---------------------------------------------------------------------------
/* private */ u32
PCIO2CommDeviceImpl::Write(const void* data, u32 size)
{
    const char *const pSrc = static_cast<const char*>(data);
    u32 offset = 0;

    while (offset < size)
    {
        const u32 restSize = size - offset;

        int res = send(m_Socket, pSrc + offset, restSize, 0);
        if (res == SOCKET_ERROR)
        {
            int errCode = WSAGetLastError();

            switch (errCode)
            {
            case WSAENETRESET: // Keep-alive activity detecting a failure.
            case WSAECONNABORTED: // Data transmission time-out, or protocol error.
            case WSAECONNRESET: // Connection reset by peer.
                // 接続が切れた場合。
                this->Disconnect();
                m_NegoPhase = NEGOPHASE_NOATTACH;
                return MCS_ERROR_NOTCONNECT;

            case WSAEWOULDBLOCK: // Resource temporarily unavailable.
                // バッファがいっぱい？
                if (u32 errorCode = this->WaitSendData()) // 書き込みできるようになるまで待つ
                {
                    return errorCode;
                }
                res = 0;
                break;

            default: // 予期しないエラー。
                NW_LOG("send() failed. (%d)\n", errCode);
                m_NegoPhase = NEGOPHASE_ERROR;
                return MCS_ERROR_COMERROR;
            }
        }
        else if (res == 0)
        {
            if (u32 errorCode = this->WaitSendData()) // 書き込みできるようになるまで待つ
            {
                return errorCode;
            }
        }

        offset -= (u32)res;
    }

    return MCS_ERROR_SUCCESS;   // 最後まで書き込めたら成功を返す
}


//---------------------------------------------------------------------------
/* private */ u32
PCIO2CommDeviceImpl::Send(const void* data, u32 size)
{
    const char *const pSrc = static_cast<const char*>(data);
    u32 offset = 0;

    while (offset < size)
    {
        const u32 restSize = size - offset;

        int res = send(m_Socket, pSrc + offset, restSize, 0);
        if (res == SOCKET_ERROR)
        {
            int errCode = WSAGetLastError();

            switch (errCode)
            {
            case WSAENETRESET: // Keep-alive activity detecting a failure.
            case WSAECONNABORTED: // Data transmission time-out, or protocol error.
            case WSAECONNRESET: // Connection reset by peer.
                // 接続が切れた場合。
                this->Disconnect();
                m_NegoPhase = NEGOPHASE_NOATTACH;
                return MCS_ERROR_NOTCONNECT;

            default: // 予期しないエラー。
                NW_LOG("send() failed. (%d)\n", errCode);
                m_NegoPhase = NEGOPHASE_ERROR;
                return MCS_ERROR_COMERROR;
            }
        }

        offset -= (u32)res;
    }

    return MCS_ERROR_SUCCESS;   // 最後まで書き込めたら成功を返す
}

//---------------------------------------------------------------------------
/* private */ void
PCIO2CommDeviceImpl::Abort()
{
    m_NegoPhase = NEGOPHASE_ERROR;
}

//---------------------------------------------------------------------------
/* private */ ChannelInfo*
PCIO2CommDeviceImpl::GetChannelInfo(ChannelType channel)
{
    ChannelInfo* pInfo = 0;

    while (0 != (pInfo = m_ChannelInfoList.GetNext(pInfo)))
    {
        if (pInfo->channel == channel)
        {
            return pInfo;
        }
    }

    return 0;
}

//---------------------------------------------------------------------------
/* private */ void
PCIO2CommDeviceImpl::InitRegisteredBuffer()
{
    ChannelInfo* pInfo = 0;

    while (0 != (pInfo = m_ChannelInfoList.GetNext(pInfo)))
    {
        pInfo->recvBuf.Discard();
    }
}

//---------------------------------------------------------------------------
/* private */ u32
PCIO2CommDeviceImpl::WaitSendData()
{
    if (u32 errorCode = this->Polling())  // PC側が書き込み待ちでブロックしてしまわないようにしておく
    {
        return errorCode;
    }

    SleepThread(8);

    return MCS_ERROR_SUCCESS;
}

//---------------------------------------------------------------------------
/* private */ u32
PCIO2CommDeviceImpl::WaitRecvData()
{
    if (u32 errorCode = this->Polling())  // PC側が書き込み待ちでブロックしてしまわないようにしておく
    {
        return errorCode;
    }

    SleepThread(8);

    return MCS_ERROR_SUCCESS;
}

//---------------------------------------------------------------------------
/* public, static */ bool
PCIO2CommDevice::EnumerateDevice(IDeviceEnumerate& enumerate)
{
    DeviceInfo deviceInfo = { DEVICETYPE_PCIO2 };
    return enumerate.Find(deviceInfo);
}

//---------------------------------------------------------------------------
/* public, static */ u32
PCIO2CommDevice::GetDeviceObjectMemSize(const DeviceInfo& deviceInfo)
{
    (void)deviceInfo;
    return sizeof(PCIO2CommDeviceImpl);
}

//---------------------------------------------------------------------------
/* public, static */ CommDevice*
PCIO2CommDevice::Construct(
    void* mem,
    const DeviceInfo& deviceInfo)
{
    return new(mem) PCIO2CommDeviceImpl(deviceInfo);
}

}   // namespace internal
}   // namespace mcs
}   // namespace nw

#endif  // #if defined(NW_MCS_ENABLE)
