﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nw/types.h>
#include <nw/mcs/mcs_Base.h>
#include <nw/mcs/mcs_InputDevice.h>
#include <nw/mcs/mcs_Common.h>
#include <nw/ut/ut_ScopedLock.h>

#include <cstring>

#if defined( NW_PLATFORM_WIN32 ) || defined(NW_USE_NINTENDO_SDK)
// TODO: NintendoSdk 対応後、このコメントを削除してください。
    #include <windows.h>
#endif

namespace nw
{
namespace mcs
{

void        InputEventHandler::KeyDownEvent(const KeyEventArg& /* arg */)       {}
void        InputEventHandler::KeyUpEvent(const KeyEventArg& /* arg */)         {}
void        InputEventHandler::CharEvent(const CharEventArg& /* arg */)         {}

void        InputEventHandler::MouseMoveEvent(const MouseEventArg& /* arg */)   {}
void        InputEventHandler::MouseDownEvent(const MouseEventArg& /* arg */)   {}
void        InputEventHandler::MouseUpEvent(const MouseEventArg& /* arg */)     {}
void        InputEventHandler::MouseDoubleClickEvent(const MouseEventArg& /* arg */)   {}
void        InputEventHandler::MouseWheelEvent(const MouseEventArg& /* arg */)  {}

}   // namespace mcs
}   // namespace nw

#if defined(NW_MCS_ENABLE)

namespace
{

using namespace nw::mcs;
using namespace nw::mcs::internal;


/* ========================================================================
    定数定義
   ======================================================================== */

enum EVENTTYPE
{
    EVENTTYPE_KEYDOWN,
    EVENTTYPE_KEYUP,
    EVENTTYPE_CHAR,

    EVENTTYPE_MOUSEMOVE,
    EVENTTYPE_MOUSEDOWN,
    EVENTTYPE_MOUSEUP,
    EVENTTYPE_MOUSEDOUBLECLICK,
    EVENTTYPE_MOUSEWHEEL,

    EVENTTYPE_CAPTURESTART,
    EVENTTYPE_CAPTURESTOP,

    EVENTTYPE_Max
};

enum REQUESTTYPE
{
    REQUESTTYPE_STOPCAPTURE,

    REQUESTTYPE_Max
};

/* ========================================================================
    staticな変数
   ======================================================================== */

int                 s_InitializedCount  = 0;        // 初期化回数の参照カウント
nw::ut::Mutex       s_Mutex;
InputDeviceWork*    s_pInputDeviceWork;
void*               s_RegisteredBuffer = NULL;


/* ========================================================================
    staticな関数
   ======================================================================== */

void
ProcessEvent(const InputEventChunk& inputChunk)
{
    NW_ASSERT_NOT_NULL(s_pInputDeviceWork);
    NW_ASSERT_NOT_NULL(s_pInputDeviceWork->pInputEventHandler);

    InputEventHandler *const pInputEventHandler = s_pInputDeviceWork->pInputEventHandler;

    switch (inputChunk.eventType)
    {
    case EVENTTYPE_KEYDOWN:          { pInputEventHandler->KeyDownEvent(inputChunk.keyEventArg);            break; }
    case EVENTTYPE_KEYUP:            { pInputEventHandler->KeyUpEvent(inputChunk.keyEventArg);              break; }
    case EVENTTYPE_CHAR:             { pInputEventHandler->CharEvent(inputChunk.charEventArg);              break; }

    case EVENTTYPE_MOUSEMOVE:        { pInputEventHandler->MouseMoveEvent(inputChunk.mouseEventArg);        break; }
    case EVENTTYPE_MOUSEDOWN:        { pInputEventHandler->MouseDownEvent(inputChunk.mouseEventArg);        break; }
    case EVENTTYPE_MOUSEUP:          { pInputEventHandler->MouseUpEvent(inputChunk.mouseEventArg);          break; }
    case EVENTTYPE_MOUSEDOUBLECLICK: { pInputEventHandler->MouseDoubleClickEvent(inputChunk.mouseEventArg); break; }
    case EVENTTYPE_MOUSEWHEEL:       { pInputEventHandler->MouseWheelEvent(inputChunk.mouseEventArg);       break; }

    case EVENTTYPE_CAPTURESTART:     { pInputEventHandler->CaptureStartEvent(inputChunk.captureEventArg);   break; }
    case EVENTTYPE_CAPTURESTOP:      { pInputEventHandler->CaptureStopEvent(inputChunk.captureEventArg);    break; }
    }
}

#if 0 // unused function.
u16 NetToHost(u16 value)
{
#ifdef NW_PLATFORM_CAFE
    return value;
#endif

#if defined(NW_PLATFORM_WIN32) || defined(NW_USE_NINTENDO_SDK)
// TODO: NintendoSdk 対応後、このコメントを削除してください。
    return ntohs(value);
#endif
}
#endif

}   // namespace


/* ========================================================================
    外部関数
   ======================================================================== */

namespace nw
{
namespace mcs
{

//---------------------------------------------------------------------------
void
McsHID_Initialize()
{
    if (s_InitializedCount)
    {
        ut::ScopedLock<ut::Mutex> lockObj(s_Mutex);
        ++s_InitializedCount;
        return;
    }

    ++s_InitializedCount;
    s_Mutex.Initialize();
}

//---------------------------------------------------------------------------
void
McsHID_Finalize()
{
    {
        ut::ScopedLock<ut::Mutex> lockObj(s_Mutex);

        --s_InitializedCount;
        if (s_InitializedCount > 0)
        {
            return;
        }
    }

    s_Mutex.Finalize();
}

//---------------------------------------------------------------------------
bool
McsHID_IsInitialized()
{
    return (s_InitializedCount > 0) && s_pInputDeviceWork;
}

//---------------------------------------------------------------------------
void
McsHID_RegisterBuffer(void* buf, u32 bufSize)
{
    ut::ScopedLock<ut::Mutex> lockObj(s_Mutex);

    NW_ASSERT(s_pInputDeviceWork == 0);

    void *const newBuf = RoundUp(buf, 4);
    const u32 newBufSize = u32(RoundDown(static_cast<u8*>(buf) + bufSize - static_cast<u8*>(newBuf), 4));

    NW_ASSERT(MCSHID_RECEIVEBUFFER_MINSIZE <= newBufSize);

    s_pInputDeviceWork = static_cast<InputDeviceWork*>(newBuf);
    Mcs_RegisterBuffer(MCSHID_CHANNEL, s_pInputDeviceWork + 1, newBufSize - sizeof(*s_pInputDeviceWork));

    s_pInputDeviceWork->pInputEventHandler = 0;

    s_RegisteredBuffer = buf;
}

//---------------------------------------------------------------------------
void
McsHID_UnregisterBuffer()
{
    ut::ScopedLock<ut::Mutex> lockObj(s_Mutex);

    NW_ASSERT_NOT_NULL(s_pInputDeviceWork);

    Mcs_UnregisterBuffer(MCSHID_CHANNEL);
    s_pInputDeviceWork = 0;
    s_RegisteredBuffer = NULL;
}

//---------------------------------------------------------------------------
void*
McsHID_GetRegisteredBuffer()
{
    return s_RegisteredBuffer;
}

//---------------------------------------------------------------------------
void
McsHID_RegisterEventHandler(InputEventHandler* pInputEventHandler)
{
    ut::ScopedLock<ut::Mutex> lockObj(s_Mutex);

    NW_ASSERT_NOT_NULL(s_pInputDeviceWork);

    s_pInputDeviceWork->pInputEventHandler= pInputEventHandler;
}

//---------------------------------------------------------------------------
u32
McsHID_ProcessEvent()
{
    ut::ScopedLock<ut::Mutex> lockObj(s_Mutex);

    if (! s_pInputDeviceWork)
    {
        return MCSHID_ERROR_SUCCESS;
    }

    InputEventChunk inputEventChunk;

    u32 readableBytes = Mcs_GetReadableBytes(MCSHID_CHANNEL);
    for (; readableBytes >= sizeof(inputEventChunk); readableBytes -= sizeof(inputEventChunk))
    {
        if (u32 errorCode = Mcs_Read(MCSHID_CHANNEL, &inputEventChunk, sizeof(inputEventChunk)))
        {
            return errorCode;
        }

        if (s_pInputDeviceWork->pInputEventHandler)   // イベントハンドラが登録されていれば
        {
            ProcessEvent(inputEventChunk);
        }

    }

    return MCSHID_ERROR_SUCCESS;
}

//---------------------------------------------------------------------------
u32
McsHID_Polling()
{
    if (u32 errorCode = Mcs_Polling())
    {
        return errorCode;
    }

    return McsHID_ProcessEvent();
}

//---------------------------------------------------------------------------
u32
McsHID_StopCapture()
{
    ut::ScopedLock<ut::Mutex> lockObj(s_Mutex);

    if (! s_pInputDeviceWork)
    {
        return MCSHID_ERROR_NOTCONNECT;
    }

    InputRequestChunk inputRequestChunk = { REQUESTTYPE_STOPCAPTURE, 0 };

    u32 result = Mcs_Write(MCSHID_CHANNEL, &inputRequestChunk, sizeof(inputRequestChunk));

    if (result <= MCSHID_ERROR_LAST)
    {
        return result;
    }
    else
    {
        return MCSHID_ERROR_COMERROR;
    }
}

}   // namespace mcs
}   // namespace nw

// #if defined(NW_MCS_ENABLE)
#endif
