﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nw/types.h>
#include "./mcs_Hio2RingBuffer.cpp"

namespace nw
{
namespace mcs
{
namespace internal
{

//---------------------------------------------------------------------------
//! @brief        共有メモリからデータを読み込みます。
//!               ポインタの更新を行いません。
//!
//! @param[out]   pbAvailable
//!
//! @return       関数が成功すれば true、失敗すれば falseを返します。
//---------------------------------------------------------------------------
bool
Hio2RingBuffer::Peek(bool* pbAvailable)
{
    NW_ASSERT(pbAvailable != 0);

    if (! UpdateWritePoint_())
    {
        return false;
    }

    *pbAvailable = m_ReadPoint != m_WritePoint;

    if (! *pbAvailable)
    {
        return true;
    }

    m_ProcBytes = 0;
    m_TransBytes = (m_WritePoint + m_BufferSize - m_ReadPoint) % m_BufferSize;

    u32 currReadPoint = m_ReadPoint;
    u8 *const msgBuf = static_cast<u8*>(m_TempBuf) + HeaderSize;
    u32 tmpOfs = 0;

    for (u32 restBytes = m_TransBytes; restBytes > 0;)
    {
        const u32 readBytes = m_WritePoint < currReadPoint ? m_BufferSize - currReadPoint: restBytes;
        if (!ReadSection_(m_BaseOffset + HeaderSize + currReadPoint, msgBuf + tmpOfs, readBytes))
        {
            return false;
        }
        currReadPoint = (currReadPoint + readBytes) % m_BufferSize;
        tmpOfs += readBytes;
        restBytes -= readBytes;
    }

    NW_ASSERT(currReadPoint == m_WritePoint);

//    MCS_REPORT1("DEBUG: Read: Point:%08x\n" , m_ReadPoint);

    return true;
}

//---------------------------------------------------------------------------
//! @brief        次のパケットまで、ポインタを移動させます。
//!
//! @return       関数が成功すれば true、失敗すれば falseを返します。
//---------------------------------------------------------------------------
bool
Hio2RingBuffer::Skip()
{
    u8 *const msgStart = static_cast<u8*>(m_TempBuf) + HeaderSize + m_ProcBytes;
    const MessageHeader *const pMsgHeader = reinterpret_cast<MessageHeader*>(msgStart);

    DataPointer* dataPtr = static_cast<DataPointer*>(m_TempBuf);
    u32 skipBytes = RoundUp(sizeof(*pMsgHeader) + NetToHost(pMsgHeader->size), CopyAlignment);
    m_ReadPoint =  (m_ReadPoint + skipBytes) % m_BufferSize;
    DataPointer_Init(dataPtr, m_ReadPoint);

    if (!WriteSection_(m_BaseOffset + ReadInfoOffset, dataPtr, sizeof(*dataPtr)))
    {
        return false;
    }

    MCS_REPORT1("DEBUG: Read: Point:%08x\n" , m_ReadPoint);
    m_ProcBytes += skipBytes;

    return true;
}

//---------------------------------------------------------------------------
//! @brief       メッセージのデータを取得します。
//!              取得しても次のメッセージ位置へ移動しません。
//!              取得できるメッセージが無いときは NULL を返します。
//!
//! @param[out]  pChannel     メッセージのチャンネル値を格納する変数への
//!                           ポインタ。
//! @param[out]  pTotalSize   メッセージの総サイズを格納する変数へのポインタ。
//!
//! @return      取得できるメッセージが無いときは NULL、
//!               そうでない場合は、メッセージデータへのポインタを返します。
//---------------------------------------------------------------------------
void*
Hio2RingBuffer::PeekMessage(
    u32*    pChannel,
    u32*    pTotalSize
)
{
    if (m_ProcBytes >= m_TransBytes)
    {
        return NULL;
    }

    u8 *const msgStart = static_cast<u8*>(m_TempBuf) + HeaderSize + m_ProcBytes;
    const MessageHeader *const pMsgHeader = reinterpret_cast<MessageHeader*>(msgStart);

    *pChannel = NetToHost(pMsgHeader->channel);
    *pTotalSize = NetToHost(pMsgHeader->size);

    //m_ProcBytes += RoundUp(sizeof(*pMsgHeader) + *pTotalSize, CopyAlignment);
    return msgStart + sizeof(*pMsgHeader);
}

}   // namespace internal
}   // namespace mcs
}   // namespace nw
