﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nw/assert.h>
#include <nw/ut/ut_Inlines.h>
#include <nw/font/font_ResFont.h>

namespace nw {
namespace font {

namespace {

/*!--------------------------------------------------------------------------*
  @brief        オフセットで格納されているポインタを解決します。

  @param[in,out]  ptr   解決するポインタ。
  @param[in]      base  オフセットの基準となるアドレス。
 *---------------------------------------------------------------------------*/
template <typename T>
inline void
ResolveOffset(T*& ptr, void* base)
{
    ptr = reinterpret_cast<T*>(
            reinterpret_cast<u8*>(base) + *reinterpret_cast<ut::ResU32*>(&ptr)
    );
}

template <typename T>
inline void
DeResolveOffset(T*& ptr, void* base)
{
    ptr = reinterpret_cast<T*>(reinterpret_cast<u32>(ptr) - reinterpret_cast<u32>(base));
}

}   // namespace


void
ResFont::RevertResource(void* resource)
{
    ut::BinaryFileHeader* fileHeader =
        reinterpret_cast<ut::BinaryFileHeader*>(resource);
    NW_POINTER_ASSERT(fileHeader);
    NW_ALIGN_ASSERT(fileHeader, GlyphDataAlignment);

    // 初期化済みの場合すぐ返る。
    if (fileHeader->signature != BINFILE_SIG_FONT_RESOLEVED) { return; }

    //----------------------------------------
    // オフセット値を戻す
    ut::BinaryBlockHeader* blockHeader;
    FontInformation* info = NULL;
    int nBlocks = 0;

    blockHeader = reinterpret_cast<ut::BinaryBlockHeader*>(
        reinterpret_cast<u8*>(fileHeader) + fileHeader->headerSize
        );

    while (nBlocks < fileHeader->dataBlocks)
    {
        NW_POINTER_ASSERT(blockHeader);
        switch (blockHeader->kind)
        {
        //--------------------------------------------------
        // INFO ブロック
        case BINBLOCK_SIG_FINF:
            // INFOブロックは1つでなければならない
            NW_ASSERT(info == NULL);
            {
                info = reinterpret_cast<FontInformation*>(
                    reinterpret_cast<u8*>(blockHeader) + sizeof(*blockHeader)
                );

                NW_ASSERT(info->fontType == FONT_TYPE_TEXTURE || info->fontType == FONT_TYPE_PACKED_TEXTURE);
                NW_ASSERT(info->alterCharIndex != GLYPH_INDEX_NOT_FOUND);

                // pGlyph は必須
                NW_NULL_ASSERT(info->pGlyph);
                DeResolveOffset(info->pGlyph, fileHeader);

                // pWidth と pMap はなくても良い
                if (info->pWidth != NULL)
                {
                    DeResolveOffset(info->pWidth, fileHeader);
                }
                if (info->pMap != NULL)
                {
                    DeResolveOffset(info->pMap, fileHeader);
                }
            }
            break;

        //--------------------------------------------------
        // TGLP ブロック
        case BINBLOCK_SIG_TGLP:
            // TGLP ブロックも1つでなければならないが複数存在しても致命的ではない
            {
                FontTextureGlyph* glyph = reinterpret_cast<FontTextureGlyph*>(
                    reinterpret_cast<u8*>(blockHeader) + sizeof(*blockHeader)
                );

                NW_NULL_ASSERT(glyph->sheetImage);
                DeResolveOffset(glyph->sheetImage, fileHeader);
            }
            break;

        //--------------------------------------------------
        // CWDHブロック
        case BINBLOCK_SIG_CWDH:
            {
                FontWidth* width = reinterpret_cast<FontWidth*>(
                    reinterpret_cast<u8*>(blockHeader) + sizeof(*blockHeader)
                );

                NW_ASSERT( width->indexBegin <= width->indexEnd );

                if (width->pNext != NULL)
                {
                    DeResolveOffset(width->pNext, fileHeader);
                }
            }
            break;

        //--------------------------------------------------
        // CMAPブロック
        case BINBLOCK_SIG_CMAP:
            {
                FontCodeMap* map = reinterpret_cast<FontCodeMap*>(
                    reinterpret_cast<u8*>(blockHeader) + sizeof(*blockHeader)
                );

                NW_ASSERT( map->ccodeBegin <= map->ccodeEnd );
                NW_ASSERT( (map->mappingMethod == FONT_MAPMETHOD_DIRECT)
                                || (map->mappingMethod == FONT_MAPMETHOD_TABLE)
                                || (map->mappingMethod == FONT_MAPMETHOD_SCAN) );

                if (map->pNext != NULL)
                {
                    DeResolveOffset(map->pNext, fileHeader);
                }
            }
            break;

        //--------------------------------------------------
        // KRNGブロック
        case BINBLOCK_SIG_KRNG:
            {
                // ここでは何も行わない
            }
            break;

        //--------------------------------------------------
        // GLGR ブロック
        case BINBLOCK_SIG_GLGR:
            {
                // ResFont では GLGR ブロックは参照しないので何もする必要はない
            }
            break;

        //--------------------------------------------------
        // unknown
        default:
            NW_ASSERTMSG(false, "The font has unknown block('%c%c%c%c').",
                (blockHeader->kind >> 24) & 0xFF,
                (blockHeader->kind >> 16) & 0xFF,
                (blockHeader->kind >>  8) & 0xFF,
                (blockHeader->kind >>  0) & 0xFF
                );
            return;
        }

        blockHeader = reinterpret_cast<ut::BinaryBlockHeader*>(
            reinterpret_cast<u8*>(blockHeader) + blockHeader->size
        );
        nBlocks++;
    }

    //---- シグネチャを変更して元に戻す
    fileHeader->signature = BINFILE_SIG_FONT;
}

/* ------------------------------------------------------------------------
        コンストラクタ/デストラクタ
   ------------------------------------------------------------------------ */

ResFont::ResFont()
 : ResFontBase()
{
}

ResFont::~ResFont()
{
    if (!IsManaging(NULL))
    {
        RemoveResource();
    }
}



/* ------------------------------------------------------------------------
        構築
   ------------------------------------------------------------------------ */

bool
ResFont::SetResource(void* bfnt)
{
    NW_ASSERT_VALID_POINTER(bfnt);
    NW_ASSERT_ALIGN(bfnt, GlyphDataAlignment);

    FontInformation* pFontInfo = NULL;
    ut::BinaryFileHeader* fileHeader =
        reinterpret_cast<ut::BinaryFileHeader*>(bfnt);

    //---- 既にリソースがセットされていないか
    if (! IsManaging(NULL))
    {
        NW_LOG("Font resource already atached.");
        return false;
    }

    //---- オフセット解決済みのフォントか？
    if (fileHeader->signature == BINFILE_SIG_FONT_RESOLEVED)
    {
        // FontInfomation を探す
        pFontInfo = static_cast<nw::font::FontInformation *>(FindBlock(fileHeader, BINBLOCK_SIG_FINF));
    }
    else
    {
        //---- 正しいファイルヘッダを持っているか

        // サポートしてるバージョンかチェック
        if (! IsValidBinaryFile(fileHeader, BINFILE_SIG_FONT, FONT_FILE_VERSION, 2))
        {
            // サポートしていない形式
            NW_WARNING(false, "Invalid font resource.");
            return false;
        }

        // 再構築する
        pFontInfo = Rebuild(fileHeader);
    }

    if (pFontInfo == NULL)
    {
        return false;
    }

    SetResourceBuffer(bfnt, pFontInfo, static_cast<nw::font::FontKerningTable *>(FindBlock(fileHeader, BINBLOCK_SIG_KRNG)));

    GenTextureNames();

    return true;
}

void* ResFont::FindBlock(ut::BinaryFileHeader* fileHeader, ut::SigWord sigword)
{
    ut::BinaryBlockHeader* blockHeader;
    int nBlocks = 0;

    blockHeader = reinterpret_cast<ut::BinaryBlockHeader*>(
        reinterpret_cast<u8*>(fileHeader) + fileHeader->headerSize
    );

    while (nBlocks < fileHeader->dataBlocks)
    {
        NW_ASSERT_VALID_POINTER( blockHeader );
        if (blockHeader->kind == sigword)
        {
            return reinterpret_cast<u8*>(blockHeader) + sizeof(*blockHeader);
        }

        blockHeader = reinterpret_cast<ut::BinaryBlockHeader*>(
            reinterpret_cast<u8*>(blockHeader) + blockHeader->size
        );
        nBlocks++;
    }

    return NULL;
}

void*
ResFont::RemoveResource()
{
    if (IsManaging(NULL))
    {
        NW_LOG("ResFont::RemoveResource(): Res font is not loaded.\n");
        return 0;
    }

    return RemoveResourceBuffer();
}

FontInformation*
ResFont::Rebuild(ut::BinaryFileHeader* fileHeader)
{
    NW_ASSERT_VALID_POINTER(fileHeader);
    NW_ASSERT_ALIGN(fileHeader, GlyphDataAlignment);

    ut::BinaryBlockHeader* blockHeader;
    FontInformation* info = NULL;
    int nBlocks = 0;

    blockHeader = reinterpret_cast<ut::BinaryBlockHeader*>(
        reinterpret_cast<u8*>(fileHeader) + fileHeader->headerSize
    );

    while (nBlocks < fileHeader->dataBlocks)
    {
        NW_ASSERT_VALID_POINTER(blockHeader);
        switch (blockHeader->kind)
        {
        //--------------------------------------------------
        // INFO ブロック
        case BINBLOCK_SIG_FINF:
            // INFOブロックは1つでなければならない
            NW_ASSERT(info == NULL);
            {
                info = reinterpret_cast<FontInformation*>(
                    reinterpret_cast<u8*>(blockHeader) + sizeof(*blockHeader)
                );

                NW_ASSERT(info->fontType == FONT_TYPE_TEXTURE || info->fontType == FONT_TYPE_PACKED_TEXTURE);
                NW_ASSERT(info->alterCharIndex != GLYPH_INDEX_NOT_FOUND);

                // pGlyph は必須
                NW_ASSERT_NOT_NULL(info->pGlyph);
                ResolveOffset(info->pGlyph, fileHeader);
                NW_ASSERT_VALID_POINTER(info->pGlyph);

                // pWidth と pMap はなくても良い
                if (info->pWidth != NULL)
                {
                    ResolveOffset(info->pWidth, fileHeader);
                    NW_ASSERT_VALID_POINTER(info->pWidth);
                }
                if (info->pMap != NULL)
                {
                    ResolveOffset(info->pMap, fileHeader);
                    NW_ASSERT_VALID_POINTER(info->pMap);
                }
            }
            break;

        //--------------------------------------------------
        // TGLP ブロック
        case BINBLOCK_SIG_TGLP:
            // TGLP ブロックも1つでなければならないが複数存在しても致命的ではない
            {
                FontTextureGlyph* glyph = reinterpret_cast<FontTextureGlyph*>(
                    reinterpret_cast<u8*>(blockHeader) + sizeof(*blockHeader)
                );

                NW_ASSERT_NOT_NULL(glyph->sheetImage);
                ResolveOffset(glyph->sheetImage, fileHeader);
                NW_ASSERT_VALID_POINTER(glyph->sheetImage);

                //---- 32x32 の I4 が最小、1024x1024 の RGBA8 が最大
                NW_FONT_MIN_ASSERT       ( glyph->cellWidth,     1 );
                NW_FONT_MIN_ASSERT       ( glyph->cellHeight,    1 );
                NW_FONT_MINMAX_ASSERT    ( glyph->sheetSize,     32 * 32 / 2,    1024 * 1024 * 4 );
                NW_FONT_MIN_ASSERT       ( glyph->sheetNum,      1 );
                NW_FONT_MIN_ASSERT       ( glyph->sheetRow,      1 );
                NW_FONT_MIN_ASSERT       ( glyph->sheetLine,     1 );
                NW_FONT_MINMAX_ASSERT    ( glyph->sheetWidth,    32,             1024 );
                NW_FONT_MINMAX_ASSERT    ( glyph->sheetHeight,   32,             1024 );
            }
            break;

        //--------------------------------------------------
        // CWDHブロック
        case BINBLOCK_SIG_CWDH:
            {
                FontWidth* width = reinterpret_cast<FontWidth*>(
                    reinterpret_cast<u8*>(blockHeader) + sizeof(*blockHeader)
                );

                NW_ASSERT( width->indexBegin <= width->indexEnd );

                if (width->pNext != NULL)
                {
                    ResolveOffset(width->pNext, fileHeader);
                    NW_ASSERT_VALID_POINTER( width->pNext );
                }
            }
            break;

        //--------------------------------------------------
        // CMAPブロック
        case BINBLOCK_SIG_CMAP:
            {
                FontCodeMap* map = reinterpret_cast<FontCodeMap*>(
                    reinterpret_cast<u8*>(blockHeader) + sizeof(*blockHeader)
                );

                NW_ASSERT( map->ccodeBegin <= map->ccodeEnd );
                NW_ASSERT( (map->mappingMethod == FONT_MAPMETHOD_DIRECT)
                                || (map->mappingMethod == FONT_MAPMETHOD_TABLE)
                                || (map->mappingMethod == FONT_MAPMETHOD_SCAN) );

                if (map->pNext != NULL)
                {
                    ResolveOffset(map->pNext, fileHeader);
                    NW_ASSERT_VALID_POINTER(map->pNext);
                }
            }
            break;

        //--------------------------------------------------
        // KRNGブロック
        case BINBLOCK_SIG_KRNG:
            {
                // ここでは何も行わない
            }
            break;

        //--------------------------------------------------
        // GLGR ブロック
        case BINBLOCK_SIG_GLGR:
            {
                // ResFont では GLGR ブロックは参照しないので何もする必要はない
            }
            break;

        //--------------------------------------------------
        // unknown
        default:
            NW_ASSERTMSG(false, "The font has unknown block('%c%c%c%c').",
                (blockHeader->kind >> 24) & 0xFF,
                (blockHeader->kind >> 16) & 0xFF,
                (blockHeader->kind >>  8) & 0xFF,
                (blockHeader->kind >>  0) & 0xFF
                );
            return NULL;
        }

        blockHeader = reinterpret_cast<ut::BinaryBlockHeader*>(
            reinterpret_cast<u8*>(blockHeader) + blockHeader->size
        );
        nBlocks++;
    }


    //---- シグネチャを変更してオフセット解決済みにマークする
    fileHeader->signature = BINFILE_SIG_FONT_RESOLEVED;

    return info;
}

}   // namespace font
}   // namespace nw
