﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nw/gfnd.h>
#include <nw/dev.h>
#include <nw/dw/system/dw_IUIRenderContext.h>
#include <nw/dw/system/dw_NwUIRenderer.h>
#include <nw/dw/system/dw_NwTypeUtility.h>

namespace nw {
namespace internal {
namespace dw {

NwUIRenderer::NwUIRenderer() :
m_pPrimitiveRenderer(NULL),
m_pTextRenderer(NULL)
{
}

nw::dev::PrimitiveRenderer* NwUIRenderer::GetPrimitiveRenderer() const
{
    return m_pPrimitiveRenderer;
}

void NwUIRenderer::SetPrimitiveRenderer(nw::dev::PrimitiveRenderer* value)
{
    m_pPrimitiveRenderer = value;
}

NwTextRenderer* NwUIRenderer::GetTextWriter() const
{
    return m_pTextRenderer;
}

void NwUIRenderer::SetTextRenderer(NwTextRenderer* value)
{
    m_pTextRenderer = value;
}

void NwUIRenderer::BeginDraw()
{
    SetState(IDLE);
}

void NwUIRenderer::EndDraw()
{
    SetState(TEXT_WRITING);
    m_pTextRenderer->Flush();
    SetState(IDLE);
}

void NwUIRenderer::ClearBuffer()
{
    if(m_pTextRenderer != NULL)
    {
        m_pTextRenderer->Clear();
    }
}

void NwUIRenderer::DrawLineInternal(const IUIRenderContext* pRenderContext, const DrawLineArgs& arg)
{
    NW_NULL_ASSERT(m_pPrimitiveRenderer);

    DrawLineArgs viewportArg = arg;

    if(pRenderContext != NULL)
    {
        const nw::math::Vector2& origin = pRenderContext->GetRenderOffset();

        viewportArg.from.x += origin.x;
        viewportArg.from.y += origin.y;
        viewportArg.to.x   += origin.x;
        viewportArg.to.y   += origin.y;
    }

    m_pPrimitiveRenderer->DrawLine(
        NwTypeUtility::ToVector3(viewportArg.from),
        NwTypeUtility::ToVector3(viewportArg.to),
        viewportArg.color0,
        viewportArg.color1,
        viewportArg.lineWidth);
}

void NwUIRenderer::DrawLineInternal(const IUIRenderContext* pRenderContext, const DrawMultiLineArgs& arg)
{
    NW_NULL_ASSERT(m_pPrimitiveRenderer);

    // 現在のモデル行列を取得
    nw::math::MTX34 lastModelMtx = m_pPrimitiveRenderer->GetModelMatrix();
    nw::math::MTX34 modelMtx = lastModelMtx;

    // モデル行列を生成
    if(pRenderContext != NULL)
    {
        const nw::math::Vector2& origin = pRenderContext->GetRenderOffset();

        modelMtx.SetIdentity();
        modelMtx.SetTranslate(nw::math::VEC3(origin.x, origin.y, 0.0f));
    }

    // モデル行列を設定
    m_pPrimitiveRenderer->SetModelMatrix(modelMtx);
    m_pPrimitiveRenderer->DrawLine(
        arg.color,
        arg.buffer,
        arg.lineWidth
    );

    // モデル行列を元に戻す
    m_pPrimitiveRenderer->SetModelMatrix(lastModelMtx);
}

void NwUIRenderer::DrawRectangleInternal(const IUIRenderContext* pRenderContext, const DrawRectangleArgs& arg)
{
    if(arg.IsSizeZero())
    {
        return;
    }

    DrawRectangleArgs viewportArg = arg;

    if(pRenderContext != NULL)
    {
        const nw::math::Vector2& origin = pRenderContext->GetRenderOffset();

        viewportArg.topLeft.x += origin.x;
        viewportArg.topLeft.y += origin.y;
    }

    NW_NULL_ASSERT(m_pPrimitiveRenderer);
    m_pPrimitiveRenderer->DrawBox(NwTypeUtility::ToQuadArg(viewportArg));
}

void NwUIRenderer::FillRectangleInternal(const IUIRenderContext* pRenderContext, const DrawRectangleArgs& arg)
{
    if(arg.IsSizeZero())
    {
        return;
    }

    DrawRectangleArgs viewportArg = arg;

    if(pRenderContext != NULL)
    {
        const nw::math::Vector2& origin = pRenderContext->GetRenderOffset();

        viewportArg.topLeft.x += origin.x;
        viewportArg.topLeft.y += origin.y;
    }

    NW_NULL_ASSERT(m_pPrimitiveRenderer);

    if(viewportArg.texture == NULL)
    {
        m_pPrimitiveRenderer->DrawQuad(NwTypeUtility::ToQuadArg(viewportArg));
    }
    else
    {
        m_pPrimitiveRenderer->DrawQuad(*viewportArg.texture, NwTypeUtility::ToQuadArg(viewportArg));
    }
}

void NwUIRenderer::DrawTextureInternal(const IUIRenderContext* pRenderContext, const DrawRectangleArgs& arg)
{
    if(arg.IsSizeZero())
    {
        return;
    }

    if(arg.texture == NULL)
    {
        return;
    }

    DrawRectangleArgs viewportArg = arg;

    if(pRenderContext != NULL)
    {
        const nw::math::Vector2& origin = pRenderContext->GetRenderOffset();

        viewportArg.topLeft.x += origin.x;
        viewportArg.topLeft.y += origin.y;
    }

    NW_NULL_ASSERT(m_pPrimitiveRenderer);
    m_pPrimitiveRenderer->DrawQuad(*viewportArg.texture, NwTypeUtility::ToQuadArg(viewportArg));
}

void NwUIRenderer::DrawTextInternal(const IUIRenderContext* pRenderContext, const DrawTextArgs& arg, const char* pText)
{
    DrawTextArgs viewportArg = arg;

    if(pRenderContext != NULL)
    {
        const nw::math::Vector2& origin = pRenderContext->GetRenderOffset();

        viewportArg.topLeft.x += origin.x;
        viewportArg.topLeft.y += origin.y;
    }

    NW_NULL_ASSERT(m_pTextRenderer);
    m_pTextRenderer->DrawText(viewportArg, pText);
}

void NwUIRenderer::DrawTextFormatInternal(const IUIRenderContext* pRenderContext, const DrawTextArgs& arg, const char* pFormat, va_list valist)
{
    DrawTextArgs viewportArg = arg;

    if(pRenderContext != NULL)
    {
        const nw::math::Vector2& origin = pRenderContext->GetRenderOffset();

        viewportArg.topLeft.x += origin.x;
        viewportArg.topLeft.y += origin.y;
    }

    NW_NULL_ASSERT(m_pTextRenderer);
    m_pTextRenderer->VDrawTextFormat(arg, pFormat, valist);
}

nw::math::Vector2 NwUIRenderer::MeasureTextInternal(const DrawTextArgs& arg, const char* pText)
{
    NW_NULL_ASSERT(m_pTextRenderer);
    return m_pTextRenderer->MeasureText(arg, pText);
}

nw::math::Vector2 NwUIRenderer::MeasureTextFormatInternal(const DrawTextArgs& arg, const char* pFormat, va_list valist)
{
    NW_NULL_ASSERT(m_pTextRenderer);

    return m_pTextRenderer->VMeasureTextFormat(arg, pFormat, valist);
}

UIRenderer::State NwUIRenderer::ChangeState(State state)
{
    SetIdle();

    State result = IDLE;

    switch( state )
    {
    case IDLE:
        break;

    case PRIMITIVE_RENDERING:
        if(m_pPrimitiveRenderer == NULL)
        {
            break;
        }

        {
            nw::gfnd::Graphics* graphics = nw::gfnd::Graphics::GetInstance();
            graphics->SetBlendEnable(true);
            graphics->SetBlendFactor(nw::gfnd::Graphics::BLEND_FACTOR_SRC_ALPHA, nw::gfnd::Graphics::BLEND_FACTOR_INV_SRC_ALPHA);
            graphics->SetBlendEquation(nw::gfnd::Graphics::BLEND_EQUATION_ADD);
            graphics->SetDepthTestEnable(true);
            graphics->SetDepthWriteEnable(true);
            graphics->SetCullingMode(nw::gfnd::Graphics::CULLING_MODE_NONE);
        }

        m_pPrimitiveRenderer->SetProjectionMtx(&GetProjectionMatrix());
        m_pPrimitiveRenderer->SetViewMtx(&GetViewMatrix());
        m_pPrimitiveRenderer->SetModelMatrix( nw::math::MTX34::Identity() );

        m_pPrimitiveRenderer->Begin();

        result = PRIMITIVE_RENDERING;
        break;

    case TEXT_WRITING:
        {
            if(m_pTextRenderer == NULL)
            {
                break;
            }

            {
                nw::gfnd::Graphics* graphics = nw::gfnd::Graphics::GetInstance();
                graphics->SetBlendEnable(true);
                graphics->SetBlendFactor(nw::gfnd::Graphics::BLEND_FACTOR_SRC_ALPHA, nw::gfnd::Graphics::BLEND_FACTOR_INV_SRC_ALPHA);
                graphics->SetBlendEquation(nw::gfnd::Graphics::BLEND_EQUATION_ADD);
                graphics->SetDepthTestEnable(false);
                graphics->SetDepthWriteEnable(false);
                graphics->SetCullingMode(nw::gfnd::Graphics::CULLING_MODE_NONE);
            }

            m_pTextRenderer->SetProjectionMatrix(GetProjectionMatrix());
            m_pTextRenderer->SetViewMatrix(GetViewMatrix());

            result = TEXT_WRITING;
        }
        break;

    default:
        NW_ASSERT(false);
        return result;
    }

    return result;
}

void NwUIRenderer::SetIdle()
{
    switch(GetState())
    {
    case IDLE:
        return;

    case PRIMITIVE_RENDERING:
        m_pPrimitiveRenderer->End();
        break;

    case TEXT_WRITING:
        break;
    }
}

} // dw
} // internal
} // nw
