﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nw/dw/control/dw_ListBox.h>
#include <nw/dw/control/dw_UIElementUtility.h>
#include <nw/dw/control/dw_HighestLevelFocusableElementFinder.h>

#include <nw/dw/system/dw_NwTypeUtility.h>

namespace nw {
namespace internal {
namespace dw {

ListBox::ListBoxItemPlaceHolder::ListBoxItemPlaceHolder() :
m_isSelected(false),
m_isSelectable(true),
m_pOwner(NULL)
{
    // TODO : DefaultStyle が実装されたら削除する
    SetMargin(Thickness(0.f));
    SetPadding(Thickness(0.f));
    SetIsFocusable(true);

    SetContents(m_Contents);
}

bool ListBox::ListBoxItemPlaceHolder::GetIsSelected() const
{
    return m_isSelected;
}

void ListBox::ListBoxItemPlaceHolder::SetIsSelected(bool value)
{
    if(m_isSelected != value)
    {
        if(value && !m_isSelectable)
        {
            return;
        }

        m_isSelected = value;
        CallbackSelectionChanged();
    }
}

bool ListBox::ListBoxItemPlaceHolder::GetIsSelectable() const
{
    return m_isSelectable;
}

void ListBox::ListBoxItemPlaceHolder::SetIsSelectable(bool value)
{
    if(m_isSelectable != value)
    {
        m_isSelectable = value;
        SetIsFocusable(value);

        if(!m_isSelectable)
        {
            SetIsSelected(false);
        }
    }
}

UIElement* ListBox::ListBoxItemPlaceHolder::GetContent() const
{
    if(m_Contents.GetCount() == 0)
    {
        return NULL;
    }
    else
    {
        NW_ASSERT(m_Contents.GetCount() == 1);
        return m_Contents[0];
    }
}

void ListBox::ListBoxItemPlaceHolder::SetContent(UIElement* pContent)
{
    if(m_Contents.GetCount() == 0)
    {
        m_Contents.AddItem(pContent);
    }
    else
    {
        NW_ASSERT(m_Contents.GetCount() == 1);
        m_Contents.SetItem(0, pContent);
    }
}

bool ListBox::ListBoxItemPlaceHolder::OnUpdatePointerInput(const nw::internal::dw::Inputs& inputs)
{
    if(inputs.GetMouse() == NULL)
    {
        return false;
    }

    const nw::dev::Mouse& mouse = *inputs.GetMouse();

    if ( ! mouse.IsDoubleClick() && mouse.IsTrig(nw::dev::Mouse::MASK_LBUTTON) )
    {
        SetIsSelected(true);
        SetFocus();
        return true;
    }

    return false;
}

UIElementRenderArgs& ListBox::ListBoxItemPlaceHolder::OnPrepareRender(const UIElementTreeContext& context)
{
    UIElementRenderArgs& result = Base::OnPrepareRender(context);

    if(m_isSelected)
    {
        result.SetBackgroundColor(
            GetIsContainsFocus() ? NwTypeUtility::SRGBToLinear(nw::ut::Color4f::X_CORNFLOWER_BLUE()) : NwTypeUtility::SRGBToLinear(nw::ut::Color4f::X_DIM_GRAY()));
    }

    if(GetIsContainsPointerOver())
    {
        result.
            SetIsBordered(true).
            SetBorderColor(NwTypeUtility::SRGBToLinear(nw::ut::Color4f::X_LIGHT_BLUE()));
    }

    return result;
}

void ListBox::ListBoxItemPlaceHolder::SetOwner(ListBox* pOwner)
{
    NW_ASSERT(pOwner != NULL);
    m_pOwner = pOwner;
}

void ListBox::ListBoxItemPlaceHolder::CallbackSelectionChanged()
{
    NW_ASSERTMSG(m_pOwner != NULL, "ListBoxItemPlaceHolder is not initialized.");
    m_pOwner->CallbackSelectionChanged(*this);
}

//---------------------------------------------------------------------------

ListBox::ListBox() :
m_FocusedItemIndex(INVALID_INDEX)
{
    SetIsFocusable(false);
    SetIsBordered(false);
}

s32 ListBox::GetItemCount() const
{
    return GetItems().GetCount();
}

UIElement* ListBox::GetItem(s32 index) const
{
    NW_ASSERT(0 <= index && index < GetItemCount());
    return GetItemPlaceHolder(index).GetContent();
}

ListBox& ListBox::AddItem(UIElement* pItem, bool isSelectable /*= true*/)
{
    NW_ASSERT(pItem != NULL);

    u32 itemCount = GetItemCount();

    ListBoxItemPlaceHolder& itemPlaceHolder = GetItemPlaceHolder(itemCount);
    itemPlaceHolder.SetOwner(this);
    itemPlaceHolder.SetContent(pItem);
    itemPlaceHolder.SetIsSelectable(isSelectable);

    GetItems().AddItem(&itemPlaceHolder);

    return *this;
}

ListBox& ListBox::InsertItem(s32 index, UIElement* pItem, bool isSelectable /*= true*/)
{
    (void)index;
    (void)pItem;
    (void)isSelectable;

    // TODO : 未実装です。
    //        ついでに UIElementList のアイテム変更通知を受けて ItemPlaceHolders の内容を同期するようにしたい。
    NW_ASSERT(false);
    return *this;
}

ListBox& ListBox::SetItem(s32 index, UIElement* pItem)
{
    NW_ASSERT(0 <= index && index < GetItemCount());
    NW_ASSERT(pItem != NULL);

    ListBoxItemPlaceHolder& itemPlaceHolder = GetItemPlaceHolder(index);
    itemPlaceHolder.SetContent(pItem);

    GetItems().SetItem(index, &itemPlaceHolder);

    return *this;
}

void ListBox::ClearItems()
{
    for( s32 i = 0; i < GetItemCount(); ++i )
    {
        ListBoxItemPlaceHolder& itemPlaceHolder = GetItemPlaceHolder(i);
        itemPlaceHolder.SetIsSelected(false);
    }

    GetItems().ClearItems();
    m_FocusedItemIndex = INVALID_INDEX;
}

s32 ListBox::IndexOf(const UIElement* pItem) const
{
    NW_ASSERT(pItem != NULL);

    for(s32 i=0; i<GetItemCount(); ++i)
    {
        if(GetItem(i) == pItem ||
            &GetItemPlaceHolder(i) == pItem)
        {
            return i;
        }
    }

    return INVALID_INDEX;
}

ListBox::SelectionChangedEvent& ListBox::GetSelectionChangedEvent()
{
    return m_SelectionChangedCallbacks;
}

bool ListBox::OnUpdateFocusedInput(const nw::internal::dw::Inputs& inputs)
{
    if(inputs.GetPad() == NULL)
    {
        return false;
    }

    const nw::dev::Pad& pad = *inputs.GetPad();

    if(pad.IsTrig(nw::dev::Pad::MASK_DOWN) ||
        pad.IsRepeatAccel(nw::dev::Pad::MASK_DOWN))
    {
        u32 offset = 1;
        if(pad.IsHold(nw::dev::Pad::MASK_L))
        {
            offset *= 10;
        }

        MoveSelection(offset);
        return true;
    }

    if(pad.IsTrig(nw::dev::Pad::MASK_UP) ||
        pad.IsRepeatAccel(nw::dev::Pad::MASK_UP))
    {
        u32 offset = static_cast<u32>( -1 );
        if(pad.IsHold(nw::dev::Pad::MASK_L))
        {
            offset *= 10;
        }

        MoveSelection(offset);
        return true;
    }

    return false;
}

UIElement* ListBox::GetFocusedItem() const
{
    if(m_FocusedItemIndex == INVALID_INDEX)
    {
        return NULL;
    }

    return GetItem(m_FocusedItemIndex);
}

s32 ListBox::GetFocusedItemIndex() const
{
    return m_FocusedItemIndex;
}

void ListBox::SelectItem(s32 index)
{
    NW_ASSERT(INVALID_INDEX == index ||
        (0 <= index && index < GetItemCount()));

    if(m_FocusedItemIndex == index)
    {
        return;
    }

    if(m_FocusedItemIndex != INVALID_INDEX)
    {
        GetItemPlaceHolder(m_FocusedItemIndex).SetIsSelected(false);
    }

    if(index == INVALID_INDEX)
    {
        return;
    }

    GetItemPlaceHolder(index).SetIsSelected(true);
    NW_ASSERT(m_FocusedItemIndex == index);
}

void ListBox::MoveSelection(s32 offset)
{
    if(GetItemCount() == 0)
    {
        return;
    }

    s32 newIndex = m_FocusedItemIndex + offset;

    if(newIndex < 0)
    {
        s32 firstSelectableIndex = FindSelectableItem(0, true);
        s32 lastSelectableIndex = FindSelectableItem(GetItemCount() - 1, false);

        // 先頭のアイテムを選択中は末尾に移動します。
        // それ以外の場合は、先頭のアイテムに移動します。
        newIndex = FindSelectableItem(
            m_FocusedItemIndex == firstSelectableIndex ? lastSelectableIndex : firstSelectableIndex,
            false);
    }
    else if(GetItemCount() <= newIndex)
    {
        s32 firstSelectableIndex = FindSelectableItem(0, true);
        s32 lastSelectableIndex = FindSelectableItem(GetItemCount() - 1, false);

        // 末尾のアイテムを選択中は先頭に移動します。
        // それ以外の場合は、末尾のアイテムに移動します。
        newIndex = FindSelectableItem(
            m_FocusedItemIndex == lastSelectableIndex ? firstSelectableIndex : lastSelectableIndex,
            true);
    }
    else
    {
        newIndex = FindSelectableItem(newIndex, offset >= 0);
    }

    if(newIndex == m_FocusedItemIndex)
    {
        return;
    }

    ListBoxItemPlaceHolder& itemPlaceHolder = GetItemPlaceHolder(newIndex);

    HighestLevelFocusableElementFinder finder;
    UIElementUtility::ForAllElements(itemPlaceHolder, finder);
    UIElement* pFocusableElement = finder.GetResult();

    if(pFocusableElement != NULL)
    {
        pFocusableElement->SetFocus();
    }
    else
    {
        itemPlaceHolder.SetFocus();
    }

    SelectItem(newIndex);
    itemPlaceHolder.EnsureVisible();
}

s32 ListBox::FindSelectableItem(s32 index, bool isAscending)
{
    s32 result = index;
    s32 offset = isAscending ? 1 : -1;

    while((void)0, 1)
    {
        if(GetItemPlaceHolder(result).GetIsSelectable())
        {
            break;
        }

        result += offset;

        if(result < 0)
        {
            result = GetItemCount() - 1;
        }

        if(GetItemCount() <= result)
        {
            result = 0;
        }

        if(result == index)
        {
            break;
        }
    }

    return result;
}

void ListBox::CallbackSelectionChanged(ListBoxItemPlaceHolder& target)
{
    if(target.GetIsSelected())
    {
        s32 oldIndex = m_FocusedItemIndex;
        m_FocusedItemIndex = UIElementUtility::IndexOf(GetItems(), target);

        if(oldIndex != m_FocusedItemIndex)
        {
            // HACK : 単一選択処理
            // 古い選択アイテムを非選択状態にします。
            if(oldIndex != INVALID_INDEX)
            {
                GetItemPlaceHolder(oldIndex).SetIsSelected(false);
            }
        }
    }

    m_SelectionChangedCallbacks.Invoke();
}

} // dw
} // internal
} // nw
