﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nw/dev/gl/dev_PrimitiveRendererGL.h>

namespace nw
{
namespace dev
{

#define __NW_TO_TEXT( ... )   #__VA_ARGS__
    static const char VS_SOURCE[] =
    {
#include "../shaders/dev_PrimitiveRenderer_vsh_win32.hpp"
    };

    static const char PS_SOURCE[] =
    {
#include "../shaders/dev_PrimitiveRenderer_psh_win32.hpp"
    };
#undef __NW_TO_TEXT

const s32 MAX_SPHERE_X = 16;
const s32 MAX_SPHERE_Y = 8;
const s32 MAX_DISK_DIVISION = 32;
const s32 MAX_SPHERE_VTX_NUM = MAX_SPHERE_X * MAX_SPHERE_Y + 2;
const s32 MAX_SPHERE_IDX_NUM = 3 * ( MAX_SPHERE_X + MAX_SPHERE_X * 2 * ( MAX_SPHERE_Y - 1 ) + MAX_SPHERE_X );
const s32 MAX_DISK_VTX_NUM = MAX_DISK_DIVISION + 1;
const s32 MAX_DISK_IDX_NUM = MAX_DISK_DIVISION * 3;

//--------------------------------------------------------------------------
PrimitiveRendererGL::PrimitiveRendererGL( nw::ut::IAllocator* /*allocator*/ )
      : m_VertexShader( 0 ),
        m_FragmentShader( 0 ),
        m_ShaderProgram( 0 ),
        m_Sampler( 0 ),
        m_ViewMtx( NULL ),
        m_ProjMtx( NULL ),
        m_ParamWVP( 0 ),
        m_ParamUser( 0 ),
        m_ParamRate( 0 ),
        m_ParamTex( 0 ),
        m_ParamColor0( 0 ),
        m_ParamColor1( 0 ),
        m_ParamUVSrc( 0 ),
        m_ParamUVSize( 0 ),
        m_AttrVertex( 0 ),
        m_AttrTexCoord0( 0 ),
        m_AttrColorRate( 0 )
{
}

//--------------------------------------------------------------------------
PrimitiveRendererGL::~PrimitiveRendererGL()
{
}

//--------------------------------------------------------------------------
void
PrimitiveRendererGL::InitializeFromBinaryImpl(
    nw::ut::IAllocator* allocator,
    const void* /*binaryData*/,
    u32 /*binarySize*/
)
{
    // VertexShader
    {
        NW_ASSERT( m_VertexShader == 0 );
        m_VertexShader = glCreateShader( GL_VERTEX_SHADER );
        NW_GL_ASSERT();

        bool result = CreateShader( m_VertexShader, VS_SOURCE, allocator );
        if ( !result )
        {
            NW_FATAL_ERROR( "Can't create VertexShader." );
            glDeleteShader( m_VertexShader );
            m_VertexShader = 0;
            return;
        }
    }

    // FragmentShader
    {
        NW_ASSERT( m_FragmentShader == 0 );
        m_FragmentShader = glCreateShader( GL_FRAGMENT_SHADER );
        NW_GL_ASSERT();

        bool result = CreateShader( m_FragmentShader, PS_SOURCE, allocator );
        if ( !result )
        {
            NW_FATAL_ERROR( "Can't create FragmentShader." );
            glDeleteShader( m_FragmentShader );
            m_FragmentShader = 0;
            glDeleteShader( m_VertexShader );
            m_VertexShader = 0;
            return;
        }
    }

    // リンク
    {
        NW_ASSERT( m_ShaderProgram == 0 );

        m_ShaderProgram = glCreateProgram();
        glAttachShader( m_ShaderProgram, m_VertexShader );
        glAttachShader( m_ShaderProgram, m_FragmentShader );
        NW_GL_ASSERT();

        glLinkProgram( m_ShaderProgram );
        GLint result;
        glGetProgramiv( m_ShaderProgram, GL_LINK_STATUS, &result );
        NW_GL_ASSERT();
        if ( result == GL_FALSE )
        {
            GLint logSize;
            glGetProgramiv( m_ShaderProgram, GL_INFO_LOG_LENGTH, &logSize );
            char* buffer = static_cast<char*>( allocator->Alloc( logSize ) );
            glGetProgramInfoLog( m_ShaderProgram, logSize, &logSize, buffer );
            NW_FATAL_ERROR( "%s", buffer );

            glDeleteProgram( m_ShaderProgram );
            m_ShaderProgram = 0;
            glDeleteShader( m_FragmentShader );
            m_FragmentShader = 0;
            glDeleteShader( m_VertexShader );
            m_VertexShader = 0;

            allocator->Free( buffer );

            return;
        }
    }

    // サンプラー作成、及び設定
    {
        glGenSamplers(1, &m_Sampler);
        glSamplerParameteri( m_Sampler, GL_TEXTURE_MAG_FILTER, GL_LINEAR );
        glSamplerParameteri( m_Sampler, GL_TEXTURE_MIN_FILTER, GL_LINEAR );
        glSamplerParameteri( m_Sampler, GL_TEXTURE_WRAP_S, GL_CLAMP_TO_EDGE );
        glSamplerParameteri( m_Sampler, GL_TEXTURE_WRAP_T, GL_CLAMP_TO_EDGE );
    }

    // ユニフォーム/頂点属性位置のキャッシュを取得
    {
        m_ParamWVP = glGetUniformLocation( m_ShaderProgram, "wvp" );
        m_ParamUser = glGetUniformLocation( m_ShaderProgram, "user" );
        m_ParamRate = glGetUniformLocation( m_ShaderProgram, "rate" );
        m_ParamTex = glGetUniformLocation( m_ShaderProgram, "texture0" );
        m_ParamColor0 = glGetUniformLocation( m_ShaderProgram, "color0" );
        m_ParamColor1 = glGetUniformLocation( m_ShaderProgram, "color1" );
        m_ParamUVSrc = glGetUniformLocation( m_ShaderProgram, "uv_src" );
        m_ParamUVSize = glGetUniformLocation( m_ShaderProgram, "uv_size" );
        m_AttrVertex = glGetAttribLocation( m_ShaderProgram, "Vertex" );
        m_AttrTexCoord0 = glGetAttribLocation( m_ShaderProgram, "TexCoord0" );
        m_AttrColorRate = glGetAttribLocation( m_ShaderProgram, "ColorRate" );
    }

    // バッファ生成
    {
        {
            Vertex vtx[ 4 ];
            static const u16 idx[ 3 ] = { 0, 2, 1 };
            nw::dev::internal::SetQuadVertex( vtx, NULL );
            CreateVertexBuffer( m_TriangleBuf, vtx, 4, idx, 3 );
        }

        {
            Vertex vtx[ 4 ];
            u16 idx[ 3 * 2 ];
            nw::dev::internal::SetQuadVertex( vtx, idx );
            CreateVertexBuffer( m_QuadBuf, vtx, 4, idx, 3 * 2 );
        }

        {
            Vertex vtx[ 4 ];
            static const u16 idx[ 4 ] = { 0, 1, 3, 2 };
            nw::dev::internal::SetQuadVertex( vtx, NULL );
            CreateVertexBuffer( m_BoxBuf, vtx, 4, idx, 4 );
        }

        {
            Vertex vtx[ 8 ];
            u16 idx[ 3 * 2 * 6 ];
            nw::dev::internal::SetCubeVertex( vtx, idx );
            CreateVertexBuffer( m_CubeBuf, vtx, 8, idx, 3 * 2 * 6 );
        }

        {
            Vertex vtx[ 8 ];
            u16 idx[ 17 ];
            nw::dev::internal::SetWireCubeVertex( vtx, idx );
            CreateVertexBuffer( m_WireCubeBuf, vtx, 8, idx, 17 );
        }

        {
            Vertex vtx[ 2 ];
            u16 idx[ 2 ];
            nw::dev::internal::SetLineVertex( vtx, idx );
            CreateVertexBuffer( m_LineBuf, vtx, 2, idx, 2 );
        }

        {
            Vertex vtx[ MAX_SPHERE_VTX_NUM ];
            u16 idx[ MAX_SPHERE_IDX_NUM ];
            nw::dev::internal::SetSphereVertex( vtx, idx, MAX_SPHERE_X / 2, MAX_SPHERE_Y / 2 );
            CreateVertexBuffer(
                m_SphereSBuf,
                vtx,
                nw::dev::internal::CalcSphereVertexNum( MAX_SPHERE_X / 2, MAX_SPHERE_Y / 2 ),
                idx,
                nw::dev::internal::CalcSphereIndexNum( MAX_SPHERE_X / 2, MAX_SPHERE_Y / 2 )
            );
        }

        {
            Vertex vtx[ MAX_SPHERE_VTX_NUM ];
            u16 idx[ MAX_SPHERE_IDX_NUM ];
            nw::dev::internal::SetSphereVertex( vtx, idx, MAX_SPHERE_X, MAX_SPHERE_Y );
            CreateVertexBuffer(
                m_SphereLBuf,
                vtx,
                nw::dev::internal::CalcSphereVertexNum( MAX_SPHERE_X, MAX_SPHERE_Y ),
                idx,
                nw::dev::internal::CalcSphereIndexNum( MAX_SPHERE_X, MAX_SPHERE_Y )
            );
        }

        {
            Vertex vtx[ MAX_DISK_VTX_NUM ];
            u16 idx[ MAX_DISK_DIVISION / 2 ];

            for ( u16 i = 0; i < MAX_DISK_DIVISION / 2; ++i )
            {
                idx[ i ] = i;
            }

            nw::dev::internal::SetDiskVertex( vtx, NULL, MAX_DISK_DIVISION / 2 );
            CreateVertexBuffer( m_CircleSBuf, vtx, MAX_DISK_DIVISION / 2, idx, MAX_DISK_DIVISION / 2 );
        }

        {
            Vertex vtx[ MAX_DISK_VTX_NUM ];
            u16 idx[ MAX_DISK_DIVISION ];

            for ( u16 i = 0; i < MAX_DISK_DIVISION; ++i )
            {
                idx[ i ] = i;
            }

            nw::dev::internal::SetDiskVertex( vtx, NULL, MAX_DISK_DIVISION );
            CreateVertexBuffer( m_CircleLBuf, vtx, MAX_DISK_DIVISION, idx, MAX_DISK_DIVISION );
        }
    }
}

//--------------------------------------------------------------------------
void
PrimitiveRendererGL::SetViewportImpl( const s32 x, const s32 y, const s32 width, const s32 height )
{
    glViewport( x, y, width, height );
    NW_GL_ASSERT();

    glScissor( x, y, width, height );
    NW_GL_ASSERT();
}

//--------------------------------------------------------------------------
#if defined(_WIN32)
#pragma warning(push)
#pragma warning(disable:4238)
#endif

// nw::math::MTX44Mult( &wvp, m_ProjMtx, &math::MTX44( *m_ViewMtx ) ); の部分で警告がでるので抑制

void
PrimitiveRendererGL::BeginImpl()
{
    // WVP 行列生成
    nw::math::MTX44 wvp;
    nw::math::MTX44Mult( &wvp, m_ProjMtx, &math::MTX44( *m_ViewMtx ) );

    // Program使用開始
    glUseProgram( m_ShaderProgram );

    // 初期値設定
    glUniformMatrix4fv( m_ParamWVP, 1, GL_FALSE, wvp );
    glUniformMatrix4fv( m_ParamUser, 1, GL_FALSE, nw::math::MTX44::Identity() );
    NW_GL_ASSERT();

    glBindVertexArray(0);

    // 属性設定
    {
        glEnableVertexAttribArray( m_AttrVertex );
        NW_GL_ASSERT();
        glEnableVertexAttribArray( m_AttrTexCoord0 );
        NW_GL_ASSERT();
        glEnableVertexAttribArray( m_AttrColorRate );
        NW_GL_ASSERT();
    }
}
#if defined(_WIN32)
#pragma warning(pop)
#endif
//--------------------------------------------------------------------------
void
PrimitiveRendererGL::EndImpl()
{
    // 属性位置無効化
    {
        glDisableVertexAttribArray( m_AttrVertex );
        NW_GL_ASSERT();
        glDisableVertexAttribArray( m_AttrTexCoord0 );
        NW_GL_ASSERT();
        glDisableVertexAttribArray( m_AttrColorRate );
        NW_GL_ASSERT();
    }

    glBindBuffer( GL_ARRAY_BUFFER, GL_NONE );
    glBindBuffer( GL_ELEMENT_ARRAY_BUFFER, GL_NONE );
}

//--------------------------------------------------------------------------
void
PrimitiveRendererGL::DrawTriangleImpl(
    const nw::math::MTX34& modelMtx,
    const nw::ut::Color4u8& color
)
{
    DrawPolygon( modelMtx, m_TriangleBuf, 3, color, color );
}

//--------------------------------------------------------------------------
void
PrimitiveRendererGL::DrawTriangleImpl(
    const nw::math::MTX34& modelMtx,
    const nw::ut::Color4u8& color,
    const DrawBuffer& buf
)
{
    DrawPolygon( modelMtx, const_cast<GLuint*>(buf.buffer), buf.indexNum, color, color );
}

//--------------------------------------------------------------------------
void
PrimitiveRendererGL::DrawQuadImpl(
    const nw::math::MTX34& modelMtx,
    const nw::ut::Color4u8& colorLeft,
    const nw::ut::Color4u8& colorRight
)
{
    DrawPolygon( modelMtx, m_QuadBuf, 6, colorLeft, colorRight );
}

//--------------------------------------------------------------------------
void
PrimitiveRendererGL::DrawQuadImpl(
    const nw::math::MTX34& modelMtx,
    const nw::gfnd::Texture& texture,
    const nw::ut::Color4u8& colorLeft,
    const nw::ut::Color4u8& colorRight,
    const nw::math::VEC2& uvSrc,
    const nw::math::VEC2& uvSize
)
{
    const nw::gfnd::TextureGL* tex = static_cast<const nw::gfnd::TextureGL*>( &texture );

    DrawVertexBuffer( modelMtx, m_QuadBuf, colorLeft, colorRight );

    glUniform1f( m_ParamRate, 1.f );
    glUniform1i( m_ParamTex, 0 );
    glUniform2fv( m_ParamUVSrc, 1, uvSrc );
    glUniform2fv( m_ParamUVSize, 1, uvSize );

    glEnable( GL_TEXTURE_2D );
    glActiveTexture( GL_TEXTURE0 );
    glBindSampler( 0, m_Sampler );
    glBindTexture( GL_TEXTURE_2D, tex->GetID() );

    glDrawElements( GL_TRIANGLES, 6, GL_UNSIGNED_SHORT, 0 );
    NW_GL_ASSERT();
}

//--------------------------------------------------------------------------
void
PrimitiveRendererGL::DrawBoxImpl(
    const nw::math::MTX34& modelMtx,
    const nw::ut::Color4u8& colorLeft,
    const nw::ut::Color4u8& colorRight
)
{
    DrawLines( modelMtx, GL_LINE_LOOP, m_BoxBuf, 4, colorLeft, colorRight );
}

//--------------------------------------------------------------------------
void
PrimitiveRendererGL::DrawCubeImpl(
    const nw::math::MTX34& modelMtx,
    const nw::ut::Color4u8& color0,
    const nw::ut::Color4u8& color1
)
{
    DrawPolygon( modelMtx, m_CubeBuf, 3 * 2 * 6, color0, color1 );
}

//--------------------------------------------------------------------------
void
PrimitiveRendererGL::DrawWireCubeImpl(
    const nw::math::MTX34& modelMtx,
    const nw::ut::Color4u8& color0,
    const nw::ut::Color4u8& color1
)
{
    DrawLines( modelMtx, GL_LINE_LOOP, m_WireCubeBuf, 17, color0, color1 );
}

//--------------------------------------------------------------------------
void
PrimitiveRendererGL::DrawLineImpl(
    const nw::math::MTX34& modelMtx,
    const nw::ut::Color4u8& color0,
    const nw::ut::Color4u8& color1,
    const f32 lineWidth /* = 1.f */
)
{
    DrawLines( modelMtx, GL_LINES, m_LineBuf, 2, color0, color1, lineWidth );
}

//--------------------------------------------------------------------------
void
PrimitiveRendererGL::DrawLineImpl(
    const nw::math::MTX34& modelMtx,
    const nw::ut::Color4u8& color,
    const DrawBuffer& buf,
    const f32 lineWidth /* = 1.f */
)
{
    DrawLines( modelMtx, GL_LINES, const_cast<GLuint*>(buf.buffer), buf.indexNum, color, color, lineWidth );
}

//--------------------------------------------------------------------------
void
PrimitiveRendererGL::DrawSphere4x8Impl(
    const nw::math::MTX34& modelMtx,
    const nw::ut::Color4u8& north,
    const nw::ut::Color4u8& south
)
{
    DrawPolygon(
        modelMtx,
        m_SphereSBuf,
        nw::dev::internal::CalcSphereIndexNum( MAX_SPHERE_X / 2, MAX_SPHERE_Y / 2 ),
        north,
        south
    );
}

//--------------------------------------------------------------------------
void
PrimitiveRendererGL::DrawSphere8x16Impl(
    const nw::math::MTX34& modelMtx,
    const nw::ut::Color4u8& north,
    const nw::ut::Color4u8& south
)
{
    DrawPolygon(
        modelMtx,
        m_SphereLBuf,
        nw::dev::internal::CalcSphereIndexNum( MAX_SPHERE_X, MAX_SPHERE_Y ),
        north,
        south
    );
}

//--------------------------------------------------------------------------
void
PrimitiveRendererGL::DrawCircle16Impl(
    const nw::math::MTX34& modelMtx,
    const nw::ut::Color4u8& edge
)
{
    DrawLines( modelMtx, GL_LINE_LOOP, m_CircleSBuf, MAX_DISK_DIVISION / 2, edge, edge );
}

//--------------------------------------------------------------------------
void
PrimitiveRendererGL::DrawCircle32Impl(
    const nw::math::MTX34& modelMtx,
    const nw::ut::Color4u8& edge
)
{
    DrawLines( modelMtx, GL_LINE_LOOP, m_CircleLBuf, MAX_DISK_DIVISION, edge, edge );
}



//--------------------------------------------------------------------------
bool
PrimitiveRendererGL::CreateShader( GLuint shader, const char* source, nw::ut::IAllocator* allocator )
{
    GLint length = ::strlen( source ) + 1;

    // コンパイル
    glShaderSource( shader, 1, &source, &length );
    glCompileShader( shader );

    // 結果をチェック
    GLint result;
    glGetShaderiv( shader, GL_COMPILE_STATUS, &result );
    if ( result != GL_TRUE )
    {
        GLint logSize;
        glGetShaderiv( shader, GL_INFO_LOG_LENGTH, &logSize );
        char* buffer = static_cast<char*>( allocator->Alloc( logSize ) );
        glGetShaderInfoLog( shader, logSize, &logSize, buffer );
        NW_FATAL_ERROR( "%s", buffer );
        allocator->Free( buffer );

        return false;
    }

    return true;
}

//--------------------------------------------------------------------------
void
PrimitiveRendererGL::CreateVertexBuffer(
    GLuint* vertexBuffer,
    const Vertex* vtx,
    s32 vtxNum,
    const u16* idx,
    s32 idxNum
)
{
    glGenBuffers( 2, vertexBuffer );

    glBindBuffer( GL_ARRAY_BUFFER, vertexBuffer[ 0 ] );
    glBufferData( GL_ARRAY_BUFFER, sizeof( Vertex ) * vtxNum, vtx, GL_STATIC_DRAW );
    NW_GL_ASSERT();

    glBindBuffer( GL_ELEMENT_ARRAY_BUFFER, vertexBuffer[ 1 ] );
    glBufferData( GL_ELEMENT_ARRAY_BUFFER, sizeof( u16 ) * idxNum, idx, GL_STATIC_DRAW );
    NW_GL_ASSERT();
}

//--------------------------------------------------------------------------
void
PrimitiveRendererGL::DrawVertexBuffer(
    const nw::math::MTX34& modelMtx,
    GLuint* vb,
    const nw::ut::Color4u8& color0,
    const nw::ut::Color4u8& color1
)
{
    nw::math::MTX44 mat( modelMtx );
    glUniformMatrix4fv( m_ParamUser, 1, GL_FALSE, mat );

    glUniform4fv( m_ParamColor0, 1, static_cast<nw::ut::FloatColor>( color0 ) );
    glUniform4fv( m_ParamColor1, 1, static_cast<nw::ut::FloatColor>( color1 ) );
    NW_GL_ASSERT();

    glUniform1f( m_ParamRate, 0.f );
    NW_GL_ASSERT();

    glBindBuffer( GL_ARRAY_BUFFER, vb[ 0 ] );
    glBindBuffer( GL_ELEMENT_ARRAY_BUFFER, vb[ 1 ] );

    // 属性設定
    {
        const char* dummy = NULL;
        glVertexAttribPointer( m_AttrVertex, 3, GL_FLOAT, GL_FALSE, sizeof( Vertex ), dummy + 0 );
        NW_GL_ASSERT();

        glVertexAttribPointer( m_AttrTexCoord0, 2, GL_FLOAT, GL_FALSE, sizeof( Vertex ), dummy + sizeof( float ) * 3 );
        NW_GL_ASSERT();

        glVertexAttribPointer( m_AttrColorRate, 4, GL_FLOAT, GL_FALSE, sizeof( Vertex ), dummy + sizeof( float ) * 5 );
        NW_GL_ASSERT();
    }
}

//--------------------------------------------------------------------------
void
PrimitiveRendererGL::DrawPolygon(
    const nw::math::MTX34& modelMtx,
    GLuint* vb,
    s32 idxNum,
    const nw::ut::Color4u8& color0,
    const nw::ut::Color4u8& color1
)
{
    DrawVertexBuffer( modelMtx, vb, color0, color1 );

    glUniform2fv( m_ParamUVSrc, 1, nw::math::VEC2(0.f, 0.f) );
    glUniform2fv( m_ParamUVSize, 1, nw::math::VEC2(1.f, 1.f) );

    glDrawElements( GL_TRIANGLES, idxNum, GL_UNSIGNED_SHORT, 0 );
    NW_GL_ASSERT();
}

//--------------------------------------------------------------------------
void
PrimitiveRendererGL::DrawLines(
    const nw::math::MTX34& modelMtx,
    GLenum lineType,
    GLuint* vb,
    s32 idxNum,
    const nw::ut::Color4u8& color0,
    const nw::ut::Color4u8& color1,
    const f32 lineWidth /* = 1.f */
)
{
    DrawVertexBuffer( modelMtx, vb, color0, color1 );

    glLineWidth( lineWidth );
    glDrawElements( lineType, idxNum, GL_UNSIGNED_SHORT, 0 );
    NW_GL_ASSERT();
}

} // namespace dev
} // namespace nw
