﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nw/dev/dev_PrimitiveRendererUtil.h>
#include <nw/ut/ut_Memory.h>

namespace nw
{
namespace dev
{
namespace internal
{

//--------------------------------------------------------------------------
void
SetQuadVertex( Vertex* vtxBuf, u16* idxBuf )
{
    static const Vertex vtx[ 4 ] =
    {
        { nw::math::VEC3( -0.5f,  0.5f, 0.f ), nw::math::VEC2( 0.f, 1.f ), nw::math::VEC4( 0.f, 0.f, 0.f, 0.f ) },
        { nw::math::VEC3(  0.5f,  0.5f, 0.f ), nw::math::VEC2( 1.f, 1.f ), nw::math::VEC4( 0.f, 0.f, 0.f, 0.f ) },
        { nw::math::VEC3( -0.5f, -0.5f, 0.f ), nw::math::VEC2( 0.f, 0.f ), nw::math::VEC4( 1.f, 0.f, 0.f, 0.f ) },
        { nw::math::VEC3(  0.5f, -0.5f, 0.f ), nw::math::VEC2( 1.f, 0.f ), nw::math::VEC4( 1.f, 0.f, 0.f, 0.f ) }
    };

    static const u16 idx[ 6 ] =
    {
        0, 2, 1,
        1, 2, 3
    };

    if ( vtxBuf )
    {
        nw::ut::MemCpy( vtxBuf, vtx, sizeof( vtx ) );
    }

    if ( idxBuf )
    {
        nw::ut::MemCpy( idxBuf, idx, sizeof( idx ) );
    }
}

//--------------------------------------------------------------------------
void
SetCubeVertex( Vertex* vtxBuf, u16* idxBuf )
{
    static const Vertex vtx[ 8 ] =
    {
        { nw::math::VEC3( -0.5f, -0.5f, -0.5f ), nw::math::VEC2( 0.f, 0.f ), nw::math::VEC4( 1.f / 3.f, 0.f, 0.f, 0.f ) },
        { nw::math::VEC3( -0.5f,  0.5f, -0.5f ), nw::math::VEC2( 0.f, 1.f ), nw::math::VEC4(       0.f, 0.f, 0.f, 0.f ) },
        { nw::math::VEC3( -0.5f,  0.5f,  0.5f ), nw::math::VEC2( 1.f, 1.f ), nw::math::VEC4( 1.f / 3.f, 0.f, 0.f, 0.f ) },
        { nw::math::VEC3( -0.5f, -0.5f,  0.5f ), nw::math::VEC2( 1.f, 0.f ), nw::math::VEC4( 2.f / 3.f, 0.f, 0.f, 0.f ) },
        { nw::math::VEC3(  0.5f, -0.5f,  0.5f ), nw::math::VEC2( 0.f, 0.f ), nw::math::VEC4(       1.f, 0.f, 0.f, 0.f ) },
        { nw::math::VEC3(  0.5f,  0.5f,  0.5f ), nw::math::VEC2( 0.f, 1.f ), nw::math::VEC4( 2.f / 3.f, 0.f, 0.f, 0.f ) },
        { nw::math::VEC3(  0.5f,  0.5f, -0.5f ), nw::math::VEC2( 1.f, 1.f ), nw::math::VEC4( 1.f / 3.f, 0.f, 0.f, 0.f ) },
        { nw::math::VEC3(  0.5f, -0.5f, -0.5f ), nw::math::VEC2( 1.f, 0.f ), nw::math::VEC4( 2.f / 3.f, 0.f, 0.f, 0.f ) }
    };

    static const u16 idx[ 36 ] =
    {
        2, 1, 0,
        3, 2, 0,
        5, 2, 3,
        4, 5, 3,
        6, 5, 7,
        7, 5, 4,
        1, 6, 0,
        6, 7, 0,
        0, 7, 3,
        3, 7, 4,
        1, 2, 6,
        2, 5, 6
    };

    if ( vtxBuf )
    {
        nw::ut::MemCpy( vtxBuf, vtx, sizeof( vtx ) );
    }

    if ( idxBuf )
    {
        nw::ut::MemCpy( idxBuf, idx, sizeof( idx ) );
    }
}

//--------------------------------------------------------------------------
void
SetWireCubeVertex( Vertex* vtxBuf, u16* idxBuf )
{
    SetCubeVertex( vtxBuf, NULL );

    static const u16 idx[ 17 ] =
    {
        0, 1, 2, 3, 0, 7, 6, 1,
        2, 5, 6, 7, 4, 5, 4, 3, 0
    };

    if ( idxBuf )
    {
        nw::ut::MemCpy( idxBuf, idx, sizeof( idx ) );
    }
}

//--------------------------------------------------------------------------
void
SetLineVertex( Vertex* vtxBuf, u16* idxBuf )
{
    static const Vertex vtx[ 2 ] =
    {
        { nw::math::VEC3( -0.5f, 0.f, 0.f ), nw::math::VEC2( 0.f, 0.5f ), nw::math::VEC4( 0.f, 0.f, 0.f, 0.f ) },
        { nw::math::VEC3(  0.5f, 0.f, 0.f ), nw::math::VEC2( 1.f, 0.5f ), nw::math::VEC4( 1.f, 0.f, 0.f, 0.f ) }
    };

    static const u16 idx[ 2 ] =
    {
        0, 1
    };

    if ( vtxBuf )
    {
        nw::ut::MemCpy( vtxBuf, vtx, sizeof( vtx ) );
    }

    if ( idxBuf )
    {
        nw::ut::MemCpy( idxBuf, idx, sizeof( idx ) );
    }
}

//--------------------------------------------------------------------------
void
SetSphereVertex( Vertex* vtxBuf, u16* idxBuf, s32 sizeX, s32 sizeY )
{
    if ( vtxBuf )
    {
        for ( s32 y = 0; y < sizeY; ++y )
        {
            f32 y_angle = ( ( y + 1 ) / ( sizeY + 1.f ) - 0.5f ) * NW_MATH_DEG_TO_RAD( 180.f );
            f32 ypos = nw::math::SinRad( y_angle ) * 0.5f;
            f32 radius = nw::math::CosRad( y_angle ) * 0.5f;

            for ( s32 x = 0; x < sizeX; ++x )
            {
                s32 i = y * sizeX + x ;
                f32 rad = NW_MATH_DEG_TO_RAD( 360.f ) * x / sizeX;

                // y が偶数のときは半分ひねる
                if ( y % 2 == 0 )
                {
                    rad -= NW_MATH_DEG_TO_RAD( 360.f ) / sizeX / 2;
                }

                f32 xpos = nw::math::CosRad( rad ) * radius;
                f32 zpos = nw::math::SinRad( rad ) * radius;

                vtxBuf[ i ].pos.Set( xpos, ypos, zpos );
                vtxBuf[ i ].uv.Set( 0.5f + ypos, x / static_cast<f32>( sizeX ) );
                vtxBuf[ i ].color.Set( -ypos + 0.5f, 0.f, 0.f, 0.f );
            }
        }

        // 北極と南極
        {
            f32 ypos = -0.5f;
            s32 i = sizeX * sizeY;
            vtxBuf[ i ].pos.Set( 0.f, ypos, 0.f );
            vtxBuf[ i ].uv.Set( 0.5f + ypos, 0.5f );
            vtxBuf[ i ].color.Set( 1.f, 0.f, 0.f, 0.f );
        }

        {
            f32 ypos = 0.5f;
            s32 i = sizeX * sizeY + 1;
            vtxBuf[ i ].pos.Set( 0.f, ypos, 0.f );
            vtxBuf[ i ].uv.Set( 0.5f + ypos, 0.5f );
            vtxBuf[ i ].color.Set( 0.f, 0.f, 0.f, 0.f );
        }
    }

    if ( idxBuf )
    {
        for ( s32 i = 0; i < sizeX; ++i )
        {
            idxBuf[ i * 3 ]     = static_cast<u16>( sizeX * sizeY );
            idxBuf[ i * 3 + 1 ] = static_cast<u16>( i );
            idxBuf[ i * 3 + 2 ] = static_cast<u16>( ( i + 1 ) % sizeX );
        }

        for ( s32 y = 0; y < ( sizeY - 1 ); ++y )
        {
            for ( s32 x = 0; x < sizeX; ++x )
            {
                // y が奇数のときは、次の頂点がひねられていることを考慮してずらす
                s32 offset = ( y % 2 );
                s32 i = ( y * sizeX * 6 ) + x * 6 + sizeX * 3;
                idxBuf[ i + 0 ] = static_cast<u16>( y * sizeX + x );
                idxBuf[ i + 1 ] = static_cast<u16>( ( y + 1 ) * sizeX + ( ( x + offset ) % sizeX ) );
                idxBuf[ i + 2 ] = static_cast<u16>( y * sizeX + ( ( x + 1 ) % sizeX ) );
                idxBuf[ i + 3 ] = static_cast<u16>( ( y + 1 ) * sizeX + ( ( x + offset ) % sizeX ) );
                idxBuf[ i + 4 ] = static_cast<u16>( ( y + 1 ) * sizeX + ( ( x + 1 + offset ) % sizeX ) );
                idxBuf[ i + 5 ] = static_cast<u16>( y * sizeX + ( ( x + 1 ) % sizeX ) );
            }
        }

        for ( s32 i = 0; i < sizeX; ++i )
        {
            s32 base = 3 * sizeX * ( sizeY - 1 ) * 2 + sizeX * 3;
            idxBuf[ i * 3 + 0 + base ] = static_cast<u16>( sizeX * sizeY + 1 );
            idxBuf[ i * 3 + 1 + base ] = static_cast<u16>( sizeX * ( sizeY - 1 ) + ( ( i + 1 ) % sizeX ) );
            idxBuf[ i * 3 + 2 + base ] = static_cast<u16>( sizeX * ( sizeY - 1 ) + i );
        }
    }
}

//--------------------------------------------------------------------------
void
SetDiskVertex( Vertex* vtxBuf, u16* idxBuf, s32 divNum )
{
    if ( vtxBuf )
    {
        for( s32 i = 0; i < divNum; ++i )
        {
            f32 rad = NW_MATH_DEG_TO_RAD( 360.f ) * i / divNum;

            vtxBuf[ i ].pos.x = nw::math::CosRad( rad ) * 0.5f;
            vtxBuf[ i ].pos.y = nw::math::SinRad( rad ) * 0.5f;
            vtxBuf[ i ].pos.z = 0.f;
            vtxBuf[ i ].uv.x = vtxBuf[ i ].pos.x;
            vtxBuf[ i ].uv.y = 1.f - vtxBuf[ i ].pos.y;
            vtxBuf[ i ].color.Set( 1.f, 0.f, 0.f, 0.f );
        }

        // 中央
        {
            s32 i = divNum;
            // f32 rad = NW_MATH_DEG_TO_RAD( 360.f ) * i / divNum;

            vtxBuf[ i ].pos.Set( 0.f, 0.f, 0.f );
            vtxBuf[ i ].uv.Set( 0.5f, 0.5f );
            vtxBuf[ i ].color.Set( 1.f, 0.f, 0.f, 0.f );
        }
    }

    if ( idxBuf )
    {
        for( s32 i = 0; i < divNum; ++i )
        {
            idxBuf[ i * 3 + 0 ] = static_cast<u16>( i );
            idxBuf[ i * 3 + 1 ] = static_cast<u16>( ( i + 1 ) % divNum );
            idxBuf[ i * 3 + 2 ] = static_cast<u16>( divNum );
        }
    }
}

} // namespace internal
} // namespace dev
} // namespace nw
