﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nw/dev/dev_MapFile.h>
#include <nw/ut/ut_ScopedLock.h>
#include <cstring>

namespace nw
{
namespace dev
{

// NOTE: MapFile::QuerySymbol 関数の実行の際 static 変数を使用している為
//       スレッドセーフではありません。
u8 MapFile::s_MapBuf[MapFile::MAP_BUFFER_SIZE];
s32 MapFile::s_MapBufOffset = -1;
u32 MapFile::s_FileLength = 0;
FSClient* MapFile::s_Client = NULL;
FSCmdBlock* MapFile::s_CmdBlock = NULL;
FSFileHandle MapFile::s_FileHandle = 0;
nw::ut::Mutex MapFile::s_Mutex;
bool MapFile::s_MutexInitialized = false;

#if defined(NW_MAP_ENABLE)
//--------------------------------------------------------------------------
MapFile::MapFileHandle
MapFile::RegistOnMem(
    void*       buffer,
    u8*         mapDataBuf,
    u32         mapDataLength
)
{
    NW_ASSERT_NOT_NULL( buffer );
    NW_ASSERT_NOT_NULL( mapDataBuf );

    MapFileBuf* pMapFile = static_cast<MapFileBuf*>( buffer );

    // データの末尾を NUL 文字で埋めておく
    mapDataBuf[ mapDataLength - 1 ] = '\0';

    pMapFile->mapBuf       = mapDataBuf;
    pMapFile->client       = NULL;
    pMapFile->fileHandle   = NULL;
    pMapFile->next         = NULL;

    AppendMapFile( pMapFile );

    if ( ! s_MutexInitialized )
    {
        s_Mutex.Initialize();
        s_MutexInitialized = true;
    }

    return static_cast<MapFileHandle>( pMapFile );
}

//--------------------------------------------------------------------------
MapFile::MapFileHandle
MapFile::RegistOnDvd(
    void*            buffer,
    FSClient*        client,
    FSCmdBlock*      cmdBlock,
    FSFileHandle     fileHandle
)
{
    NW_ASSERT_NOT_NULL( buffer );

    MapFileBuf* pMapFile = static_cast<MapFileBuf*>( buffer );

    pMapFile->mapBuf       = NULL;
    pMapFile->client       = client;
    pMapFile->cmdBlock     = cmdBlock;
    pMapFile->fileHandle   = fileHandle;
    pMapFile->next         = NULL;

    NW_ASSERT( pMapFile->client != NULL );

    AppendMapFile( pMapFile );

    if ( ! s_MutexInitialized )
    {
        s_Mutex.Initialize();
        s_MutexInitialized = true;
    }

    return static_cast<MapFileHandle>( pMapFile );
}

//--------------------------------------------------------------------------
bool
MapFile::QuerySymbol(
    u32 address,
    u8* strBuf,
    u32 strBufSize
)
{
    MapFileBuf* pMap = m_MapFileList;

    while ( pMap != NULL )
    {
        if ( QuerySymbolToSingleMapFile( pMap, address, strBuf, strBufSize ) )
        {
            return true;
        }

        pMap = pMap->next;
    }

    return false;
}



//--------------------------------------------------------------------------
void
MapFile::AppendMapFile( MapFile::MapFileBuf* pMapFileBuf )
{
    NW_ASSERT_NOT_NULL( pMapFileBuf );

    // 一つも登録されていなければリストの先頭へ登録する。
    if ( m_MapFileList == NULL )
    {
        m_MapFileList = pMapFileBuf;
        return;
    }

    {
        pMapFileBuf->next = m_MapFileList;
        m_MapFileList  = pMapFileBuf;
    }
}

//--------------------------------------------------------------------------
void
MapFile::RemoveMapFile( MapFileBuf* pMapFileBuf )
{
    NW_ASSERT_NOT_NULL( pMapFileBuf );

    if ( pMapFileBuf == m_MapFileList )
    {
        m_MapFileList = m_MapFileList->next;
        return;
    }

    for ( MapFileBuf* map = m_MapFileList; map != NULL; map = map->next )
    {
        if ( map->next == pMapFileBuf )
        {
            map->next = pMapFileBuf->next;
            return;
        }
    }
}

//--------------------------------------------------------------------------
bool
MapFile::QuerySymbolToSingleMapFile(
    MapFileBuf* pMapFile,
    u32         address,
    u8*         strBuf,
    u32         strBufSize
)
{
    NW_ASSERT_NOT_NULL( pMapFile );
    NW_ASSERT_NOT_NULL( strBuf );

    if ( pMapFile->mapBuf != NULL )
        // マップファイルがメモリ上に置かれている場合の問い合わせ
    {
        GetCharPtr = &MapFile::GetCharOnMem;

        return QuerySymbolToMapFile( pMapFile->mapBuf, NULL, address, strBuf, strBufSize );
    }
    else if ( pMapFile->client != NULL )
        // マップファイルが DVD 上に置かれている場合の問い合わせ
    {
        // ファイルのオフセット情報は、オフセット 0 の場合に NULL として扱われないように、
        // 最上位ビットを立ててあります。
        u8 *buf = (u8*)0x80000000;

        s_Client = pMapFile->client;
        s_CmdBlock = pMapFile->cmdBlock;
        s_FileHandle = pMapFile->fileHandle;

        FSStat stat;
        FSStatus status = FS_STATUS_OK;
        NW_UNUSED_VARIABLE( status ); // Release では使用しない。

        status = FSGetStatFile(pMapFile->client, pMapFile->cmdBlock, pMapFile->fileHandle, &stat, FS_RET_ALL_ERROR);
        NW_ASSERT(status >= FS_STATUS_OK);

        s_FileLength = stat.size;

        GetCharPtr = &MapFile::GetCharOnDvd;

        return QuerySymbolToMapFile( buf, NULL, address, strBuf, strBufSize );
    }

    // メモリ上にも DVD 上にもない場合は問い合わせ失敗とする
    *strBuf = '\0';
    return false;
}

//--------------------------------------------------------------------------
bool
MapFile::QuerySymbolToMapFile(
    u8*         buf,
    const void* moduleInfo,
    u32         address,
    u8*         strBuf,
    u32         strBufSize
)
{
    // TODO: 再配置可能モジュールに対応する。

    static const s32 PARAM_SECTION          = 0;
    static const s32 PARAM_ADDRESS_AND_SIZE = 1;
    static const s32 PARAM_SYMBOL           = 2;
    static const s32 PARAM_SIZE = 1;
    static const s32 HEADER_LINES = 2;
    static const char* PARAGRAPH_NAME = "Global Symbols";

    NW_ASSERT_NOT_NULL( strBuf );
    NW_ASSERT( strBufSize > 0 );

    {
        // 再配置可能モジュール用の開始アドレスオフセット
        // static なモジュールの場合は、オフセットは必要ないので 0 にしておく
        u32 offset = 0;

        // Global Symbols の先頭を探す
        buf = SearchParagraph( buf, PARAGRAPH_NAME );
        // ヘッダ行を読み飛ばす
        buf = SearchNextLine( buf, HEADER_LINES - 1 );

        while ( true )
        {

            u32 startAddr = 0;
            u32 size = 0;

            // 次の行を検索する
            buf = SearchNextLine( buf, 1 );

            if ( buf == NULL )
            {
                return false;
            }

            {
                u8* paramSection;
                if ( ( paramSection = SearchParam( buf, PARAM_SECTION, ' ' ) ) == NULL )
                {
                    break;
                }

                if ( ( this->*GetCharPtr )( paramSection ) != '.' )
                {
                    continue;           // 次の行へ
                }

                u8* paramAddr;
                if ( ( paramAddr = SearchParam( buf, PARAM_ADDRESS_AND_SIZE, ' ' ) ) == NULL )
                {
                    break;
                }
                startAddr = XStrToU32( paramAddr );

                if ( startAddr == 0 )
                {
                    continue;           // 次の行へ
                }

                u8* paramSize;
                if ( ( paramSize = SearchParam( paramAddr, PARAM_SIZE, '+' ) ) == NULL )
                {
                    break;
                }
                size = XStrToU32( paramSize );
            }

            startAddr += offset;
            if ( address < startAddr || startAddr + size <= address )
            {
                continue;           // 次の行へ
            }

            {
                u8* paramSymbol;

                if ( ( paramSymbol = SearchParam( buf, PARAM_SYMBOL, ' ' ) ) == NULL )
                {
                    // アドレスは見つかったがシンボル名が見つからなかった場合
                    *strBuf = '\0';
                    return true;
                }

                (void)CopySymbol( paramSymbol, strBuf, strBufSize, '\n' );

                return true;
            }
        }
    }

    return false;
}

//--------------------------------------------------------------------------
u8*
MapFile::SearchNextLine( u8* buf, s32 lines )
{
    u8 c;

    NW_ASSERT_NOT_NULL( this->GetCharPtr );

    if ( buf == NULL )
    {
        return NULL;
    }

    while ( (c = ( this->*GetCharPtr )( buf )) != '\0' )
    {
        if ( c == '\n' )
        {
            if ( --lines <= 0 )
            {
                return (buf + 1);
            }
        }

        ++buf;
    }

    return NULL;
}

//--------------------------------------------------------------------------
u8*
MapFile::SearchParagraph( u8* buf, const char* name )
{
    NW_ASSERT_NOT_NULL( this->GetCharPtr );

    const size_t nameLen = ::std::strlen( name );

    while ( true )
    {
        buf = SearchNextLine( buf, 1 );

        if ( buf == NULL )
        {
            return NULL;
        }

        u8* bufPos = buf;
        u8* namePos = reinterpret_cast<u8*>( const_cast<char*>( name ) );
        s32 count = 0;

        while ( count < nameLen )
        {
            u8 c = ( this->*GetCharPtr )( bufPos );

            if( c != *namePos )
            {
                break;
            }

            if ( c == '\0' )
            {
                return NULL;
            }

            ++bufPos ;
            ++namePos;
            ++count;
        }

        if ( count >= nameLen )
        {
            return buf;
        }
    }
}

//--------------------------------------------------------------------------
u8*
MapFile::SearchParam( u8* lineTop, u32 argNum, u8 splitter )
{
    bool inArg = false;
    u8*  buf   = lineTop;

    NW_ASSERT_NOT_NULL( this->GetCharPtr );

    if ( buf == NULL )
    {
        return NULL;
    }

    while ( true )
    {
        u8 c = ( this->*GetCharPtr )( buf );

        if ( c == '\0' || c == '\n' )
        {
            return NULL;
        }

        if ( inArg )
        {
            if ( c == splitter )
            {
                inArg = false;
            }
        }
        else
        {
            if ( c != splitter )
            {
                if ( argNum-- == 0 )
                {
                    return buf;
                }

                inArg = true;
            }
        }

        ++buf;
    }
}

//--------------------------------------------------------------------------
u32
MapFile::XStrToU32( const u8* str )
{
    u32 val = 0;
    NW_ASSERT_NOT_NULL( str );
    NW_ASSERT_NOT_NULL( this->GetCharPtr );

    while ( true )
    {
        u32 num;
        u8  c = ( this->*GetCharPtr )( str );

        if ( '0' <= c && c <= '9' )
        {
            num = (u32)(c - '0');
        }
        else if ( 'a' <= c && c <= 'z' )
        {
            num = (u32)(c - 'a' + 0xA);
        }
        else if ( 'A' <= c && c <= 'Z' )
        {
            num = (u32)(c - 'A' + 0xA);
        }
        else
        {
            return val;
        }

        if ( val >= (0x1 << 28) )
        {
            // オーバーフロー
            return 0;
        }
        val = val * 0x10 + num;
        ++str;
    }
}

//--------------------------------------------------------------------------
u32
MapFile::CopySymbol(
    const u8*   buf,
    u8*         str,
    u32         strLenMax,
    u8          splitter
)
{
    u32 cnt = 0;

    NW_ASSERT_NOT_NULL( buf );
    NW_ASSERT_NOT_NULL( str );
    NW_ASSERT_NOT_NULL( this->GetCharPtr );

    while ( true )
    {
        u8 c = ( this->*GetCharPtr )( buf++ );

        if ( c == splitter || c == '\0' || c == '\n' )
        {
            *str = '\0';
            return cnt;
        }

        *str++ = c;

        if ( ++cnt >= strLenMax - 1 )
        {
            *str = '\0';
            return cnt;
        }
    }
}

//--------------------------------------------------------------------------
u8
MapFile::GetCharOnMem( const u8* buf )
{
    return *buf;
}

//--------------------------------------------------------------------------
u8
MapFile::GetCharOnDvd( const u8* buf )
{
    // ファイルのオフセット情報は、オフセット0の場合にNULLとして扱われないように、
    // 最上位ビットを立ててあります。
    s32 address = (s32)buf & 0x7FFFFFFF;    // 最上位ビットを下げる
    s32 offset  = address - s_MapBufOffset;

    if ( address >= s_FileLength )
    {
        return '\0';
    }

    if ( s_MapBufOffset < 0 ||
         offset < 0 ||
         offset >= MAP_BUFFER_SIZE )
    {
        s32 len;
        u32 size = MAP_BUFFER_SIZE;
        FSStatus status = FS_STATUS_OK;
        NW_UNUSED_VARIABLE( status ); // Release では使用しない。

        s_MapBufOffset = nw::ut::RoundDown( address, PPC_IO_BUFFER_ALIGN );
        offset = address - s_MapBufOffset;

        if ( s_MapBufOffset + MAP_BUFFER_SIZE >= s_FileLength )
        {
            size = nw::ut::RoundUp( s_FileLength - s_MapBufOffset, PPC_IO_BUFFER_ALIGN );
        }

        {
            nw::ut::ScopedLock<nw::ut::Mutex> lockObj(s_Mutex);

            status = FSSetPosFile(s_Client, s_CmdBlock, s_FileHandle, s_MapBufOffset, FS_RET_ALL_ERROR);
            NW_ASSERT( status == FS_STATUS_OK );

            len = FSReadFile(
                s_Client,
                s_CmdBlock,
                s_MapBuf,
                sizeof(*s_MapBuf),
                size / sizeof(*s_MapBuf),
                s_FileHandle,
                0,
                FS_RET_ALL_ERROR
            );

            NW_ASSERT(len >= FS_STATUS_OK);
        }

        if ( len <= 0 )
        {
            return '\0';
        }
    }

    return s_MapBuf[ offset ];
}

#endif // if defined( NW_MAP_ENABLE )

} // namespace dev
} // namespace nw
