﻿<#
    Copyright (C)Nintendo All rights reserved.

    These coded instructions, statements, and computer programs contain proprietary
    information of Nintendo and/or its licensed developers and are protected by
    national and international copyright laws. They may not be disclosed to third
    parties or copied or duplicated in any form, in whole or in part, without the
    prior written consent of Nintendo.

    The content herein is highly confidential and should be handled accordingly.
#>

param
(
    [Parameter(Mandatory=$true)][ValidateSet("Win32", "Cafe")][alias("t")]$target
)

# -------------------------------------------------------------------------------

function Get-NintendoSdkRootPath
{
    $callerPath = $MyInvocation.ScriptName
    $dirName    = [System.IO.Path]::GetDirectoryName($callerPath)
    while ( $dirName )
    {
        if ( Test-Path "${dirName}\NintendoSdkRootMark" )
        {
            return $dirName
        }
        $dirName = [System.IO.Path]::GetDirectoryName($dirName)
    }

    $message = "NintendoSdkRootMark not found"
    throw New-Object "System.IO.FileNotFoundException" $message
}

function Get-ProgramRootPath
{
    $NintendoSdkRootPath = Get-NintendoSdkRootPath
    if ( Test-Path "${NintendoSdkRootPath}\SigloRootMark" )
    {
        return "${NintendoSdkRootPath}\Programs\Iris"
    }
    return "${NintendoSdkRootPath}"
}

# -------------------------------------------------------------------------------

$BinaryExt = ".bnsh"
$HeaderExt = ".h"

$ScriptPath          = $MyInvocation.MyCommand.Path
$ScriptDirectoryPath = [System.IO.Path]::GetDirectoryName($scriptPath)

$NintendoSdkRoot     = Get-NintendoSdkRootPath
$ProgramRoot         = Get-ProgramRootPath
$GshCompileExe       = "${env:CAFE_ROOT}\system\bin\win64\gshCompile.exe"
$Text2CppBat         = "${ScriptDirectoryPath}\text2cpp.bat"

$SrcResourcePath    = "${ProgramRoot}\Sources\Resources\DevShaders"
$DstResourcePath    = "${ProgramRoot}\Resources\DevShaders"
$CommitResourcePath = "${ProgramRoot}\Sources\Libraries\Sys\dev\shaders"
$IntermediatePath   = "${ProgramRoot}\Intermediates\${target}\Libraries\DevShaders"

$VertexShaderSrcFile = "dev_PrimitiveRenderer.vsh"
$PixelShaderSrcFile  = "dev_PrimitiveRenderer.psh"

$GfxVertexShaderSrcFile = "dev_PrimitiveRendererGfx.vsh"
$GfxPixelShaderSrcFile  = "dev_PrimitiveRendererGfx.psh"
$GfxOutputDir       = $DstResourcePath
$GfxCommitDir       = $CommitResourcePath

# -------------------------------------------------------------------------------

$VertexShaderSrcPath = "${SrcResourcePath}\${VertexShaderSrcFile}"
$PixelShaderSrcPath  = "${SrcResourcePath}\${PixelShaderSrcFile}"
$GfxVertexShaderSrcPath = "${SrcResourcePath}\${GfxVertexShaderSrcFile}"
$GfxPixelShaderSrcPath  = "${SrcResourcePath}\${GfxPixelShaderSrcFile}"

# -------------------------------------------------------------------------------

$PrimitiveRendererOutputDirectoryWin32  = $CommitResourcePath
$VertexPrimitiveRendererOutputPathWin32 = "${CommitResourcePath}\dev_PrimitiveRenderer_vsh_win32.hpp"
$PixelPrimitiveRendererOutputPathWin32  = "${CommitResourcePath}\dev_PrimitiveRenderer_psh_win32.hpp"

$PrimitiveRendererGfxVertexOutputFilenameWin32 = "dev_PrimitiveRendererGfx_generic_vsh"
$PrimitiveRendererGfxPixelOutputFilenameWin32  = "dev_PrimitiveRendererGfx_generic_psh"
$PrimitiveRendererGfxVertexOutputSymbolWin32   = "dev_PrimitiveRendererGfx_generic_VS"
$PrimitiveRendererGfxPixelOutputSymbolWin32    = "dev_PrimitiveRendererGfx_generic_PS"

$PrimitiveRendererOutputDirectoryCafe = ${DstResourcePath}
$PrimitiveRendererShaderFilenameCafe  = "dev_PrimitiveRenderer"
$PrimitiveRendererOutputExtensionCafe = ".gsh"

$PrimitiveRendererGfxOutputFilenameCafe = "dev_PrimitiveRendererGfx_cafe_gsh"

# -------------------------------------------------------------------------------

function Convert-PrimitiveShaderWin32 ()
{
    $outDir = $PrimitiveRendererOutputDirectoryWin32
    if (!(Test-Path "${outDir}"))
    {
        mkdir "${outDir}"
    }

    $commandPixel  = "${Text2CppBat} ${PixelShaderSrcPath} ${PixelPrimitiveRendererOutputPathWin32}"
    Write-Output $commandPixel
    Invoke-Expression "${commandPixel}"

    $commandVertex = "${Text2CppBat} ${VertexShaderSrcPath} ${VertexPrimitiveRendererOutputPathWin32}"
    Write-Output $commandVertex
    Invoke-Expression "${commandVertex}"
}

# -------------------------------------------------------------------------------

function Convert-PrimitiveShaderCafe ()
{
    $outDir = $PrimitiveRendererOutputDirectoryCafe
    if (!(Test-Path "${outDir}"))
    {
        mkdir "${outDir}"
    }
    $outPath = "${outDir}\${PrimitiveRendererShaderFilenameCafe}${PrimitiveRendererOutputExtensionCafe}"

    $CompileCommand = "${GshCompileExe}" +
                      " -gpu 2" +
                      " -v ${VertexShaderSrcPath}" +
                      " -p ${PixelShaderSrcPath}" +
                      " -o ${outPath}"
    Write-Output $CompileCommand
    Invoke-Expression $CompileCommand
}

# -------------------------------------------------------------------------------

function Add-HeaderToFile ($ReadPath, $WritePath, $Header)
{
    Write-Output "Add-HeaderToFile"
    Write-Output "  In :${ReadPath}"
    Write-Output "  Out:${WritePath}"

    $outDir = [System.IO.Path]::GetDirectoryName($WritePath)
    if (!(Test-Path "${outDir}"))
    {
        mkdir "${outDir}"
    }

    $reader = $null
    $writer = $null
    try
    {
        $reader = New-Object "System.IO.StreamReader" ([string]$ReadPath)
        $writer = New-Object "System.IO.StreamWriter" ([string]$WritePath)
        if($Headers -ne $null)
        {
            foreach($header in $Headers)
            {
                $writer.WriteLine($header)
            }
        }
        while(-not $reader.EndOfStream)
        {
            $writer.WriteLine($reader.ReadLine())
        }
    }
    finally
    {
        if ($reader -ne $null)
        {
            $reader.Close()
        }
        if ($writer -ne $null)
        {
            $writer.Close()
        }
    }
}

function Convert-ShaderToHeader ($ReadPath, $WritePath, $Symbol)
{
    Write-Output "Convert-ShaderToHeader"
    Write-Output "  In :${ReadPath}"
    Write-Output "  Out:${WritePath}"

    $reader = $null
    $writer = $null
    try
    {
        $reader = New-Object "System.IO.StreamReader" ([string]$ReadPath)
        $writer = New-Object "System.IO.StreamWriter" ([String]$WritePath)
        $writer.WriteLine("// This file is automatically generated.")
        $writer.WriteLine("")
        $writer.WriteLine("static char " + $Symbol + "[] =")
        while(-not $reader.EndOfStream)
        {
            $writer.WriteLine("`"" + $reader.ReadLine() + "\n`"")
        }
        $writer.WriteLine(";")
    }
    finally
    {
        if ($reader -ne $null)
        {
            $reader.Close()
        }
        if ($writer -ne $null)
        {
            $writer.Close()
        }
    }
}

function Convert-ToCommitableSource ($Files)
{
    Write-Output "Convert-ToCommitableSource"
    Write-Output "  In/Out:${Files}"

    $trimChars = [char[]]([char]" ")
    foreach($file in $Files.Split([char]";"))
    {
        $content = ""
        $reader = $null
        try
        {
            $reader = New-Object "System.IO.StreamReader" ($file)
            while(-not $reader.EndOfStream)
            {
                $content += $reader.ReadLine().TrimEnd($trimChars) + "`n"
            }
        }
        finally
        {
            if($reader -ne $null)
            {
                $reader.Dispose()
            }
        }
        $writer = $null
        try
        {
            $writer = New-Object "System.IO.StreamWriter" ($file)
            $writer.Write($content)
        }
        finally
        {
            if($writer -ne $null)
            {
                $writer.Dispose()
            }
        }
    }
}

# -------------------------------------------------------------------------------

function Convert-PrimitiveShaderGfxWin32 ()
{
    if ((-not (Test-Path $GfxVertexShaderSrcPath)) -or (-not (Test-Path $GfxPixelShaderSrcPath)))
    {
        return
    }

    $outDir = $GfxOutputDir
    if (!(Test-Path "${outDir}"))
    {
        mkdir "${outDir}"
    }

    $headers = [string[]]"#version 430"

    $vertexOutputPath = "${outDir}\${PrimitiveRendererGfxVertexOutputFilenameWin32}${HeaderExt}"
    $pixelOutputPath  = "${outDir}\${PrimitiveRendererGfxPixelOutputFilenameWin32}${HeaderExt}"
    $vertexIntPath = "${IntermediatePath}\${GfxVertexShaderSrcFile}"
    $pixelIntPath = "${IntermediatePath}\${GfxPixelShaderSrcFile}"


    Add-HeaderToFile $GfxVertexShaderSrcPath $vertexIntPath $headers
    Add-HeaderToFile $GfxPixelShaderSrcPath  $pixelIntPath  $headers

    Convert-ShaderToHeader $vertexIntPath $vertexOutputPath $PrimitiveRendererGfxVertexOutputSymbolWin32
    Convert-ShaderToHeader $pixelIntPath  $pixelOutputPath  $PrimitiveRendererGfxPixelOutputSymbolWin32

    Convert-ToCommitableSource $vertexOutputPath
    Convert-ToCommitableSource $pixelOutputPath
}

# -------------------------------------------------------------------------------

function Convert-PrimitiveShaderGfxCafe ()
{
    if ((-not (Test-Path $GfxVertexShaderSrcPath)) -or (-not (Test-Path $GfxPixelShaderSrcPath)))
    {
        return
    }

    $outDir = $GfxOutputDir
    if (!(Test-Path "${outDir}"))
    {
        mkdir "${outDir}"
    }

    $headers =
        "#version 330",
        "#extension GL_ARB_separate_shader_objects : enable"
    $outPath = "${outDir}\${PrimitiveRendererGfxOutputFilenameCafe}${HeaderExt}"
    $vertexIntPath = "${IntermediatePath}\${GfxVertexShaderSrcFile}"
    $pixelIntPath  = "${IntermediatePath}\${GfxPixelShaderSrcFile}"

    Add-HeaderToFile $GfxVertexShaderSrcPath $vertexIntPath $headers
    Add-HeaderToFile $GfxPixelShaderSrcPath  $pixelIntPath  $headers

    $compileCommand =
        "${GshCompileExe}" +
        " -gpu 2" +
        " -v ${vertexIntPath}" +
        " -p ${pixelIntPath}" +
        " -oh ${outPath}"
    Write-Output $compileCommand
    Invoke-Expression $compileCommand

    Convert-ToCommitableSource $outPath
}

# -------------------------------------------------------------------------------

if ($target -eq "Win32")
{
    Convert-PrimitiveShaderWin32
    Convert-PrimitiveShaderGfxWin32
}
if ($target -eq "Cafe")
{
    Convert-PrimitiveShaderCafe
    Convert-PrimitiveShaderGfxCafe
}

exit 0
