﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "../../../../precompiled.h"

#include <nw/snd/spy/fnd/os/sndspyfnd_Thread.h>

namespace nw {
namespace snd {
namespace spy {
namespace internal {
namespace fnd {

//---------------------------------------------------------------------------
//! @brief  スレッドのメイン関数オブジェクトです。
//---------------------------------------------------------------------------
class Thread::ThreadMain
{
public:
    static int Run(int intArg, void* ptrArg)
    {
        Thread* owner = reinterpret_cast<Thread*>(ptrArg);

        NW_ASSERT_NOT_NULL(owner);
        NW_ASSERT_NOT_NULL(owner->m_Handler);

        owner->OnRun();

        u32 result = owner->m_Handler->Run(owner->m_Param);

        owner->OnExit();

        return result;
    }
};

//---------------------------------------------------------------------------
Thread::Thread() :
m_State(STATE_NOT_RUN),
m_ID(INVALID_ID),
m_Priority(DEFAULT_THREAD_PRIORITY),
m_Name(""),
m_Handler(NULL)
{
    m_Handle = RoundUpTo<Handle>(m_HandleBuffer, OSTHREAD_ALIGNMENT);
}

//---------------------------------------------------------------------------
bool
Thread::RunArgs::IsValid() const
{
    if(stack == NULL) { return false; }
    if(stackSize == 0) { return false; }
    if(priority > MAX_THREAD_PRIORITY) { return false; }
    if(handler == NULL) { return false; }
    return true;
}

//---------------------------------------------------------------------------
void
Thread::SetPriority(s32 value)
{
    OSSetThreadPriority(m_Handle, value);
}

//---------------------------------------------------------------------------
void
Thread::Sleep(const TimeSpan& timeSpan)
{
    OSSleepTicks(timeSpan.GetTick());
}

//---------------------------------------------------------------------------
bool
Thread::Create(Handle& handle, u32& id, const RunArgs& args)
{
    if(!OSCreateThread(
        handle,
        ThreadMain::Run,
        1,
        this,
        AddOffsetToPtr(args.stack, args.stackSize),
        args.stackSize,
        args.priority,
        0))
    {
        m_ID = INVALID_ID;
        return false;
    }

    // OSThread へのポインタを ID として利用します。
    m_ID = reinterpret_cast<u32>(&handle);

    return true;
}

//---------------------------------------------------------------------------
void
Thread::Detach()
{
    if(IsTerminated())
    {
        return;
    }

    OSDetachThread(m_Handle);
}

//---------------------------------------------------------------------------
void
Thread::SetName(const char* name)
{
    OSSetThreadName(m_Handle, name == NULL ? "" : name);
}

//---------------------------------------------------------------------------
void
Thread::SetAffinityMask(AFFINITY_MASK value)
{
    // OS_THREAD_ATTR_AFFINITY_CORE0 ～ OS_THREAD_ATTR_AFFINITY_CORE2 と同じ値
    OSSetThreadAffinity(m_Handle, static_cast<u16>(value));
}

//---------------------------------------------------------------------------
void
Thread::Resume()
{
    OSResumeThread(m_Handle);
}

//---------------------------------------------------------------------------
void
Thread::Join()
{
    // TODO : ★Cafe版で結果を返せるか検討
    OSJoinThread(m_Handle, NULL);
}

//---------------------------------------------------------------------------
bool
Thread::IsTerminated() const
{
    return OSIsThreadTerminated(m_Handle);
}

} // namespace nw::snd::spy::internal::fnd
} // namespace nw::snd::spy::internal
} // namespace nw::snd::spy
} // namespace nw::snd
} // namespace nw
