﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "../../../../precompiled.h"

#ifdef NW_SND_SPY_ENABLE

#include <nw/snd/spy/fnd/hio/sndspyfnd_HioChannel.h>

#include <cafe/os.h>
#include <cafe/hio.h>
#include <nw/snd/spy/fnd/string/sndspyfnd_String.h>

#if defined(NW_DEBUG)
#define STATUS_DEBUG_ENABLED
#endif

namespace nw {
namespace snd {
namespace spy {
namespace internal {
namespace fnd {

namespace {

static const HIOHandle INVALID_HANDLE = -1;

}

//----------------------------------------------------------
HioChannel::HioChannel() :
m_Handle(INVALID_HANDLE),
m_IsConnected(false),
m_ConnectEvent(true, true)
{
    m_Port[0] = '\0';
}

//----------------------------------------------------------
bool
HioChannel::Open(const PortType port)
{
    NW_ASSERT_NOT_NULL(port);
    NW_ASSERT(port[0] != '\0');

    if(IsOpened())
    {
        return true;
    }

    HIOHandle handle = HIOOpen(port, ConnectionCallback, this, HIO_CHANNEL_OPTION_READ_WRITE, 0);

    if(handle < 0)
    {
        NW_WARNING(false, "failed to HioOpen : port=%s.\n", port);
        return false;
    }

    String::Copy(m_Port, MAX_PORT_LENGTH, port);
    m_Handle = handle;

    m_ConnectEvent.Reset();

#ifdef STATUS_DEBUG_ENABLED
    NW_LOG(
        "[nw::snd::spy::internal::fnd::HioChannel] Open(%s) : isOpened=%d, isConnected=%d.\n",
        port,
        IsOpened(),
        IsConnected());
#endif

    return true;
}

//----------------------------------------------------------
void
HioChannel::Close()
{
    if(m_Handle != INVALID_HANDLE)
    {
        HIOClose(m_Handle);

        m_Handle = INVALID_HANDLE;
        m_Port[0] = '\0';
        m_IsConnected = false;

#ifdef STATUS_DEBUG_ENABLED
        NW_LOG(
            "[nw::snd::spy::internal::fnd::HioChannel] Close : isOpened=%d, isConnected=%d.\n",
            IsOpened(),
            IsConnected());
#endif

        m_ConnectEvent.Set();
    }
}

//----------------------------------------------------------
bool
HioChannel::Connect()
{
    while(IsOpened() && !m_IsConnected)
    {
        // コールバックを受け取るまで待機する
        m_ConnectEvent.Wait();
    }

    return m_IsConnected;
}

//----------------------------------------------------------
void
HioChannel::Disconnect()
{
    // 何もしない
}

//----------------------------------------------------------
bool
HioChannel::IsOpened() const
{
    return m_Port[0] != '\0';
}

//----------------------------------------------------------
bool
HioChannel::IsConnected() const
{
    return m_IsConnected;
}

//----------------------------------------------------------
HioChannel::ConstPortType
HioChannel::GetPort() const
{
    return reinterpret_cast<ConstPortType>(m_Port);
}

//----------------------------------------------------------
u32
HioChannel::Read(void* buffer, u32 length)
{
    NW_ASSERT_NOT_NULL(buffer);

    if(!IsOpened())
    {
        return 0;
    }

    if(length == 0)
    {
        return 0;
    }

    HIOStatus result = HIO_STATUS_OK;

    for(;;)
    {
        result = HIORead(m_Handle, length, buffer);

        if(result != HIO_STATUS_NO_CLIENT_TXN_BUF_AVAILABLE)
        {
            if (result < 0)
            {
                // 読み込みに失敗しました。
                return static_cast<u32>(-1);
            }

            break;
        }
    }

    return result;
}

//----------------------------------------------------------
u32
HioChannel::Write(const void* buffer, u32 length)
{
    NW_ASSERT_NOT_NULL(buffer);

    if(!IsOpened())
    {
        return 0;
    }

    if(length == 0)
    {
        return 0;
    }

    HIOStatus result = HIO_STATUS_OK;

    for(;;)
    {
        result = HIOWrite(m_Handle, length, buffer);

        if(result != HIO_STATUS_NO_CLIENT_TXN_BUF_AVAILABLE)
        {
            break;
        }
    }

    return result;
}

//----------------------------------------------------------
void
HioChannel::ConnectionCallback(HIOStatus status, void* context)
{
    HioChannel* target = reinterpret_cast<HioChannel*>(context);
    NW_ASSERT_NOT_NULL(target);

#ifdef STATUS_DEBUG_ENABLED
    NW_LOG("[nw::snd::spy::internal::fnd::HioChannel] ConnectionCallback : port=%s, status=%d.\n", target->m_Port, status);
#endif

    // 新たに接続された。
    if(status == HIO_STATUS_NEW_CONNECTION)
    {
        target->m_IsConnected = true;
        target->m_ConnectEvent.Set();

#ifdef STATUS_DEBUG_ENABLED
        NW_LOG(
            "[nw::snd::spy::internal::fnd::HioChannel] HIO_STATUS_NEW_CONNECTION : port=%s, isOpened=%d, isConnected=%d.\n",
            target->m_Port,
            target->IsOpened(),
            target->IsConnected());
#endif
        return;
    }

    // すべての接続が切断された。
    // Cafe の HIO では、複数の PC アプリが同一ポートに接続可能。
    if(status == HIO_STATUS_NO_CONNECTIONS )
    {
        target->m_IsConnected = false;

#ifdef STATUS_DEBUG_ENABLED
        NW_LOG(
            "[nw::snd::spy::internal::fnd::HioChannel] HIO_STATUS_NO_CONNECTIONS : port=%s, isOpened=%d, isConnected=%d.\n",
            target->m_Port,
            target->IsOpened(),
            target->IsConnected());
#endif

        target->Close();
        target->m_ConnectEvent.Set();
        return;
    }

    // エラー。
    if(status < 0)
    {
        target->m_IsConnected = false;

#ifdef STATUS_DEBUG_ENABLED
        NW_LOG(
            "[nw::snd::spy::internal::fnd::HioChannel] ConnectionCallback : port=%s, isOpened=%d, isConnected=%d.\n",
            target->m_Port,
            target->IsOpened(),
            target->IsConnected());
#endif
        return;
    }
}

} // namespace nw::snd::spy::internal::fnd
} // namespace nw::snd::spy::internal
} // namespace nw::snd::spy
} // namespace nw::snd
} // namespace nw

#endif // NW_SND_SPY_ENABLE
