﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nw/snd/edit/hio/sndedit_HioPacketStream.h>

#ifdef NW_SND_CONFIG_ENABLE_DEV

#include <nw/snd/edit/hio/sndedit_HioStream.h>

namespace nw {
namespace snd {
namespace edit {
namespace internal {

//----------------------------------------------------------
HioPacketStream::HioPacketStream() :
m_ReadState(READ_STATE_NOT_READ),
m_Stream(NULL),
m_ReadBodySize(0)
{
}

//----------------------------------------------------------
HioResult
HioPacketStream::Initialize(HioStream& stream, void* buffer, u32 bufferSize)
{
    if(IsInitialized())
    {
        NW_FATAL_ERROR("HioPacketStream is already initialized.\n");
        return HioResult(HIO_RESULT_TARGET_NOT_INITIALIZED);
    }

    if(buffer == NULL ||
        bufferSize < sizeof(HioPacket))
    {
        NW_FATAL_ERROR("invalid arguments.\n");
        return HioResult(HIO_RESULT_FAILED);
    }

    m_Stream = &stream;
    m_Buffer = reinterpret_cast<HioPacket*>(buffer);
    m_BufferSize = bufferSize;
    m_ReadState = READ_STATE_NOT_READ;

    return HioResult(HIO_RESULT_TRUE);
}

//----------------------------------------------------------
void
HioPacketStream::Finalize()
{
    m_Stream = NULL;
    m_Buffer = NULL;
    m_BufferSize = 0;
    m_ReadState = READ_STATE_NOT_READ;
}

//----------------------------------------------------------
bool
HioPacketStream::IsAvailable() const
{
    return IsInitialized() && m_Stream->IsAvailable();
}

//----------------------------------------------------------
u32
HioPacketStream::GetReadableBytes() const
{
    return m_Stream->GetReadableBytes();
}

//----------------------------------------------------------
const HioPacketHeader*
HioPacketStream::GetReadHeader() const
{
    if(!IsInitialized() ||
        m_ReadState == READ_STATE_NOT_READ)
    {
        NW_FATAL_ERROR("HioPacketStream has invalid state.\n");
        return NULL;
    }

    return &m_Buffer->GetHeader();
}

//----------------------------------------------------------
HioPacket*
HioPacketStream::GetReadingPacket() const
{
    if(!IsInitialized() ||
        m_ReadState == READ_STATE_NOT_READ)
    {
        NW_FATAL_ERROR("HioPacketStream has invalid state.\n");
        return NULL;
    }

#if defined(NW_SNDEDIT_HIO_PACKET_DEBUG_DUMP)
    m_Buffer->Dump();
#endif

    return m_Buffer;
}

//----------------------------------------------------------
const HioPacket*
HioPacketStream::GetReadPacket() const
{
    if(!IsInitialized() ||
        m_ReadState != READ_STATE_BODY_READ)
    {
        NW_FATAL_ERROR("HioPacketStream has invalid state.\n");
        return NULL;
    }

#if defined(NW_SNDEDIT_HIO_PACKET_DEBUG_DUMP)
    m_Buffer->Dump();
#endif

    return m_Buffer;
}

//----------------------------------------------------------
bool
HioPacketStream::CanReadNewPacket() const
{
    return m_ReadState == READ_STATE_NOT_READ || m_ReadState == READ_STATE_BODY_READ;
}

//----------------------------------------------------------
bool
HioPacketStream::CanReadBody() const
{
    return m_ReadState == READ_STATE_HEADER_READ || m_ReadState == READ_STATE_BODY_READING;
}

//----------------------------------------------------------
HioResult
HioPacketStream::TryReadPacket()
{
    if(!IsInitialized())
    {
        NW_FATAL_ERROR("HioPacketStream is not initialized.\n");
        return HioResult(HIO_RESULT_TARGET_NOT_INITIALIZED);
    }

    // 中途半端な手動読み込み中は失敗させます。
    if(m_ReadState == READ_STATE_BODY_READING)
    {
        NW_FATAL_ERROR("HioPacketStream has invalid state.\n");
        return HioResult(HIO_RESULT_FAILED);
    }

    // メッセージヘッダを受信します。
    if(CanReadNewPacket())
    {
        HioResult result = TryReadHeader();

        if(!result.IsTrue())
        {
            return result;
        }
    }

    const HioPacketHeader* header = GetReadHeader();
    NW_ASSERT_NOT_NULL(header);

    // メッセージボディを受信しきれない場合は処理しません。
    if(m_Stream->GetReadableBytes() < header->bodySize)
    {
        return HioResult(HIO_RESULT_FALSE);
    }

    // メモリ不足の場合は、そのメッセージ全体をスキップします。
    if(m_BufferSize < sizeof(HioPacketHeader) + header->bodySize)
    {
        SkipBody();
        return HioResult(HIO_RESULT_TARGET_OUT_OF_MEMORY);
    }

    // メッセージボディを受信します。
    return TryReadBody();
}

//----------------------------------------------------------
HioResult
HioPacketStream::TryReadHeader()
{
    if(!IsInitialized())
    {
        NW_FATAL_ERROR("HioPacketStream is not initialized.\n");
        return HioResult(HIO_RESULT_TARGET_NOT_INITIALIZED);
    }

    if(!CanReadNewPacket())
    {
        NW_FATAL_ERROR("HioPacketStream has invalid state.\n");
        return HioResult(HIO_RESULT_FAILED);
    }

    m_ReadState = READ_STATE_NOT_READ;
    m_ReadBodySize = 0;

    // 受信しきれない場合は処理しません。
    if(m_Stream->GetReadableBytes() < sizeof(HioPacketHeader))
    {
        return HioResult(HIO_RESULT_FALSE);
    }

    s32 readSize = m_Stream->Read(m_Buffer, sizeof(HioPacketHeader));

    if(readSize < static_cast<s32>(sizeof(HioPacketHeader)))
    {
        return HioResult(HIO_RESULT_TARGET_RECIEVE_ERROR);
    }

    if(!m_Buffer->GetHeader().IsValid())
    {
        NW_WARNING(false, "[nw::snd::edit::internal::HioPacketStream] receive invalid packet header.\n");
        return HioResult(HIO_RESULT_TARGET_RECIEVE_ERROR);
    }

    m_ReadState = READ_STATE_HEADER_READ;

    return HioResult(HIO_RESULT_TRUE);
}

//----------------------------------------------------------
HioResult
HioPacketStream::TryReadBody()
{
    if(!IsInitialized())
    {
        NW_FATAL_ERROR("HioPacketStream is not initialized.\n");
        return HioResult(HIO_RESULT_TARGET_NOT_INITIALIZED);
    }

    // ReadBodyPart() による読み込み途中の場合は失敗させます。
    if(m_ReadState != READ_STATE_HEADER_READ)
    {
        NW_FATAL_ERROR("HioPacketStream has invalid state.\n");
        return HioResult(HIO_RESULT_FAILED);
    }

    const HioPacketHeader* header = GetReadHeader();
    NW_ASSERT_NOT_NULL(header);

    NW_ASSERT(header->bodySize >= m_ReadBodySize);
    u32 restBodySize = header->bodySize - m_ReadBodySize;

    // 受信しきれない場合は処理しません。
    if(m_Stream->GetReadableBytes() < restBodySize)
    {
        return HioResult(HIO_RESULT_FALSE);
    }

    // メモリ不足の場合は処理しません。
    if(m_BufferSize < sizeof(HioPacketHeader) + header->bodySize)
    {
        SkipBody();
        return HioResult(HIO_RESULT_TARGET_OUT_OF_MEMORY);
    }

    // メッセージボディを受信します。
    if(restBodySize > 0)
    {
        s32 readSize = m_Stream->Read(
            ut::AddOffsetToPtr(m_Buffer->GetBody(), m_ReadBodySize),
            restBodySize);

        if(static_cast<u32>(readSize) != restBodySize)
        {
            return HioResult(HIO_RESULT_TARGET_RECIEVE_ERROR);
        }
    }

    m_ReadBodySize = header->bodySize;
    m_ReadState = READ_STATE_BODY_READ;

    return HioResult(HIO_RESULT_TRUE);
}

//----------------------------------------------------------
HioResult
HioPacketStream::TryReadBodyPart(void* buffer, u32 readLength)
{
    if(GetReadableBytes() < readLength)
    {
        return HioResult(HIO_RESULT_FALSE);
    }

    return ReadBodyPart(buffer, readLength);
}

//----------------------------------------------------------
HioResult
HioPacketStream::ReadBodyPart(void* buffer, u32 readLength)
{
    NW_ASSERT_NOT_NULL(buffer);

    if(!IsInitialized())
    {
        NW_FATAL_ERROR("HioPacketStream is not initialized.\n");
        return HioResult(HIO_RESULT_TARGET_NOT_INITIALIZED);
    }

    if(!CanReadBody())
    {
        NW_FATAL_ERROR("HioPacketStream has invalid state.\n");
        return HioResult(HIO_RESULT_FAILED);
    }

    if(readLength == 0)
    {
        return HioResult(HIO_RESULT_FALSE);
    }

    const HioPacketHeader* header = GetReadHeader();
    NW_ASSERT_NOT_NULL(header);

    NW_ASSERT(header->bodySize > m_ReadBodySize);
    u32 restBodySize = header->bodySize - m_ReadBodySize;

    // パケットサイズを超える場合は受信しません。
    if(readLength > restBodySize)
    {
        return HioResult(HIO_RESULT_FAILED);
    }

    // メッセージボディを受信します。
    s32 readSize = m_Stream->Read(buffer, readLength);

    if(static_cast<u32>(readSize) != readLength)
    {
        return HioResult(HIO_RESULT_TARGET_RECIEVE_ERROR);
    }

    m_ReadBodySize += readLength;

    if(m_ReadBodySize == header->bodySize)
    {
        m_ReadState = READ_STATE_BODY_READ;
    }

    return HioResult(HIO_RESULT_TRUE);
}

//----------------------------------------------------------
HioResult
HioPacketStream::SkipHeader()
{
    if(!IsInitialized())
    {
        NW_FATAL_ERROR("HioPacketStream is not initialized.\n");
        return HioResult(HIO_RESULT_TARGET_NOT_INITIALIZED);
    }

    // パケット読み込みが完了していない場合は失敗させます。
    switch(m_ReadState)
    {
    case READ_STATE_NOT_READ:
    case READ_STATE_BODY_READ:
        break;

    default:
        NW_FATAL_ERROR("HioPacketStream has invalid state.\n");
        return HioResult(HIO_RESULT_FAILED);
    }

    s32 skipedSize = m_Stream->Skip(sizeof(HioPacketHeader));

    if(static_cast<u32>(skipedSize) != sizeof(HioPacketHeader))
    {
        return HioResult(HIO_RESULT_TARGET_RECIEVE_ERROR);
    }

    return HioResult(HIO_RESULT_TRUE);
}

//----------------------------------------------------------
HioResult
HioPacketStream::SkipBody()
{
    if(!IsInitialized())
    {
        NW_FATAL_ERROR("HioPacketStream is not initialized.\n");
        return HioResult(HIO_RESULT_TARGET_NOT_INITIALIZED);
    }

    // ヘッダ読み込みが完了していない場合は失敗させます。
    switch(m_ReadState)
    {
    case READ_STATE_HEADER_READ:
    case READ_STATE_BODY_READING:
    case READ_STATE_BODY_READ:
        break;

    default:
        NW_FATAL_ERROR("HioPacketStream has invalid state.\n");
        return HioResult(HIO_RESULT_FAILED);
    }

    const HioPacketHeader* header = GetReadHeader();
    NW_NULL_ASSERT(header);
    NW_ASSERT(header->bodySize >= m_ReadBodySize);
    u32 skipSize = header->bodySize - m_ReadBodySize;
    s32 skipedSize = m_Stream->Skip(skipSize);

    m_ReadBodySize += skipedSize;

    if(static_cast<u32>(skipedSize) != skipSize)
    {
        m_ReadState = READ_STATE_ERROR;
        return HioResult(HIO_RESULT_TARGET_RECIEVE_ERROR);
    }

    m_ReadState = READ_STATE_BODY_READ;
    return HioResult(HIO_RESULT_TRUE);
}

//----------------------------------------------------------
HioResult
HioPacketStream::SkipBodyPart(u32 length)
{
    if(!IsInitialized())
    {
        NW_FATAL_ERROR("HioPacketStream is not initialized.\n");
        return HioResult(HIO_RESULT_TARGET_NOT_INITIALIZED);
    }

    // ボディ読み込み中でない場合は失敗させます。
    switch(m_ReadState)
    {
    case READ_STATE_HEADER_READ:
    case READ_STATE_BODY_READING:
        break;

    default:
        NW_FATAL_ERROR("HioPacketStream has invalid state.\n");
        return HioResult(HIO_RESULT_FAILED);
    }

    const HioPacketHeader* header = GetReadHeader();
    NW_NULL_ASSERT(header);
    NW_ASSERT(header->bodySize >= m_ReadBodySize + length);
    s32 skipedSize = m_Stream->Skip(length);

    m_ReadBodySize += skipedSize;

    if(static_cast<u32>(skipedSize) != length)
    {
        m_ReadState = READ_STATE_ERROR;
        return HioResult(HIO_RESULT_TARGET_RECIEVE_ERROR);
    }

    if(m_ReadBodySize == header->bodySize)
    {
        m_ReadState = READ_STATE_BODY_READ;
    }

    return HioResult(HIO_RESULT_TRUE);
}


//----------------------------------------------------------
HioResult
HioPacketStream::WritePacket(const HioPacket& packet)
{
    if(!IsInitialized())
    {
        NW_FATAL_ERROR("HioPacketStream is not initialized.\n");
        return HioResult(HIO_RESULT_TARGET_NOT_INITIALIZED);
    }

    u32 packetSize = packet.GetSize();

    if(m_Stream->Write(&packet, packetSize) != static_cast<s32>(packetSize))
    {
        return HioResult(HIO_RESULT_TARGET_SEND_ERROR);
    }

    return HioResult(HIO_RESULT_TRUE);
}

//----------------------------------------------------------
void
HioPacketStream::ClearBuffer()
{
    m_ReadState = READ_STATE_NOT_READ;
    m_ReadBodySize = 0;
}

} // namespace nw::snd::edit::internal
} // namespace nw::snd::edit
} // namespace nw::snd
} // namespace nw

#endif // NW_SND_CONFIG_ENABLE_DEV
