﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nw/snd.h>
#include <nw/snd/dw/snddw_PerformancePanel.h>

#include <nw/dw/system/dw_InputSettings.h>

namespace nw {
namespace snd {
namespace dw {

const char* PerformancePanel::SOUNDTHREAD_CPU_LABEL     = "CPU  ";
const char* PerformancePanel::SOUNDTHREAD_SND_LABEL     = " SND ";
const char* PerformancePanel::SOUNDTHREAD_SNDCB_LABEL   = "  CB ";
const char* PerformancePanel::SOUNDTHREAD_SNDTH_LABEL   = "  Th.";
const char* PerformancePanel::SOUNDTHREAD_AUX_LABEL     = "  AUX";
const char* PerformancePanel::SOUNDTHREAD_AX_LABEL      = " AX  ";
const char* PerformancePanel::SOUNDTHREAD_DSP_LABEL     = "DSP  ";
const char* PerformancePanel::SOUNDTHREAD_SNDINT_LABEL  = "ThInt";
const char* PerformancePanel::SOUNDTHREAD_AXINT_LABEL   = "CBInt";

PerformancePanel::PerformancePanel() :
m_IsVideoFrameProfile(true)
#if defined(NW_PLATFORM_CAFE)
, m_PrevAxFrameProcessBegin(0)
, m_PrevNwFrameProcessBegin(0)
#endif
{
    SetMargin(nw::internal::dw::Thickness(0.f));
    SetIsFocusable(true);

    m_Contents.AddItem(&m_Container);
    SetContents(m_Contents);

    m_Container.SetMeasurement(nw::internal::dw::MEASUREMENT_AUTO_HEIGHT);
    m_Container.SetOrientation(nw::internal::dw::VERTICAL);

    nw::internal::dw::UIElementList& containerItems = m_Container.GetContents();

    // パフォーマンスメーターを初期化します。
    for(s32 i=0; i<PERFORMANCE_BAR_COUNT; ++i)
    {
        internal::FixedLabelPeakMeter<LABEL_LENGTH>& performanceMeter = m_PerformanceMeters[i];

        performanceMeter.SetMargin(nw::internal::dw::Thickness(3.f, 1.f));
        performanceMeter.SetMeasurement(nw::internal::dw::MEASUREMENT_AUTO_HEIGHT);
        performanceMeter.SetOrientation(nw::internal::dw::HORIZONTAL);
        performanceMeter.SetMaximumValue(100.f);
        performanceMeter.SetPeakHoldChange(1.f);

        containerItems.AddItem(&performanceMeter);
    }

    m_PerformanceMeters[SOUNDTHREAD_CPU_INDEX].SetLabelText(SOUNDTHREAD_CPU_LABEL);
    m_PerformanceMeters[SOUNDTHREAD_SND_INDEX].SetLabelText(SOUNDTHREAD_SND_LABEL);
    m_PerformanceMeters[SOUNDTHREAD_SNDCB_INDEX].SetLabelText(SOUNDTHREAD_SNDCB_LABEL);
    m_PerformanceMeters[SOUNDTHREAD_SNDTH_INDEX].SetLabelText(SOUNDTHREAD_SNDTH_LABEL);
    m_PerformanceMeters[SOUNDTHREAD_AUX_INDEX].SetLabelText(SOUNDTHREAD_AUX_LABEL);
    m_PerformanceMeters[SOUNDTHREAD_AX_INDEX].SetLabelText(SOUNDTHREAD_AX_LABEL);
    m_PerformanceMeters[SOUNDTHREAD_DSP_INDEX].SetLabelText(SOUNDTHREAD_DSP_LABEL);
    m_PerformanceMeters[SOUNDTHREAD_SNDINT_INDEX].SetLabelText(SOUNDTHREAD_SNDINT_LABEL);
    m_PerformanceMeters[SOUNDTHREAD_AXINT_INDEX].SetLabelText(SOUNDTHREAD_AXINT_LABEL);

#if defined(NW_PLATFORM_CAFE)
    nw::snd::SoundSystem::RegisterProfileReader(m_ProfileReader);
#endif
}

bool PerformancePanel::OnUpdateFocusedInput(const nw::internal::dw::Inputs& inputs)
{
    if(inputs.GetPad() == NULL)
    {
        return false;
    }

    const nw::dev::Pad& pad = *inputs.GetPad();


    if (IsPressWindowControlKey(pad))
    {
        // プロファイリングモードの変更
        if(pad.IsTrig(nw::dev::Pad::MASK_LEFT) ||
            pad.IsTrig(nw::dev::Pad::MASK_RIGHT))
        {
            SetProfileMode(!m_IsVideoFrameProfile);
            return true;
        }
    }

    return false;
}

void PerformancePanel::OnUpdate(const nw::internal::dw::UIElementTreeContext& context)
{
    (void)context;
#if defined(NW_PLATFORM_CAFE)

    static nw::snd::SoundProfile profile[nw::snd::internal::MAX_PROFILE_COUNT];

    u32 profileCount = m_ProfileReader.Read(profile, nw::snd::internal::MAX_PROFILE_COUNT);

    // ビデオフレームでの処理負荷を表示
    if ( m_IsVideoFrameProfile )
    {
        UpdateVideoFrameProfiles(profile, profileCount);
    }
    // オーディオフレーム (3ms) 単位での処理負荷を表示
    else
    {
        UpdateAudioFrameProfiles(profile, profileCount);
    }

#endif
}

bool PerformancePanel::IsPressWindowControlKey(const nw::dev::Pad& pad) const
{
    return pad.IsHoldAll( nw::internal::dw::InputSettings::GetInstance().GetWindowManagerSetting().GetWindowControlModifier() );
}

#if defined(NW_PLATFORM_CAFE)
void PerformancePanel::UpdateVideoFrameProfiles(const nw::snd::SoundProfile* pProfile, u32 profileCount)
{
    NW_NULL_ASSERT(pProfile);

    ProfileData sumProf;

    for ( u32 i = 0; i < profileCount; i++ )
    {
        const nw::snd::SoundProfile& prof = pProfile[i];

        // ビデオフレーム単位で表示
        sumProf.axFrame          += ut::TimeSpan::FromNanoSeconds(OSTicksToNanoseconds(prof.axFrameProcessTick));
        sumProf.axFrameInterval  += prof.GetAxFrameProcessInterval(m_PrevAxFrameProcessBegin);
        sumProf.auxProcessing    += prof.auxProcess.GetSpan();
        sumProf.userCallback     += prof.nwVoiceParamUpdate.GetSpan();
        sumProf.sndFrame         += ut::TimeSpan::FromNanoSeconds(OSTicksToNanoseconds(prof.nwFrameProcessTick));
        sumProf.sndFrameInterval += prof.GetNwFrameProcessInterval(m_PrevNwFrameProcessBegin);
        sumProf.dspFrame         += prof.dspFrameProcess.GetSpan();

        m_PrevAxFrameProcessBegin = prof.axFrameProcess.begin;
        m_PrevNwFrameProcessBegin = prof.nwFrameProcess.begin;
    }

    // 積算値を表示
    {
        const f32 SUM_INTERVAL_USEC =
            static_cast<f32>(nw::snd::SoundSystem::SOUND_THREAD_INTERVAL_USEC * profileCount);

        f32 usec = static_cast<f32>((sumProf.axFrame + sumProf.sndFrame ).GetMicroSeconds());
        SetMeter( SOUNDTHREAD_CPU_INDEX,   usec / SUM_INTERVAL_USEC );

        usec = static_cast<f32>((sumProf.sndFrame + sumProf.userCallback + sumProf.auxProcessing ).GetMicroSeconds());
        SetMeter( SOUNDTHREAD_SND_INDEX,   usec / SUM_INTERVAL_USEC );

        usec = static_cast<f32>(sumProf.userCallback.GetMicroSeconds());
        SetMeter( SOUNDTHREAD_SNDCB_INDEX, usec / SUM_INTERVAL_USEC );

        usec = static_cast<f32>(sumProf.sndFrame.GetMicroSeconds());
        SetMeter( SOUNDTHREAD_SNDTH_INDEX, usec / SUM_INTERVAL_USEC );

        usec = static_cast<f32>(sumProf.auxProcessing.GetMicroSeconds());
        SetMeter( SOUNDTHREAD_AUX_INDEX,   usec / SUM_INTERVAL_USEC);

        usec = static_cast<f32>((sumProf.axFrame - sumProf.userCallback - sumProf.auxProcessing).GetMicroSeconds());
        SetMeter( SOUNDTHREAD_AX_INDEX,    usec / SUM_INTERVAL_USEC);

// HACK : CafeSDK 2.05 では正しい数値が採れないので、無効化しています。
#if 0
        usec = static_cast<f32>(sumProf.sndFrameInterval.GetMicroSeconds());
        SetMeter( SOUNDTHREAD_SNDINT_INDEX,
            ut::Abs( usec - SUM_INTERVAL_USEC ) / SUM_INTERVAL_USEC );
#endif

        usec = static_cast<f32>(sumProf.axFrameInterval.GetMicroSeconds());
        SetMeter( SOUNDTHREAD_AXINT_INDEX,
            ut::Abs( usec - SUM_INTERVAL_USEC ) / SUM_INTERVAL_USEC );

        usec = static_cast<f32>(sumProf.dspFrame.GetMicroSeconds());
        SetMeter( SOUNDTHREAD_DSP_INDEX, usec / SUM_INTERVAL_USEC );
    }
}

void PerformancePanel::UpdateAudioFrameProfiles(const nw::snd::SoundProfile* pProfile, u32 profileCount)
{
    NW_NULL_ASSERT(pProfile);

    for ( u32 i=0; i<profileCount; i++ )
    {
        const nw::snd::SoundProfile& prof = pProfile[i];

        ProfileData sumProf;

        sumProf.axFrame          = ut::TimeSpan::FromNanoSeconds(OSTicksToNanoseconds(prof.axFrameProcessTick));
        sumProf.axFrameInterval  = prof.GetAxFrameProcessInterval(m_PrevAxFrameProcessBegin);
        sumProf.auxProcessing    = prof.auxProcess.GetSpan();
        sumProf.userCallback     = prof.nwVoiceParamUpdate.GetSpan();
        sumProf.sndFrame         = ut::TimeSpan::FromNanoSeconds(OSTicksToNanoseconds(prof.nwFrameProcessTick));
        sumProf.sndFrameInterval = prof.GetNwFrameProcessInterval(m_PrevNwFrameProcessBegin);
        sumProf.dspFrame         = prof.dspFrameProcess.GetSpan();

        m_PrevAxFrameProcessBegin = prof.axFrameProcess.begin;
        m_PrevNwFrameProcessBegin = prof.nwFrameProcess.begin;

        f32 usec = static_cast<f32>(( sumProf.axFrame + sumProf.sndFrame ).GetMicroSeconds());
        SetMeter(
            SOUNDTHREAD_CPU_INDEX,
            usec / nw::snd::SoundSystem::SOUND_THREAD_INTERVAL_USEC
            );

        usec = static_cast<f32>((sumProf.sndFrame + sumProf.userCallback + sumProf.auxProcessing ).GetMicroSeconds());
        SetMeter(
            SOUNDTHREAD_SND_INDEX,
            usec / nw::snd::SoundSystem::SOUND_THREAD_INTERVAL_USEC
            );

        usec = static_cast<f32>(sumProf.userCallback.GetMicroSeconds());
        SetMeter(
            SOUNDTHREAD_SNDCB_INDEX,
            usec / nw::snd::SoundSystem::SOUND_THREAD_INTERVAL_USEC
            );

        usec = static_cast<f32>(sumProf.sndFrame.GetMicroSeconds());
        SetMeter(
            SOUNDTHREAD_SNDTH_INDEX,
            usec / nw::snd::SoundSystem::SOUND_THREAD_INTERVAL_USEC
            );

        usec = static_cast<f32>(sumProf.auxProcessing.GetMicroSeconds());
        SetMeter(
            SOUNDTHREAD_AUX_INDEX,
            usec / nw::snd::SoundSystem::SOUND_THREAD_INTERVAL_USEC
            );

        usec = static_cast<f32>((sumProf.axFrame - sumProf.userCallback - sumProf.auxProcessing).GetMicroSeconds());
        SetMeter(
            SOUNDTHREAD_AX_INDEX,
            usec / nw::snd::SoundSystem::SOUND_THREAD_INTERVAL_USEC
            );

        usec = static_cast<f32>(sumProf.dspFrame.GetMicroSeconds());
        SetMeter(
            SOUNDTHREAD_DSP_INDEX,
            usec / nw::snd::SoundSystem::SOUND_THREAD_INTERVAL_USEC
            );

// HACK : CafeSDK 2.05 では正しい数値が採れないので、無効化しています。
#if 0
        usec = static_cast<f32>(sumProf.sndFrameInterval.GetMicroSeconds());
        SetMeter(
            SOUNDTHREAD_SNDINT_INDEX,
            ut::Abs( usec - nw::snd::SoundSystem::SOUND_THREAD_INTERVAL_USEC ) /
            nw::snd::SoundSystem::SOUND_THREAD_INTERVAL_USEC
            );
#endif

        usec = static_cast<f32>(sumProf.axFrameInterval.GetMicroSeconds());
        SetMeter(
            SOUNDTHREAD_AXINT_INDEX,
            ut::Abs( usec - nw::snd::SoundSystem::SOUND_THREAD_INTERVAL_USEC ) /
            nw::snd::SoundSystem::SOUND_THREAD_INTERVAL_USEC
            );
    }
}
#endif

void PerformancePanel::SetMeter(int index, f32 value)
{
    m_PerformanceMeters[index].SetValue(value * 100.f);
}

} // dw
} // snd
} // nw
