﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nw/snd/dw/snddw_MasterOutputPanel.h>

#include <nw/dw/system/dw_InputSettings.h>

#if defined( NW_PLATFORM_WIN32 )
using namespace nw::internal::winext;
#endif

namespace nw {
namespace snd {
namespace dw {

const f32 MasterOutputPanel::MAX_SCALE_X = 4.f;
const f32 MasterOutputPanel::MIN_SCALE_X = 1.f / 32.f;

MasterOutputPanel* MasterOutputPanel::s_pThis = NULL;

MasterOutputPanel::MasterOutputPanel() :
m_SampleCount(0),
m_CurrentSampleIndex(0),
m_DrcCurrentSampleIndex(0),
m_CurrentOutputMode(nw::snd::OUTPUT_MODE_STEREO),
m_IsWaveGraphPaused(false),
m_IsMonitoring(false)
{
    SetMargin(nw::internal::dw::Thickness(0.f));
    SetIsFocusable(true);

    m_Contents.AddItem(&m_Container);
    SetContents(m_Contents);

    nw::internal::dw::UIElementList& containerItems = m_Container.GetContents();

    for(u32 i = 0; i < MAX_CHANNEL_COUNT; ++i)
    {
        containerItems.AddItem(&m_Channels[i]);

        if(i < MAX_CHANNEL_COUNT - 1)
        {
            containerItems.AddItem(&m_Separators[i]);
        }
    }

    for(u32 i = 0; i < MAX_CHANNEL_COUNT - 1; ++i)
    {
        m_Separators[i].SetMargin(nw::internal::dw::Thickness(2.f, 1.f));
        m_Separators[i].SetMeasurement(nw::internal::dw::MEASUREMENT_MANUAL);
        m_Separators[i].SetHeight(0.5f);
        m_Separators[i].SetOrientation(nw::internal::dw::HORIZONTAL);
        m_Separators[i].SetBorderColor(nw::ut::Color4f(0.455f, 0.486f, 0.560f, 1.f));
    }

    m_SampleCount = sizeof(m_Samples[0]) / sizeof(s16);

    for(u32 i = 0; i < MAX_CHANNEL_COUNT; ++i)
    {
        memset(m_Samples[i], 0, sizeof(m_Samples[i]));
        m_WaveGraphs[i].SetSamplesPerSec(SAMPLES_PER_SEC);
        m_WaveGraphs[i].SetWaveData(m_Samples[i], m_SampleCount);
        m_WaveGraphs[i].SetWaveCacheBuffer(m_GraphCacheBuffers[i], GRAPH_CACHE_BUFFER_SIZE);

        m_WaveGraphs[i].SetMargin(nw::internal::dw::Thickness(1.f, 1.f, 1.f, 0.f));
        m_WaveGraphs[i].SetMeasurement(nw::internal::dw::MEASUREMENT_MANUAL);
        m_WaveGraphs[i].SetHorizontalAlignment(nw::internal::dw::HORIZONTAL_STRETCH);
        m_WaveGraphs[i].SetHeight(30);

        m_ChannelLabels[i].SetMeasurement(nw::internal::dw::MEASUREMENT_AUTO);
        m_ChannelLabels[i].SetHorizontalAlignment(nw::internal::dw::HORIZONTAL_LEFT);
        m_ChannelLabels[i].SetVerticalAlignment(nw::internal::dw::VERTICAL_TOP);
        m_ChannelLabels[i].SetBackgroundColor(nw::ut::Color4f::X_DIM_GRAY());
        m_ChannelLabels[i].SetTextScale(0.7f);

        m_Channels[i].SetMargin(nw::internal::dw::Thickness(0.f));
        m_Channels[i].SetPadding(nw::internal::dw::Thickness(0.f));
        m_Channels[i].SetMeasurement(nw::internal::dw::MEASUREMENT_MANUAL);
        m_Channels[i].SetHorizontalAlignment(nw::internal::dw::HORIZONTAL_STRETCH);
        m_Channels[i].SetHeight(30);

        m_ChannelContentsArray[i].AddItem(&m_WaveGraphs[i]);
        m_ChannelContentsArray[i].AddItem(&m_ChannelLabels[i]);
        m_Channels[i].SetContents(m_ChannelContentsArray[i]);
    }

    m_ChannelLabels[MAIN_FL_INDEX].SetText("Main FL");
    m_ChannelLabels[MAIN_FR_INDEX].SetText("Main FR");
    m_ChannelLabels[MAIN_FC_INDEX].SetText("Main FC");
    m_ChannelLabels[MAIN_LFE_INDEX].SetText("Main LFE");
    m_ChannelLabels[MAIN_RL_INDEX].SetText("Main RL");
    m_ChannelLabels[MAIN_RR_INDEX].SetText("Main RR");
    m_ChannelLabels[DRC_FL_INDEX].SetText("DRC FL");
    m_ChannelLabels[DRC_FR_INDEX].SetText("DRC FR");
    m_ChannelLabels[DRC_RL_INDEX].SetText("DRC RL");
    m_ChannelLabels[DRC_RR_INDEX].SetText("DRC RR");

    // HACK : MainC, MainLfe は SDK未対応のため、グレー表示しています。
    m_ChannelLabels[MAIN_FC_INDEX].SetTextColor(nw::ut::Color4f::X_GRAY());
    m_ChannelLabels[MAIN_LFE_INDEX].SetTextColor(nw::ut::Color4f::X_GRAY());

    // HACK : 現状は２インスタンス以上の生成をサポートしていません。
    NW_ASSERT(s_pThis == NULL);

    if(s_pThis == NULL)
    {
        s_pThis = this;
    }
}

void MasterOutputPanel::StartMonitoring()
{
    if (!m_IsMonitoring)
    {
        m_IsMonitoring = true;
        snd::internal::driver::HardwareManager::GetInstance().AppendReadOnlyFinalMixCallback(s_pThis);
    }
}

void MasterOutputPanel::StopMonitoring()
{
    if (m_IsMonitoring)
    {
        snd::internal::driver::HardwareManager::GetInstance().EraseReadOnlyFinalMixCallback(s_pThis);
        m_IsMonitoring = false;
    }
}

void
MasterOutputPanel::Initialize( void* memory, u32 memorySize )
{
    m_DrawBufferAllocator.Initialize(memory, memorySize);

    for(u32 i = 0; i < MAX_CHANNEL_COUNT; ++i)
    {
        m_WaveGraphs[i].Initialize(m_DrawBufferAllocator);
    }
}

void
MasterOutputPanel::Finalize()
{
    for(u32 i = 0; i < MAX_CHANNEL_COUNT; ++i)
    {
        m_WaveGraphs[i].Finalize();
    }

    m_DrawBufferAllocator.Finalize();
}

u32
MasterOutputPanel::GetRequiredSize() const
{
    u32 requiredSize = 0;

    for(u32 i = 0; i < MAX_CHANNEL_COUNT; ++i)
    {
        requiredSize += m_WaveGraphs[i].GetRequiredDrawBufferSize();
    }

    return requiredSize;
}

MasterOutputPanel::~MasterOutputPanel()
{
    if(s_pThis != NULL)
    {
        StopMonitoring();
        s_pThis = NULL;
    }
}

f32 MasterOutputPanel::GetScaleX() const
{
    return m_WaveGraphs[0].GetScaleX();
}

void MasterOutputPanel::SetScaleX(f32 value)
{
    for(u32 i = 0; i < MAX_CHANNEL_COUNT; ++i)
    {
        m_WaveGraphs[i].SetScaleX(value);
    }
}

void MasterOutputPanel::Pause()
{
    m_IsWaveGraphPaused = true;
}

void MasterOutputPanel::Resume()
{
    if(!m_IsWaveGraphPaused)
    {
        return;
    }

    ClearSamples();
    m_IsWaveGraphPaused = false;
}

bool MasterOutputPanel::IsPaused() const
{
    return m_IsWaveGraphPaused;
}

bool MasterOutputPanel::OnUpdateFocusedInput(const nw::internal::dw::Inputs& inputs)
{
    if(inputs.GetPad() == NULL)
    {
        return false;
    }

    const nw::dev::Pad& pad = *inputs.GetPad();

    if (IsPressWindowControlKey(pad))
    {
        // 波形表示スケール * 2
        if(pad.IsTrig(nw::dev::Pad::MASK_RIGHT))
        {
            f32 scaleX = GetScaleX() * 2;

            if(scaleX > MAX_SCALE_X)
            {
                scaleX = MIN_SCALE_X;
            }

            SetScaleX(scaleX);
            return true;
        }

        // 波形表示スケール / 2
        if(pad.IsTrig(nw::dev::Pad::MASK_LEFT))
        {
            f32 scaleX = GetScaleX() / 2;

            if(scaleX < MIN_SCALE_X)
            {
                scaleX = MAX_SCALE_X;
            }

            SetScaleX(scaleX);
            return true;
        }

        // 波形表示の一時停止、再開
        if(pad.IsTrig(nw::dev::Pad::MASK_START))
        {
            if(IsPaused())
            {
                Resume();
            }
            else
            {
                Pause();
            }

            return true;
        }
    }

    return false;
}

void MasterOutputPanel::OnUpdate(const nw::internal::dw::UIElementTreeContext& context)
{
    (void)context;
    nw::snd::OutputMode outputMode = nw::snd::SoundSystem::GetOutputMode();

    if(m_CurrentOutputMode != outputMode)
    {
        m_CurrentOutputMode = outputMode;

        // 波形グラフのサンプリングレートを調整します。
        for(u32 i = MAIN_CHANNEL_START; i < MAIN_CHANNEL_START + MAX_MAIN_CHANNEL_COUNT; ++i)
        {
            m_WaveGraphs[i].SetSamplesPerSec(SAMPLES_PER_SEC);
        }

        // Main FC, Main LFE, Main RL, MainRR の波形バッファをクリアします。
        memset(m_Samples[MAIN_FC_INDEX], 0, sizeof(m_Samples[MAIN_FC_INDEX]));
        memset(m_Samples[MAIN_LFE_INDEX], 0, sizeof(m_Samples[MAIN_LFE_INDEX]));
        memset(m_Samples[MAIN_RL_INDEX], 0, sizeof(m_Samples[MAIN_RL_INDEX]));
        memset(m_Samples[MAIN_RR_INDEX], 0, sizeof(m_Samples[MAIN_RR_INDEX]));
    }
}

void MasterOutputPanel::OnFinalMix(OutputDevice device, const FinalMixData* data)
{
    NW_NULL_ASSERT(s_pThis);
    NW_NULL_ASSERT(data);
    if (device == OUTPUT_DEVICE_MAIN)
    {
        s_pThis->UpdateWaveGraph(data);
    }
    else if (device == OUTPUT_DEVICE_DRC0)
    {
        s_pThis->UpdateDrcWaveGraph(data);
    }
}

s16* MasterOutputPanel::GetSamples(u32 channel)
{
    NW_ASSERT(channel < MAX_CHANNEL_COUNT);
    return m_Samples[channel];
}

void MasterOutputPanel::UpdateWaveGraph(const FinalMixData* data)
{
    if(m_IsWaveGraphPaused)
    {
        return;
    }

    bool isSurroundEnabled =
        nw::snd::SoundSystem::GetOutputMode() == nw::snd::OUTPUT_MODE_SURROUND;

#ifdef NW_PLATFORM_WIN32
    // PC版では、サラウンドモードに対応していません。
    isSurroundEnabled = false;
#endif

    const u32 deviceIndex = 0;
    u32 restCopySamples = data[deviceIndex].sampleCount;
    u32 sampleOffset = 0;

    while(restCopySamples > 0)
    {
        const u32 copySamples = nw::ut::Min(restCopySamples, m_SampleCount - m_CurrentSampleIndex);

        for(u32 i = 0; i < copySamples ; ++i)
        {
            u32 srcOffset = sampleOffset + i;
            u32 destOffset = m_CurrentSampleIndex + i;

            if (data[deviceIndex].frontLeft != NULL)
            {
                GetSamples(MAIN_FL_INDEX)[destOffset] = static_cast<s16>(data[deviceIndex].frontLeft[srcOffset]);
            }
            if (data[deviceIndex].frontRight != NULL)
            {
                GetSamples(MAIN_FR_INDEX)[destOffset] = static_cast<s16>(data[deviceIndex].frontRight[srcOffset]);
            }

            if(isSurroundEnabled)
            {
                if (data[deviceIndex].frontCenter != NULL)
                {
                    GetSamples(MAIN_FC_INDEX)[destOffset]  = static_cast<s16>(data[deviceIndex].frontCenter[srcOffset]);
                }
                if (data[deviceIndex].lfe != NULL)
                {
                    GetSamples(MAIN_LFE_INDEX)[destOffset] = static_cast<s16>(data[deviceIndex].lfe[srcOffset]);
                }
                if (data[deviceIndex].rearLeft != NULL)
                {
                    GetSamples(MAIN_RL_INDEX)[destOffset]  = static_cast<s16>(data[deviceIndex].rearLeft[srcOffset]);
                }
                if (data[deviceIndex].rearRight != NULL)
                {
                    GetSamples(MAIN_RR_INDEX)[destOffset]  = static_cast<s16>(data[deviceIndex].rearRight[srcOffset]);
                }
            }
        }

        restCopySamples -= copySamples;
        sampleOffset += copySamples;
        m_CurrentSampleIndex += copySamples;

        if ( m_CurrentSampleIndex >= m_SampleCount )
        {
            m_CurrentSampleIndex = 0;
        }
    }

    for(u32 i = MAIN_CHANNEL_START; i < MAIN_CHANNEL_START + MAX_MAIN_CHANNEL_COUNT; ++i)
    {
        m_WaveGraphs[i].SetWaveDataStartIndex(m_CurrentSampleIndex);
    }
}

void MasterOutputPanel::UpdateDrcWaveGraph(const FinalMixData* data)
{
    if(m_IsWaveGraphPaused)
    {
        return;
    }

    const u32 deviceIndex = 0;
    u32 restCopySamples = data[deviceIndex].sampleCount;
    u32 sampleOffset = 0;

    while(restCopySamples > 0)
    {
        const u32 copySamples = nw::ut::Min(restCopySamples, m_SampleCount - m_DrcCurrentSampleIndex);

        for(u32 i = 0; i < copySamples ; ++i)
        {
            u32 srcOffset = sampleOffset + i;
            u32 destOffset = m_DrcCurrentSampleIndex + i;

            if (data[deviceIndex].frontLeft != NULL)
            {
                GetSamples(DRC_FL_INDEX)[destOffset] = static_cast<s16>(data[deviceIndex].frontLeft[srcOffset]);
            }
            if (data[deviceIndex].frontRight != NULL)
            {
                GetSamples(DRC_FR_INDEX)[destOffset] = static_cast<s16>(data[deviceIndex].frontRight[srcOffset]);
            }
            if (data[deviceIndex].rearLeft != NULL)
            {
                GetSamples(DRC_RL_INDEX)[destOffset] = static_cast<s16>(data[deviceIndex].rearLeft[srcOffset]);
            }
            if (data[deviceIndex].rearRight != NULL)
            {
                GetSamples(DRC_RR_INDEX)[destOffset] = static_cast<s16>(data[deviceIndex].rearRight[srcOffset]);
            }
        }

        restCopySamples -= copySamples;
        sampleOffset += copySamples;
        m_DrcCurrentSampleIndex += copySamples;

        if ( m_DrcCurrentSampleIndex >= m_SampleCount )
        {
            m_DrcCurrentSampleIndex = 0;
        }
    }

    for(u32 i = DRC_CHANNEL_START; i < DRC_CHANNEL_START + MAX_DRC_CHANNEL_COUNT; ++i)
    {
        m_WaveGraphs[i].SetWaveDataStartIndex(m_DrcCurrentSampleIndex);
    }
}

void MasterOutputPanel::ClearSamples()
{
    // 必ず一時停止中にコールします。
    NW_ASSERT(m_IsWaveGraphPaused);

    for(u32 i = 0; i < MAX_CHANNEL_COUNT; ++i)
    {
        memset(m_Samples[i], 0, sizeof(m_Samples[i]));
        m_WaveGraphs[i].SetWaveDataStartIndex(0);
    }
}

bool MasterOutputPanel::IsPressWindowControlKey(const nw::dev::Pad& pad) const
{
    return pad.IsHoldAll( nw::internal::dw::InputSettings::GetInstance().GetWindowManagerSetting().GetWindowControlModifier() );
}

} // dw
} // snd
} // nw
