﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nw/snd/ctrl/sndctrl_SoundControlSession.h>

#ifdef NW_SND_CONFIG_ENABLE_DEV

#include <nw/ut/ut_Inlines.h>
#include <nw/snd/fnd/basis/sndfnd_Memory.h>
#include <nw/snd/edit/sndedit_IErrorProvider.h>
#include <nw/snd/edit/hio/sndedit_HioAsyncChannel.h>
#include <nw/snd/ctrl/sndctrl_SoundObjectController.h>
#include <nw/snd/ctrl/protocol/sndctrl_PlaySoundPacket.h>
#include <nw/snd/ctrl/protocol/sndctrl_StopSoundPacket.h>
#include <nw/snd/ctrl/protocol/sndctrl_PauseSoundPacket.h>
#include <nw/snd/ctrl/protocol/sndctrl_UpdateSoundInfoPacket.h>

#if !defined(NW_RELEASE)
//#define NW_ENABLE_COM_DEBUG
#endif

namespace nw {
namespace snd {
namespace ctrl {

//----------------------------------------------------------
SoundControlSession::SoundControlSession() :
m_State(STATE_NOT_INITIALIZED),
m_Port(0),
m_SyncTimeout(0),
m_SoundObjectController(NULL)
{
}

//----------------------------------------------------------
SoundControlSession::~SoundControlSession()
{
    // Finalize() 済みであることを確認します。
    NW_ASSERTMSG(!IsInitialized(), "SoundControlSession is not finalized.\n");
}

//----------------------------------------------------------
void
SoundControlSession::Finalize()
{
    Close();

    UnregisterSoundObjectController(NULL);

    FinalizeChannel();

    m_HioManager.Finalize();

    m_SyncTimeout = 0;
    m_State = STATE_NOT_INITIALIZED;
}

//----------------------------------------------------------
edit::Result
SoundControlSession::RegisterSoundObjectController(SoundObjectController* soundObjectController)
{
    NW_ASSERTMSG(IsInitialized(), "SoundControlSession is not initialized.\n");
    NW_ASSERTMSG(m_SoundObjectController == NULL, "SoundObjectController is already regiestered.\n");

    if(soundObjectController == NULL ||
        !soundObjectController->IsInitialized())
    {
        NW_FATAL_ERROR("invalid arguments.\n");
        return edit::Result(edit::SNDEDIT_RESULT_FAILED);
    }

    m_SoundObjectController = soundObjectController;

    m_PlaySoundHandler.SetSoundObjectController(soundObjectController);
    m_StopSoundHandler.SetSoundObjectController(soundObjectController);
    m_PauseSoundHandler.SetSoundObjectController(soundObjectController);

    return edit::Result(edit::SNDEDIT_RESULT_TRUE);
}

//----------------------------------------------------------
void
SoundControlSession::UnregisterSoundObjectController(SoundObjectController* soundObjectController)
{
    // TODO : 複数サウンドアーカイブに対応したら利用します。
    (void)soundObjectController;

    NW_ASSERTMSG(IsInitialized(), "SoundControlSession is not initialized.\n");
    m_SoundObjectController = NULL;

    m_PlaySoundHandler.SetSoundObjectController(NULL);
    m_StopSoundHandler.SetSoundObjectController(NULL);
    m_PauseSoundHandler.SetSoundObjectController(NULL);
}

//----------------------------------------------------------
void
SoundControlSession::Open()
{
    NW_ASSERTMSG(IsInitialized(), "SoundControlSession is not initialized.\n");

    if(IsOpened())
    {
        return;
    }

    ClearBuffer();
    m_State = STATE_OPENED;

    m_SyncStopWatch.Start();
}

//----------------------------------------------------------
void
SoundControlSession::Close()
{
    NW_ASSERTMSG(IsInitialized(), "SoundControlSession is not initialized.\n");

    m_SyncStopWatch.Stop();

    m_State = STATE_INITIALIZED;
}

//----------------------------------------------------------
void
SoundControlSession::Update()
{
    NW_ASSERTMSG(IsInitialized(), "SoundControlSession is not initialized.\n");

    if(!IsOpened())
    {
        return;
    }

#if !defined(NW_SND_EDIT_USE_MCS)
    if(!m_Channel.IsOpened())
    {
        if(!m_Channel.Open(GetChannelInfo(edit::internal::HIO_SNDEDIT_CTRL_CHANNEL)))
        {
            return;
        }
    }
#endif

    m_HioManager.Update();

    // SYNC タイムアウトを経過したら、同期します。
    if(m_SyncStopWatch.GetElapsedTime().ToMilliSeconds() >= m_SyncTimeout)
    {
        m_SyncStopWatch.Reset();
        Sync();
    }
}

//----------------------------------------------------------
edit::Result
SoundControlSession::InitializeChannel(
    snd::internal::fnd::FrameHeap& allocator,
    u32 recvStreamBufferSize,
    u32 recvPacketBufferSize)
{
    void* recvStreamBuffer = allocator.Alloc(recvStreamBufferSize);
    void* recvPacketBuffer = allocator.Alloc(recvPacketBufferSize);

#if !defined(NW_SND_EDIT_USE_MCS)
    void* workBuffer = allocator.Alloc(GetRequiredWorkBufferSize());
#endif

    if(recvStreamBuffer == NULL || recvPacketBuffer == NULL)
    {
        return edit::Result(edit::SNDEDIT_RESULT_OUT_OF_MEMORY);
    }

#if defined(NW_SND_EDIT_USE_MCS)
    edit::internal::HioResult result = m_Channel.Initialize(
        recvStreamBuffer,
        recvStreamBufferSize,
        recvPacketBuffer,
        recvPacketBufferSize);
#else
    edit::internal::HioResult result = m_Channel.Initialize(
        recvStreamBuffer,
        recvStreamBufferSize,
        recvPacketBuffer,
        recvPacketBufferSize,
        workBuffer,
        GetRequiredWorkBufferSize()
    );
#endif

    if(result.IsFailed())
    {
        return edit::Result(edit::ResultType(result));
    }

    if(!m_Channel.Open(GetChannelInfo(edit::internal::HIO_SNDEDIT_CTRL_CHANNEL)))
    {
        return edit::Result(edit::SNDEDIT_RESULT_FALSE);
    }

    m_HioManager.RegisterChannel(m_Channel);

    InitializeHandlers();

    return edit::Result(edit::SNDEDIT_RESULT_TRUE);
}

//----------------------------------------------------------
void
SoundControlSession::FinalizeChannel()
{
    m_HioManager.UnregisterChannel(m_Channel);

    FinalizeHandlers();

    m_Channel.Close();
    m_Channel.Finalize();
}

//----------------------------------------------------------
void
SoundControlSession::InitializeHandlers()
{
    NW_ASSERT(m_Channel.IsInitialized());

    m_Channel.RegisterMessageHandler(m_PlaySoundHandler);
    m_Channel.RegisterMessageHandler(m_StopSoundHandler);
    m_Channel.RegisterMessageHandler(m_PauseSoundHandler);
}

//----------------------------------------------------------
void
SoundControlSession::FinalizeHandlers()
{
    if(m_Channel.IsInitialized())
    {
        m_Channel.UnregisterMessageHandler(m_PlaySoundHandler);
        m_Channel.UnregisterMessageHandler(m_StopSoundHandler);
        m_Channel.UnregisterMessageHandler(m_PauseSoundHandler);
    }
}

//----------------------------------------------------------
u32
SoundControlSession::GetChannelRecvPacketBufferSize(u32 maxItemName) const
{
    u32 result = 0;

    // TODO : チャンネルで扱うパケットの種類が増えたら、ここに追加します。
    // TODO : チャンネルで扱うパケットのサイズが変わったら、ここを編集します。
    result = ut::Max(result, internal::PlaySoundPacket::GetRequiredSize(maxItemName));
    result = ut::Max(result, internal::StopSoundPacket::GetRequiredSize());
    result = ut::Max(result, internal::PauseSoundPacket::GetRequiredSize());
    result = ut::Max(result, internal::UpdateSoundInfoPacket::GetRequiredSize());

    return ut::RoundUp(result, snd::internal::fnd::MemoryTraits::DEFAULT_ALIGNMENT);
}

//----------------------------------------------------------
void
SoundControlSession::ClearBuffer()
{
    m_Channel.ClearBuffer();
}

//----------------------------------------------------------
void
SoundControlSession::Sync()
{
    internal::UpdateSoundInfoPacket packet;

    if(m_SoundObjectController != NULL)
    {
        for(u32 index = 0; index < SoundObjectController::SOUND_CONTROLLER_COUNT; ++index)
        {
            internal::UpdateSoundInfoPacket::SoundState state =
                static_cast<internal::UpdateSoundInfoPacket::SoundState>(
                m_SoundObjectController->GetSoundController(index)->GetState()
                );

            packet.GetBody().SetSoundState(index, state);
        }
    }

    m_Channel.Send(packet);
}

} // namespace nw::snd::ctrl
} // namespace nw::snd
} // namespace nw

#endif // NW_SND_CONFIG_ENABLE_DEV
