﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nw/snd/snd_WaveSound.h>

#include <nw/snd/snd_SoundInstanceManager.h>
#include <nw/snd/snd_WaveFileReader.h>
#include <nw/snd/snd_DriverCommand.h>

namespace nw {
namespace snd {
namespace internal {

/* ========================================================================
        member function
   ======================================================================== */

/*---------------------------------------------------------------------------*
  Name:         WaveSound

  Description:  コンストラクタ

  Arguments:    None.

  Returns:      None.
 *---------------------------------------------------------------------------*/
WaveSound::WaveSound( WaveSoundInstanceManager& manager )
: m_Manager( manager ),
  m_InitializeFlag( false ),
  m_IsCalledPrepare( false ),
  m_ChannelCount( 0 )
{
}

void WaveSound::Initialize()
{
    BasicSound::Initialize();

    m_pTempSpecialHandle = NULL;
    m_pWaveFile = NULL;
    m_WaveType = WAVE_TYPE_INVALID;

    m_IsCalledPrepare = false;
    m_InitializeFlag = true;
    m_ChannelCount = 0;
}

/*---------------------------------------------------------------------------*
  Name:         Shutdown

  Description:  サウンドの終了

  Arguments:    None.

  Returns:      None.
 *---------------------------------------------------------------------------*/
void WaveSound::Finalize()
{
    if ( ! m_InitializeFlag )
    {
        return;
    }

    m_InitializeFlag = false;
    m_IsCalledPrepare = false;
    m_ChannelCount = 0;

    BasicSound::Finalize();
    m_Manager.Free( this );
}

/*---------------------------------------------------------------------------*
  Name:         Prepare

  Description:  サウンド開始

  Arguments:    seqDataBase   - シーケンスデータベースアドレス
                seqDataOffset - シーケンスデータのオフセット
                bank          - バンクデータ

  Returns:      None.
 *---------------------------------------------------------------------------*/
void WaveSound::Prepare(
        const driver::WaveSoundPlayer::StartInfo& startInfo,
        const driver::WaveSoundPlayer::PrepareArg& arg)
{
    NW_NULL_ASSERT( arg.wsdFile );
    NW_ASSERT( m_InitializeFlag );

    // コマンド
    {
        DriverCommand& cmdmgr = DriverCommand::GetInstance();
        DriverCommandWaveSoundPrepare* command =
            cmdmgr.AllocCommand<DriverCommandWaveSoundPrepare>();
        command->id = DRIVER_COMMAND_WSD_PREPARE;
        command->player = &m_PlayerInstance;
        command->startInfo = startInfo; // 構造体コピー
        command->arg = arg; // 構造体コピー
        cmdmgr.PushCommand(command);
    }

    m_WaveType = arg.waveType;
    m_pWaveFile = arg.waveFile;
    m_IsCalledPrepare = true;

    WaveFileReader reader( m_pWaveFile, m_WaveType );
    WaveInfo waveInfo;
    if ( reader.ReadWaveInfo( &waveInfo ) )
    {
        m_ChannelCount = static_cast<u32>( ut::Min( waveInfo.channelCount, 2 ) );
    }
}

void WaveSound::RegisterDataLoadTask(
        const driver::WaveSoundLoader::LoadInfo& loadInfo,
        const driver::WaveSoundPlayer::StartInfo& startInfo )
{
    // コマンド
    {
        DriverCommand& cmdmgr = DriverCommand::GetInstance();
        DriverCommandWaveSoundLoad* command =
            cmdmgr.AllocCommand<DriverCommandWaveSoundLoad>();
        command->id = DRIVER_COMMAND_WSD_LOAD;
        command->player = &m_PlayerInstance;
        command->startInfo = startInfo; // 構造体コピー
        command->arg.soundDataManager = loadInfo.soundDataManager;
        command->arg.soundArchive = loadInfo.soundArchive;
        command->arg.soundPlayer = loadInfo.soundPlayer;
        command->arg.loadInfoWsd = *(loadInfo.loadInfoWsd); // 構造体コピー
        command->arg.index = startInfo.index;

        cmdmgr.PushCommand(command);
    }
}

void WaveSound::SetChannelPriority( int priority )
{
    NW_MINMAX_ASSERT( priority, 0, 127 );
    u8 u8_prio = static_cast<u8>(priority);
    {
        DriverCommand& cmdmgr = DriverCommand::GetInstance();
        DriverCommandWaveSoundChannelPrio* command =
            cmdmgr.AllocCommand<DriverCommandWaveSoundChannelPrio>();
        command->id = DRIVER_COMMAND_WSD_CHANNEL_PRIO;
        command->player = &m_PlayerInstance;
        command->priority = u8_prio;
        cmdmgr.PushCommand(command);
    }
}

void WaveSound::InitializeChannelParam(int priority, bool isReleasePriorityFix)
{
    NW_MINMAX_ASSERT( priority, 0, 127 );
    u8 u8_prio = static_cast<u8>(priority);

    {
        DriverCommand& cmdmgr = DriverCommand::GetInstance();
        DriverCommandWaveSoundChannelParam* command =
            cmdmgr.AllocCommand<DriverCommandWaveSoundChannelParam>();
        command->id = DRIVER_COMMAND_WSD_CHANNEL_PARAM;
        command->player = &m_PlayerInstance;
        command->priority = u8_prio;
        command->isReleasePriorityFix = isReleasePriorityFix;
        cmdmgr.PushCommand(command);
    }
}

/*---------------------------------------------------------------------------*
  Name:         OnUpdatePlayerPriority

  Description:  プレイヤープライオリティを変更

  Arguments:

  Returns:      None.
 *---------------------------------------------------------------------------*/
void WaveSound::OnUpdatePlayerPriority()
{
    int priority = CalcCurrentPlayerPriority();
    m_Manager.UpdatePriority( this, priority );
}

bool WaveSound::IsAttachedTempSpecialHandle()
{
    return m_pTempSpecialHandle != NULL;
}

void WaveSound::DetachTempSpecialHandle()
{
    // TODO: m_pTempSpecialHandle->DetachSound();
}

bool WaveSound::ReadWaveSoundDataInfo( WaveSoundDataInfo* info ) const
{
    NW_NULL_ASSERT( info );

    const void* waveFile = m_pWaveFile;
    if (waveFile == NULL)
    {
        if (IsPlayerAvailable() == true)
        {
            waveFile = m_PlayerInstance.GetWaveFile();
        }
    }

    if (waveFile == NULL)
    {
        return false;
    }

    if (m_WaveType != WAVE_TYPE_NWWAV)
    {
        return false;
    }

    WaveFileReader reader( m_pWaveFile );
    WaveInfo waveInfo;
    if ( ! reader.ReadWaveInfo( &waveInfo ) )
    {
        return false;
    }
    info->loopFlag = ( waveInfo.loopFlag != 0 );
    info->sampleRate = static_cast<int>( waveInfo.sampleRate );
    info->loopStart = waveInfo.originalLoopStartFrame;
    info->loopEnd = waveInfo.loopEndFrame - (waveInfo.loopStartFrame - waveInfo.originalLoopStartFrame);
    info->compatibleLoopStart = waveInfo.loopStartFrame;
    info->compatibleLoopEnd = waveInfo.loopEndFrame;
    return true;
}

long WaveSound::GetPlaySamplePosition(bool isOriginalSamplePosition) const
{
    if ( ! IsPlayerAvailable() ) return 0;

    driver::SoundThreadLock lock;
    return m_PlayerInstance.GetPlaySamplePosition(isOriginalSamplePosition);
}

} // namespace nw::snd::internal
} // namespace nw::snd
} // namespace nw

