﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nw/ut.h>
#include <nw/snd/snd_WavOutFileStream.h>
#include <nw/snd/snd_WavBinary.h>

namespace nw {
namespace snd {
namespace internal {

#ifdef NW_PLATFORM_CAFE
const u32 WavOutFileStream::FILE_IO_BUFFER_ALIGNMENT = PPC_IO_BUFFER_ALIGN;
#else
const u32 WavOutFileStream::FILE_IO_BUFFER_ALIGNMENT = 1;
#endif

//---------------------------------------------------------------------------
WavOutFileStream::WavOutFileStream() :
m_Stream(NULL),
m_WaveDataSize(0),
m_IsWaveDataSizeCalculating(false),
m_Buffer(NULL),
m_BufferLength(0),
m_ValidBufferLength(0)
{
}

//---------------------------------------------------------------------------
WavOutFileStream::~WavOutFileStream()
{
}

//---------------------------------------------------------------------------
bool
WavOutFileStream::Open(ut::FileStream& stream, u32 channels, u32 samplesPerSec)
{
    NW_ASSERT(stream.IsAvailable());

    m_Stream = &stream;

    if(!WriteHeader(channels, samplesPerSec))
    {
        return false;
    }

    // data チャンクのサイズ計算を開始します。
    m_WaveDataSize = 0;
    m_IsWaveDataSizeCalculating = true;

    return true;
}

//---------------------------------------------------------------------------
void
WavOutFileStream::Close()
{
    if(!IsAvailable())
    {
        return;
    }

    // 閉じる前にバッファの残りを書き出します。
    FlushBuffer();

    m_IsWaveDataSizeCalculating = false;

    // 正確な出力サイズを取得するために、Flush() します。
    FlushBuffer();

    // ヘッダサイズ分を出力できていない場合は、ヘッダの更新を中止します。
    if(m_Stream->GetSize() < CalcRiffChunkSize(0))
    {
        m_Stream = NULL;
        return;
    }

    if(!UpdateRiffChunkSize())
    {
        m_Stream = NULL;
        return;
    }

    if(!UpdateDataChunkSize())
    {
        m_Stream = NULL;
        return;
    }

    m_Stream = NULL;
    m_WaveDataSize = 0;
}

//---------------------------------------------------------------------------
s32
WavOutFileStream::Write(const void* buf, u32 length)
{
    if(!IsAvailable())
    {
        // IOStream の実装次第でエラーコードが変わるので、とりあえず -1 を返しています。
        return -1;
    }

    u32 writtenLength = 0;

    while(writtenLength < length)
    {
        const u32 restLength = length - writtenLength;

        if(restLength > m_BufferLength)
        {
            s32 result = WriteDirect(buf, restLength);

            if(result < 0 )
            {
                return result;
            }

            writtenLength += result;
            continue;
        }

        u32 copyLength = nw::ut::Min(restLength, m_BufferLength - m_ValidBufferLength);

        memcpy(
            nw::ut::AddOffsetToPtr(m_Buffer, m_ValidBufferLength),
            nw::ut::AddOffsetToPtr(buf, length - restLength),
            copyLength);

        m_ValidBufferLength += copyLength;
        writtenLength += copyLength;

        if(m_ValidBufferLength == m_BufferLength)
        {
            FlushBuffer();
        }
    }

    if(m_IsWaveDataSizeCalculating)
    {
        m_WaveDataSize += length;
    }

    return length;
}

//---------------------------------------------------------------------------
bool
WavOutFileStream::Seek(s32 offset, u32 origin)
{
    if(!CanSeek())
    {
        return false;
    }

    // シーク前にバッファの残りを書き出します。
    FlushBuffer();

    return m_Stream->Seek(offset, origin);
}

//---------------------------------------------------------------------------
void
WavOutFileStream::SetCacheBuffer(u8* buf, u32 length)
{
    m_Buffer = buf;
    m_BufferLength = length;
}

//---------------------------------------------------------------------------
bool
WavOutFileStream::WriteHeader(u32 channels, u32 samplesPerSec)
{
    NW_ASSERT(IsAvailable());

    u8 buffer[sizeof(WaveBinaryHeader) + FILE_IO_BUFFER_ALIGNMENT];
    void* alignedBuffer = ut::RoundUp(buffer, FILE_IO_BUFFER_ALIGNMENT);

    WaveBinaryHeader* header = new(alignedBuffer) WaveBinaryHeader();

    header->riffChunk.header.size = CalcRiffChunkSize(0);

    header->fmtChunk.formatTag      = FmtChunk::FORMAT_PCM;
    header->fmtChunk.channels       = static_cast<fnd::PcBinU16>(channels);
    header->fmtChunk.samplesPerSec  = samplesPerSec;
    header->fmtChunk.bitsPerSample  = 16;
    header->fmtChunk.blockAlign     = static_cast<u16>(header->fmtChunk.channels * (header->fmtChunk.bitsPerSample / 8));
    header->fmtChunk.avgBytesPerSec = header->fmtChunk.samplesPerSec * header->fmtChunk.blockAlign;

    s32 result = Write(alignedBuffer, sizeof(WaveBinaryHeader));
    return result == sizeof(WaveBinaryHeader);
}

//---------------------------------------------------------------------------
bool
WavOutFileStream::UpdateRiffChunkSize()
{
    if(!IsAvailable())
    {
        return false;
    }

    static const s32 riffChunkSizePosition = 4;

    if(!Seek(riffChunkSizePosition, ut::FILE_STREAM_SEEK_BEGIN))
    {
        NW_WARNING(false, "[WavOutFileStream] failed to update wave header.\n");
        return false;
    }

    u8 buffer[sizeof(fnd::PcBinU32) + FILE_IO_BUFFER_ALIGNMENT];
    void* alignedBuffer = ut::RoundUp(buffer, FILE_IO_BUFFER_ALIGNMENT);

    fnd::PcBinU32& size = *new(alignedBuffer) fnd::PcBinU32();

    size = CalcRiffChunkSize(m_WaveDataSize);

    s32 result = WriteDirect(&size, sizeof(size));
    if(result < 0)
    {
        NW_WARNING(false, "[WavOutFileStream] failed to update wave header. : %d\n", result);
        return false;
    }

    return true;
}

//---------------------------------------------------------------------------
bool
WavOutFileStream::UpdateDataChunkSize()
{
    if(!IsAvailable())
    {
        return false;
    }

    static const s32 dataChunkSizePosition = sizeof(RiffChunk) + sizeof(FmtChunk) + 4;

    if(!Seek(dataChunkSizePosition, ut::FILE_STREAM_SEEK_BEGIN))
    {
        NW_WARNING(false, "[WavOutFileStream] failed to update wave header.\n");
        return false;
    }

    u8 buffer[sizeof(fnd::PcBinU32) + FILE_IO_BUFFER_ALIGNMENT];
    void* alignedBuffer = ut::RoundUp(buffer, FILE_IO_BUFFER_ALIGNMENT);

    fnd::PcBinU32& size = *new(alignedBuffer) fnd::PcBinU32();

    size = m_WaveDataSize;

    s32 result = WriteDirect(&size, sizeof(size));

    if(result < 0)
    {
        NW_WARNING(false, "[WavOutFileStream] failed to update wave header. : %d\n", result);
        return false;
    }

    return true;
}

//---------------------------------------------------------------------------
u32
WavOutFileStream::CalcRiffChunkSize(u32 dataSize)
{
    return
        sizeof(RiffChunk)   // RiffChunk サイズ
        - 8                 // RiffChunk の chunkID, chunkSize 分
        + sizeof(FmtChunk)  // FmtChunk サイズ
        + sizeof(DataChunk) // DataChunk サイズ
        + dataSize;         // データサイズ
}

//---------------------------------------------------------------------------
s32
WavOutFileStream::WriteDirect(const void* buf, u32 length)
{
    if(!IsAvailable())
    {
        // IOStream の実装次第でエラーコードが変わるので、とりあえず -1 を返しています。
        return -1;
    }

    // キャッシュは存在しないはず
    NW_ASSERT(m_ValidBufferLength == 0);

    const void* alignedBuffer = ut::RoundUp(buf, FILE_IO_BUFFER_ALIGNMENT);

    u32 unAlignedLength = ut::GetOffsetFromPtr(buf, alignedBuffer);
    u32 alignedBufferLength = length - unAlignedLength;

    if(unAlignedLength > 0)
    {
        // アライメント調整されていない場合は、２回に分けて出力します。
        u8 bufferForAlignment[FILE_IO_BUFFER_ALIGNMENT + FILE_IO_BUFFER_ALIGNMENT];
        u8* alignedBufferForAlignment = ut::RoundUpTo<u8*>(bufferForAlignment, FILE_IO_BUFFER_ALIGNMENT);

        memcpy(alignedBufferForAlignment, buf, unAlignedLength);

        s32 result = m_Stream->Write(alignedBufferForAlignment, unAlignedLength);

        if( result < 0)
        {
            return result;
        }

        if( result != static_cast<s32>(unAlignedLength) )
        {
            // IOStream の実装次第でエラーコードが変わるので、とりあえず -1 を返しています。
            return -1;
        }
    }

    s32 result = m_Stream->Write(alignedBuffer, alignedBufferLength);

    if( result < 0)
    {
        return result;
    }

    return result + unAlignedLength;
}

//---------------------------------------------------------------------------
s32
WavOutFileStream::FlushBuffer()
{
    s32 length = m_ValidBufferLength;
    m_ValidBufferLength = 0;

    if(length == 0)
    {
        return 0;
    }

    return WriteDirect(m_Buffer, length);
}

} // internal
} // snd
} // nw
