﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nw/snd/snd_Config.h>
#include <nw/snd/snd_StreamSoundPlayer.h>
#include <nw/snd/snd_StreamBufferPool.h>
#include <nw/snd/snd_TaskManager.h>
#include <nw/snd/snd_MultiVoiceManager.h>
#include <nw/snd/snd_DriverCommand.h>
#include <nw/snd/snd_SoundSystem.h>
#include <nw/snd/snd_WaveFileReader.h>

#if defined( NW_PLATFORM_WIN32 )
using namespace nw::internal::winext;
#elif defined( NW_USE_NINTENDO_SDK )
// TODO: nn_audio
using namespace nw::internal::winext;
#endif

// #define DEBUG_STRM
// #define ENABLE_PRINT_STREAM_INFO

namespace
{
    const u8 ADTS_FILE_TYPE = static_cast<u8>(nw::snd::internal::STREAM_FILE_BFSTM) + 1;
    const f32 AAC_PITCH_MAX = 3.0f;

#ifdef DEBUG_STRM
    FILE* s_pFile;
    const char* DUMP_FILE = "_StreamSoundPlayer.dump";
#endif

    const u32 LOOP_REGION_SIZE_MIN = nw::snd::internal::DATA_BLOCK_SIZE_MARGIN_SAMPLES;

#if defined(NW_DEBUG) || defined(NW_DEVELOP)
    const u32 PCM8_SAMPLES_PER_BLOCK = 8 * 1024;
    const u32 PCM16_SAMPLES_PER_BLOCK = 4 * 1024;
    const u32 ADPCM_SAMPLES_PER_BLOCK = 14336;
    void CheckSampleLength(u32 length, const nw::snd::internal::StreamDataInfoDetail& info, u8 fileType)
    {
        bool doCheck = true;

        // 下記の条件の時は、バッファジャンプが起こらないのでチェック不要
        if (info.loopFlag == false)
        {
            u32 sampleCountPerBlock = 0;
            switch (info.sampleFormat)
            {
            case nw::snd::SAMPLE_FORMAT_PCM_S8:
                sampleCountPerBlock = PCM8_SAMPLES_PER_BLOCK;
                break;
            case nw::snd::SAMPLE_FORMAT_PCM_S16:
                sampleCountPerBlock = PCM16_SAMPLES_PER_BLOCK;
                break;
            case nw::snd::SAMPLE_FORMAT_DSP_ADPCM:
                sampleCountPerBlock = ADPCM_SAMPLES_PER_BLOCK;
                break;
            default:
                NW_ASSERTMSG(false, "SampleFormat(%d) is invalid", info.sampleFormat);
                break;
            }
            if (length <= sampleCountPerBlock)
            {
                doCheck = false;
            }
        }

        if (doCheck)
        {
            u32 dataBlockSizeMarginSamples = nw::snd::internal::DATA_BLOCK_SIZE_MARGIN_SAMPLES;

            if (fileType == ADTS_FILE_TYPE)
            {
                dataBlockSizeMarginSamples = 48000 * 7 * 3 / 1000; // 1008 sample
            }

            NW_ASSERT(length >= dataBlockSizeMarginSamples);
        }
    }
#endif

} // anonymous namespace


namespace nw {
namespace snd {
namespace internal {
namespace driver {


u16 StreamSoundPlayer::s_TaskRequestIndexCount = 0;

StreamSoundPlayer::StreamSoundPlayer()
: m_ChannelCount(0)
, m_TrackCount(0)
, m_IsInitialized(false)
, m_IsRegisterPlayerCallback(false)
, m_pLoaderManager(NULL)
, m_pLoader(NULL)
, m_IsSucceedPrepare(false)
, m_pStreamPrefetchFile(NULL)
{
}

StreamSoundPlayer::~StreamSoundPlayer()
{
    Finalize();
}

void StreamSoundPlayer::Initialize()
{
    BasicSoundPlayer::Initialize();

    m_IsInitialized = false;

    m_IsPrepared          = false;
    m_PauseStatus         = false;
    m_LoadWaitFlag        = false;
    m_LoadFinishFlag      = false;
    m_IsFinalizing        = false;
    m_IsPreparedPrefetch  = false;

    m_LoopCounter         = 0;
    m_PrefetchOffset      = 0;

    m_DelayCount          = 0;
    m_UseDelayCount       = false;

    if (TryAllocLoader() == true)
    {
        m_pLoader->Initialize();
    }

    // アイテム（トラックの親）データパラメータの初期化
    m_ItemData.pitch = 1.0f;
    m_ItemData.mainSend = 1.0f;
    for ( int i = 0; i < AUX_BUS_NUM; i++ )
    {
        m_ItemData.fxSend[ i ] = 0.0f;
    }

    // トラックの初期化
    for ( int trackIndex = 0; trackIndex < STRM_TRACK_NUM; trackIndex++ )
    {
        StreamTrack& track  = m_Tracks[ trackIndex ];
        track.m_ActiveFlag  = false;
        track.m_Volume      = 1.0f;
        track.m_OutputLine  = -1;

        track.m_TvParam.Initialize();
        for ( int i = 0; i < DRC_OUT_COUNT; i++ )
        {
            track.m_DrcParam[i].Initialize();
        }
    }

    // チャンネルの初期化
    for ( int channelIndex = 0; channelIndex < STRM_CHANNEL_NUM; channelIndex++ )
    {
        StreamChannel& channel = m_Channels[ channelIndex ];
        channel.m_pBufferAddress = NULL;
        channel.m_pVoice = NULL;
    }
}

/*---------------------------------------------------------------------------*
  Name:         Finalize

  Description:  ストリームプレイヤーのシャットダウン

  Arguments:    None.

  Returns:      None.
 *---------------------------------------------------------------------------*/
void StreamSoundPlayer::Finalize()
{
    FinishPlayer();

    if ( m_IsInitialized )
    {
        m_IsFinalizing = true;

        // ボイスとチャンネルの解放
        FreeStreamBuffers();
        FreeVoices();

        FreeLoader();

        m_pBufferPool = NULL;

    #ifdef DEBUG_STRM
        std::fclose( s_pFile );
    #endif

        BasicSoundPlayer::Finalize();
        m_ActiveFlag = false;
        m_IsInitialized = false;
    }
}

/*---------------------------------------------------------------------------*
  Name:         Setup

  Description:  ストリームプレイヤーのセットアップ

  Arguments:    pBufferPool - ストリームバッファのメモリプール

  Returns:      セットアップに成功したかどうか
 *---------------------------------------------------------------------------*/
void StreamSoundPlayer::Setup(const SetupArg& arg)
{
    NW_NULL_ASSERT( arg.pBufferPool );

    if (m_pLoader == NULL)
    {
        m_SetupArg = arg;
        return;
    }

    m_FileType = arg.fileType;
    m_LoopFlag = arg.loopFlag;
    m_LoopStart = arg.loopStart;
    m_LoopEnd = arg.loopEnd;

    // bXsar にあるストリームサウンドの情報を設定
    m_ItemData.Set( arg );

    // bXsar にあるストリームサウンドトラックの情報を StreamDataInfoDetail に設定
    if ( !SetupTrack( arg ) )
    {
        return;
    }

    m_pBufferPool = arg.pBufferPool;
    // NN_LOG("m_pBufferPool %08x\n",m_pBufferPool);

    m_IsInitialized = true;

#ifdef DEBUG_STRM
    s_pFile = std::fopen( DUMP_FILE, "wb" );
#endif
}

void StreamSoundPlayer::Prepare(const PrepareArg& arg)
{
    // プレイヤー登録
    if ( !m_IsRegisterPlayerCallback )
    {
        SoundThread::GetInstance().RegisterPlayerCallback( this );
        m_IsRegisterPlayerCallback = true;
    }

    if ( m_pLoader == NULL )
    {
        m_PrepareArg = arg;
        m_IsSucceedPrepare = false;
        return;
    }

    if ( ! m_IsInitialized )
    {
        return;
    }

    SetPrepareBaseArg(arg.baseArg);

    m_ReportLoadingDelayFlag = false;
    m_IsSucceedPrepare = true;

    RequestLoadHeader(arg);
}

void StreamSoundPlayer::PreparePrefetch(const PreparePrefetchArg& arg)
{
    if ( ! m_IsInitialized )
    {
        return;
    }

    // プリフェッチデータ読み込み
    m_pStreamPrefetchFile = arg.strmPrefetchFile;
    StreamSoundPrefetchFileReader reader;
    reader.Initialize( m_pStreamPrefetchFile );
    if ( !ReadPrefetchFile(reader) )
    {
        return;
    }

    SetPrepareBaseArg(arg.baseArg);

    if ( !ApplyStreamDataInfo(m_StreamDataInfo) )
    {
        return;
    }

    // ストリームプレイヤーのセットアップ
    if ( ! SetupPlayer() )
    {
        return;
    }

    // ボイス確保
    if ( ! AllocVoices() )
    {
        FreeStreamBuffers();
        return;
    }

    m_IsPreparedPrefetch = true;

    // waveBuffer の追加
    if ( !LoadPrefetchBlocks( reader ) )
    {
        return;
    }
}

void StreamSoundPlayer::Start()
{
    if ( !m_UseDelayCount && !m_StartedFlag )
    {
        StartPlayer();
    }
}

/*---------------------------------------------------------------------------*
  Name:         Stop

  Description:  ストリームの再生停止

  Arguments:    None.

  Returns:      None.
  *---------------------------------------------------------------------------*/
void StreamSoundPlayer::Stop()
{
    FinishPlayer();
}

/*---------------------------------------------------------------------------*
  Name:         Pause

  Description:  ストリームの一時停止／再開

  Arguments:    flag - trueで停止、falseで再開

  Returns:      None.
 *---------------------------------------------------------------------------*/

void StreamSoundPlayer::Pause( bool flag )
{
    m_PauseFlag = flag;
    if ( flag ) m_LoadWaitFlag = true;

    UpdatePauseStatus();
}

/*---------------------------------------------------------------------------*
  Name:         GetStreamDataInfo

  Description:  ストリームデータの情報取得

  Arguments:    info - ストリームデータ情報構造体

  Returns:      現在の再生しているストリームデータの情報を取得する
 *---------------------------------------------------------------------------*/
bool StreamSoundPlayer::ReadStreamDataInfo( nw::snd::StreamDataInfo* info ) const
{
    SoundThreadLock lock;

    if ( ! m_IsPrepared )
    {
        NW_WARNING(false, "ReadStreamDataInfo failed. Please use this API after playback preparing finished.");
        return false;
    }

    info->loopFlag              = m_StreamDataInfo.loopFlag;
    info->sampleRate            = m_StreamDataInfo.sampleRate;
    info->loopStart             = m_StreamDataInfo.originalLoopStart;
    info->loopEnd               = m_StreamDataInfo.originalLoopEnd;
    info->compatibleLoopStart   = m_StreamDataInfo.loopStart;
    info->compatibleLoopEnd     = m_StreamDataInfo.sampleCount;
    return true;
}

/*---------------------------------------------------------------------------*
  Name:         GetPlaySamplePosition

  Description:  現在の再生位置の取得

  Arguments:    なし

  Returns:      現在の再生位置をサンプル単位で返す
                再生していない場合は、負の値を返す
 *---------------------------------------------------------------------------*/
long StreamSoundPlayer::GetPlaySamplePosition(bool isOriginalSamplePosition) const
{
    SoundThreadLock lock;

    if ( ! m_ActiveFlag )
    {
        return -1;
    }
    if ( ! m_Tracks[ 0 ].m_ActiveFlag )
    {
        return -1;
    }
    if ( ! m_IsPrepared )
    {
        return 0;
    }

    long samplePosition;

    const StreamChannel& strmChannel = m_Channels[0];

    u32 bufferBlockIndex = m_PlayingBufferBlockIndex;
    const WaveBuffer* waveBuffer = &strmChannel.m_WaveBuffer[bufferBlockIndex];

    // m_PlayingBufferBlockIndexが更新される前の対策
    if ( waveBuffer->status == WaveBuffer::STATUS_DONE )
    {
        ++bufferBlockIndex;
        if ( bufferBlockIndex >= m_BufferBlockCount ) bufferBlockIndex = 0;
        waveBuffer = &strmChannel.m_WaveBuffer[bufferBlockIndex];
    }

    if ( waveBuffer->status == WaveBuffer::STATUS_PLAY ||
         waveBuffer->status == WaveBuffer::STATUS_WAIT )
    {
        const WaveBufferInfo* waveBufferInfo = &m_WaveBufferInfo[bufferBlockIndex];
        samplePosition = waveBufferInfo->sampleBegin;
        if ( strmChannel.m_pVoice != NULL )
        {
            samplePosition += static_cast<long>( strmChannel.m_pVoice->GetCurrentPlayingSample() );
        }
    }
    else
    {
        const WaveBufferInfo* waveBufferInfo = &m_WaveBufferInfo[m_LastPlayFinishBufferBlockIndex];
        samplePosition = waveBufferInfo->sampleBegin + waveBufferInfo->sampleLength;
    }

    if ( isOriginalSamplePosition && m_StreamDataInfo.loopFlag )
    {
        samplePosition = GetOriginalPlaySamplePosition(samplePosition, m_StreamDataInfo);
    }

    return samplePosition;
}

/*---------------------------------------------------------------------------*
  Name:         GetFilledBufferPercentage

  Description:  ストリームバッファ中でデータがロードされている割合を取得する

  Arguments:    なし

  Returns:      データがロードされている割合を百分率で返す
 *---------------------------------------------------------------------------*/
f32 StreamSoundPlayer::GetFilledBufferPercentage() const
{
    SoundThreadLock lock;

    if ( ! m_ActiveFlag || ! m_Tracks[ 0 ].m_ActiveFlag )
    {
        return 0.0f;
    }
    if ( m_LoadFinishFlag )
    {
        return 1.0f;
    }
    if ( ! m_IsPrepared )
    {
        if ( m_BufferBlockCount == 0 )
        {
            return 0.0f;
        }
        return static_cast<f32>( m_BufferBlockCount - m_PrepareCounter )
            * 100.f / m_BufferBlockCount;
    }

    s32 restSamples = 0;        // まだ再生していないサンプル数
    s32 entireSamples = 0;      // 全バッファのサンプル数
    for ( unsigned int i = 0; i < m_BufferBlockCount; i++ )
    {
        s32 bufferSamples = m_Channels[0].m_WaveBuffer[i].sampleLength;
        switch ( m_Channels[0].m_WaveBuffer[i].status )
        {
        case nw::snd::WaveBuffer::STATUS_PLAY:
            restSamples += (bufferSamples -
                    static_cast<s32>(m_Channels[0].m_pVoice->GetCurrentPlayingSample()));
            entireSamples += bufferSamples;
            break;
        case nw::snd::WaveBuffer::STATUS_WAIT:
            restSamples += bufferSamples;
            entireSamples += bufferSamples;
            break;
        default:
            // NOTE: 波形末尾の半端なブロックは考慮できていないが、これはあきらめる
            entireSamples += m_StreamDataInfo.blockSampleCount;
            break;
        }
    }
    f32 percentage = 100.f * restSamples / entireSamples;
    return percentage;
}

bool StreamSoundPlayer::LoadHeader(
    bool result,
    AdpcmParam* adpcmParam[],
    u16 requestIndex
)
{
    if ( ! m_IsInitialized )
    {
        // 終了処理後に、コマンドバッファにたまっていたコマンドが処理され
        // ここに来る場合がある
        return false;
    }

    if ( requestIndex != m_TaskRequestIndex )
    {
        // ヘッダーロードリクエストしたときとは別のサウンドとしてアロックされた
        return false;
    }
    if ( ! result ) {
        NW_WARNING( false, "Task error is occured." );
        m_FinishFlag = true;
        Stop();
        return false;
    }

    m_ChannelCount = ut::Min( m_StreamDataInfo.channelCount, STRM_CHANNEL_NUM );

    // ストリームバッファ確保
    if ( ! AllocStreamBuffers() )
    {
        NW_WARNING(false, "Failed to start stream sound for not enough stream channel instance." );
        Finalize();
        m_IsFinalizedForCannotAllocateResource = true;
        return SETUP_ERR_CANNOT_ALLOCATE_BUFFER;
    }

    if ( !ApplyStreamDataInfo(m_StreamDataInfo) )
    {
        return false;
    }

    if ( !m_IsPreparedPrefetch )
    {
        // ストリームプレイヤーのセットアップ
        if ( ! SetupPlayer() )
        {
            return false;
        }

        // ボイス確保
        if ( ! AllocVoices() )
        {
            FreeStreamBuffers();
            return false;
        }

        // セットアップコールバック呼び出し
        m_PrepareCounter = 0;
        for ( unsigned int i = 0; i < m_BufferBlockCount; i++ )
        {
            UpdateLoadingBlockIndex();
            m_PrepareCounter++;
        }
    }
    else
    {
        m_IsPrepared = true;
    }

    for ( unsigned int ch = 0; ch < m_ChannelCount; ch++ )
    {
        AdpcmParam* pAdpcmParam = adpcmParam[ch];
        if ( pAdpcmParam != NULL )
        {
            m_Channels[ ch ].m_pVoice->SetAdpcmParam( 0, *pAdpcmParam );
        }
    }

    return true;
}

bool StreamSoundPlayer::LoadStreamData(
    bool result,
    const LoadDataParam& loadDataParam
)
{
    return LoadStreamData(
        result,
        loadDataParam,
        false,
        0,
        0
    );
}

bool StreamSoundPlayer::LoadStreamData(
    bool result,
    const LoadDataParam& loadDataParam,
    bool usePrefetchFlag,
    u32 currentPrefetchBlockIndex,
    u32 currentPrefetchBlockBytes
)
{
    if ( ! m_IsInitialized )
    {
        // 終了処理後に、コマンドバッファにたまっていたコマンドが処理され
        // ここに来る場合がある
        return false;
    }
    if ( ! result ) {
        NW_WARNING( false, "Task error is occured." );
        m_FinishFlag = true;
        Stop();
        return false;
    }

    if ( loadDataParam.samples > 0 )
    {
        for ( unsigned int ch = 0; ch < m_ChannelCount ; ch++ )
        {
            const void* bufferAddress;

            if ( ! usePrefetchFlag )
            {
                NW_UNUSED_VARIABLE(currentPrefetchBlockIndex); NW_UNUSED_VARIABLE(currentPrefetchBlockBytes);
                bufferAddress = ut::AddOffsetToPtr(
                    m_Channels[ ch ].m_pBufferAddress,
                    ( m_StreamDataInfo.blockSize + StreamSoundLoader::DATA_BLOCK_SIZE_MARGIN ) * loadDataParam.blockIndex );
            }
            else
            {
                bufferAddress = ut::AddOffsetToPtr(
                    m_PrefetchDataInfo.dataAddress,
                    m_StreamDataInfo.blockSize * m_ChannelCount * currentPrefetchBlockIndex + currentPrefetchBlockBytes * ch );
            }

            // WaveBuffer への割り付け
            WaveBuffer* waveBuffer =
                &m_Channels[ch].m_WaveBuffer[loadDataParam.blockIndex];
            AdpcmContext* pAdpcmContext = NULL;
            if ( loadDataParam.adpcmContextEnable )
            {
                pAdpcmContext = &m_Channels[ch].m_AdpcmContext[loadDataParam.blockIndex];
                *pAdpcmContext = loadDataParam.adpcmContext[ch];
            #ifdef ENABLE_PRINT_STREAM_INFO
                NW_LOG("[%s] predScale(%04x) yn1(%6d) yn2(%6d)\n", __FUNCTION__,
                        pAdpcmContext->pred_scale, pAdpcmContext->yn1, pAdpcmContext->yn2);
            #endif
            }
            waveBuffer->Initialize();
            waveBuffer->bufferAddress = bufferAddress;
            waveBuffer->sampleLength = loadDataParam.samples;
            waveBuffer->sampleOffset = loadDataParam.sampleOffset;
            waveBuffer->pAdpcmContext = pAdpcmContext;

            // StreamChannel へのアペンド
        #ifdef ENABLE_PRINT_STREAM_INFO
            NW_LOG("AppendWaveBuffer ch(%d) bufferAddr(%08x) length(%d) ofs(%d) blockIdx(%d)\n",
                   ch, bufferAddress, loadDataParam.samples, loadDataParam.sampleOffset, loadDataParam.blockIndex);
        #endif
        #if defined(NW_DEBUG) || defined(NW_DEVELOP)
            CheckSampleLength(waveBuffer->sampleLength, m_StreamDataInfo, m_FileType);
        #endif
            m_Channels[ch].AppendWaveBuffer( waveBuffer, loadDataParam.lastBlockFlag );
        }

        WaveBufferInfo* waveBufferInfo = &m_WaveBufferInfo[loadDataParam.blockIndex];
        waveBufferInfo->sampleBegin = loadDataParam.sampleBegin;
        waveBufferInfo->sampleLength = loadDataParam.samples;
        waveBufferInfo->loopCount = loadDataParam.loopCount;
    }

    if ( loadDataParam.lastBlockFlag )
    {
        m_LoadFinishFlag = true;
    }

    // バッファ書き換え完了
    if ( ! m_IsPrepared )
    {
        if ( ! usePrefetchFlag )
        {
            m_PrepareCounter--;
            if ( m_PrepareCounter == 0 || m_LoadFinishFlag )
            {
                m_IsPrepared = true;
            }
        }
    }

    return true;
}

void StreamSoundPlayer::StartPlayer()
{
    for ( unsigned int trackIndex = 0; trackIndex < m_TrackCount; trackIndex++ )
    {
        StreamTrack& track = m_Tracks[ trackIndex ];
        if ( ! track.m_ActiveFlag ) continue;

        // ボイスセットアップ
        for ( int ch = 0; ch < track.channelCount; ch++ )
        {
            StreamChannel* channel = track.m_pChannels[ch];
            if ( channel == NULL ) continue;

            MultiVoice* voice = channel->m_pVoice;
            if ( voice != NULL )
            {
                // blockOffset は使われない。かわりに LoadStreamData 内で設定される。
                voice->SetSampleFormat( m_StreamDataInfo.sampleFormat );
                voice->SetSampleRate( m_StreamDataInfo.sampleRate );
                voice->Start();
            }
        }
    }

    UpdatePauseStatus();

    m_StartedFlag = true;
}

void StreamSoundPlayer::FinishPlayer()
{
    if (m_pLoader != NULL)
    {
        m_pLoader->CancelRequest();
    }

    // ボイスの停止
    for ( unsigned int ch = 0; ch < m_ChannelCount; ch++ )
    {
        MultiVoice* voice = m_Channels[ch].m_pVoice;
        if ( voice != NULL )
        {
            voice->Stop();
        }
    }

    // プレイヤー登録の解除
    if ( m_IsRegisterPlayerCallback )
    {
        SoundThread::GetInstance().UnregisterPlayerCallback( this );
        m_IsRegisterPlayerCallback = false;
    }

    if ( m_StartedFlag )
    {
        m_StartedFlag = false;
    }
}

/*---------------------------------------------------------------------------*
  Name:         SetupPlayer

  Description:  ストリームのセットアップ

  Arguments:    header - ストリームファイルヘッダ

  Returns:      成功したかどうか
 *---------------------------------------------------------------------------*/
bool StreamSoundPlayer::SetupPlayer()
{
    NW_NULL_ASSERT( m_pBufferPool );
    const size_t strmBufferSize = m_pBufferPool->GetBlockSize();    // 1ch 分のバッファサイズ

    // ブロックサイズの取得
    if ( m_StreamDataInfo.blockSize > StreamSoundLoader::DATA_BLOCK_SIZE_BASE )
    {
        NW_WARNING( false, "Too large stream data block size." );
        return false;
    }
    m_BufferBlockCount = strmBufferSize / ( m_StreamDataInfo.blockSize + StreamSoundLoader::DATA_BLOCK_SIZE_MARGIN );
    if ( m_BufferBlockCount < 4 )
    {
        NW_WARNING( false, "Too small stream buffer size." );
        return false;
    }
    if ( m_BufferBlockCount > STRM_DATA_LOAD_TASK_MAX )
    {
        // 非同期ロードコマンド発行のため、上限を設定
        m_BufferBlockCount = STRM_DATA_LOAD_TASK_MAX;
    }

    m_LoadingBufferBlockIndex = 0;
    m_PlayingBufferBlockIndex = 0;
    m_LastPlayFinishBufferBlockIndex = 0;

    return true;
}

/*---------------------------------------------------------------------------*
  Name:         AllocStreamBuffers

  Description:  全チャンネルのストリームバッファを確保

  Arguments:    なし

  Returns:      成功したらtrue
 *---------------------------------------------------------------------------*/
bool StreamSoundPlayer::AllocStreamBuffers()
{
    // 全チャンネル分のストリームバッファ確保
    for( unsigned index = 0; index < m_ChannelCount; index++ )
    {
        // 1 ブロック分 (1ch 分) のバッファを確保
        void* strmBuffer = m_pBufferPool->Alloc();
        if ( strmBuffer == NULL )
        {
            // ひとつでも失敗したら、すべてのバッファを返却し false を返す
            for( unsigned int i = 0 ; i < index ; i++ )
            {
                m_pBufferPool->Free( m_Channels[i].m_pBufferAddress );
                m_Channels[i].m_pBufferAddress = NULL;
            }
            return false;
        }

        m_Channels[ index ].m_pBufferAddress = strmBuffer;
    }
    return true;
}

/*---------------------------------------------------------------------------*
  Name:         FreeStreamBuffers

  Description:  全チャンネルのストリームバッファを開放

  Arguments:    なし

  Returns:      なし
 *---------------------------------------------------------------------------*/
void StreamSoundPlayer::FreeStreamBuffers()
{
    // ストリームバッファ開放
    for( unsigned int index = 0; index < m_ChannelCount; index++ )
    {
        if ( m_Channels[ index ].m_pBufferAddress == NULL )
        {
            continue;
        }

        m_pBufferPool->Free( m_Channels[index].m_pBufferAddress );
        m_Channels[ index ].m_pBufferAddress = NULL;
    }
}

// 全 StreamTrack に対し、Voice を割り当てる
bool StreamSoundPlayer::AllocVoices()
{
    for ( unsigned int channelIndex = 0; channelIndex < m_ChannelCount; channelIndex++ )
    {
        StreamChannel& channel = m_Channels[ channelIndex ];

        // MultiVoice 確保
        MultiVoice* voice = MultiVoiceManager::GetInstance().AllocVoice(
            1,      // channelCount
            MultiVoice::PRIORITY_NODROP,
            VoiceCallbackFunc,
            &channel,
            GetVoiceRendererType()
        );
        if ( voice == NULL )
        {
            for ( unsigned int i = 0; i < channelIndex; i++ )
            {
                StreamChannel& c = m_Channels[ i ];
                if ( c.m_pVoice != NULL )
                {
                    c.m_pVoice->Free();
                    c.m_pVoice = NULL;
                }
            }
            return false;
        }
        channel.m_pVoice = voice;
    }
    return true;
}

/*---------------------------------------------------------------------------*
  Name:         FreeAllTrackVoice

  Description:  ストリーム用チャンネルの解放

  Arguments:    player - ストリームオブジェクト

  Returns:      None.
 *---------------------------------------------------------------------------*/
void StreamSoundPlayer::FreeVoices()
{
    for ( unsigned int ch = 0; ch < m_ChannelCount; ch++ )
    {
        StreamChannel& channel = m_Channels[ ch ];
        if ( channel.m_pVoice != NULL )
        {
            channel.m_pVoice->Free();
            channel.m_pVoice = NULL;
        }
    }
}

bool StreamSoundPlayer::TryAllocLoader()
{
    if (m_pLoader != NULL)
    {
        return true;    // 割り当て済み
    }

    if (m_pLoaderManager == NULL)
    {
        return false;
    }

    StreamSoundLoader* loader = m_pLoaderManager->Alloc();
    if (loader == NULL)
    {
        return false;
    }

    m_pLoader = loader;
    return true;
}

void StreamSoundPlayer::FreeLoader()
{
    if (m_pLoader == NULL)
    {
        return;
    }

    if (m_pLoaderManager == NULL)
    {
        return;
    }

    m_pLoaderManager->Free(m_pLoader);
    m_pLoader = NULL;
}

/*---------------------------------------------------------------------------*
  Name:         Update

  Description:  パラメータ更新
                （サウンドスレッドから呼びだされる）

  Arguments:    None.

  Returns:      None.
 *---------------------------------------------------------------------------*/
void StreamSoundPlayer::Update()
{
    if (TryAllocLoader() == true)
    {
        if (!m_IsSucceedPrepare)
        {
            m_pLoader->Initialize();
            Setup(m_SetupArg);
            Prepare(m_PrepareArg);
        }
    }
    if (m_pLoader == NULL)
    {
        return;
    }

    m_pLoader->Update();

    if (m_DelayCount > 0)
    {
        --m_DelayCount;
        return;
    }

    // フラグチェック
    if (m_UseDelayCount)
    {
        if ( m_IsPrepared && !m_StartedFlag )
        {
            StartPlayer();
        }
    }

    UpdateBuffer();

    // ストリームチャンネルの停止チェック
    if ( m_StartedFlag )
    {
        for ( unsigned int ch = 0; ch < m_ChannelCount; ch++ )
        {
            StreamChannel& channel = m_Channels[ ch ];
            if ( channel.m_pVoice == NULL )
            {
                Stop();
                m_FinishFlag = true;
                return;
            }
        }

        // パラメータのアップデート
        for ( unsigned int trackIndex = 0; trackIndex < m_TrackCount; trackIndex++ )
        {
            UpdateVoiceParams( &m_Tracks[ trackIndex ] );
        }
    }

    // ロード待ちが解消されたかをチェック
    if ( m_LoadWaitFlag )
    {
        if ( ! m_pLoader->IsBusy() && ! CheckDiskDriveError() )
        {
            m_LoadWaitFlag = false;
            UpdatePauseStatus();
        }
    }

    // ロード遅延メッセージ出力
    if ( m_ReportLoadingDelayFlag )
    {
        NW_WARNING( false, "Pause stream because of loading delay." );
        m_ReportLoadingDelayFlag = false;
    }
}

void StreamSoundPlayer::UpdateVoiceParams( StreamTrack* track )
{
    if ( ! track->m_ActiveFlag )
    {
        return;
    }

    TrackData trackData;
    trackData.Set(track);

    // volume
    f32 volume = 1.0f;
    volume *= GetVolume();
    volume *= trackData.volume;
    volume *= track->m_Volume;

    // pitch
    f32 pitchRatio = 1.0f;
    pitchRatio *= GetPitch();
    pitchRatio *= m_ItemData.pitch;

    if (m_FileType == ADTS_FILE_TYPE)
    {
        if (pitchRatio > AAC_PITCH_MAX)
        {
            NW_WARNING(false, "If file type is aac, pitch must be less than %lf.", AAC_PITCH_MAX);
            pitchRatio = AAC_PITCH_MAX;
        }
    }

    // lpf freq
    f32 lpfFreq = trackData.lpfFreq;
    lpfFreq += GetLpfFreq();

    // biquad filter
    int biquadType = trackData.biquadType;
    f32 biquadValue = trackData.biquadValue;
    int handleBiquadType = GetBiquadFilterType();
    if ( handleBiquadType != static_cast<int>(BIQUAD_FILTER_TYPE_INHERIT) )
    {
        biquadType  = handleBiquadType;
        biquadValue = GetBiquadFilterValue();
    }

    // outputLine
    u32 outputLine = 0;
    if ( track->m_OutputLine != -1 )
    {
        outputLine = track->m_OutputLine;
    }
    else
    {
        outputLine = GetOutputLine();
    }

    // TV 出力向けパラメータ
    OutputParam tvParam;
    {
        tvParam = GetTvParam();
        SetOutputParam( &tvParam, track->m_TvParam, trackData );
    }

    // DRC 出力向けパラメータ
    OutputParam drcParam[DRC_OUT_COUNT];
    for ( u32 drc = 0; drc < DRC_OUT_COUNT; drc++ )
    {
        OutputParam& param = drcParam[drc];
        param = GetDrcParam( drc );
        SetOutputParam( &param, track->m_DrcParam[drc], trackData );
    }

    for ( int ch = 0; ch < track->channelCount; ch++ )
    {
        MultiVoice* voice = track->m_pChannels[ch]->m_pVoice;

        if ( voice != NULL )
        {
            voice->SetVolume( volume );
            voice->SetPitch( pitchRatio );
            voice->SetLpfFreq( lpfFreq );
            voice->SetBiquadFilter( biquadType, biquadValue );
            voice->SetFrontBypass(track->flags != 0);
            voice->SetRemoteFilter( GetRemoteFilter() );
            voice->SetOutputLine( outputLine );

            if ( track->channelCount == 1 )
            {
                voice->SetTvParam( tvParam );
                for ( u32 drc = 0; drc < DRC_OUT_COUNT; drc++ )
                {
                    voice->SetDrcParam( drc, drcParam[drc] );
                }
            }
            else if ( track->channelCount == 2 )
            {
                ApplyTvOutputParamForMultiChannel( &tvParam, voice, ch, static_cast<MixMode>(tvParam.mixMode) );
                for ( u32 drc = 0; drc < DRC_OUT_COUNT; drc++ )
                {
                    ApplyDrcOutputParamForMultiChannel( &drcParam[drc], voice, ch, static_cast<MixMode>(tvParam.mixMode), drc );
                }
            }

            for ( u32 i = 0; i < REMOTE_OUT_COUNT; i++ )
            {
                voice->SetRemoteParam( i, GetRemoteParam(i) );
            }

        }
    }
}

void StreamSoundPlayer::SetOutputParam(OutputParam* pOutputParam, const OutputParam& trackParam, const TrackData& trackData)
{
    pOutputParam->volume *= trackParam.volume;

    for ( int i = 0; i < WAVE_CHANNEL_MAX; i++ )
    {
        for ( int j = 0; j < CHANNEL_INDEX_NUM; j++ )
        {
            pOutputParam->mixParameter[i].ch[j] *= trackParam.mixParameter[i].ch[j];
        }
    }
    pOutputParam->pan += trackData.pan + trackParam.pan;
    pOutputParam->span += trackData.span + trackParam.span;

    pOutputParam->mainSend += trackParam.mainSend;
    pOutputParam->mainSend += trackData.mainSend + m_ItemData.mainSend;

    for ( int i = 0; i < AUX_BUS_NUM; i++ )
    {
        pOutputParam->fxSend[i] += trackParam.fxSend[i];
        pOutputParam->fxSend[i] += trackData.fxSend[i] + m_ItemData.fxSend[i];
    }
}

void StreamSoundPlayer::MixSettingForOutputParam(OutputParam* pOutputParam, s32 channelIndex, MixMode mixMode)
{
    if ( mixMode == MIX_MODE_PAN )
    {
        // パン補正
#if 0
        // 1trk == 2ch の場合は、パンモード別のパン計算を行う
        switch ( GetPanMode() )
        {
        case nw::snd::PAN_MODE_BALANCE:
            if ( channelIndex == 0 ) { pan = -1.0f; } // L に振る
            if ( channelIndex == 1 ) { pan =  1.0f; } // R に振る
            break;
        case nw::snd::PAN_MODE_DUAL:
            if ( channelIndex == 0 ) { pan -= 1.0f; } // L に寄せる
            if ( channelIndex == 1 ) { pan += 1.0f; } // R に寄せる
            break;
        }
#else
        // TODO: DualPan, BalancePan に対応するには、
        //       StreamTrack に Voiceを持たせる形にする必要がある。
        //       上記の #if 0 の部分は、実装イメージ。
        //       BalancePan で pan を -1.0f や 1.0f に固定してしまうと、
        //       GetPan() やトラックごとのデータパン・API パンが反映されなく鳴る。
        if ( channelIndex == 0 )
        {
            pOutputParam->pan -= 1.0f;
        }
        else if ( channelIndex == 1 )
        {
            pOutputParam->pan += 1.0f;
        }
#endif
    }
    else
    {
        // ミックスモード補正

        // ストリームサウンドでは、波形のチャンネル毎にボイスを分けるため
        // サラウンドボリュームを 0 ch 側に設定し直す。
        if ( channelIndex == 1 )
        {
            for ( u32 i = 0; i < CHANNEL_INDEX_NUM; i++ )
            {
                pOutputParam->mixParameter[0].ch[i] = pOutputParam->mixParameter[channelIndex].ch[i];
                pOutputParam->mixParameter[channelIndex].ch[i] = 0.0f;
            }
        }
    }
}

void StreamSoundPlayer::ApplyTvOutputParamForMultiChannel(OutputParam* pOutputParam, MultiVoice* pVoice, s32 channelIndex, MixMode mixMode)
{
    // 一時保存
    f32 originalPan = pOutputParam->pan;
    MixParameter originalMixParameter[WAVE_CHANNEL_MAX];
    for ( u32 i = 0; i < WAVE_CHANNEL_MAX; i++ )
    {
        for ( u32 j = 0; j < CHANNEL_INDEX_NUM; j++ )
        {
            originalMixParameter[i].ch[j] = pOutputParam->mixParameter[i].ch[j];
        }
    }

    MixSettingForOutputParam(pOutputParam, channelIndex, mixMode);

    // pVoice へのパラメータセット
    pVoice->SetTvParam( *pOutputParam );

    // 書き戻し
    pOutputParam->pan = originalPan;
    for ( u32 i = 0; i < WAVE_CHANNEL_MAX; i++ )
    {
        for ( u32 j = 0; j < CHANNEL_INDEX_NUM; j++ )
        {
            pOutputParam->mixParameter[i].ch[j] = originalMixParameter[i].ch[j];
        }
    }
}

void StreamSoundPlayer::ApplyDrcOutputParamForMultiChannel(OutputParam* pOutputParam, MultiVoice* pVoice, s32 channelIndex, MixMode mixMode, u32 drcIndex)
{
    // 一時保存
    f32 originalPan = pOutputParam->pan;
    MixParameter originalMixParameter[WAVE_CHANNEL_MAX];
    for ( u32 i = 0; i < WAVE_CHANNEL_MAX; i++ )
    {
        for ( u32 j = 0; j < CHANNEL_INDEX_NUM; j++ )
        {
            originalMixParameter[i].ch[j] = pOutputParam->mixParameter[i].ch[j];
        }
    }

    MixSettingForOutputParam(pOutputParam, channelIndex, mixMode);

    // pVoice へのパラメータセット
    pVoice->SetDrcParam( drcIndex, *pOutputParam );

    // 書き戻し
    pOutputParam->pan = originalPan;
    for ( u32 i = 0; i < WAVE_CHANNEL_MAX; i++ )
    {
        for ( u32 j = 0; j < CHANNEL_INDEX_NUM; j++ )
        {
            pOutputParam->mixParameter[i].ch[j] = originalMixParameter[i].ch[j];
        }
    }
}

void StreamSoundPlayer::ItemData::Set( const SetupArg& arg )
{
    // pitch
    pitch = arg.pitch;

    // send
    mainSend = static_cast<f32>( arg.mainSend ) / 127.0f - 1.0f;
    for ( int i = 0; i < AUX_BUS_NUM; i++ )
    {
        fxSend[ i ] = static_cast<f32>( arg.fxSend[i] ) / 127.0f;
    }
}

void StreamSoundPlayer::TrackData::Set( const StreamTrack* track )
{
    // volume
    volume = static_cast<f32>( track->volume ) / 127.0f;

    // lpf freq
    lpfFreq = static_cast<f32>( track->lpfFreq ) / 64.0f;

    // biquad filter;
    biquadType = static_cast<int>( track->biquadType );
    biquadValue = static_cast<f32>( track->biquadValue ) / 127.0f;

    // pan
    if ( track->pan <= 1 )
    {
        pan = static_cast<f32>( static_cast<int>( track->pan ) - 63 ) / 63.0f;
    }
    else
    {
        pan = static_cast<f32>( static_cast<int>( track->pan ) - 64 ) / 63.0f;
    }

    // span
    if ( track->span <= 63 )
    {
        span = static_cast<f32>( track->span ) / 63.0f;
    }
    else
    {
        span = static_cast<f32>( track->span + 1 ) / 64.0f;
    }

    // send
    mainSend = static_cast<f32>( track->mainSend ) / 127.0f - 1.0f;
    for ( int i = 0; i < AUX_BUS_NUM; i++ )
    {
        fxSend[i] = static_cast<f32>( track->fxSend[i] ) / 127.0f;
    }
}

/*---------------------------------------------------------------------------*
  Name:         StreamSoundPlayer::CheckDiskDriveError

  Description:  ディスクドライブのエラーが発生しているかどうかチェックします。

  Arguments:    None.

  Returns:      エラーが発生しているかどうかを返します。
                DVDストリームでない場合は、falseを返します。
  *---------------------------------------------------------------------------*/
bool StreamSoundPlayer::CheckDiskDriveError() const
{
    // TODO: 関数名と挙動があっていない
    return nw::snd::SoundSystem::detail_IsStreamLoadWait();
}

/*---------------------------------------------------------------------------*
  Name:         UpdateBuffer

  Description:  再生バッファの更新
                （AXコールバックから呼びだされる）

  Arguments:    None.

  Returns:      None.
 *---------------------------------------------------------------------------*/
void StreamSoundPlayer::UpdateBuffer()
{
    if ( ! m_StartedFlag )
    {
        return;
    }
    if ( ! m_Tracks[ 0 ].m_ActiveFlag )
    {
        return;
    }

    if ( m_IsPreparedPrefetch && !m_IsPrepared )
    {
        return;
    }

    if ( CheckDiskDriveError() )
    {
        m_LoadWaitFlag = true;
        UpdatePauseStatus();
    }

    // -------------------------
    // バッファのアップデート
    // バッファブロックの境界を越えたかチェックし、ファイルロードを発行する
    for ( ;; )
    {
        bool isAllDone = true;
        for ( u32 ch = 0; ch < m_ChannelCount; ch++ )
        {
            if ( m_Channels[ch].m_WaveBuffer[m_PlayingBufferBlockIndex].status != WaveBuffer::STATUS_DONE )
            {
                isAllDone = false;
                break;
            }
        }
        if ( isAllDone == false )
        {
            break;
        }

        // すべてのチャンネルの WaveBuffer が STATUS_DONE になってから更新する
        for ( u32 ch = 0; ch < m_ChannelCount; ch++ )
        {
            m_Channels[ch].m_WaveBuffer[m_PlayingBufferBlockIndex].Initialize();
        }

        // 再生ブロックの更新
        const WaveBufferInfo* waveBufferInfo = &m_WaveBufferInfo[m_PlayingBufferBlockIndex];
        m_LoopCounter += waveBufferInfo->loopCount;

        // ロードブロックの更新
        UpdateLoadingBlockIndex();

        // バッファブロック番号更新
        m_LastPlayFinishBufferBlockIndex = m_PlayingBufferBlockIndex;
        m_PlayingBufferBlockIndex++;
        if ( m_PlayingBufferBlockIndex >= m_BufferBlockCount ) m_PlayingBufferBlockIndex = 0;
    }

    NW_ASSERT(m_pLoader != NULL);

    // -------------------------
    // ロードが追いついているかをチェック
    if ( ! m_LoadWaitFlag && m_pLoader->IsBusy() && IsBufferEmpty() )
    {
        // ロードが追いついてない
        m_ReportLoadingDelayFlag = true;
        m_LoadWaitFlag = true;
        UpdatePauseStatus();
    }
}

bool StreamSoundPlayer::IsBufferEmpty() const
{
    for ( unsigned int i = 0; i < m_BufferBlockCount; i++ )
    {
        switch ( m_Channels[0].m_WaveBuffer[i].status )
        {
        case nw::snd::WaveBuffer::STATUS_WAIT:
        case nw::snd::WaveBuffer::STATUS_PLAY:
            return false;
        default:
            break;
        }
    }
    return true;
}

// SetupTrack (サウンドアーカイブバイナリ → StreamDataInfoDetail)
// ApplyTrackDataInfo (StreamDataInfoDetail → StreamTrack)
bool StreamSoundPlayer::SetupTrack( const SetupArg& arg )
{
    // 有効なトラックを順次アクティブ化
    u32 bitMask = arg.allocTrackFlag;
    u32 trackIndex = 0;
    while ( bitMask != 0 )
    {
        if ( bitMask & 0x01 )
        {
            if ( trackIndex >= STRM_TRACK_NUM )
            {
                NW_WARNING(
                        false, "Too large track index (%d). Max track index is %d.",
                        trackIndex, STRM_TRACK_NUM-1
                        );
                break;
            }

            m_Tracks[ trackIndex ].m_ActiveFlag = true;
        }

        bitMask >>= 1;
        trackIndex++;
    }

    m_TrackCount = ut::Min( trackIndex, STRM_TRACK_NUM );
    if ( m_TrackCount == 0 )
    {
        Finalize();
        return false;
    }

    // サウンドアーカイブ (*.bXsar) 埋め込みのトラック情報反映
    for (int i = 0; i < STRM_TRACK_NUM; i++)
    {
        TrackDataInfo& data = m_StreamDataInfo.trackInfo[i];
        data = arg.trackInfos.track[i];
    }

    return true;
}

void StreamSoundPlayer::SetPrepareBaseArg(const PrepareBaseArg& baseArg)
{
    m_DelayCount      = ToDelayCount(baseArg.delayTime);
    m_UseDelayCount   = m_DelayCount > 0;
    m_StartOffsetType = baseArg.startOffsetType;
    m_StartOffset     = baseArg.offset;

    m_ActiveFlag      = true;
}

void StreamSoundPlayer::RequestLoadHeader(const PrepareArg& arg)
{
    NW_ASSERT(m_pLoader != NULL);

    // タスク
    m_TaskRequestIndex = s_TaskRequestIndexCount++;

    m_pLoader->m_StreamRegionCallbackFunc = arg.baseArg.regionCallback;
    m_pLoader->m_StreamRegionCallbackArg = arg.baseArg.regionCallbackArg;

    m_pLoader->m_PlayerHandle = this;
    m_pLoader->m_DataInfo = &m_StreamDataInfo;
    m_pLoader->m_FileType = static_cast<StreamFileType>(m_FileType);
    m_pLoader->m_LoopFlag = m_LoopFlag;
#if defined(NW_RELEASE)
    m_pLoader->m_IsStreamOpenFailureHalt = true;
#else
    m_pLoader->m_IsStreamOpenFailureHalt = nw::snd::SoundSystem::detail_IsStreamOpenFailureHaltEnabled();
#endif
    m_pLoader->m_LoopStart = m_LoopStart;
    m_pLoader->m_LoopEnd = m_LoopEnd;
    m_pLoader->m_RequestIndex = m_TaskRequestIndex;
    ut::strcpy( m_pLoader->m_FilePath, FILE_PATH_MAX, arg.baseArg.filePath );
    m_pLoader->m_pFileStream = NULL;
    m_pLoader->m_FileStreamHookParam = arg.baseArg.fileStreamHookParam;
    m_pLoader->m_pCacheBuffer = arg.cacheBuffer;
    m_pLoader->m_CacheSize = arg.cacheSize;
#if defined(NW_PLATFORM_CAFE)
    m_pLoader->m_pFsClient = arg.baseArg.fsClient;
    m_pLoader->m_pFsCommandBufferPool = arg.baseArg.fsCommandBufferPool;
    m_pLoader->m_FsPriority = arg.baseArg.fsPriority;
#endif
    m_pLoader->RequestLoadHeader();
}

bool StreamSoundPlayer::ReadPrefetchFile(StreamSoundPrefetchFileReader& reader)
{
    // プリフェッチ情報の読み込み
    StreamSoundPrefetchFileReader::PrefetchDataInfo prefetchInfo;
    if ( !reader.ReadPrefetchDataInfo(&prefetchInfo, 0) ) // 現状プリフェッチデータは１つのみ
    {
        return false;
    }

    m_PrefetchDataInfo.startFrame   = prefetchInfo.startFrame;
    m_PrefetchDataInfo.prefetchSize = prefetchInfo.prefetchSize;
    m_PrefetchDataInfo.dataAddress  = prefetchInfo.dataAddress;

    // ストリームサウンド情報の読み込み
    StreamSoundFile::StreamSoundInfo info;
    reader.ReadStreamSoundInfo(&info);

    m_StreamDataInfo.channelCount   = reader.GetChannelCount();
    m_StreamDataInfo.SetStreamSoundInfo(info);

    m_ChannelCount = ut::Min( m_StreamDataInfo.channelCount, STRM_CHANNEL_NUM );

    return true;
}

bool StreamSoundPlayer::ApplyStreamDataInfo(const StreamDataInfoDetail& streamDataInfo)
{
    if ( streamDataInfo.loopFlag )
    {
        NW_ASSERT( streamDataInfo.sampleCount - streamDataInfo.loopStart >= DATA_BLOCK_SIZE_MARGIN_SAMPLES );
    }
    NW_ASSERT_ALIGN32( static_cast<u32>(streamDataInfo.blockSampleCount) );

    // ループなし＋波形終端以降のオフセット再生の場合は失敗
    if ( !IsValidStartOffset(streamDataInfo) )
    {
        m_FinishFlag = true;
        Stop();
        return false;
    }

#ifdef ENABLE_PRINT_STREAM_INFO
    // DEBUG: ストリーム情報ダンプ
    {
        NW_LOG("*** STREAM INFO ***\n");
        NW_LOG("  fmt(%d) rate(%d) loopFlag(%d) loopStart(%d) sampleCount(%d) ch(%d) track(%d)\n",
               streamDataInfo.sampleFormat, streamDataInfo.sampleRate,
               streamDataInfo.loopFlag, streamDataInfo.loopStart,
               streamDataInfo.sampleCount,
               streamDataInfo.channelCount, streamDataInfo.trackCount);
        NW_LOG("  [normal block] sampleCount(%d) size(%d)\n",
               streamDataInfo.blockSampleCount, streamDataInfo.blockSize);
        NW_LOG("  [last block  ] sampleCount(%d) size(%d)\n",
               streamDataInfo.lastBlockSampleCount, streamDataInfo.lastBlockSize);

        for ( int i = 0; i < STRM_TRACK_NUM; i++ )
        {
            const TrackDataInfo& trk = streamDataInfo.trackInfo[i];
            NW_LOG("  trk[%d] vol(%3d) pan(%4d) ch(%d) chIdx(%d, %d)\n",
                    i, trk.volume, trk.pan, trk.channelCount,
                    trk.channelIndex[0], trk.channelIndex[1] );
        }
    }
#endif

    // トラックの情報を読み取る
    ApplyTrackDataInfo(streamDataInfo);

    return true;
}

u32 StreamSoundPlayer::GetOriginalPlaySamplePosition( u32 playSamplePosition, const StreamDataInfoDetail& streamDataInfo ) const
{
    if ( playSamplePosition > streamDataInfo.originalLoopEnd )
    {
        u32 loopRegionSize = streamDataInfo.originalLoopEnd - streamDataInfo.originalLoopStart;

        if ( loopRegionSize < LOOP_REGION_SIZE_MIN )
        {
            return streamDataInfo.originalLoopStart + ( playSamplePosition - streamDataInfo.originalLoopStart ) % loopRegionSize;
        }
        else
        {
            return streamDataInfo.originalLoopStart + ( playSamplePosition - streamDataInfo.originalLoopEnd );
        }
    }

    return playSamplePosition;
}

u32 StreamSoundPlayer::GetStartOffsetSamples(const StreamDataInfoDetail& streamDataInfo)
{
    u32 startOffsetSamples = 0;
    if ( m_StartOffsetType == START_OFFSET_TYPE_SAMPLE )
    {
        startOffsetSamples = m_StartOffset;
    }
    else if ( m_StartOffsetType == START_OFFSET_TYPE_MILLISEC )
    {
        startOffsetSamples =
            static_cast<u32>(ut::Clamp( static_cast<u64>(m_StartOffset) * streamDataInfo.sampleRate / 1000,
                        static_cast<u64>(0), static_cast<u64>(UINT_MAX) ));
    }

    return startOffsetSamples;
}

bool StreamSoundPlayer::IsValidStartOffset(const StreamDataInfoDetail& streamDataInfo)
{
    if (streamDataInfo.loopFlag == false)
    {
        if (GetStartOffsetSamples(streamDataInfo) >= streamDataInfo.sampleCount)
        {
            return false;
        }
    }

    return true;
}

void StreamSoundPlayer::ApplyTrackDataInfo(const StreamDataInfoDetail& streamDataInfo)
{
    for ( unsigned int i = 0; i < m_TrackCount; i++ )
    {
        const TrackDataInfo& trackInfo = streamDataInfo.trackInfo[i];
        m_Tracks[i].volume = trackInfo.volume;
        m_Tracks[i].pan = trackInfo.pan;
        m_Tracks[i].span = trackInfo.span;
        m_Tracks[i].mainSend = trackInfo.mainSend;
        for ( int j = 0; j < AUX_BUS_NUM; j++ )
        {
            m_Tracks[i].fxSend[j] = trackInfo.fxSend[j];
        }
        m_Tracks[i].lpfFreq = trackInfo.lpfFreq;
        m_Tracks[i].biquadType = trackInfo.biquadType;
        m_Tracks[i].biquadValue = trackInfo.biquadValue;
        m_Tracks[i].flags = trackInfo.flags;
        m_Tracks[i].channelCount = trackInfo.channelCount;

        // StreamChannel を StreamTrack に割り当て
        for ( int ch = 0; ch < trackInfo.channelCount; ch++ )
        {
            m_Tracks[i].m_pChannels[ch] = &m_Channels[ trackInfo.channelIndex[ch] ];
        }
    }
}

bool StreamSoundPlayer::LoadPrefetchBlocks(StreamSoundPrefetchFileReader& reader)
{
    PrefetchIndexInfo indexInfo;
    indexInfo.Initialize( m_StreamDataInfo );
    u32 sampleBeginPosition = 0;

    for ( u32 blockIndex = 0; blockIndex < m_BufferBlockCount; blockIndex++ )
    {
        PrefetchLoadDataParam loadDataParam;
        loadDataParam.Initialize();

        loadDataParam.prefetchBlockIndex = 0;
        loadDataParam.prefetchBlockBytes = 0;

        loadDataParam.sampleBegin = sampleBeginPosition;

        u32  blockOffsetFromLoopEnd = 0;
        if ( indexInfo.IsOverLastBlock( blockIndex ) )
        {
            blockOffsetFromLoopEnd = indexInfo.GetBlockOffsetFromLoopEnd( blockIndex );
        }

        if ( indexInfo.IsLastBlock( blockIndex, blockOffsetFromLoopEnd ) )
        {
            PreparePrefetchOnLastBlock( &loadDataParam, indexInfo );
        }
        else if ( m_StreamDataInfo.loopFlag && indexInfo.IsOverLastBlock( blockIndex ) )
        {
            if ( indexInfo.IsLoopStartBlock( blockOffsetFromLoopEnd ) )
            {
                // ループ開始ブロックの場合はサンプル開始位置をリセットする。loadDataParam.sampleBegin も上書きされるため注意
                sampleBeginPosition = 0;
                if ( !PreparePrefetchOnLoopStartBlock( &loadDataParam, indexInfo, reader ) )
                {
                    return false;
                }
            }
            else
            {
                PreparePrefetchOnLoopBlock( &loadDataParam, indexInfo, blockOffsetFromLoopEnd );
            }
        }
        else
        {
            if ( !PreparePrefetchOnNormalBlock( &loadDataParam, blockIndex, reader ) )
            {
                return false;
            }
        }

        // TODO: startOffset 対応時には、regionInfo を加味した adpcmContext の生成が必要
        loadDataParam.blockIndex = m_LoadingBufferBlockIndex;
        loadDataParam.sampleOffset = 0;
        sampleBeginPosition += loadDataParam.samples;

        LoadStreamData(
            true,
            loadDataParam,
            true,
            loadDataParam.prefetchBlockIndex,
            loadDataParam.prefetchBlockBytes
            );

        // ロードバッファブロックインデックスの更新
        // プリフェッチデータ使用時も更新するのは、waveBuffer の管理インデックスとしても使用されているため。
        m_LoadingBufferBlockIndex++;
        if ( m_LoadingBufferBlockIndex >= m_BufferBlockCount )
        {
            // ロードバッファループ
            m_LoadingBufferBlockIndex = 0;
        }

        if ( loadDataParam.lastBlockFlag )
        {
            break;
        }
    }

    return true;
}

void StreamSoundPlayer::PreparePrefetchOnLastBlock(PrefetchLoadDataParam* param, const PrefetchIndexInfo& indexInfo)
{
    // 終端ブロック処理
    param->samples            = m_StreamDataInfo.lastBlockSampleCount;
    param->prefetchBlockBytes = m_StreamDataInfo.lastBlockSize;
    param->prefetchBlockIndex = indexInfo.lastBlockIndex;

    if ( !m_StreamDataInfo.loopFlag )
    {
        param->lastBlockFlag = true;

        m_PrefetchOffset     = 0;
    }
    else
    {
        param->loopCount++;

        m_PrefetchOffset     = m_StreamDataInfo.loopStart;
    }
}

bool StreamSoundPlayer::PreparePrefetchOnLoopStartBlock(PrefetchLoadDataParam* param, const PrefetchIndexInfo& indexInfo, StreamSoundPrefetchFileReader& reader)
{
    param->samples            = m_StreamDataInfo.blockSampleCount;
    param->prefetchBlockBytes = m_StreamDataInfo.blockSize;
    param->prefetchBlockIndex = indexInfo.loopStartBlockIndex;
    param->sampleBegin        = indexInfo.loopStartInBlock;
    m_PrefetchOffset          = m_StreamDataInfo.blockSampleCount * ( indexInfo.loopStartBlockIndex + 1 );

    if ( m_StreamDataInfo.sampleFormat == SAMPLE_FORMAT_DSP_ADPCM )
    {
        // ループ先頭部分の ADPCM 情報を読み取る
        if ( !SetAdpcmLoopInfo(reader, m_StreamDataInfo, m_PrefetchAdpcmParam, param->adpcmContext) )
        {
            return false;
        }

        param->adpcmContextEnable = true;
    }

    return true;
}

void StreamSoundPlayer::PreparePrefetchOnLoopBlock(PrefetchLoadDataParam* param, const PrefetchIndexInfo& indexInfo, u32 blockOffsetFromLoopEnd)
{
    param->samples             = m_StreamDataInfo.blockSampleCount;
    param->prefetchBlockBytes  = m_StreamDataInfo.blockSize;
    param->prefetchBlockIndex  = indexInfo.loopStartBlockIndex + blockOffsetFromLoopEnd - 1;
    m_PrefetchOffset          += m_StreamDataInfo.blockSampleCount;
}

bool StreamSoundPlayer::PreparePrefetchOnNormalBlock(PrefetchLoadDataParam* param, u32 blockIndex, StreamSoundPrefetchFileReader& reader)
{
    param->samples             = m_StreamDataInfo.blockSampleCount;
    param->prefetchBlockBytes  = m_StreamDataInfo.blockSize;
    param->prefetchBlockIndex  = blockIndex;
    m_PrefetchOffset          += m_StreamDataInfo.blockSampleCount;

    if ( m_StreamDataInfo.sampleFormat == SAMPLE_FORMAT_DSP_ADPCM && param->prefetchBlockIndex == 0 )
    {
        // 先頭サンプル部分の ADPCM 情報を読み取る
        if ( !SetAdpcmInfo(reader, m_StreamDataInfo, m_PrefetchAdpcmParam, param->adpcmContext) )
        {
            return false;
        }

        param->adpcmContextEnable = true;
    }

    return true;
}

void StreamSoundPlayer::PrefetchIndexInfo::Initialize( const StreamDataInfoDetail& streamDataInfo )
{
    lastBlockIndex      = streamDataInfo.GetLastBlockIndex();
    loopStartInBlock    = streamDataInfo.GetLoopStartInBlock();
    loopStartBlockIndex = streamDataInfo.GetLoopStartBlockIndex(loopStartInBlock);
    loopBlockNum        = lastBlockIndex - loopStartBlockIndex + 1;
}

bool StreamSoundPlayer::SetAdpcmInfo(
    StreamSoundPrefetchFileReader& reader,
    const StreamDataInfoDetail& streamDataInfo,
    AdpcmParam* adpcmParam,
    AdpcmContext* adpcmContext)
{
    NW_ASSERT( streamDataInfo.channelCount <= STRM_CHANNEL_NUM );

    for ( unsigned int ch = 0; ch < streamDataInfo.channelCount; ch++ )
    {
        DspAdpcmParam dspAdpcmParam;
        DspAdpcmLoopParam dspAdpcmLoopParam;

        if ( ! reader.ReadDspAdpcmChannelInfo(
            &dspAdpcmParam,
            &dspAdpcmLoopParam,
            ch ) )
        {
            return false;
        }

        for ( int i= 0; i < 8; i++ )
        {
            adpcmParam[ch].coef[i][0] = dspAdpcmParam.coef[i][0];
            adpcmParam[ch].coef[i][1] = dspAdpcmParam.coef[i][1];
        }

        m_Channels[ ch ].m_pVoice->SetAdpcmParam( 0, adpcmParam[ch] );

        adpcmContext[ch].pred_scale = dspAdpcmParam.predScale;
        adpcmContext[ch].yn1        = dspAdpcmParam.yn1;
        adpcmContext[ch].yn2        = dspAdpcmParam.yn2;
    }

    return true;
}

bool StreamSoundPlayer::SetAdpcmLoopInfo(
    StreamSoundPrefetchFileReader& reader,
    const StreamDataInfoDetail& streamDataInfo,
    AdpcmParam* adpcmParam,
    AdpcmContext* adpcmContext)
{
    NW_ASSERT( streamDataInfo.channelCount <= STRM_CHANNEL_NUM );

    for ( unsigned int ch = 0; ch < streamDataInfo.channelCount; ch++ )
    {
        DspAdpcmParam dspAdpcmParam;
        DspAdpcmLoopParam dspAdpcmLoopParam;

        if ( ! reader.ReadDspAdpcmChannelInfo(
            &dspAdpcmParam,
            &dspAdpcmLoopParam,
            ch ) )
        {
            return false;
        }

        for ( int i= 0; i < 8; i++ )
        {
            adpcmParam[ch].coef[i][0] = dspAdpcmParam.coef[i][0];
            adpcmParam[ch].coef[i][1] = dspAdpcmParam.coef[i][1];
        }

        m_Channels[ ch ].m_pVoice->SetAdpcmParam( 0, adpcmParam[ch] );

        adpcmContext[ch].pred_scale = dspAdpcmLoopParam.loopPredScale;
        adpcmContext[ch].yn1        = dspAdpcmLoopParam.loopYn1;
        adpcmContext[ch].yn2        = dspAdpcmLoopParam.loopYn2;
    }

    return true;
}

/*---------------------------------------------------------------------------*
  Name:         UpdateLoadingBlockIndex

  Description:  ロードブロックインデックスの更新

  Arguments:    None.

  Returns:      None.
 *---------------------------------------------------------------------------*/
void StreamSoundPlayer::UpdateLoadingBlockIndex()
{
    void* bufferAddress[ STRM_CHANNEL_NUM ];
    for( unsigned int ch=0 ; ch < m_ChannelCount ; ch++){
        NW_ASSERT(ch < STRM_CHANNEL_NUM);
        bufferAddress[ ch ] = ut::AddOffsetToPtr(
            m_Channels[ ch ].m_pBufferAddress,
            (m_StreamDataInfo.blockSize + StreamSoundLoader::DATA_BLOCK_SIZE_MARGIN) * m_LoadingBufferBlockIndex);
    }

    u32 startOffsetSamples = 0;
    if ( m_StartOffsetType == START_OFFSET_TYPE_SAMPLE )
    {
        startOffsetSamples = m_StartOffset;
    }
    else if ( m_StartOffsetType == START_OFFSET_TYPE_MILLISEC )
    {
        startOffsetSamples =
            static_cast<u32>(ut::Clamp( static_cast<u64>(m_StartOffset) * m_StreamDataInfo.sampleRate / 1000,
                    static_cast<u64>(0), static_cast<u64>(UINT_MAX) ));
        // NW_LOG("UINT_MAX(%d) (%X)\n", UINT_MAX, UINT_MAX);
    }
    u32 prefetchOffsetSamples = m_PrefetchOffset;

    m_StartOffset = 0;
    m_PrefetchOffset = 0;

    NW_ASSERT(m_pLoader != NULL);

    m_pLoader->RequestLoadData(
        bufferAddress,
        m_LoadingBufferBlockIndex,
        startOffsetSamples,
        prefetchOffsetSamples,
        m_StartedFlag ? internal::TaskManager::PRIORITY_HIGH : internal::TaskManager::PRIORITY_MIDDLE
    );

    // ロードバッファブロックインデックスの更新
    m_LoadingBufferBlockIndex++;
    if ( m_LoadingBufferBlockIndex >= m_BufferBlockCount )
    {
        // ロードバッファループ
        m_LoadingBufferBlockIndex = 0;
    }
}

/*---------------------------------------------------------------------------*
  Name:         UpdatePauseStatus

  Description:  ボイスのポーズステータスを更新

  Arguments:    None.

  Returns:      None.
 *---------------------------------------------------------------------------*/
void StreamSoundPlayer::UpdatePauseStatus()
{
    bool pauseStatus = false;
    if ( m_PauseFlag )
    {
        pauseStatus = true;
    }
    if ( m_LoadWaitFlag )
    {
        pauseStatus = true;
    }

    if ( pauseStatus != m_PauseStatus )
    {
        for ( unsigned int ch = 0; ch < m_ChannelCount; ch++ )
        {
            MultiVoice* voice = m_Channels[ ch ].m_pVoice;
            if ( voice != NULL )
            {
                voice->Pause( pauseStatus );
            }
        }
        m_PauseStatus = pauseStatus;
    }
}

/*---------------------------------------------------------------------------*
  Name:         VoiceCallbackFunc

  Description:  ボイスから呼びだされるコールバック関数

  Arguments:    voice - ボイス
                statuc - コールバックステータス
                arg - ユーザー引数

  Returns:      None.
 *---------------------------------------------------------------------------*/
void StreamSoundPlayer::VoiceCallbackFunc(
    MultiVoice* voice,
    MultiVoice::VoiceCallbackStatus status,
    void* arg
)
{
    NW_NULL_ASSERT( arg );
    StreamChannel* channel = reinterpret_cast<StreamChannel*>( arg );

    NW_ASSERT( channel->m_pVoice == voice );

    switch ( status )
    {
    case MultiVoice::CALLBACK_STATUS_FINISH_WAVE:
    case MultiVoice::CALLBACK_STATUS_CANCEL:
        voice->Free();
        channel->m_pVoice = NULL;
        break;
    case MultiVoice::CALLBACK_STATUS_DROP_VOICE:
    case MultiVoice::CALLBACK_STATUS_DROP_DSP:
        channel->m_pVoice = NULL;
        break;
    default:
        NW_ASSERTMSG( false, "Unknown Voice callback status %d", status );
        return;
    }
}

void StreamSoundPlayer::SetTrackVolume( u32 trackBitFlag, f32 volume )
{
    for( unsigned int trackNo = 0;
         trackNo < m_TrackCount && trackBitFlag != 0;
         trackNo++, trackBitFlag >>= 1
    )
    {
        if ( ( trackBitFlag & 0x01 ) == 0 )
        {
            continue;
        }
        m_Tracks[ trackNo ].m_Volume = volume;
    }
}

void StreamSoundPlayer::SetTrackInitialVolume( u32 trackBitFlag, u32 volume )
{
    for( unsigned int trackNo = 0;
         trackNo < m_TrackCount && trackBitFlag != 0;
         trackNo++, trackBitFlag >>= 1
    )
    {
        if ( ( trackBitFlag & 0x01 ) == 0 )
        {
            continue;
        }
        m_Tracks[ trackNo ].volume = static_cast<u8>(volume);
    }
}

void StreamSoundPlayer::SetTrackOutputLine( u32 trackBitFlag, u32 outputLine )
{
    for( unsigned int trackNo = 0;
         trackNo < m_TrackCount && trackBitFlag != 0;
         trackNo++, trackBitFlag >>= 1
    )
    {
        if ( ( trackBitFlag & 0x01 ) == 0 )
        {
            continue;
        }
        m_Tracks[ trackNo ].m_OutputLine = outputLine;
    }
}

void StreamSoundPlayer::ResetTrackOutputLine( u32 trackBitFlag )
{
    for( unsigned int trackNo = 0;
         trackNo < m_TrackCount && trackBitFlag != 0;
         trackNo++, trackBitFlag >>= 1
    )
    {
        if ( ( trackBitFlag & 0x01 ) == 0 )
        {
            continue;
        }
        m_Tracks[ trackNo ].m_OutputLine = -1;
    }
}




void StreamSoundPlayer::SetTrackTvVolume( u32 trackBitFlag, f32 volume )
{
    for( unsigned int trackNo = 0;
         trackNo < m_TrackCount && trackBitFlag != 0;
         trackNo++, trackBitFlag >>= 1
    )
    {
        if ( ( trackBitFlag & 0x01 ) == 0 )
        {
            continue;
        }
        m_Tracks[ trackNo ].m_TvParam.volume = volume;
    }
}
void StreamSoundPlayer::SetTrackChannelTvMixParameter( u32 trackBitFlag, u32 srcChNo, const MixParameter& param )
{
    for( unsigned int trackNo = 0;
        trackNo < m_TrackCount && trackBitFlag != 0;
        trackNo++, trackBitFlag >>= 1
        )
    {
        if ( ( trackBitFlag & 0x01 ) == 0 )
        {
            continue;
        }
        for ( int i = 0; i < CHANNEL_INDEX_NUM; i++ )
        {
            m_Tracks[ trackNo ].m_TvParam.mixParameter[ srcChNo ].ch[i] = param.ch[i];
        }
    }
}
void StreamSoundPlayer::SetTrackTvPan( u32 trackBitFlag, f32 pan )
{
    for( unsigned int trackNo = 0;
         trackNo < m_TrackCount && trackBitFlag != 0;
         trackNo++, trackBitFlag >>= 1
    )
    {
        if ( ( trackBitFlag & 0x01 ) == 0 )
        {
            continue;
        }
        m_Tracks[ trackNo ].m_TvParam.pan = pan;
    }
}
void StreamSoundPlayer::SetTrackTvSurroundPan( u32 trackBitFlag, f32 span )
{
    for( unsigned int trackNo = 0;
         trackNo < m_TrackCount && trackBitFlag != 0;
         trackNo++, trackBitFlag >>= 1
    )
    {
        if ( ( trackBitFlag & 0x01 ) == 0 )
        {
            continue;
        }
        m_Tracks[ trackNo ].m_TvParam.span = span;
    }
}
void StreamSoundPlayer::SetTrackTvMainSend( u32 trackBitFlag, f32 send )
{
    for( unsigned int trackNo = 0;
         trackNo < m_TrackCount && trackBitFlag != 0;
         trackNo++, trackBitFlag >>= 1
    )
    {
        if ( ( trackBitFlag & 0x01 ) == 0 )
        {
            continue;
        }
        m_Tracks[ trackNo ].m_TvParam.mainSend = send;
    }
}
void StreamSoundPlayer::SetTrackTvFxSend( u32 trackBitFlag, AuxBus bus, f32 send )
{
    for( unsigned int trackNo = 0;
         trackNo < m_TrackCount && trackBitFlag != 0;
         trackNo++, trackBitFlag >>= 1
    )
    {
        if ( ( trackBitFlag & 0x01 ) == 0 )
        {
            continue;
        }
        m_Tracks[ trackNo ].m_TvParam.fxSend[bus] = send;
    }
}



void StreamSoundPlayer::SetTrackDrcVolume( u32 drcIndex, u32 trackBitFlag, f32 volume )
{
    for( unsigned int trackNo = 0;
         trackNo < m_TrackCount && trackBitFlag != 0;
         trackNo++, trackBitFlag >>= 1
    )
    {
        if ( ( trackBitFlag & 0x01 ) == 0 )
        {
            continue;
        }
        m_Tracks[ trackNo ].m_DrcParam[drcIndex].volume = volume;
    }
}
void StreamSoundPlayer::SetTrackChannelDrcMixParameter( u32 drcIndex, u32 trackBitFlag, u32 srcChNo, const MixParameter& param )
{
    for( unsigned int trackNo = 0;
        trackNo < m_TrackCount && trackBitFlag != 0;
        trackNo++, trackBitFlag >>= 1
        )
    {
        if ( ( trackBitFlag & 0x01 ) == 0 )
        {
            continue;
        }
        for ( int i = 0; i < CHANNEL_INDEX_NUM; i++ )
        {
            m_Tracks[ trackNo ].m_DrcParam[drcIndex].mixParameter[ srcChNo ].ch[i] = param.ch[i];
        }
    }
}
void StreamSoundPlayer::SetTrackDrcPan( u32 drcIndex, u32 trackBitFlag, f32 pan )
{
    for( unsigned int trackNo = 0;
         trackNo < m_TrackCount && trackBitFlag != 0;
         trackNo++, trackBitFlag >>= 1
    )
    {
        if ( ( trackBitFlag & 0x01 ) == 0 )
        {
            continue;
        }
        m_Tracks[ trackNo ].m_DrcParam[drcIndex].pan = pan;
    }
}
void StreamSoundPlayer::SetTrackDrcSurroundPan( u32 drcIndex, u32 trackBitFlag, f32 span )
{
    for( unsigned int trackNo = 0;
         trackNo < m_TrackCount && trackBitFlag != 0;
         trackNo++, trackBitFlag >>= 1
    )
    {
        if ( ( trackBitFlag & 0x01 ) == 0 )
        {
            continue;
        }
        m_Tracks[ trackNo ].m_DrcParam[drcIndex].span = span;
    }
}
void StreamSoundPlayer::SetTrackDrcMainSend( u32 drcIndex, u32 trackBitFlag, f32 send )
{
    for( unsigned int trackNo = 0;
         trackNo < m_TrackCount && trackBitFlag != 0;
         trackNo++, trackBitFlag >>= 1
    )
    {
        if ( ( trackBitFlag & 0x01 ) == 0 )
        {
            continue;
        }
        m_Tracks[ trackNo ].m_DrcParam[drcIndex].mainSend = send;
    }
}
void StreamSoundPlayer::SetTrackDrcFxSend( u32 drcIndex, u32 trackBitFlag, AuxBus bus, f32 send )
{
    for( unsigned int trackNo = 0;
         trackNo < m_TrackCount && trackBitFlag != 0;
         trackNo++, trackBitFlag >>= 1
    )
    {
        if ( ( trackBitFlag & 0x01 ) == 0 )
        {
            continue;
        }
        m_Tracks[ trackNo ].m_DrcParam[drcIndex].fxSend[bus] = send;
    }
}

StreamTrack* StreamSoundPlayer::GetPlayerTrack( int trackNo )
{
    if ( trackNo > STRM_TRACK_NUM - 1 )
    {
        return NULL;
    }

    return &m_Tracks[ trackNo ];
}

const StreamTrack* StreamSoundPlayer::GetPlayerTrack( int trackNo ) const
{
    if ( trackNo > STRM_TRACK_NUM - 1 )
    {
        return NULL;
    }

    return &m_Tracks[ trackNo ];
}

} // namespace nw::snd::internal::driver
} // namespace nw::snd::internal
} // namespace nw::snd
} // namespace nw

