﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nw/snd/snd_StreamSoundLoader.h>
#include <nw/snd/snd_DriverCommand.h>
#include <nw/snd/snd_TaskManager.h>
#include <nw/snd/snd_WaveFileReader.h>

#if defined( NW_PLATFORM_WIN32 )
using namespace nw::internal::winext;
#elif defined( NW_USE_NINTENDO_SDK )
#endif

//#define NW_SND_ENABLE_DEBUG_STREAM_SOUND_LOADER
#ifdef NW_SND_ENABLE_DEBUG_STREAM_SOUND_LOADER
    #define NW_DEBUG_TRACE_FUNC() NW_LOG("[%p] %s\n", this, __FUNCTION__)
#else
    #define NW_DEBUG_TRACE_FUNC() ((void)0)
#endif // NW_SND_ENABLE_DEBUG_STREAM_SOUND_LOADER

namespace {

const u8 DEFAULT_LPF_FREQ       = 64;   // フィルタがかかっていない
const u8 DEFAULT_BIQUAD_TYPE    = 0;    // フィルタ未使用
const u8 DEFAULT_BIQUAD_VALUE   = 0;    // フィルタがかかっていない

} // anonymous namespace

namespace nw {
namespace snd {
namespace internal {

void StreamDataInfoDetail::SetStreamSoundInfo(const StreamSoundFile::StreamSoundInfo& info)
{
    sampleFormat = WaveFileReader::GetSampleFormat( info.encodeMethod );
    sampleRate = info.sampleRate;
    loopFlag = (info.isLoop != 0);
    loopStart = info.loopStart;
    sampleCount = info.frameCount;
    originalLoopStart = info.originalLoopStart;
    originalLoopEnd = info.originalLoopEnd;
    blockSampleCount = info.oneBlockSamples;
    blockSize = info.oneBlockBytes;
    lastBlockSampleCount = info.lastBlockSamples;
    lastBlockSize = info.lastBlockPaddedBytes;
}

namespace driver {

NW_ALIGN_VARIABLE(u8 StreamSoundLoader::s_LoadBuffer[ LOAD_BUFFER_SIZE ],256);
StreamDataDecoder* StreamSoundLoader::s_StreamDataDecoder = NULL;

StreamSoundLoader::StreamSoundLoader()
    : m_pFileStream(NULL), m_pDecodeWorkBuffer(NULL)
{
    u32 taskCount = m_StreamDataLoadTaskPool.Create(
        m_StreamDataLoadTaskArea,
        sizeof(m_StreamDataLoadTaskArea)
    );
    NW_ASSERT( taskCount == STRM_DATA_LOAD_TASK_MAX );
    std::memset(m_FilePath, 0, sizeof(char) * FILE_PATH_MAX);
}

StreamSoundLoader::~StreamSoundLoader()
{
    WaitFinalize();

    if (s_StreamDataDecoder != NULL && m_pDecodeWorkBuffer != NULL)
    {
        s_StreamDataDecoder->FreeWorkBuffer(m_pDecodeWorkBuffer);
        m_pDecodeWorkBuffer = NULL;
    }

    NW_ASSERT( m_StreamDataLoadTaskPool.Count() == STRM_DATA_LOAD_TASK_MAX );
    m_StreamDataLoadTaskPool.Destroy();
}

void StreamSoundLoader::Initialize()
{
    NW_DEBUG_TRACE_FUNC();

    WaitFinalize();

    m_LoadingDataBlockIndex = 0;
    m_LastBlockIndex = 0xffffffff;
    m_LoopStartBlockIndex = 0;
    m_LoopStartFilePos = 0;
    m_LoopStartBlockSampleOffset = 0;
    m_LoopJumpFlag = false;
    m_LoadFinishFlag = false;
    m_IsRegionInfoEnabled = false;

    m_SampleFormat = SAMPLE_FORMAT_DSP_ADPCM;

    m_AdpcmContextForStartOffsetFrame = 0xffffffff;

    m_StreamRegionCallbackFunc = NULL;
    m_StreamRegionCallbackArg = NULL;
}

void StreamSoundLoader::Finalize()
{
    NW_DEBUG_TRACE_FUNC();

    CancelRequest();
    RequestClose();
}

void StreamSoundLoader::WaitFinalize()
{
    NW_DEBUG_TRACE_FUNC();

    m_StreamHeaderLoadTask.Wait();
    m_StreamCloseTask.Wait();
    for ( StreamDataLoadTaskList::Iterator itr = m_StreamDataLoadTaskList.GetBeginIter();
          itr != m_StreamDataLoadTaskList.GetEndIter();
    )
    {
        StreamDataLoadTaskList::Iterator curItr = itr++;
        StreamDataLoadTask* task = &*curItr;
        task->Wait();
        m_StreamDataLoadTaskList.Erase( task );
        m_StreamDataLoadTaskPool.Free( task );
        // NN_LOG( "Task Cancel %08x\n", task );
    }
}

void StreamSoundLoader::CancelRequest()
{
    NW_DEBUG_TRACE_FUNC();
    TaskManager::GetInstance().CancelTaskById( reinterpret_cast<u32>(this) );
}

void StreamSoundLoader::RequestLoadHeader()
{
    NW_DEBUG_TRACE_FUNC();

    m_StreamHeaderLoadTask.m_pLoader = this;

    m_CurrentRegionNo = 0;

    m_StreamHeaderLoadTask.SetId(reinterpret_cast<u32>(this));
    // NW_LOG("[%s](%p) AppendTask ", __FUNCTION__, this);
    internal::TaskManager::GetInstance().AppendTask(
            &m_StreamHeaderLoadTask,
            internal::TaskManager::PRIORITY_MIDDLE );
    // NW_LOG(" -- [%s](%p) end\n", __FUNCTION__, this);
}

void StreamSoundLoader::RequestClose()
{
    NW_DEBUG_TRACE_FUNC();

    m_StreamCloseTask.Wait();
    m_StreamCloseTask.m_pLoader = this;
    internal::TaskManager::GetInstance().AppendTask(
            &m_StreamCloseTask,
            internal::TaskManager::PRIORITY_MIDDLE );
}


void StreamSoundLoader::RequestLoadData( void* bufferAddress[], u32 bufferBlockIndex, u32 startOffsetSamples, u32 prefetchOffsetSamples, int priority )
{
    // ロードタスクの発行
    StreamDataLoadTask* task = m_StreamDataLoadTaskPool.Alloc();
    NW_NULL_ASSERT( task );

    task->m_pLoader = this;
    task->m_BufferBlockIndex = bufferBlockIndex;
    task->m_StartOffsetSamples = startOffsetSamples;
    task->m_PrefetchOffsetSamples = prefetchOffsetSamples;
    task->SetId(reinterpret_cast<u32>(this));

    for( unsigned int ch=0 ; ch < m_ChannelCount ; ch++){
        NW_ASSERT(ch < STRM_CHANNEL_NUM);
        task->m_BufferAddress[ch] = bufferAddress[ch];
    }

    m_StreamDataLoadTaskList.PushBack( task );
    // NW_LOG("[%s](%p) AppendTask ", __FUNCTION__, this);
    internal::TaskManager::GetInstance().AppendTask( task, internal::TaskManager::TaskPriority(priority) );
    // NW_LOG(" -- [%s](%p) end\n", __FUNCTION__, this);

    // NN_LOG("loadtask append buf(%d)\n", m_LoadingBufferBlockIndex);
}

/*---------------------------------------------------------------------------*
  Name:         Update

  Description:  タスクの状態をチェックし、完了したものの解放処理を行う

  Arguments:    None.

  Returns:      None.
 *---------------------------------------------------------------------------*/
void StreamSoundLoader::Update()
{
    for ( StreamDataLoadTaskList::Iterator itr = m_StreamDataLoadTaskList.GetBeginIter();
          itr != m_StreamDataLoadTaskList.GetEndIter();
        )
    {
        StreamDataLoadTaskList::Iterator curItr = itr++;
        StreamDataLoadTask* task = &*curItr;
        if ( task->GetStatus() != Task::STATUS_DONE && task->GetStatus() != Task::STATUS_CANCEL )
        {
            break;
        }

        m_StreamDataLoadTaskList.Erase( task );
        m_StreamDataLoadTaskPool.Free( task );

        // NN_LOG("Free %08x %d\n",task,m_StreamDataLoadTaskList.size());
    }
}

void StreamSoundLoader::ForceFinish()
{
    DriverCommand& cmdmgr = DriverCommand::GetInstanceForTaskThread();
    DriverCommandStreamSoundForceFinish* command =
        cmdmgr.AllocCommand<DriverCommandStreamSoundForceFinish>();
    command->id = DRIVER_COMMAND_STRM_FORCE_FINISH;
    command->player = m_PlayerHandle;
    cmdmgr.PushCommand(command);
    cmdmgr.FlushCommand(true);
}

bool StreamSoundLoader::IsBusy() const
{
    if ( m_LoadFinishFlag ) return false;
    if ( ! m_StreamDataLoadTaskList.IsEmpty() ) return true;
    return false;
}

bool StreamSoundLoader::IsInUse()
{
    Update();

    if ( !m_StreamDataLoadTaskList.IsEmpty() ) return true;

    return false;
}

bool StreamSoundLoader::Open()
{
    NW_DEBUG_TRACE_FUNC();

    // フックが有効な時はそちらからのオープンを試みる
    if (m_FileStreamHookParam.IsHookEnabled())
    {
        m_pFileStream = m_FileStreamHookParam.pSoundArchiveFilesHook->OpenFile(
                m_FileStreamBuffer,
                FILE_STREAM_BUFFER_SIZE,
                m_pCacheBuffer,
                m_CacheSize,
                m_FileStreamHookParam.itemLabel,
                internal::SoundArchiveFilesHook::FILE_TYPE_STREAM_BINARY);
    }

    // フックされてないときか、オープンできなかったときは通常のオープン
    if (m_pFileStream == NULL)
    {
        // NW_LOG("(%p) [%s] m_pFilePath(%p:%s)\n", this, __FUNCTION__, m_pFilePath, m_pFilePath);
        // ファイルのオープン
        internal::CachedFsFileStream* stream = new(m_FileStreamBuffer) CachedFsFileStream();
        NW_ASSERT_NOT_NULL(stream);
    #if defined(NW_PLATFORM_CAFE)
        stream->SetPriority(m_FsPriority);
        bool result = stream->Open(m_pFsClient, m_FilePath, "r", m_pFsCommandBufferPool);
    #else
        bool result = stream->Open(m_FilePath, fnd::File::ACCESS_MODE_READ);
    #endif
        if (!result)
        {
            return false;
        }
        // キャッシュ用バッファを設定
        if (stream->IsAvailable())
        {
            if ((m_pCacheBuffer != NULL) && (m_CacheSize > 0))
            {
                stream->SetCacheBuffer(m_pCacheBuffer, m_CacheSize);
            }
        }

        m_pFileStream = stream;
        // NW_LOG("(%p) [%s] stream(%p)\n", this, __FUNCTION__, stream);
    }

    m_FileLoader.Initialize( m_pFileStream );

    // デコーダ用のバッファの確保
    if (s_StreamDataDecoder != NULL && m_pDecodeWorkBuffer == NULL)
    {
        m_pDecodeWorkBuffer = s_StreamDataDecoder->AllocWorkBuffer();
    }

    return true;
}

void StreamSoundLoader::Close()
{
    NW_DEBUG_TRACE_FUNC();

    if (s_StreamDataDecoder != NULL && m_pDecodeWorkBuffer != NULL)
    {
        s_StreamDataDecoder->FreeWorkBuffer(m_pDecodeWorkBuffer);
        m_pDecodeWorkBuffer = NULL;
    }

    if (m_pFileStream)
    {
        m_pFileStream->Close();
        m_pFileStream = NULL;
        m_FileLoader.Finalize();
    }
}

void StreamSoundLoader::LoadHeader()
{
    NW_DEBUG_TRACE_FUNC();

    // ヘッダーロード
    DriverCommand& cmdmgr = DriverCommand::GetInstanceForTaskThread();
    DriverCommandStreamSoundLoadHeader* command =
        cmdmgr.AllocCommand<DriverCommandStreamSoundLoadHeader>();
    command->id = DRIVER_COMMAND_STRM_LOADHEADER;

    bool result;

    switch( static_cast<s32>(m_FileType) )
    {
    case STREAM_FILE_BFSTM:
        result = LoadHeader1( command );
        break;
    case 1:
        result = LoadHeader2( command );
        break;
    default:
        result = false;
        break;
    }
    command->result = result;

    cmdmgr.PushCommand(command);
    cmdmgr.FlushCommand(true);
}

bool StreamSoundLoader::LoadHeader1( DriverCommandStreamSoundLoadHeader* command )
{
    NW_DEBUG_TRACE_FUNC();

    // ファイルヘッダのロード
    StreamSoundFileReader reader;
    if ( ! m_FileLoader.LoadFileHeader( &reader, s_LoadBuffer, LOAD_BUFFER_SIZE ) )
    {
        return false;
    }

    command->player = m_PlayerHandle;
    command->requestIndex = m_RequestIndex;

    // ストリーム情報の読み取り
    internal::StreamSoundFile::StreamSoundInfo info;
    if ( ! reader.ReadStreamSoundInfo( &info ) )
    {
        return false;
    }

    u32 channelCount = reader.GetChannelCount();
    m_ChannelCount = channelCount;
    m_DataInfo->channelCount = channelCount;
    m_DataInfo->SetStreamSoundInfo(info);

    // bXstm 内にトラック情報が埋められている場合 (バイナリバージョン 0.2.0.0 まで)
    if ( reader.IsTrackInfoAvailable() )
    {
        if ( !ReadTrackInfoFromStreamSoundFile(reader) )
        {
            return false;
        }
    }

    // ADPCM
    m_SampleFormat = m_DataInfo->sampleFormat;
    if ( m_SampleFormat == SAMPLE_FORMAT_DSP_ADPCM )
    {
        // ADPCM の情報を読み取る
        if ( !SetAdpcmInfo(reader, channelCount, command->adpcmParam) )
        {
            return false;
        }
    }
    else
    {
        for ( unsigned int ch = 0; ch < channelCount; ch++ )
        {
            command->adpcmParam[ch] = NULL;
        }
    }

    m_DataStartFilePos = reader.GetSampleDataOffset();
    m_LastBlockIndex = m_DataInfo->GetLastBlockIndex();
    m_LoopStartBlockIndex = m_DataInfo->GetLoopStartBlockIndex(0);
    m_LoopStartFilePos = m_DataStartFilePos + m_DataInfo->blockSize * m_ChannelCount * m_LoopStartBlockIndex;
    m_LoopStartBlockSampleOffset = 0;

    NW_ASSERT( m_DataInfo->GetLoopStartInBlock() == 0 );
    NW_ASSERT( m_LastBlockIndex == info.blockCount - 1 );

    if ( !SetupRegionInfo() )
    {
        return false;
    }

    UpdateLoadingDataBlockIndex();

    return true;
}

bool StreamSoundLoader::LoadHeader2( DriverCommandStreamSoundLoadHeader* command )
{
    NW_DEBUG_TRACE_FUNC();

    if ( s_StreamDataDecoder == NULL ) return false;

    command->player = m_PlayerHandle;
    command->requestIndex = m_RequestIndex;

    // ストリーム情報の読み取り
    StreamDataDecoder::DataInfo info;
    bool result = s_StreamDataDecoder->ReadDataInfo( m_pFileStream, &info, m_pDecodeWorkBuffer );
    if ( ! result )
    {
        return false;
    }

    u32 channelCount = info.channelCount;
    m_ChannelCount = channelCount;
    m_DataInfo->channelCount = channelCount;
    SetStreamSoundInfo2(info);

    // 最後のブロックのサンプル数は計算で求める
    m_LastBlockIndex = m_DataInfo->GetLastBlockIndex();
    if ( m_DataInfo->loopFlag )
    {
        m_DataInfo->lastBlockSampleCount =
            m_DataInfo->sampleCount - m_DataInfo->blockSampleCount * m_LastBlockIndex;
    }
    else
    {
        m_DataInfo->lastBlockSampleCount = m_DataInfo->blockSampleCount;
    }

    // トラックの情報は読み込まない。（bXstmではないのでトラック情報は含まれていない）
    // （StreamSoundPlayer::Setup のタイミングで bXsar から読み込んだ値を設定しています。）

    // ADPCMパラメータを無効化
    for ( unsigned int ch = 0; ch < channelCount; ch++ )
    {
        command->adpcmParam[ch] = NULL;
    }

    // ループ先補正前の情報を取得
    u32 loopStartBlockIndex = m_DataInfo->GetLoopStartBlockIndex(0);
    u32 loopStartBlockSampleOffset = m_DataInfo->loopStart - loopStartBlockIndex * m_DataInfo->blockSampleCount;

    // ループ先補正
    u32 extFrameSize = m_DataInfo->blockSampleCount - m_DataInfo->lastBlockSampleCount;
    u32 loopRestFrameSize  = m_DataInfo->blockSampleCount - loopStartBlockSampleOffset;
    if (extFrameSize <= loopRestFrameSize)
    {
        m_DataInfo->loopStart += loopRestFrameSize;
        m_DataInfo->lastBlockSampleCount += loopRestFrameSize;
    }
    else
    {
        m_DataInfo->loopStart += loopRestFrameSize + m_DataInfo->blockSampleCount;
        m_DataInfo->lastBlockSampleCount += (loopRestFrameSize + m_DataInfo->blockSampleCount);
    }

    // m_DataInfo->loopStart が補正されるため再計算
    loopStartBlockIndex = m_DataInfo->GetLoopStartBlockIndex(0);
    loopStartBlockSampleOffset = m_DataInfo->loopStart - loopStartBlockIndex * m_DataInfo->blockSampleCount;

    m_LoopStartBlockIndex = loopStartBlockIndex;
    m_LoopStartBlockSampleOffset = loopStartBlockSampleOffset;

    if ( !SetupRegionInfo() )
    {
        return false;
    }

    return true;
}

void StreamSoundLoader::LoadData( void* bufferAddress[], u32 bufferBlockIndex, u32 startOffsetSamples, u32 prefetchOffsetSamples )
{
    if ( m_LoadFinishFlag )
    {
        return;
    }

    // NN_LOG("exec(0x%p): bufIdx(%d)\n", this, m_BufferBlockIndex );

    DriverCommand& cmdmgr = DriverCommand::GetInstanceForTaskThread();
    DriverCommandStreamSoundLoadData* command =
        cmdmgr.AllocCommand<DriverCommandStreamSoundLoadData>();
    command->id = DRIVER_COMMAND_STRM_LOADDATA;

    // ★ 見直しが必要
    bool result;
    if (m_pFileStream != NULL)
    {
        switch( static_cast<s32>(m_FileType) )
        {
        case STREAM_FILE_BFSTM:
            result = LoadData1( command, bufferAddress, bufferBlockIndex, startOffsetSamples, prefetchOffsetSamples );
            break;
        case 1:
            result = LoadData2( command, bufferAddress, bufferBlockIndex, startOffsetSamples, prefetchOffsetSamples );
            break;
        default:
            result = false;
            break;
        }
    }
    else
    {
        result = false;
    }
    command->result = result;
    command->player = m_PlayerHandle;

    cmdmgr.PushCommand(command);
    cmdmgr.FlushCommand(true);
}

bool StreamSoundLoader::LoadData1(
    DriverCommandStreamSoundLoadData* command,
    void* bufferAddress[],
    u32 bufferBlockIndex,
    u32 startOffsetSamples,
    u32 prefetchOffsetSamples
)
{
    LoadDataParam& loadDataParam = command->loadDataParam;

    // 開始オフセット
    u32 startOffsetSamplesInFrame = 0; // 波形バッファ先頭からのサンプル位置
    bool updateAdpcmContext = false; // ADPCMコンテキストの更新が必要かどうか
    u32 loopCount = 0;

    // 開始オフセットが指定されている場合
    if ( startOffsetSamples > 0 || prefetchOffsetSamples > 0 )
    {
        if ( !ApplyStartOffset(startOffsetSamples + prefetchOffsetSamples, &loopCount) )
        {
            // 範囲外のオフセットが指定された
            loadDataParam.samples = 0;
            return true;
        }

        UpdateLoadingDataBlockIndex();

        if ( m_SampleFormat == SAMPLE_FORMAT_DSP_ADPCM )
        {
            updateAdpcmContext = true;
        }
    }

    u32 totalBlockSamples = 0;
    u32 destAddressOffset = 0;
    u32 sampleBegin = m_Region.current;

    // 現在のブロックサンプルが小さいとき、または、
    // 次のブロックサンプルが小さいとき、一緒にロードする
    bool firstBlock = true;
    // プリフェッチデータからストリームデータへの移行時は最初のブロックとして扱わない
    if ( prefetchOffsetSamples > 0 )
    {
        firstBlock = false;
    }
    while( totalBlockSamples == 0 ||
           totalBlockSamples < DATA_BLOCK_SIZE_MARGIN_SAMPLES ||
           m_Region.Rest() < DATA_BLOCK_SIZE_MARGIN_SAMPLES )
    {
        BlockInfo blockInfo;
        CalculateBlockInfo(blockInfo);

        if ( firstBlock )
        {
            startOffsetSamplesInFrame = blockInfo.GetStartOffsetInFrame();

            if ( updateAdpcmContext )
            {
                if ( !LoadAdpcmContextForStartOffset() )
                {
                    return false;
                }
            }
        }

        if ( !LoadOneBlockData(bufferAddress, blockInfo, destAddressOffset, firstBlock, updateAdpcmContext) )
        {
            return false;
        }

        totalBlockSamples += blockInfo.samples;
        destAddressOffset += blockInfo.copyByte;
        m_Region.current += blockInfo.samples;

        // ロードデータブロックインデックスの更新
        NW_ASSERT( m_LoadingDataBlockIndex <= m_LastBlockIndex );
        m_LoadingDataBlockIndex++;

        // リージョン終端処理
        if ( m_Region.IsEnd() )
        {
            if ( MoveNextRegion(&loopCount) )
            {
                UpdateLoadingDataBlockIndex();
            }

            // ADPCMコンテキストの更新ができないため
            // ブロックを跨いだ一括ロードは禁止
            // PCM8/PCM16は可能であるが非対応
            // 対応する場合、１バッファで複数回loopCountを増やす対応が必要
            break;
        }

        firstBlock = false;
    } // while(ブロック)

    loadDataParam.adpcmContextEnable = false;
    if ( m_SampleFormat == SAMPLE_FORMAT_DSP_ADPCM )
    {
        if ( sampleBegin == 0 )
        {
            for ( unsigned int ch = 0 ; ch < m_ChannelCount ; ch++ )
            {
                loadDataParam.adpcmContext[ch] = m_AdpcmInfo[ch].beginContext;
            }
            loadDataParam.adpcmContextEnable = true;
        }
        else if ( m_DataInfo->loopFlag && sampleBegin == m_DataInfo->loopStart )
        {
            for ( unsigned int ch = 0 ; ch < m_ChannelCount ; ch++ )
            {
                loadDataParam.adpcmContext[ch] = m_AdpcmInfo[ch].loopContext;
            }
            loadDataParam.adpcmContextEnable = true;
        }
        else if ( sampleBegin == m_AdpcmContextForStartOffsetFrame )
        {
            for ( unsigned int ch = 0; ch < m_ChannelCount; ch++ )
            {
                loadDataParam.adpcmContext[ch] = m_AdpcmContextForStartOffset[ch];
            }
            loadDataParam.adpcmContextEnable = true;
        }
    }

    loadDataParam.blockIndex = bufferBlockIndex; // バッファ識別番号
    loadDataParam.samples = totalBlockSamples;
    loadDataParam.sampleBegin = sampleBegin; // バッファ先頭のサンプル位置
    loadDataParam.sampleOffset = startOffsetSamplesInFrame;
    loadDataParam.loopCount = loopCount;
    loadDataParam.lastBlockFlag = m_LoadFinishFlag;

    // NW_LOG("[%s] bufBlock(%d) loopCount(%d) lastBlock(%d) adpcmContextEnable(%d)\n",
    //         __FUNCTION__, loadDataParam.bufferBlockIndex, loadDataParam.loopCount,
    //         loadDataParam.astBlockFlag, loadDataParam.adpcmContextEnable );

    return true;
}

bool StreamSoundLoader::LoadData2(
    DriverCommandStreamSoundLoadData* command,
    void* bufferAddress[],
    u32 bufferBlockIndex,
    u32 startOffsetSamples,
    u32 prefetchOffsetSamples
)
{
    NW_UNUSED_VARIABLE(prefetchOffsetSamples);
    LoadDataParam& loadDataParam = command->loadDataParam;

    u32 startOffsetSamplesInFrame = 0; // 波形バッファ先頭からのサンプル位置
    u32 loopCount = 0;

    // 最後のブロックを超えたとき
    if ( m_LoadingDataBlockIndex > m_LastBlockIndex )
    {
        if ( m_DataInfo->loopFlag )
        {
            m_LoadingDataBlockIndex = m_LoopStartBlockIndex;
            m_pFileStream->Seek( m_LoopStartFilePos, ut::FILE_STREAM_SEEK_BEGIN );

            // ループスタートブロックより１ブロック前の場所を空デコード
            if ( m_LoopStartBlockIndex > 0 )
            {
                if ( !DecodeStreamData(StreamDataDecoder::DECODE_TYPE_IDLING, bufferAddress) )
                {
                    return false;
                }
            }
            m_Region.current = m_DataInfo->loopStart;
            m_LoopJumpFlag = true;
        }
    }

    // 開始オフセットが指定されている場合
    if ( startOffsetSamples > 0 )
    {
        if ( !ApplyStartOffset(startOffsetSamples, &loopCount) )
        {
            // 範囲外のオフセットが指定された
            loadDataParam.samples = 0;
            return true;
        }

        UpdateLoadingDataBlockIndex2();
        startOffsetSamplesInFrame = m_Region.current - m_LoadingDataBlockIndex * m_DataInfo->blockSampleCount;
    }
    else if ( m_LoopJumpFlag )
    {
        startOffsetSamplesInFrame = m_LoopStartBlockSampleOffset;
    }

    // ループスタートブロックより１ブロック前の場所を記憶
    if ( IsLoopStartFilePos(m_LoadingDataBlockIndex) )
    {
        m_LoopStartFilePos = m_pFileStream->Tell();
    }

    u32 sampleBegin = m_Region.current;

    NW_ALIGN32_ASSERT( m_DataInfo->blockSize );   // TODO: このチェックが本当に必要か確認
    u32 blockSamples = 0;
    StreamDataDecoder::DecodeType decodeType;
    if ( m_LoadingDataBlockIndex != m_LastBlockIndex )
    {
        // 通常ブロック
        blockSamples = m_DataInfo->blockSampleCount;
        decodeType = StreamDataDecoder::DECODE_TYPE_NORMAL;
    }
    else
    {
        // 最終ブロック
        blockSamples = m_DataInfo->lastBlockSampleCount;
        decodeType = StreamDataDecoder::DECODE_TYPE_LOOP;
    }

    if ( m_pFileStream->Tell() >= m_pFileStream->GetSize() )
    {
        blockSamples = 0;
    }

    // データをデコードしつつロード
    if ( blockSamples > 0 )
    {
        if ( !DecodeStreamData(decodeType, bufferAddress) )
        {
            return false;
        }
        for (u32 i = 0; i < m_ChannelCount; ++i)
        {
            nw::snd::internal::driver::HardwareManager::FlushDataCache(bufferAddress[i], blockSamples * sizeof(s16));
        }

        // ループ波形でないときは、m_LoopEnd に -1 が入るため、
        // LoadHeader で m_LastBlockIndex の計算ができないのでこの処理が必要
        // ループ波形の時はループエンド以降にデータがあるのでやるとまずい
        if ( !m_DataInfo->loopFlag )
        {
            if ( m_pFileStream->Tell() >= m_pFileStream->GetSize() )
            {
                m_LastBlockIndex = m_LoadingDataBlockIndex;
            }
        }
    }

    m_Region.current += blockSamples;
    m_LoopJumpFlag = false;

    // ロードデータブロックインデックスの更新
    m_LoadingDataBlockIndex++;
    if ( m_LoadingDataBlockIndex > m_LastBlockIndex )
    {
        if ( !m_DataInfo->loopFlag )
        {
            m_LoadFinishFlag = true;
        }
    }

    loadDataParam.adpcmContextEnable = false;
    loadDataParam.blockIndex = bufferBlockIndex;
    loadDataParam.samples = blockSamples;
    loadDataParam.sampleBegin = sampleBegin;
    loadDataParam.sampleOffset = startOffsetSamplesInFrame;
    loadDataParam.loopCount = loopCount;
    loadDataParam.lastBlockFlag = m_LoadFinishFlag;

    return true;
}

void StreamSoundLoader::SetStreamSoundInfo2(const StreamDataDecoder::DataInfo& info)
{
    m_DataInfo->sampleFormat = SAMPLE_FORMAT_PCM_S16;
    m_DataInfo->sampleRate = info.sampleRate;
    m_DataInfo->loopFlag = m_LoopFlag;
    m_DataInfo->loopStart = m_LoopStart;
    m_DataInfo->sampleCount = m_LoopEnd;
    m_DataInfo->originalLoopStart = m_LoopStart;
    m_DataInfo->originalLoopEnd = m_LoopEnd;
    m_DataInfo->blockSampleCount = info.blockSampleCount;
    m_DataInfo->blockSize = info.blockSize;
    m_DataInfo->lastBlockSize = m_DataInfo->blockSize;
}

bool StreamSoundLoader::ApplyStartOffset(u32 startOffsetSamples, u32* loopCount)
{
    u32 startOffsetSamplesInRegion = startOffsetSamples; // リージョン先頭からのサンプル位置

    while ( !m_Region.IsIn(startOffsetSamplesInRegion) )
    {
        startOffsetSamplesInRegion -= m_Region.Rest();

        if ( !MoveNextRegion(loopCount) )
        {
            return false;
        }
    }

    m_Region.current += startOffsetSamplesInRegion;

    return true;
}

bool StreamSoundLoader::MoveNextRegion(u32* loopCount)
{
    if ( UpdateRegion() )
    {
        // 次のリージョン
        ++*loopCount;
    }
    else
    {
        // 終端
        m_LoadFinishFlag = true;

        return false;
    }

    return true;
}

void StreamSoundLoader::ChangeRegionInfo(u32 regionNo)
{
    StreamSoundFile::RegionInfo regionInfo;
    bool result = m_FileLoader.ReadRegionInfo( &regionInfo, regionNo );
    if ( result )
    {
        m_Region.begin = regionInfo.start;
        m_Region.end = regionInfo.end;

        if ( m_SampleFormat == SAMPLE_FORMAT_DSP_ADPCM )
        {
            for( unsigned int ch = 0 ; ch < m_ChannelCount ; ch++ )
            {
                m_AdpcmContextForStartOffset[ch].pred_scale = regionInfo.adpcmContext[ch].loopPredScale;
                m_AdpcmContextForStartOffset[ch].yn1 = regionInfo.adpcmContext[ch].loopYn1;
                m_AdpcmContextForStartOffset[ch].yn2 = regionInfo.adpcmContext[ch].loopYn2;
            }
            m_AdpcmContextForStartOffsetFrame = regionInfo.start;
        }
    }
    else
    {
        m_Region.begin = 0;
        m_Region.end = m_DataInfo->sampleCount;
    }
}

bool StreamSoundLoader::SetupRegionInfo()
{
    StreamSoundFile::RegionInfo regionInfo;
    m_IsRegionInfoEnabled = m_FileLoader.ReadRegionInfo( &regionInfo, 0 );

    if ( m_IsRegionInfoEnabled == true && m_StreamRegionCallbackFunc != NULL )
    {
        StreamRegionCallbackParam param;
        param.regionNo = 0;
        if ( m_StreamRegionCallbackFunc( &param, m_StreamRegionCallbackArg ) == STREAM_REGION_CALLBACK_FINISH )
        {
            return false;
        }
        m_CurrentRegionNo = param.regionNo;
        ChangeRegionInfo(m_CurrentRegionNo);
    }
    else
    {
        if ( m_StreamRegionCallbackFunc != NULL )
        {
            NW_WARNING(false, "Cannot use regionCallback without regionInfo.[%s]\n", m_FilePath);
        }
        m_Region.begin = 0;
        m_Region.end = m_DataInfo->sampleCount;
    }

    m_Region.current = m_Region.begin;

    return true;
}

bool StreamSoundLoader::UpdateRegion()
{
    if ( m_DataInfo->loopFlag )
    {
        m_Region.begin = m_DataInfo->loopStart;
        m_Region.end = m_DataInfo->sampleCount;
        m_Region.current = m_Region.begin;

        return true;
    }

    if ( m_IsRegionInfoEnabled == true && m_StreamRegionCallbackFunc != NULL )
    {
        StreamRegionCallbackParam param;
        param.regionNo = m_CurrentRegionNo;
        if ( m_StreamRegionCallbackFunc( &param, m_StreamRegionCallbackArg ) != STREAM_REGION_CALLBACK_FINISH )
        {
            m_CurrentRegionNo = param.regionNo;
            ChangeRegionInfo(m_CurrentRegionNo);
            m_Region.current = m_Region.begin;

            return true;
        }
    }

    m_Region.current = m_Region.end;

    return false;
}

bool StreamSoundLoader::ReadTrackInfoFromStreamSoundFile(StreamSoundFileReader& reader)
{
    u32 trackCount = reader.GetTrackCount();
    if ( trackCount > STRM_TRACK_NUM ) {
        trackCount = STRM_TRACK_NUM;
    }

    // m_DataInfo->trackCount = trackCount;

    // トラックの情報を読み取る
    for ( unsigned int i = 0; i < trackCount; i++ )
    {
        StreamSoundFileReader::TrackInfo trackInfo;
        if ( ! reader.ReadStreamTrackInfo( &trackInfo, i ) )
        {
            return false;
        }

        m_DataInfo->trackInfo[i].volume = trackInfo.volume;
        m_DataInfo->trackInfo[i].pan = trackInfo.pan;
        m_DataInfo->trackInfo[i].channelCount = trackInfo.channelCount;
        for( int ch=0; ch < trackInfo.channelCount ; ch++ ) {
            m_DataInfo->trackInfo[i].channelIndex[ch] = trackInfo.globalChannelIndex[ch];
        }

        // bXstrm 0.2.0.0 には、span, flags が含まれないので読み込まない
        m_DataInfo->trackInfo[i].span = 0;
        m_DataInfo->trackInfo[i].flags = 0;

        m_DataInfo->trackInfo[i].mainSend = 127;
        for( int j=0; j < AUX_BUS_NUM; j++ ) {
            m_DataInfo->trackInfo[i].fxSend[j] = 0;
        }
        m_DataInfo->trackInfo[i].lpfFreq     = DEFAULT_LPF_FREQ;
        m_DataInfo->trackInfo[i].biquadType  = DEFAULT_BIQUAD_TYPE;
        m_DataInfo->trackInfo[i].biquadValue = DEFAULT_BIQUAD_VALUE;
    }

    return true;
}

bool StreamSoundLoader::IsLoopStartFilePos(u32 loadingDataBlockIndex)
{
    return m_LoopStartBlockIndex > 0 && loadingDataBlockIndex == m_LoopStartBlockIndex - 1;
}

void StreamSoundLoader::UpdateLoadingDataBlockIndex()
{
    m_LoadingDataBlockIndex = m_Region.current / m_DataInfo->blockSampleCount;
    u32 startFilePos = m_DataStartFilePos + m_DataInfo->blockSize * m_ChannelCount * m_LoadingDataBlockIndex;
    m_pFileStream->Seek( startFilePos, ut::FILE_STREAM_SEEK_BEGIN );
}

void StreamSoundLoader::UpdateLoadingDataBlockIndex2()
{
    m_LoadingDataBlockIndex = m_Region.current / m_DataInfo->blockSampleCount;

    m_pFileStream->Seek( 0, ut::FILE_STREAM_SEEK_BEGIN );
    for( unsigned int i = 0; i < m_LoadingDataBlockIndex; i++ )
    {
        if ( IsLoopStartFilePos(i) )
        {
            // ループスタートブロックより１ブロック前の場所を記憶
            m_LoopStartFilePos = m_pFileStream->Tell();
        }

        // 1ブロック分スキップ
        s_StreamDataDecoder->Skip( m_pFileStream );
    }
}

u32 StreamSoundLoader::GetLoadChannelCount(u32 loadStartChannel)
{
    u32 loadChannelCount = LOAD_BUFFER_CHANNEL_NUM;
    if ( loadStartChannel + loadChannelCount > m_ChannelCount )
    {
        loadChannelCount = m_ChannelCount - loadStartChannel;
    }

    return loadChannelCount;
}

bool StreamSoundLoader::LoadStreamBuffer(u8* buffer, const BlockInfo& blockInfo, u32 loadChannelCount)
{
    size_t loadSize = blockInfo.size * loadChannelCount;
    NW_ASSERT( loadSize <= LOAD_BUFFER_SIZE );

    // データのロード
    s32 resultSize = m_pFileStream->Read( buffer, loadSize );
    if ( resultSize != static_cast<s32>(loadSize) )
    {
        return false;
    }

#ifdef DEBUG_STRM
    std::fwrite( buffer, loadSize, 1, s_pFile );
#endif

    return true;
}

void StreamSoundLoader::CalculateBlockInfo(BlockInfo& blockInfo)
{
    if ( m_LoadingDataBlockIndex != m_LastBlockIndex )
    {
        // 通常ブロック
        blockInfo.size = m_DataInfo->blockSize;
        blockInfo.samples = m_DataInfo->blockSampleCount;
    }
    else
    {
        // 最終ブロック
        blockInfo.size = m_DataInfo->lastBlockSize;
        blockInfo.samples = m_DataInfo->lastBlockSampleCount;
    }

    blockInfo.startOffsetSamples = (m_Region.current % m_DataInfo->blockSampleCount);
    blockInfo.startOffsetSamplesAlign = blockInfo.startOffsetSamples;
    if ( m_SampleFormat == SAMPLE_FORMAT_DSP_ADPCM )
    {
        blockInfo.startOffsetSamplesAlign = static_cast<int>(blockInfo.startOffsetSamples / 14) * 14;
    }

    blockInfo.startOffsetByte = Util::GetByteBySample( blockInfo.startOffsetSamplesAlign, m_SampleFormat );
    blockInfo.copyByte = blockInfo.size - blockInfo.startOffsetByte;
    blockInfo.samples -= blockInfo.startOffsetSamples;

    if ( !m_Region.IsInWithBorder(blockInfo.samples) )
    {
        blockInfo.samples = m_Region.Rest();
        if (blockInfo.samples < internal::DATA_BLOCK_SIZE_MARGIN_SAMPLES)
        {
            blockInfo.copyByte = StreamSoundLoader::DATA_BLOCK_SIZE_MARGIN;
        }
    }
}

bool StreamSoundLoader::LoadOneBlockData(void* bufferAddress[], const BlockInfo& blockInfo, u32 destAddressOffset, bool firstBlock, bool updateAdpcmContext)
{
    unsigned int ch = 0;
    while ( ch < m_ChannelCount )
    {
        // 終了中であれば読み込みを中断
        NW_NULL_ASSERT( m_PlayerHandle );
        if ( m_PlayerHandle->IsFinalizing() )
        {
            return false;
        }

        // 同時に読み込むチャンネル数を確定し、ロードサイズを計算
        u32 loadChannelCount = GetLoadChannelCount(ch);
        if ( !LoadStreamBuffer(s_LoadBuffer, blockInfo, loadChannelCount) )
        {
            return false;
        }

        // 同時に読みこんだチャンネル数分のバッファを、再生用バッファにコピーしてフラッシュ
        for ( unsigned int i = 0; i < loadChannelCount; i++ )
        {
            const void* blockBegin = ut::AddOffsetToPtr( s_LoadBuffer, blockInfo.size * i );
            const void* source = ut::AddOffsetToPtr( blockBegin, blockInfo.startOffsetByte );
            void* dest = ut::AddOffsetToPtr( bufferAddress[ ch ], destAddressOffset );
            std::memcpy( dest, source, blockInfo.copyByte );
            nw::snd::internal::driver::HardwareManager::FlushDataCache( dest, blockInfo.copyByte );

            // 必要なときに、ADPCMコンテキストを更新
            if ( firstBlock && updateAdpcmContext )
            {
                UpdateAdpcmInfoForStartOffset(blockBegin, ch, blockInfo);
            }

            ++ ch;
        }

    } // while(チャンネル)

    return true;
}

bool StreamSoundLoader::LoadAdpcmContextForStartOffset()
{
    u32 fpos = m_pFileStream->Tell();

    u16 yn1[ STRM_CHANNEL_NUM ];
    u16 yn2[ STRM_CHANNEL_NUM ];
    if ( ! m_FileLoader.ReadSeekBlockData(
             yn1,
             yn2,
             m_LoadingDataBlockIndex,
             m_ChannelCount
         ) )
    {
        return false;
    }
    m_pFileStream->Seek(fpos, ut::FILE_STREAM_SEEK_BEGIN);

    for ( unsigned int ch=0; ch < m_ChannelCount; ch++ )
    {
        m_AdpcmContextForStartOffset[ch].yn1 = yn1[ch];
        m_AdpcmContextForStartOffset[ch].yn2 = yn2[ch];
    }

    return true;
}

void StreamSoundLoader::UpdateAdpcmInfoForStartOffset(const void* blockBegin, u32 channel, const BlockInfo& blockInfo)
{
    AdpcmContext& adpcmContext = m_AdpcmContextForStartOffset[channel];
    adpcmContext.pred_scale = *reinterpret_cast<const u8*>(blockBegin);

    // ADPCM フレーム (8バイト=14サンプル) の途中からはデコードできないので、
    // オフセットをフレーム境界に合わせる
    u32 offset = static_cast<u32>(blockInfo.startOffsetSamples/14)*14;
    MultiVoice::CalcOffsetAdpcmParam(
        &adpcmContext,
        m_AdpcmInfo[channel].param,
        offset,
        blockBegin
    );
    m_AdpcmContextForStartOffsetFrame = m_Region.current;
}

bool StreamSoundLoader::SetAdpcmInfo(
    StreamSoundFileReader& reader,
    u32 channelCount,
    AdpcmParam* adpcmParam[])
{

    NW_ASSERT( channelCount <= STRM_CHANNEL_NUM );

    for ( unsigned int ch = 0; ch < channelCount; ch++ )
    {
        DspAdpcmParam dspAdpcmParam;
        DspAdpcmLoopParam dspAdpcmLoopParam;
        if ( ! reader.ReadDspAdpcmChannelInfo(
                 &dspAdpcmParam,
                 &dspAdpcmLoopParam,
                 ch ) )
        {
            return false;
        }

        AdpcmInfo& adpcmInfo = m_AdpcmInfo[ch];
        for ( int i= 0; i < 8; i++ )
        {
            adpcmInfo.param.coef[i][0] = dspAdpcmParam.coef[i][0];
            adpcmInfo.param.coef[i][1] = dspAdpcmParam.coef[i][1];
        }
        adpcmInfo.beginContext.pred_scale = dspAdpcmParam.predScale;
        adpcmInfo.beginContext.yn1 = dspAdpcmParam.yn1;
        adpcmInfo.beginContext.yn2 = dspAdpcmParam.yn2;

        adpcmInfo.loopContext.pred_scale = dspAdpcmLoopParam.loopPredScale;
        adpcmInfo.loopContext.yn1 = dspAdpcmLoopParam.loopYn1;
        adpcmInfo.loopContext.yn2 = dspAdpcmLoopParam.loopYn2;

        adpcmParam[ch] = &adpcmInfo.param;

    #ifdef ENABLE_PRINT_STREAM_INFO
        NW_LOG("*** STRM ADPCM INFO ch(%d) ***\n", ch);
        NW_LOG("    [coef] %04x %04x %04x %04x %04x %04x %04x %04x\n",
                adpcmInfo.param.coef[0][0], adpcmInfo.param.coef[0][1],
                adpcmInfo.param.coef[1][0], adpcmInfo.param.coef[1][1],
                adpcmInfo.param.coef[2][0], adpcmInfo.param.coef[2][1],
                adpcmInfo.param.coef[3][0], adpcmInfo.param.coef[3][1] );
        NW_LOG("           %04x %04x %04x %04x %04x %04x %04x %04x\n",
                adpcmInfo.param.coef[4][0], adpcmInfo.param.coef[4][1],
                adpcmInfo.param.coef[5][0], adpcmInfo.param.coef[5][1],
                adpcmInfo.param.coef[6][0], adpcmInfo.param.coef[6][1],
                adpcmInfo.param.coef[7][0], adpcmInfo.param.coef[7][1] );
        NW_LOG("    [begin] predScale(%04x) yn1(%6d) yn2(%6d)\n",
                adpcmInfo.beginContext.pred_scale,
                adpcmInfo.beginContext.yn1, adpcmInfo.beginContext.yn2 );
        NW_LOG("    [loop ] predScale(%04x) yn1(%6d) yn2(%6d)\n",
                adpcmInfo.loopContext.pred_scale,
                adpcmInfo.loopContext.yn1, adpcmInfo.loopContext.yn2 );
    #endif
    }

    return true;
}

bool StreamSoundLoader::DecodeStreamData(StreamDataDecoder::DecodeType decodeType, void* bufferAddress[])
{
    s16* decodedDataArray[STRM_CHANNEL_NUM];
    for( unsigned int i = 0; i < m_ChannelCount; i++ )
    {
        decodedDataArray[i] = reinterpret_cast<s16*>(bufferAddress[i]);
    }

    return s_StreamDataDecoder->Decode( m_pFileStream, m_ChannelCount, decodeType, decodedDataArray, m_pDecodeWorkBuffer );
}

} // namespace nw::snd::internal::driver
} // namespace nw::snd::internal
} // namespace nw::snd
} // namespace nw

