﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nw/snd/snd_StreamSoundFileReader.h>
#include <nw/snd/snd_WaveFile.h>                // EncodeMethod

namespace nw {
namespace snd {
namespace internal {

namespace {

const u32 SIGNATURE_FILE_STM = NW_UT_MAKE_SIGWORD( 'F', 'S', 'T', 'M' );
const u32 SIGNATURE_INFO_BLOCK_STM = NW_UT_MAKE_SIGWORD( 'I', 'N', 'F', 'O' );
// const u32 SIGNATURE_SEEK_BLOCK_STM = NW_UT_MAKE_SIGWORD( 'S', 'E', 'E', 'K' );
// const u32 SIGNATURE_DATA_BLOCK_STM = NW_UT_MAKE_SIGWORD( 'D', 'A', 'T', 'A' );

const u32 SUPPORTED_FILE_VERSION_STM       = NW_UT_MAKE_VERSION( 0, 1, 0, 0 );
const u32 CURRENT_FILE_VERSION_STM         = NW_UT_MAKE_VERSION( 0, 4, 0, 0 );
const u32 INCLUDE_TRACKINFO_VERSION_STM    = NW_UT_MAKE_VERSION( 0, 2, 0, 0 );
const u32 INCLUDE_ORIGINALLOOP_VERSION_STM = NW_UT_MAKE_VERSION( 0, 4, 0, 0 );

} // anonymous namespace

StreamSoundFileReader::StreamSoundFileReader()
: m_pHeader( NULL ),
  m_pInfoBlockBody( NULL )
{
}

void StreamSoundFileReader::Initialize( const void* streamSoundFile )
{
    NW_NULL_ASSERT( streamSoundFile );

    if ( ! IsValidFileHeader( streamSoundFile ) ) return;

    m_pHeader =
        reinterpret_cast<const StreamSoundFile::FileHeader*>( streamSoundFile );
    const StreamSoundFile::InfoBlock* infoBlock = m_pHeader->GetInfoBlock();
    NW_ASSERT( infoBlock->header.kind == SIGNATURE_INFO_BLOCK_STM );
    if ( infoBlock->header.kind != SIGNATURE_INFO_BLOCK_STM )
    {
        return;
    }


    m_pInfoBlockBody = &infoBlock->body;
    NW_ASSERT( m_pInfoBlockBody->GetStreamSoundInfo()->oneBlockBytes % 32 == 0 );
}

void StreamSoundFileReader::Finalize()
{
    m_pHeader = NULL;
    m_pInfoBlockBody = NULL;
}

bool StreamSoundFileReader::IsTrackInfoAvailable() const
{
#if defined(NW_PLATFORM_CAFE)
    const ut::BinaryFileHeader& header =
        *reinterpret_cast<const ut::BinaryFileHeader*>( m_pHeader );
#else
    const BinaryFileHeader& header =
        *reinterpret_cast<const BinaryFileHeader*>( m_pHeader );
#endif
    if (header.version <= INCLUDE_TRACKINFO_VERSION_STM)
    {
        return true;
    }
    return false;
}

bool StreamSoundFileReader::IsOriginalLoopAvailable() const
{
#if defined(NW_PLATFORM_CAFE)
    const ut::BinaryFileHeader& header =
        *reinterpret_cast<const ut::BinaryFileHeader*>( m_pHeader );
#else
    const BinaryFileHeader& header =
        *reinterpret_cast<const BinaryFileHeader*>( m_pHeader );
#endif
    if (header.version >= INCLUDE_ORIGINALLOOP_VERSION_STM)
    {
        return true;
    }
    return false;
}

bool StreamSoundFileReader::ReadStreamSoundInfo(
        StreamSoundFile::StreamSoundInfo* strmInfo ) const
{
    NW_NULL_ASSERT( m_pInfoBlockBody );
    const StreamSoundFile::StreamSoundInfo* info =
        m_pInfoBlockBody->GetStreamSoundInfo();

    NW_ASSERT_ALIGN32( (u32)(info->oneBlockBytes) );
    NW_ASSERT_ALIGN32( (u32)(info->lastBlockPaddedBytes) );

    strmInfo->encodeMethod = info->encodeMethod;
    strmInfo->isLoop = info->isLoop;
    strmInfo->channelCount = info->channelCount;
    strmInfo->regionCount = info->regionCount;
    strmInfo->sampleRate = info->sampleRate;
    strmInfo->loopStart = info->loopStart;
    strmInfo->frameCount = info->frameCount;
    strmInfo->blockCount = info->blockCount;

    strmInfo->oneBlockBytes = info->oneBlockBytes;
    strmInfo->oneBlockSamples = info->oneBlockSamples;

    strmInfo->lastBlockBytes = info->lastBlockBytes;
    strmInfo->lastBlockSamples = info->lastBlockSamples;
    strmInfo->lastBlockPaddedBytes = info->lastBlockPaddedBytes;

    strmInfo->sizeofSeekInfoAtom = info->sizeofSeekInfoAtom;
    strmInfo->seekInfoIntervalSamples = info->seekInfoIntervalSamples;

    strmInfo->sampleDataOffset = info->sampleDataOffset;

    strmInfo->regionInfoBytes = info->regionInfoBytes;
    strmInfo->regionDataOffset = info->regionDataOffset;

    if ( IsOriginalLoopAvailable() )
    {
        strmInfo->originalLoopStart = info->originalLoopStart;
        strmInfo->originalLoopEnd = info->originalLoopEnd;
    }
    else
    {
        strmInfo->originalLoopStart = info->loopStart;
        strmInfo->originalLoopEnd = info->frameCount;
    }


    return true;
}

bool StreamSoundFileReader::ReadStreamTrackInfo(
        TrackInfo* pTrackInfo, int trackIndex ) const
{
    NW_NULL_ASSERT( m_pInfoBlockBody );
    NW_NULL_ASSERT( pTrackInfo );
    const StreamSoundFile::TrackInfoTable* table =
        m_pInfoBlockBody->GetTrackInfoTable();
    if ( table == NULL )
    {
        return false;
    }

    if ( trackIndex >= static_cast<int>( table->GetTrackCount() ) )
    {
        return false;
    }

    const StreamSoundFile::TrackInfo* src = table->GetTrackInfo( trackIndex );
    pTrackInfo->volume = src->volume;
    pTrackInfo->pan = src->pan;
    pTrackInfo->span = src->span;
    pTrackInfo->flags = src->flags;
    pTrackInfo->channelCount = static_cast<u8>( src->GetTrackChannelCount() ); // TODO: バイナリデータも 1 バイトでいいかも (NW4C-0.1.0)

    u32 count = ut::Min( static_cast<u32>( pTrackInfo->channelCount ), WAVE_CHANNEL_MAX );
    for ( u32 i = 0; i < count; i++ )
    {
        pTrackInfo->globalChannelIndex[ i ] = src->GetGlobalChannelIndex( i );
    }

    return true;
}

bool StreamSoundFileReader::ReadDspAdpcmChannelInfo(
        DspAdpcmParam* pParam,
        DspAdpcmLoopParam* pLoopParam,
        int channelIndex ) const
{
    NW_NULL_ASSERT( m_pInfoBlockBody );
    NW_NULL_ASSERT( pParam );
    NW_NULL_ASSERT( pLoopParam );

    const StreamSoundFile::DspAdpcmChannelInfo* src =
        m_pInfoBlockBody->
        GetChannelInfoTable()->
        GetChannelInfo( channelIndex )->
        GetDspAdpcmChannelInfo();
    if ( src == NULL ) return false;

    *pParam = src->param;
    *pLoopParam = src->loopParam;
    return true;
}

bool StreamSoundFileReader::IsValidFileHeader( const void* streamSoundFile ) const
{
    NW_NULL_ASSERT( streamSoundFile );

#if defined(NW_PLATFORM_CAFE)
    const ut::BinaryFileHeader& header =
        *reinterpret_cast<const ut::BinaryFileHeader*>( streamSoundFile );
#else
    const BinaryFileHeader& header =
        *reinterpret_cast<const BinaryFileHeader*>( streamSoundFile );
#endif

    // シグニチャ確認
    NW_ASSERTMSG(
            header.signature == SIGNATURE_FILE_STM,
            "invalid file signature. stream data is not available." );
    if ( header.signature != SIGNATURE_FILE_STM ) return false;

    // バージョン確認
    bool isSupportedVersion = false;
    if ( (SUPPORTED_FILE_VERSION_STM <= header.version) &&
                                   (header.version <= CURRENT_FILE_VERSION_STM) )
    {
        isSupportedVersion = true;

    }
    NW_ASSERTMSG( isSupportedVersion,
            "bfstm file is not supported version.\n"
            "please reconvert file using new version tools.\n"
            "(version condition: 0x%08x <= ... <= 0x%08x, but your version[0x%08x])\n",
            SUPPORTED_FILE_VERSION_STM, CURRENT_FILE_VERSION_STM, header.version
    );
    return isSupportedVersion;
}

} // namespace nw::snd::internal
} // namespace nw::snd
} // namespace nw

