﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nw/types.h>

#if defined( NW_PLATFORM_WIN32 ) || defined( NW_USE_NINTENDO_SDK )
#include "windows.h"
#endif

#if defined( NW_PLATFORM_CAFE )
  #include <cafe.h>
#elif defined( NW_PLATFORM_WIN32 )
  #include <winext/cafe.h>
#elif defined( NW_PLATFORM_ANDROID ) || defined( NW_PLATFORM_IOS )
  #include <winext/cafe.h>
#elif defined( NW_USE_NINTENDO_SDK )
  // TODO: nn_audio
  #include <winext/cafe.h>
  #include <nn/os/os_Thread.h>
#else
  #error
#endif

#include <nw/snd/snd_SoundSystem.h>
#include <nw/snd/snd_MultiVoice.h>
#include <nw/snd/snd_Channel.h>
#include <nw/snd/snd_ChannelManager.h>
#include <nw/snd/snd_DriverCommand.h>
#include <nw/snd/snd_SequenceSoundPlayer.h>
#include <nw/snd/snd_TaskManager.h>
#include <nw/snd/snd_NwVoice.h>
#include <nw/snd/snd_VoiceCommand.h>
#include <nw/snd/snd_CurveLfo.h>

#include <nw/middlewareSymbol.h>

namespace nw {
namespace snd {

bool SoundSystem::s_IsInitialized = false;
bool SoundSystem::s_IsStreamLoadWait = false;
bool SoundSystem::s_IsEnterSleep = false;
bool SoundSystem::s_IsInitializedDriverCommandManager = false;
void* SoundSystem::s_LoadThreadStackPtr = NULL;
u32 SoundSystem::s_LoadThreadStackSize = 0;
void* SoundSystem::s_SoundThreadStackPtr = NULL;
u32 SoundSystem::s_SoundThreadStackSize = 0;
int SoundSystem::s_MaxVoiceCount = AX_MAX_VOICES;
internal::NwVoiceRenderer* SoundSystem::s_pNwVoiceRenderer = NULL;
bool SoundSystem::s_IsInitializedNwRenderer = false;
bool SoundSystem::s_IsStreamOpenFailureHalt = true;

#if defined( NW_PLATFORM_CAFE )
const s32 SoundSystem::SoundSystemParam::DEFAULT_SOUND_THREAD_PRIORITY = 4;
const s32 SoundSystem::SoundSystemParam::DEFAULT_TASK_THREAD_PRIORITY = 3;
#elif defined( NW_PLATFORM_WIN32 )
const s32 SoundSystem::SoundSystemParam::DEFAULT_SOUND_THREAD_PRIORITY = THREAD_PRIORITY_HIGHEST;
const s32 SoundSystem::SoundSystemParam::DEFAULT_TASK_THREAD_PRIORITY = THREAD_PRIORITY_HIGHEST;
#elif defined( NW_PLATFORM_ANDROID ) || defined( NW_PLATFORM_IOS )
// TODO: 暫定処置(ut::thread がプライオリティ設定にまだ対応していない)
const s32 SoundSystem::SoundSystemParam::DEFAULT_SOUND_THREAD_PRIORITY = 0;
const s32 SoundSystem::SoundSystemParam::DEFAULT_TASK_THREAD_PRIORITY = 0;
#elif defined( NW_USE_NINTENDO_SDK )
const s32 SoundSystem::SoundSystemParam::DEFAULT_SOUND_THREAD_PRIORITY = nn::os::HighestThreadPriority;
const s32 SoundSystem::SoundSystemParam::DEFAULT_TASK_THREAD_PRIORITY = nn::os::HighestThreadPriority;
#else
#error
#endif


SoundSystem::SoundSystemParam::SoundSystemParam()
: soundThreadPriority( DEFAULT_SOUND_THREAD_PRIORITY )
, soundThreadStackSize( DEFAULT_SOUND_THREAD_STACK_SIZE )
, soundThreadCommandBufferSize( DEFAULT_SOUND_THREAD_COMMAND_BUFFER_SIZE )
#ifdef NW_SND_CONFIG_ENABLE_VOICE_COMMAND
, voiceCommandBufferSize( 512*1024 )
#endif
, taskThreadPriority( DEFAULT_TASK_THREAD_PRIORITY )
, taskThreadStackSize( DEFAULT_TASK_THREAD_STACK_SIZE )
, taskThreadCommandBufferSize( DEFAULT_TASK_THREAD_COMMAND_BUFFER_SIZE )
, enableGetSoundThreadTick( true )
, enableNwRenderer( false )
#ifdef NW_SND_CONFIG_ENABLE_VOICE_COMMAND
, nwVoiceSynthesizeBufferCount(16)
#else
#if defined( NW_PLATFORM_CAFE )
, nwVoiceSynthesizeBufferCount(2)
#elif defined( NW_PLATFORM_WIN32 )
, nwVoiceSynthesizeBufferCount(5)
#elif defined( NW_PLATFORM_ANDROID ) || defined( NW_PLATFORM_IOS )
, nwVoiceSynthesizeBufferCount(5)
#elif defined( NW_USE_NINTENDO_SDK )
// TODO: nn_audio
, nwVoiceSynthesizeBufferCount(5)
#else
#error
#endif
#endif
{
}

size_t SoundSystem::GetRequiredMemSize( const SoundSystemParam& param )
{
    NW_ASSERT_ALIGN( param.soundThreadStackSize, 8 );
    NW_ASSERT_ALIGN( param.taskThreadStackSize, 8 );

    size_t result =
        detail_GetRequiredDriverCommandManagerMemSize( param )
      + param.taskThreadStackSize
      + internal::driver::MultiVoiceManager::GetInstance().GetRequiredMemSize( s_MaxVoiceCount )
      + internal::driver::ChannelManager::GetInstance().GetRequiredMemSize( s_MaxVoiceCount );

    result += param.soundThreadStackSize;

    if ( param.enableNwRenderer )
    {
        result += internal::NwVoiceSynthesizeBuffer::GetSynthesizeBufferSize() * param.nwVoiceSynthesizeBufferCount;
        result += sizeof(internal::NwVoiceRenderer);
    }

    // 先頭アドレスをCACHE_BLOCK_SIZEアライメントするための余裕
    result += CACHE_BLOCK_SIZE - 1;

    return result;
}

size_t SoundSystem::detail_GetRequiredDriverCommandManagerMemSize(
        const SoundSystemParam& param )
{
    NW_ASSERT( param.soundThreadCommandBufferSize % 4 == 0 );
    NW_ASSERT( param.taskThreadCommandBufferSize % 4 == 0 );

    size_t result
        = param.soundThreadCommandBufferSize
        + param.taskThreadCommandBufferSize
#ifdef NW_SND_CONFIG_ENABLE_VOICE_COMMAND
        + param.voiceCommandBufferSize * 3 // ( AxVocie + NwVoice ) + VoiceReply
#endif
        ;

    return result;
}

void SoundSystem::detail_InitializeDriverCommandManager(
    const SoundSystemParam& param,
    uptr workMem,
    size_t workMemSize )
{
    if ( s_IsInitializedDriverCommandManager )
    {
        return;
    }

    NW_UNUSED_VARIABLE( workMemSize );
    NW_ASSERTMSG( workMemSize >= detail_GetRequiredDriverCommandManagerMemSize( param ),
            "workMemSize(%X) >= RequiredMemSize(%X)",
            workMemSize, detail_GetRequiredDriverCommandManagerMemSize( param ) );

    uptr ptr = workMem;

    // コマンドバッファ初期化
    uptr soundThreadCommandBufferPtr = ptr;
    ptr += param.soundThreadCommandBufferSize;

    uptr taskThreadCommandBufferPtr = ptr;
    ptr += param.taskThreadCommandBufferSize;

    internal::DriverCommand::GetInstance().Initialize(
        reinterpret_cast<void*>( soundThreadCommandBufferPtr ),
        param.soundThreadCommandBufferSize
    );
    internal::DriverCommand::GetInstanceForTaskThread().Initialize(
        reinterpret_cast<void*>( taskThreadCommandBufferPtr ),
        param.taskThreadCommandBufferSize
    );

#ifdef NW_SND_CONFIG_ENABLE_VOICE_COMMAND
    uptr axVoiceCommandBufferPtr = ptr;
    ptr += param.voiceCommandBufferSize;

    uptr nwVoiceCommandBufferPtr = ptr;
    ptr += param.voiceCommandBufferSize;

    uptr voiceReplyCommandBufferPtr = ptr;
    ptr += param.voiceCommandBufferSize;

    internal::AxVoiceCommand::GetInstance().Initialize(
        reinterpret_cast<void*>( axVoiceCommandBufferPtr ),
        param.voiceCommandBufferSize
    );
    internal::NwVoiceCommand::GetInstance().Initialize(
        reinterpret_cast<void*>( nwVoiceCommandBufferPtr ),
        param.voiceCommandBufferSize
    );
    internal::VoiceReplyCommand::GetInstance().Initialize(
        reinterpret_cast<void*>( voiceReplyCommandBufferPtr ),
        param.voiceCommandBufferSize
    );
#endif

    s_IsInitializedDriverCommandManager = true;
}

void SoundSystem::Initialize(
    const SoundSystemParam& param,
    uptr workMem,
    size_t workMemSize )
{
    NW_PUT_MIDDLEWARE_SYMBOL(snd);

    NW_ASSERT_ALIGN4( workMem );
    NW_UNUSED_VARIABLE( workMemSize );
    NW_ASSERTMSG( workMemSize >= GetRequiredMemSize( param ),
            "workMemSize(%X) >= RequiredMemSize(%X)",
            workMemSize, GetRequiredMemSize( param ) );

    // 多重初期化チェック
    if ( s_IsInitialized )
    {
        return;
    }

#if defined( NW_PLATFORM_WIN32 )
    nw::internal::winext::AXInit();
#elif defined( NW_PLATFORM_ANDROID ) || defined( NW_PLATFORM_IOS )
    nw::internal::winext::AXInit();
#elif defined( NW_USE_NINTENDO_SDK )
    // TODO: nn_audio
    nw::internal::winext::AXInit();
#endif

    bool result;
    NW_UNUSED_VARIABLE(result);

    uptr ptr = workMem;

    ptr = ut::RoundUp( ptr, CACHE_BLOCK_SIZE );

    // NwVoiceRenderer の初期化
    if ( param.enableNwRenderer )
    {
        uptr synthesizeBuffer = ptr;
        u32 synthesizeBufferSize = internal::NwVoiceSynthesizeBuffer::GetSynthesizeBufferSize() * param.nwVoiceSynthesizeBufferCount;
        ptr += synthesizeBufferSize;
        uptr nwRendererWork = ptr;
        ptr += sizeof(internal::NwVoiceRenderer);

        s_pNwVoiceRenderer = new ( reinterpret_cast<void*>(nwRendererWork) ) internal::NwVoiceRenderer();
        s_pNwVoiceRenderer->Initialize( reinterpret_cast<void*>(synthesizeBuffer), synthesizeBufferSize );

        internal::driver::SoundThread::GetInstance().detail_SetNwVoiceRenderer( s_pNwVoiceRenderer );

        s_IsInitializedNwRenderer = true;
    }
    else
    {
        s_IsInitializedNwRenderer = false;
    }

    // コマンドバッファの初期化
    if ( s_IsInitializedDriverCommandManager == false )
    {
        size_t workMemSizeForDriverCommandManager =
            detail_GetRequiredDriverCommandManagerMemSize( param );
        detail_InitializeDriverCommandManager(
                param,
                ptr,
                workMemSizeForDriverCommandManager );
        ptr += workMemSizeForDriverCommandManager;
    }

    // SDK 関連初期化
    internal::driver::HardwareManager::GetInstance().Initialize();

    // リモコンスピーカー初期化
    internal::RemoteSpeakerManager::GetInstance().Initialize();


    // nw::snd サウンドスレッドの準備
    {
        uptr soundThreadStackPtr = static_cast<uptr>(NULL);
        soundThreadStackPtr = ptr;
        ptr += param.soundThreadStackSize;
        s_SoundThreadStackPtr = reinterpret_cast<void*>(soundThreadStackPtr);
        s_SoundThreadStackSize = param.soundThreadStackSize;
    }
    internal::driver::SoundThread::GetInstance().Initialize();

    // nw::snd サウンドデータロードスレッドの準備
    {
        uptr loadThreadStackPtr = ptr;
        ptr += param.taskThreadStackSize;
        internal::TaskManager::GetInstance().Initialize();
        s_LoadThreadStackPtr = reinterpret_cast<void*>(loadThreadStackPtr);
        s_LoadThreadStackSize = param.taskThreadStackSize;
    }

    // MultiVoiceManager初期化
    {
        uptr voiceWork = ptr;
        ptr += internal::driver::MultiVoiceManager::GetInstance().GetRequiredMemSize( s_MaxVoiceCount );
        internal::driver::MultiVoiceManager::GetInstance().Initialize(
                reinterpret_cast<void*>( voiceWork ),
                internal::driver::MultiVoiceManager::GetInstance().GetRequiredMemSize( s_MaxVoiceCount )
                );
    }

    // ChannelManager初期化
    {
        uptr channelWork = ptr;
        ptr += internal::driver::ChannelManager::GetInstance().GetRequiredMemSize( s_MaxVoiceCount );
        internal::driver::ChannelManager::GetInstance().Initialize(
                reinterpret_cast<void*>( channelWork ),
                internal::driver::ChannelManager::GetInstance().GetRequiredMemSize( s_MaxVoiceCount )
                );
    }

    NW_ASSERT( ptr <= workMem + workMemSize );

    // SequenceSoundPlayer初期化
    internal::driver::SequenceSoundPlayer::InitSequenceSoundPlayer();

    u16 attribute = 0;
#if defined( NW_PLATFORM_CAFE )
    u32 coreId = OSGetCoreId();
    switch( coreId ) {
    case 0:
        attribute = OS_THREAD_ATTR_AFFINITY_CORE0;
        break;
    case 1:
        attribute = OS_THREAD_ATTR_AFFINITY_CORE1;
        break;
    case 2:
        attribute = OS_THREAD_ATTR_AFFINITY_CORE2;
        break;
    }
#endif // defined( NW_PLATFORM_CAFE )

    // データロードスレッド起動
    result = internal::TaskThread::GetInstance().Create(
        param.taskThreadPriority,
        s_LoadThreadStackPtr,
        s_LoadThreadStackSize,
        attribute
    );
    NW_ASSERT( result );

    // サウンドスレッド起動
    result = internal::driver::SoundThread::GetInstance().CreateSoundThread(
        param.soundThreadPriority,
        s_SoundThreadStackPtr,
        s_SoundThreadStackSize,
        attribute,
        param.enableGetSoundThreadTick );
    NW_ASSERT( result );

    // SEQ モジュレーションカーブテーブルの初期化
    internal::CurveLfo::InitializeCurveTable();

    // Ax ボイスのレンダリング周波数の初期化
#if defined(NW_SND_CONFIG_ENABLE_SOUND2) && defined(NW_PLATFORM_CAFE)
    internal::AxVoice::SetSamplesPerFrame( static_cast<s32>( AXGetInputSamplesPerFrame() ) );
    internal::AxVoice::SetSamplesPerSec  ( static_cast<s32>( AXGetInputSamplesPerSec() ) );
#else
    internal::AxVoice::SetSamplesPerFrame( AX_IN_SAMPLES_PER_FRAME );
    internal::AxVoice::SetSamplesPerSec  ( AX_IN_SAMPLES_PER_SEC );
#endif

    s_IsInitialized = true;
}

void SoundSystem::Finalize()
{
    if ( ! s_IsInitialized )
    {
        return;
    }

    // データロードスレッド終了
    internal::TaskManager::GetInstance().CancelAllTask();
    internal::TaskThread::GetInstance().Destroy();
    internal::TaskManager::GetInstance().Finalize();

    // サウンドスレッド停止
    internal::driver::SoundThread::GetInstance().Destroy();

    // チャンネルマネージャー破棄
    internal::RemoteSpeakerManager::GetInstance().Finalize();
    internal::driver::ChannelManager::GetInstance().Finalize();
    internal::driver::MultiVoiceManager::GetInstance().Finalize();
    internal::driver::HardwareManager::GetInstance().Finalize();

    if ( s_IsInitializedNwRenderer )
    {
        internal::driver::SoundThread::GetInstance().detail_SetNwVoiceRenderer( NULL );

        s_pNwVoiceRenderer->Finalize();
        s_IsInitializedNwRenderer = false;
    }

    // サウンドスレッド終了
    internal::driver::SoundThread::GetInstance().Finalize();

    internal::DriverCommand::GetInstance().Finalize();
    internal::DriverCommand::GetInstanceForTaskThread().Finalize();

#ifdef NW_SND_CONFIG_ENABLE_VOICE_COMMAND
    internal::AxVoiceCommand::GetInstance().Finalize();
    internal::NwVoiceCommand::GetInstance().Finalize();
    internal::VoiceReplyCommand::GetInstance().Finalize();
#endif

    s_IsInitialized = false;
    s_IsInitializedDriverCommandManager = false;

#if defined( NW_PLATFORM_WIN32 )
    nw::internal::winext::AXQuit();
#elif defined( NW_PLATFORM_ANDROID ) || defined( NW_PLATFORM_IOS )
    nw::internal::winext::AXQuit();
#elif defined( NW_USE_NINTENDO_SDK )
    // TODO: nn_audio
    nw::internal::winext::AXQuit();
#endif
}

bool SoundSystem::IsInitialized()
{
    return s_IsInitialized;
}

void SoundSystem::SetNwVoiceSampleRate( u32 sampleRate )
{
    if ( s_pNwVoiceRenderer != NULL )
    {
        if ( sampleRate == 48000 )
        {
            s_pNwVoiceRenderer->SetSampleRate( snd::internal::NwVoiceRenderer::SAMPLE_RATE_48000 );
        }
        else if ( sampleRate == 32000 )
        {
            s_pNwVoiceRenderer->SetSampleRate( snd::internal::NwVoiceRenderer::SAMPLE_RATE_32000 );
        }
    }
}


#if 0
/*---------------------------------------------------------------------------*
  Name:         WaitForResetReady

  Description:  リセット準備の完了を待つ

  Arguments:    無し

  Returns:      無し
 *---------------------------------------------------------------------------*/
void SoundSystem::WaitForResetReady()
{
    if ( ! s_IsInitialized ) return;

#ifdef NW_PLATFORM_CTRWIN
    OSTick tick = OS_GetTick();
#else
    nn::os::Tick tick = nn::os::Tick::GetSystemCurrent();
#endif

    while( ! internal::driver::HardwareManager::GetInstance().IsResetReady() )
    {
#ifdef NW_PLATFORM_CTRWIN
        s64 seconds = OS_TicksToSeconds( OS_DiffTick( OS_GetTick(), tick ) );
#else
        s64 seconds = static_cast<nn::fnd::TimeSpan>(
                nn::os::Tick::GetSystemCurrent() - tick ).GetSeconds();
#endif
        if ( seconds > 0 )
        {
            NW_WARNING( false, "SoundSystem::WaitForResetReady is TIME OUT.\n" );
            break;
        }
    }
}
#endif

bool SoundSystem::AppendEffect( AuxBus bus, FxBase* effect, OutputDevice device )
{
    NW_NULL_ASSERT( effect );

    if ( ! effect->IsValidChannelNum( device ) )
    {
        return false;
    }

    if ( ! effect->Initialize() )
    {
        return false;
    }

    internal::DriverCommand& cmdmgr = internal::DriverCommand::GetInstance();

    internal::DriverCommandEffect* command =
        cmdmgr.AllocCommand<internal::DriverCommandEffect>();
    command->id = internal::DRIVER_COMMAND_APPEND_EFFECT;
    command->bus = bus;
    command->device = device;
    command->effect = effect;
    cmdmgr.PushCommand(command);

    return true;
}

void SoundSystem::ClearEffect( AuxBus bus, int fadeTimes, OutputDevice device )
{
    internal::DriverCommand& cmdmgr = internal::DriverCommand::GetInstance();

    internal::DriverCommandEffect* command =
        cmdmgr.AllocCommand<internal::DriverCommandEffect>();
    command->id = internal::DRIVER_COMMAND_CLEAR_EFFECT;
    command->bus = bus;
    command->device = device;
    command->fadeTimes = fadeTimes;
    cmdmgr.PushCommand(command);

    u32 tag = cmdmgr.FlushCommand( true );
    cmdmgr.WaitCommandReply( tag );
}

void SoundSystem::VoiceCommandProcess(u32 audioFrameCount)
{
    if (s_IsInitialized == false)
    {
        return;
    }

    internal::NwVoiceCommand& nwVoiceCommand = internal::NwVoiceCommand::GetInstance();
    internal::driver::SoundThread& soundThread = internal::driver::SoundThread::GetInstance();
    internal::AxVoiceCommand& axVoiceCommand = internal::AxVoiceCommand::GetInstance();

    while (axVoiceCommand.GetCommandListCount() < audioFrameCount)
    {
        soundThread.FrameProcess();

        axVoiceCommand.RecvCommandReply();
        axVoiceCommand.FlushCommand( true );

        nwVoiceCommand.RecvCommandReply();
        nwVoiceCommand.FlushCommand( false ); // NwVoice はこの後に受信側処理を行うため、処理を待たないよう false を渡す
    }
    while (nwVoiceCommand.ProcessCommand())
    {
        internal::NwVoiceRenderer* nwVoiceRenderer = soundThread.detail_GetNwVoiceRenderer();
        if ( nwVoiceRenderer != NULL )
        {
            nwVoiceRenderer->UpdateAllVoices();
            nwVoiceRenderer->Synthesize();
        }
    }
}

} // namespace nw::snd
} // namespace nw

