﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nw/snd/snd_SoundPlayer.h>

#include <nw/snd/snd_SoundThread.h>

namespace nw {
namespace snd {

/* ========================================================================
        public function
   ======================================================================== */

/*---------------------------------------------------------------------------*
  Name:         SoundPlayer

  Description:  コンストラクタ

  Arguments:    None.

  Returns:      None.
 *---------------------------------------------------------------------------*/
SoundPlayer::SoundPlayer()
: m_PlayableCount( 1 ),
  m_PlayableLimit( INT_MAX ),
  m_PlayerHeapCount( 0 ),
  m_Volume( 1.0f ),
  m_LpfFreq( 0.0f ),
  m_BiquadType( BIQUAD_FILTER_TYPE_INHERIT ),
  m_BiquadValue( 0.0f ),
  m_OutputLineFlag( OUTPUT_LINE_MAIN ),
  m_IsFirstComeBased( false )
{
    for ( int i = 0; i < REMOTE_OUT_COUNT; i++ )
    {
        m_RemoteOutVolume[ i ] = 1.0f;
    }
    for ( int i = 0; i < DRC_OUT_COUNT; i++ )
    {
        m_DrcParam[i].Initialize();
    }
    m_TvParam.Initialize();
}

/*---------------------------------------------------------------------------*
  Name:         ~SoundPlayer

  Description:  デストラクタ

  Arguments:    None.

  Returns:      None.
 *---------------------------------------------------------------------------*/
SoundPlayer::~SoundPlayer()
{
    StopAllSound( 0 );
}

/*---------------------------------------------------------------------------*
  Name:         Update

  Description:  プレイヤーのフレーム処理を更新する

  Arguments:    None.

  Returns:      None.
 *---------------------------------------------------------------------------*/
void SoundPlayer::Update()
{
    // 利用済みプレイヤーヒープを m_PlayerHeapFreeList に戻す
    DoFreePlayerHeap();

    // BasicSound::Update
    for ( SoundList::Iterator itr = m_SoundList.GetBeginIter();
          itr != m_SoundList.GetEndIter();
        )
    {
        SoundList::Iterator curItr = itr++;
        curItr->Update();
    }

    // プライオリティリストを整列しなおす
    detail_SortPriorityList();
}

/*---------------------------------------------------------------------------*
  Name:         StopAllSound

  Description:  全てのサウンドを停止する

  Arguments:    fadeFrames - フェードアウトフレーム数

  Returns:      None.
 *---------------------------------------------------------------------------*/
void SoundPlayer::StopAllSound( int fadeFrames )
{
    for ( SoundList::Iterator itr = m_SoundList.GetBeginIter();
          itr != m_SoundList.GetEndIter();
        )
    {
        SoundList::Iterator curItr = itr++;
        curItr->Stop( fadeFrames );
    }
}

/*---------------------------------------------------------------------------*
  Name:         PauseAllSound

  Description:  全てのサウンドを一時停止または再開する

  Arguments:    flag       - 一時停止か再開か
                fadeFrames - フェードフレーム数

  Returns:      None.
 *---------------------------------------------------------------------------*/
void SoundPlayer::PauseAllSound( bool flag, int fadeFrames )
{
    for ( SoundList::Iterator itr = m_SoundList.GetBeginIter();
          itr != m_SoundList.GetEndIter();
        )
    {
        SoundList::Iterator curItr = itr++;
        curItr->Pause( flag, fadeFrames );
    }
}

/*---------------------------------------------------------------------------*
  Name:         SetVolume

  Description:  音量を変更

  Arguments:    volume - 音量

  Returns:      None.
 *---------------------------------------------------------------------------*/
void SoundPlayer::SetVolume( float volume )
{
    NW_ASSERT( volume >= 0.0f );
    if ( volume < 0.0f ) volume = 0.0f;
    m_Volume = volume;
}

/*---------------------------------------------------------------------------*
  Name:         SetLpfFreq

  Description:  ローパスフィルタの周波数変更

  Arguments:    lpfFreq - ローパスフィルタの周波数値

  Returns:      なし
 *---------------------------------------------------------------------------*/
void SoundPlayer::SetLpfFreq( float lpfFreq )
{
    m_LpfFreq = lpfFreq;
}

/*---------------------------------------------------------------------------*
  Name:         SetBiquadFilter

  Description:  Biquadフィルタの設定

  Arguments:    type  - フィルタの種類 (-1 - 127)
                value - フィルタの値 (0.0f - 1.0f)

  Returns:      なし
 *---------------------------------------------------------------------------*/
void SoundPlayer::SetBiquadFilter( int type, float value )
{
    m_BiquadType = type;
    m_BiquadValue = value;
}

void SoundPlayer::SetDefaultOutputLine( u32 outputLineFlag )
{
    m_OutputLineFlag = outputLineFlag;
}

void SoundPlayer::SetMainOutVolume( f32 volume )
{
    NW_ASSERT( volume >= 0.0f );
    if ( volume < 0.0f )
    {
        volume = 0.0f;
    }
    m_TvParam.volume = volume;
}

void SoundPlayer::SetRemoteOutVolume( int remoteIndex, f32 volume )
{
    NW_ASSERT_MINMAXLT( remoteIndex, 0, REMOTE_OUT_COUNT );
    NW_ASSERT( volume >= 0.0f );
    if ( volume < 0.0f )
    {
        volume = 0.0f;
    }
    m_RemoteOutVolume[ remoteIndex ] = volume;
}

f32 SoundPlayer::GetRemoteOutVolume( int remoteIndex ) const
{
    NW_ASSERT_MINMAXLT( remoteIndex, 0, REMOTE_OUT_COUNT );
    return m_RemoteOutVolume[ remoteIndex ];
}

void SoundPlayer::SetDrcOutVolume( f32 volume, u32 drcIndex )
{
    NW_ASSERT_MAXLT( drcIndex, DRC_OUT_COUNT );
    NW_ASSERT( volume >= 0.0f );
    if ( volume < 0.0f )
    {
        volume = 0.0f;
    }
    m_DrcParam[drcIndex].volume = volume;
}

void SoundPlayer::SetOutputVolume( OutputDevice device, f32 volume )
{
    NW_ASSERT_MINMAXLT( device, 0, OUTPUT_DEVICE_COUNT );
    NW_ASSERT( volume >= 0.0f );
    if ( volume < 0.0f )
    {
        volume = 0.0f;
    }

    switch ( device )
    {
    case nw::snd::OUTPUT_DEVICE_MAIN:
        m_TvParam.volume = volume;
        break;
    case nw::snd::OUTPUT_DEVICE_DRC:
        m_DrcParam[0].volume = volume;
        break;
    default:
        NW_ASSERTMSG( false, "Invalid device\n" );
        break;
    }

}

/*---------------------------------------------------------------------------*
  Name:         RemoveSoundList

  Description:  サウンドをプレイヤーリストから削除する

  Arguments:    pSound - シーケンスサウンド

  Returns:      None.
 *---------------------------------------------------------------------------*/
void SoundPlayer::RemoveSoundList( internal::BasicSound* pSound )
{
    // 再生リストから削除する
    m_SoundList.Erase( pSound );
    pSound->DetachSoundPlayer( this );
}

/*---------------------------------------------------------------------------*
  Name:         InsertPriorityList

  Description:  サウンドをプライオリティリストへ追加

  Arguments:    pSound - 追加するサウンド

  Returns:      None.
  *---------------------------------------------------------------------------*/
void SoundPlayer::InsertPriorityList( internal::BasicSound* pSound )
{
    PriorityList::Iterator itr = m_PriorityList.GetBeginIter();
    while ( itr != m_PriorityList.GetEndIter() )
    {
        if ( m_IsFirstComeBased )
        {
            if ( pSound->CalcCurrentPlayerPriority() <= itr->CalcCurrentPlayerPriority() ) break;
        }
        else
        {
            if ( pSound->CalcCurrentPlayerPriority() < itr->CalcCurrentPlayerPriority() ) break;
        }
        (void)++itr;
    }
    m_PriorityList.Insert( itr, pSound );
}

/*---------------------------------------------------------------------------*
  Name:         RemovePriorityList

  Description:  サウンドをプライオリティリストから削除

  Arguments:    pSound - 削除するサウンド

  Returns:      None.
  *---------------------------------------------------------------------------*/
void SoundPlayer::RemovePriorityList( internal::BasicSound* pSound )
{
    m_PriorityList.Erase( pSound );
}

/*---------------------------------------------------------------------------*
  Name:         detail_SortPriorityList

  Description:  指定したサウンドのプライオリティ順序を並び替える

  Arguments:    pSound - 並び替えるサウンド

  Returns:      None.
  *---------------------------------------------------------------------------*/
void SoundPlayer::detail_SortPriorityList( internal::BasicSound* pSound )
{
    RemovePriorityList( pSound );
    InsertPriorityList( pSound );
}

/*---------------------------------------------------------------------------*
  Name:         detail_SortPriorityList

  Description:  全サウンドのプライオリティ順序を並び替える

  Arguments:    reverse - 同一プライオリティ内で先着順/後着順を並び替える.

  Returns:      None.
  *---------------------------------------------------------------------------*/
void SoundPlayer::detail_SortPriorityList( bool reverse )
{
    if ( m_PriorityList.GetSize() < 2 ) return;

    static const int TMP_NUM = internal::BasicSound::PRIORITY_MAX - internal::BasicSound::PRIORITY_MIN + 1;
    static PriorityList tmplist[ TMP_NUM ]; // notice: large stack
#ifdef NW_DEBUG
    for ( int i=0; i<TMP_NUM; i++ ) NW_ASSERT( tmplist[i].IsEmpty() );
#endif

    while ( !m_PriorityList.IsEmpty() )
    {
        internal::BasicSound& front = m_PriorityList.GetFront();
        m_PriorityList.PopFront();
        tmplist[ front.CalcCurrentPlayerPriority() ].PushBack( &front );
    }
    for ( int i=0; i<TMP_NUM; i++ )
    {
        while ( !tmplist[i].IsEmpty() )
        {
            if ( reverse )
            {
                internal::BasicSound& back = tmplist[i].GetBack();
                tmplist[i].PopBack();
                m_PriorityList.PushBack( &back );
            }
            else
            {
                internal::BasicSound& front = tmplist[i].GetFront();
                tmplist[i].PopFront();
                m_PriorityList.PushBack( &front );
            }
        }
    }
}

/*---------------------------------------------------------------------------*
  Name:         detail_AppendSound

  Description:  サウンドをサウンドプレイヤーに登録する

  Arguments:    pSound - サウンド

  Returns:      登録できたらtrue
 *---------------------------------------------------------------------------*/
bool SoundPlayer::detail_AppendSound( internal::BasicSound* pSound )
{
    NW_NULL_ASSERT( pSound );

    int allocPriority = pSound->CalcCurrentPlayerPriority();

    // 最大同時再生数のチェック
    if ( GetPlayableSoundCount() == 0 )
    {
        return false;
    }
    while ( GetPlayingSoundCount() >= GetPlayableSoundCount() )
    {
        internal::BasicSound* dropSound = GetLowestPrioritySound();
        if ( dropSound == NULL )
        {
            return false;
        }
        if ( m_IsFirstComeBased )
        {
            if ( allocPriority <= dropSound->CalcCurrentPlayerPriority() ) return false;
        }
        else
        {
            if ( allocPriority < dropSound->CalcCurrentPlayerPriority() ) return false;
        }
        dropSound->Finalize();
    }

    // リストへ登録
    m_SoundList.PushBack( pSound );
    InsertPriorityList( pSound );

    pSound->AttachSoundPlayer( this );

    return true;
}

/*---------------------------------------------------------------------------*
  Name:         detail_RemoveSound

  Description:  サウンドをサウンドプレイヤーから削除する

  Arguments:    pSound - 削除するサウンド

  Returns:      None.
  *---------------------------------------------------------------------------*/
void SoundPlayer::detail_RemoveSound( internal::BasicSound* pSound )
{
    RemovePriorityList( pSound );
    RemoveSoundList( pSound );
}

/*---------------------------------------------------------------------------*
  Name:         SetPlayableSoundCount

  Description:  同時再生数を設定

  Arguments:    count - 同時再生数

  Returns:      None.
 *---------------------------------------------------------------------------*/
void SoundPlayer::SetPlayableSoundCount( int count )
{
    NW_ASSERT( count >= 0 );

    // プレイヤーヒープ使用時の上限
    NW_WARNING( count <= m_PlayableLimit, "playable sound count is over limit." );
    count = ut::Clamp( count, 0, m_PlayableLimit );

    m_PlayableCount = count;

    // 新しく設定された同時再生数を越えるサウンドを終了する
    while ( GetPlayingSoundCount() > GetPlayableSoundCount() )
    {
        internal::BasicSound* dropSound = GetLowestPrioritySound();
        NW_NULL_ASSERT( dropSound );
        dropSound->Finalize();
    }
}

/*---------------------------------------------------------------------------*
  Name:         detail_SetPlayableSoundLimit

  Description:

  Arguments:    None.

  Returns:      None.
  *---------------------------------------------------------------------------*/
void SoundPlayer::detail_SetPlayableSoundLimit( int limit )
{
    NW_ASSERT( limit >= 0 );

    m_PlayableLimit = limit;
}

/*---------------------------------------------------------------------------*
  Name:         detail_CanPlaySound

  Description:  指定したプライオリティのサウンドを再生できるかどうかを調べる

  Arguments:    startPriority - プライオリティ

  Returns:      再生可能ならtrue
 *---------------------------------------------------------------------------*/
bool SoundPlayer::detail_CanPlaySound( int startPriority )
{
    // プレイヤー毎の最大同時再生数のチェック
    if ( GetPlayableSoundCount() == 0 ) return false;
    if ( GetPlayingSoundCount() >= GetPlayableSoundCount() )
    {
        internal::BasicSound* dropSound = GetLowestPrioritySound();
        if ( dropSound == NULL ) return false;
        if ( m_IsFirstComeBased )
        {
            if ( startPriority <= dropSound->CalcCurrentPlayerPriority() ) return false;
        }
        else
        {
            if ( startPriority < dropSound->CalcCurrentPlayerPriority() ) return false;
        }
    }

    return true;
}

/*---------------------------------------------------------------------------*
  Name:         detail_AppendPlayerHeap

  Description:  プレイヤーヒープを追加

  Arguments:    heap - 追加するプレイヤーヒープ

  Returns:      None.
 *---------------------------------------------------------------------------*/
void SoundPlayer::detail_AppendPlayerHeap( internal::PlayerHeap* pHeap )
{
    NW_NULL_ASSERT( pHeap );

    pHeap->AttachSoundPlayer( this );
    m_PlayerHeapFreeList.PushBack( pHeap );
    m_PlayerHeapCount++;
}

internal::PlayerHeap* SoundPlayer::detail_AllocPlayerHeap()
{
    // プレイヤーヒープの確保
    if ( m_PlayerHeapFreeList.IsEmpty() )
    {
        return NULL;
    }
    internal::PlayerHeap& playerHeap = m_PlayerHeapFreeList.GetFront();
    m_PlayerHeapFreeList.PopFront();

    // NOTE: プレイヤーヒープ初期化 playerHeap.Clear() は、
    // 各データロードタスクの Execute あたまで呼び出す
    return &playerHeap;
}

void SoundPlayer::detail_FreePlayerHeap(internal::PlayerHeap* pHeap)
{
    // プレイヤーヒープの解放
    if (pHeap->GetState() == internal::PlayerHeap::STATE_TASK_FINISHED)
    {
        m_PlayerHeapFreeList.PushBack(pHeap);
    }
    else
    {
        // タスク未完時は、次の Update で回収を試みる
        m_PlayerHeapFreeReqList.PushBack(pHeap);
    }
}

// 利用済みプレイヤーヒープを m_PlayerHeapFreeList に戻す
void SoundPlayer::DoFreePlayerHeap()
{
    for ( PlayerHeapList::Iterator itr = m_PlayerHeapFreeReqList.GetBeginIter();
          itr != m_PlayerHeapFreeReqList.GetEndIter();
        )
    {
        PlayerHeapList::Iterator curItr = itr++;
        if (curItr->GetState() == internal::PlayerHeap::STATE_TASK_FINISHED)
        {
            m_PlayerHeapFreeReqList.Erase(&(*curItr));
            m_PlayerHeapFreeList.PushBack(&(*curItr));
        }
    }
}

} // namespace nw::snd
} // namespace nw

