﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nw/snd/snd_SoundHeap.h>

#include <nw/ut/ut_Inlines.h>
#include <nw/snd/snd_SoundThread.h>
#include <nw/snd/snd_DriverCommand.h>
#include <nw/snd/snd_Util.h>

namespace nw {
namespace snd {

/*---------------------------------------------------------------------------*
  Name:         SoundHeap

  Description:  コンストラクタ

  Arguments:    無し

  Returns:      無し
 *---------------------------------------------------------------------------*/
SoundHeap::SoundHeap()
: m_FrameHeap()
{
}

/*---------------------------------------------------------------------------*
  Name:         SoundHeap

  Description:  デストラクタ

  Arguments:    無し

  Returns:      無し
 *---------------------------------------------------------------------------*/
SoundHeap::~SoundHeap()
{
    Destroy();

//    m_CriticalSection.Finalize();
}

/*---------------------------------------------------------------------------*
  Name:         Create

  Description:  サウンドヒープを作成する

  Arguments:    startAddress - メインメモリ上のヒープ開始アドレス
                size         - メインメモリ上のヒープのサイズ

  Returns:      成功したらtrue
 *---------------------------------------------------------------------------*/
bool SoundHeap::Create(
    void* startAddress,
    size_t size
)
{
    NW_WARNING(
        snd::internal::Util::IsValidMemoryForDsp( startAddress, size ),
        "the memory area (0x%08x - 0x%08x %dbyte) provided cross a 512 MB segment.",
        startAddress,
        ut::AddOffsetToPtr( startAddress, size ),
        size );

    ut::ScopedLock<ut::CriticalSection> lock( m_CriticalSection );

    return m_FrameHeap.Create( startAddress, size );
}

/*---------------------------------------------------------------------------*
  Name:         Destroy

  Description:  ヒープを破棄する

  Arguments:    無し

  Returns:      無し
 *---------------------------------------------------------------------------*/
void SoundHeap::Destroy()
{
    ut::ScopedLock<ut::CriticalSection> lock( m_CriticalSection );

    if ( ! m_FrameHeap.IsValid() ) return;

    Clear();

    m_FrameHeap.Destroy();
}

/*---------------------------------------------------------------------------*
  Name:         Alloc

  Description:  メインメモリ上のヒープからメモリ領域を割り当てます

  Arguments:    size        - メモリ領域のサイズ

  Returns:      割り当てたメモリ領域の先頭アドレス
 *---------------------------------------------------------------------------*/
void* SoundHeap::Alloc( size_t size )
{
//    NW_LOG("SoundHeap::Alloc %d / %d\n",size,GetFreeSize());

    ut::ScopedLock<ut::CriticalSection> lock( m_CriticalSection );

    void* buffer = Alloc(
        size,
        DisposeCallbackFunc,
        NULL
    );
    // NN_LOG("[%s] free(%d %dKB %.3fMB)\n", __FUNCTION__,
    //         GetFreeSize(), GetFreeSize()/1024, GetFreeSize()/(1024.f*1024));
    return buffer;
}

/*---------------------------------------------------------------------------*
  Name:         Alloc

  Description:  メインメモリ上のヒープからメモリ領域を割り当てます

  Arguments:    size        - メモリ領域のサイズ
                callback    - メモリ領域が解放されるときに呼び出されるコールバック
                callbackArg - コールバック

  Returns:      割り当てたメモリ領域の先頭アドレス
 *---------------------------------------------------------------------------*/
void* SoundHeap::Alloc(
    size_t size,
    SoundHeap::DisposeCallback callback,
    void* callbackArg
)
{
//    NW_LOG("SoundHeap::Alloc %d / %d\n",size,GetFreeSize());

    ut::ScopedLock<ut::CriticalSection> lock( m_CriticalSection );

    return m_FrameHeap.Alloc(
        size,
        callback,
        callbackArg
    );
}

// 確保したメモリブロックを全て解放する
void SoundHeap::Clear()
{
    if ( ! m_FrameHeap.IsValid() ) return;

    {
        ut::ScopedLock<ut::CriticalSection> lock( m_CriticalSection );

        m_FrameHeap.Clear();
    }

    internal::DriverCommand& cmdmgr = internal::DriverCommand::GetInstance();
    if ( cmdmgr.IsAvailable() ) {
        u32 tag = cmdmgr.FlushCommand( true );
        cmdmgr.WaitCommandReply( tag );
    }
}

// サウンドヒープの状態を保存する
int SoundHeap::SaveState()
{
    ut::ScopedLock<ut::CriticalSection> lock( m_CriticalSection );

    return m_FrameHeap.SaveState();
}

// サウンドヒープの状態を復元する
void SoundHeap::LoadState( int level )
{
    {
        ut::ScopedLock<ut::CriticalSection> lock( m_CriticalSection );

        m_FrameHeap.LoadState( level );

        internal::DriverCommand& cmdmgr = internal::DriverCommand::GetInstance();
        if ( cmdmgr.IsAvailable() ) {
            u32 tag = cmdmgr.FlushCommand( true );
            cmdmgr.WaitCommandReply( tag );
        }
    }

    // NN_LOG("[%s] free(%d %dKB %.3fMB)\n", __FUNCTION__,
    //         GetFreeSize(), GetFreeSize()/1024, GetFreeSize()/(1024.f*1024));
}

/* ========================================================================
 private function
 ======================================================================== */

void SoundHeap::DisposeCallbackFunc( void* mem, unsigned long size, void* arg )
{
    NW_UNUSED_VARIABLE(arg);

    internal::DriverCommand& cmdmgr = internal::DriverCommand::GetInstance();

    if ( cmdmgr.IsAvailable() ) {
        internal::DriverCommandInvalidateData* command =
            cmdmgr.AllocCommand<internal::DriverCommandInvalidateData>();
        command->id = internal::DRIVER_COMMAND_INVALIDATE_DATA;
        command->mem = mem;
        command->size = size;
        cmdmgr.PushCommand(command);
    }
}

} // namespace nw::snd
} // namespace nw

