﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nw/snd/snd_SoundArchivePlayer.h>

#include <nw/snd/snd_SoundSystem.h>
#include <nw/snd/snd_SoundDataManager.h>
#include <nw/snd/snd_SoundPlayer.h>
#include <nw/snd/snd_SoundHandle.h>
#include <nw/snd/snd_SoundActor.h>
#include <nw/snd/snd_SoundArchiveFilesHook.h>
#include <nw/snd/snd_SoundArchiveParametersHook.h>
#include <nw/snd/snd_Bank.h>
#include <nw/snd/snd_StreamSoundFileReader.h>
#include <nw/snd/snd_SequenceSoundFileReader.h>
#include <nw/snd/snd_WaveSoundFileReader.h>
#include <nw/snd/snd_WaveArchiveFileReader.h>
#include <nw/snd/snd_WaveFileReader.h>
#include <nw/snd/snd_Debug.h>
#include <nw/snd/snd_Util.h>
#include <nw/snd/snd_PlayerHeap.h>
#include <nw/snd/snd_DriverCommand.h>
#include <nw/snd/snd_FsFileStream.h>
#include <nw/snd/snd_Config.h>

#include <nw/snd/fnd/string/sndfnd_String.h>

// #define NW_SND_DEBUG_PRINT_ENABLE
#ifdef NW_SND_DEBUG_PRINT_ENABLE
    #define NW_SND_INIT_LOG NW_LOG
#else
    #define NW_SND_INIT_LOG(...)
#endif

namespace
{
    const size_t DATA_ALIGN_SIZE = PPC_IO_BUFFER_ALIGN;
    const size_t MEM_ALIGN_SIZE = 32;
#if defined(NW_PLATFORM_CAFE)
    const int VOICE_OFFSET_BOUNDARY_LIMITATION_BYTES = 512 * 1024 * 1024; // バッファ配置アドレスの境界の制限

    bool IsValidVoiceOffsetBoundary(const void* bufferAddress1, const void* bufferAddress2)
    {
        // bufferAddress1 > bufferAddress2 直後の 512MB 境界、あるいは bufferAddress2 > bufferAddress1 直後の 512MB 境界であれば不正
        return ! (nw::ut::ComparePtr(bufferAddress1, nw::ut::RoundUp(bufferAddress2, VOICE_OFFSET_BOUNDARY_LIMITATION_BYTES)) > 0
               || nw::ut::ComparePtr(bufferAddress2, nw::ut::RoundUp(bufferAddress1, VOICE_OFFSET_BOUNDARY_LIMITATION_BYTES)) > 0);
    }
#endif
}

namespace nw {
namespace snd {

/*---------------------------------------------------------------------------*
  Name:         SoundArchivePlayer

  Description:  コンストラクタ

  Arguments:    無し

  Returns:      無し
 *---------------------------------------------------------------------------*/
SoundArchivePlayer::SoundArchivePlayer()
: m_pSoundArchive( NULL ),
  m_SequenceUserprocCallback( NULL ),
  m_pSequenceUserprocCallbackArg( NULL ),
  m_SoundPlayerCount( 0 ),
  m_pSoundPlayers( NULL ),
  m_pSequenceTrackAllocator( NULL ),
  m_MmlSequenceTrackAllocator( &m_MmlParser ),
  m_SoundUserParamSize( 0 ),
  m_pSoundDataManager( NULL ),
  m_pSoundArchiveFilesHook( NULL ),
  m_IsEnableWarningPrint(false),
  m_IsInitialized(false)
{
    m_SequenceCallback.Initialize(*this);
}

/*---------------------------------------------------------------------------*
  Name:         SoundArchivePlayer

  Description:  デストラクタ

  Arguments:    無し

  Returns:      無し
 *---------------------------------------------------------------------------*/
SoundArchivePlayer::~SoundArchivePlayer()
{
}

/*---------------------------------------------------------------------------*
  Name:         IsAvailable

  Description:  サウンドアーカイブプレイヤーが利用可能かどうかを調べる

  Arguments:    無し

  Returns:      利用可能かどうかを返す
 *---------------------------------------------------------------------------*/
bool SoundArchivePlayer::IsAvailable() const
{
    if ( m_pSoundArchive == NULL )
    {
        return false;
    }
    if ( ! m_pSoundArchive->IsAvailable() )
    {
        return false;
    }

    return true;
}

/*---------------------------------------------------------------------------*
  Name:         Initialize

  Description:  サウンドアーカイブ中のプレイヤー情報にしたがって
                プレイヤーをセットアップする

  Arguments:    arc - サウンドアーカイブ
                buffer - セットアップに使用するメモリ
                size - メモリサイズ
                aramBuffer - セットアップに使用するＡメモリ
                aramSize - メモリサイズ

  Returns:      セットアップに成功したらtrue
 *---------------------------------------------------------------------------*/
bool SoundArchivePlayer::Initialize(
    const SoundArchive* arc,
    const SoundDataManager* manager,
    void*   buffer,
    u32     size,
    void*   strmBuffer,
    u32     strmBufferSize,
    size_t  userParamSizePerSound
)
{
    if (m_IsInitialized)
    {
        NW_WARNING(false, "SoundArchivePlayer::Initialize failed (already initialized)\n");
        return false;
    }

    // サウンドシステムの初期化が必要
    NW_ASSERT( SoundSystem::IsInitialized() );
    if ( ! SoundSystem::IsInitialized() )
    {
        return false;
    }

    NW_ASSERT_NOT_NULL( arc );
    NW_ASSERT_NOT_NULL( manager );
    NW_ASSERT_NOT_NULL( buffer );

    if ( strmBufferSize > 0 )
    {
        NW_ASSERT_NOT_NULL( strmBuffer );
    }
    NW_ASSERT( strmBufferSize >= GetRequiredStreamBufferSize( arc ) );
    NW_WARNING( strmBufferSize >= GetRequiredStreamBufferSize( arc ) * GetRequiredStreamBufferTimes( arc ),
        "StreamBuffer is not enough for prefetch data, Please figure in StreamBufferTimes." );

    // ストリームバッファは 4 倍まで
    NW_ASSERT( strmBufferSize <= GetRequiredStreamBufferSize( arc ) * 4 );

    if ( ! SetupMram( arc, buffer, size, userParamSizePerSound ) ) {
        return false;
    }

    if ( ! SetupStreamBuffer( arc, strmBuffer, strmBufferSize ) ) {
        return false;
    }

    m_pSequenceTrackAllocator = &m_MmlSequenceTrackAllocator;
    m_pSoundDataManager = manager;

    m_IsInitialized = true;
    return true;
}

/*---------------------------------------------------------------------------*
  Name:         Finalize

  Description:  サウンドアーカイブプレイヤーの機能を停止させる

  Arguments:    無し

  Returns:      無し
 *---------------------------------------------------------------------------*/
void SoundArchivePlayer::Finalize()
{
    if (!m_IsInitialized)
    {
        return;
    }

    // 参照の破棄
    m_pSoundArchive = NULL;
    m_pSequenceTrackAllocator = NULL;

    // 全サウンドの停止
    for ( SoundArchive::ItemId playerId = 0; playerId < m_SoundPlayerCount ; ++playerId )
    {
        m_pSoundPlayers[ playerId ].StopAllSound(0);
    }

    // 停止完了待ち
    internal::DriverCommand& cmdmgr = internal::DriverCommand::GetInstance();
    u32 tag = cmdmgr.FlushCommand( true );
    cmdmgr.WaitCommandReply( tag );

    // インスタンスの破棄
    m_SoundPlayerCount = 0;
    m_pSoundPlayers = NULL;


    m_SequenceSoundInstanceManager.Destroy();
    m_SequenceSoundLoaderManager.Destroy();
    internal::driver::SoundThread::GetInstance().UnregisterSoundFrameCallback(&m_SequenceSoundLoaderManager);
    m_MmlSequenceTrackAllocator.Destroy();

    m_WaveSoundInstanceManager.Destroy();
    m_WaveSoundLoaderManager.Destroy();
    internal::driver::SoundThread::GetInstance().UnregisterSoundFrameCallback(&m_WaveSoundLoaderManager);

    m_StreamBufferPool.Finalize();
    m_StreamSoundInstanceManager.Destroy();
    m_StreamSoundLoaderManager.Destroy();
    internal::driver::SoundThread::GetInstance().UnregisterSoundFrameCallback(&m_StreamSoundLoaderManager);

    m_IsInitialized = false;
}


/*---------------------------------------------------------------------------*
  Name:         GetRequiredMemSize

  Description:  サウンドアーカイブプレイヤーで使用するメモリサイズを取得する

  Arguments:    arc - サウンドアーカイブ

  Returns:      無し
 *---------------------------------------------------------------------------*/
size_t SoundArchivePlayer::GetRequiredMemSize(
        const SoundArchive* arc, size_t userParamSizePerSound ) const
{
    NW_ASSERT_NOT_NULL( arc );

    size_t size = 0;

    // SoundPlayer
    {
        u32 playerCount = arc->GetPlayerCount();
        size += ut::RoundUp( playerCount * sizeof( SoundPlayer ), MEM_ALIGN_SIZE );
        NW_SND_INIT_LOG("[SoundPlayer] instance(%d) x playerCount(%d), ALIGN(%d) => %d\n",
                sizeof(SoundPlayer), playerCount, MEM_ALIGN_SIZE, size);

        for ( u32 playerIndex = 0; playerIndex < playerCount; ++playerIndex )
        {
            SoundArchive::PlayerInfo playerInfo;
            if ( ! arc->ReadPlayerInfo( internal::Util::GetMaskedItemId(
                            playerIndex, internal::ItemType_Player ), &playerInfo ) )
            {
                continue;
            }

            if ( playerInfo.playerHeapSize > 0 )
            {
                for ( int i = 0; i < playerInfo.playableSoundMax ; i++ )
                {
                    size += ut::RoundUp( sizeof(internal::PlayerHeap), DATA_ALIGN_SIZE );
                    NW_SND_INIT_LOG("[%d] [PlayerHeap:%d/%d] instance(%8d) ALIGN(%d) => %8d\n",
                            playerIndex, i, playerInfo.playableSoundMax,
                            sizeof(internal::PlayerHeap), DATA_ALIGN_SIZE, size);
                    size += ut::RoundUp( playerInfo.playerHeapSize, MEM_ALIGN_SIZE );
                    NW_SND_INIT_LOG("[%d] [PlayerHeap:%d/%d] heapSize(%8d) ALIGN(%d) => %8d\n",
                            playerIndex, i, playerInfo.playableSoundMax,
                            playerInfo.playerHeapSize, MEM_ALIGN_SIZE, size);
                }
            }
        }
    }

    SoundArchive::SoundArchivePlayerInfo soundArchivePlayerInfo;
    if ( arc->ReadSoundArchivePlayerInfo( &soundArchivePlayerInfo ) )
    {
        // SequenceSound
        size += ut::RoundUp(
            m_SequenceSoundInstanceManager.GetRequiredMemSize(
                soundArchivePlayerInfo.sequenceSoundMax), MEM_ALIGN_SIZE );
        NW_SND_INIT_LOG("[SEQ] instance: max(%2d) ALIGN(%d) => %8d\n",
                soundArchivePlayerInfo.sequenceSoundMax, MEM_ALIGN_SIZE, size);
        size += ut::RoundUp(
            internal::driver::SequenceSoundLoaderManager::GetRequiredMemSize(
                soundArchivePlayerInfo.sequenceSoundMax), MEM_ALIGN_SIZE );
        NW_SND_INIT_LOG("[SEQ] loader:   max(%2d) ALIGN(%d) => %8d\n",
                soundArchivePlayerInfo.sequenceSoundMax, MEM_ALIGN_SIZE, size);

        // WaveSound
        size += ut::RoundUp(
            m_WaveSoundInstanceManager.GetRequiredMemSize(
                soundArchivePlayerInfo.waveSoundMax), MEM_ALIGN_SIZE );
        NW_SND_INIT_LOG("[WSD] instance: max(%2d) ALIGN(%d) => %8d\n",
                soundArchivePlayerInfo.waveSoundMax, MEM_ALIGN_SIZE, size);
        size += ut::RoundUp(
            internal::driver::WaveSoundLoaderManager::GetRequiredMemSize(
                soundArchivePlayerInfo.waveSoundMax), MEM_ALIGN_SIZE );
        NW_SND_INIT_LOG("[WSD] loader:   max(%2d) ALIGN(%d) => %8d\n",
                soundArchivePlayerInfo.waveSoundMax, MEM_ALIGN_SIZE, size);

        // StreamSound
        size += ut::RoundUp(
            m_StreamSoundInstanceManager.GetRequiredMemSize(
                soundArchivePlayerInfo.streamSoundMax), MEM_ALIGN_SIZE );
        NW_SND_INIT_LOG("[STRM] isntance:max(%2d) ALIGN(%d) => %8d\n",
                soundArchivePlayerInfo.streamSoundMax, MEM_ALIGN_SIZE, size);
        size += ut::RoundUp(
            internal::driver::StreamSoundLoaderManager::GetRequiredMemSize(
                soundArchivePlayerInfo.streamSoundMax), MEM_ALIGN_SIZE );
        NW_SND_INIT_LOG("[STRM] loader:   max(%2d) ALIGN(%d) => %8d\n",
                soundArchivePlayerInfo.streamSoundMax, MEM_ALIGN_SIZE, size);

        // SequenceTrack
        size += ut::RoundUp(
                soundArchivePlayerInfo.sequenceTrackMax *
                sizeof( internal::driver::MmlSequenceTrack ), MEM_ALIGN_SIZE );
        NW_SND_INIT_LOG("[SEQ] track:    max(%2d) x trackSize(%d) ALIGN(%d) => %8d\n",
                soundArchivePlayerInfo.sequenceTrackMax,
                sizeof(internal::driver::MmlSequenceTrack), MEM_ALIGN_SIZE, size);

    }

    // userParam
    if (userParamSizePerSound > 0)
    {
        u32 soundCount =
            soundArchivePlayerInfo.sequenceSoundMax
          + soundArchivePlayerInfo.waveSoundMax
          + soundArchivePlayerInfo.streamSoundMax;
        size_t adjustSize = ut::RoundUp(userParamSizePerSound, 4);
        size += (adjustSize * soundCount);
        NW_SND_INIT_LOG("[UserParam] adjustSize(%d) x soundCount(%d) => size(%8d)\n",
                adjustSize, soundCount, size);
    }

    return size;
}

/*---------------------------------------------------------------------------*
  Name:         GetRequiredStreamBufferSize

  Description:  サウンドアーカイブプレイヤーで使用するＡメモリサイズを取得する

  Arguments:    arc - サウンドアーカイブ

  Returns:      無し
 *---------------------------------------------------------------------------*/
size_t SoundArchivePlayer::GetRequiredStreamBufferSize(
        const nw::snd::SoundArchive* arc ) const
{
    NW_ASSERT_NOT_NULL( arc );

    int strmChannelCount = 0;

    SoundArchive::SoundArchivePlayerInfo soundArchivePlayerInfo;
    if ( arc->ReadSoundArchivePlayerInfo( &soundArchivePlayerInfo ) )
    {
        strmChannelCount = soundArchivePlayerInfo.streamChannelMax;
    }

    size_t memSize = static_cast<size_t>(
        internal::driver::StreamSoundLoader::DATA_BLOCK_SIZE_MAX
        * DEFAULT_STREAM_BLOCK_COUNT
        * strmChannelCount
    );
    return memSize;
}

s32 SoundArchivePlayer::GetRequiredStreamBufferTimes(
    const nw::snd::SoundArchive* arc ) const
{
    NW_ASSERT_NOT_NULL( arc );

    int strmBufferTimes  = 1;

    SoundArchive::SoundArchivePlayerInfo soundArchivePlayerInfo;
    if ( arc->ReadSoundArchivePlayerInfo( &soundArchivePlayerInfo ) )
    {
        strmBufferTimes  = soundArchivePlayerInfo.streamBufferTimes;
    }

    return strmBufferTimes;
}

size_t SoundArchivePlayer::GetRequiredStreamCacheSize(const SoundArchive* arc, size_t cacheSizePerSound) const
{
    NW_ASSERT_NOT_NULL(arc);

    int strmSoundCount = 0;

    SoundArchive::SoundArchivePlayerInfo soundArchivePlayerInfo;
    if ( arc->ReadSoundArchivePlayerInfo( &soundArchivePlayerInfo ) )
    {
        strmSoundCount = soundArchivePlayerInfo.streamSoundMax;
    }

    size_t memSize = strmSoundCount * cacheSizePerSound;
    return memSize;
}

bool SoundArchivePlayer::Initialize(const InitializeParam& param)
{
    bool result = Initialize(
            param.soundArchive,
            param.soundDataManager,
            param.setupBuffer,
            param.setupBufferSize,
            param.streamBuffer,
            param.streamBufferSize,
            param.userParamSizePerSound);
    if (result == false)
    {
        return false;
    }

    if (param.streamCacheBuffer)
    {
        NW_ASSERT_ALIGN(param.streamCacheBuffer, internal::FsFileStream::BUFFER_ALIGN_SIZE);

        int strmSoundCount = 0;
        SoundArchive::SoundArchivePlayerInfo soundArchivePlayerInfo;
        if ( param.soundArchive->ReadSoundArchivePlayerInfo( &soundArchivePlayerInfo ) )
        {
            strmSoundCount = soundArchivePlayerInfo.streamSoundMax;
        }
        else
        {
            //冒頭の Initialize で true になってしまうので、改めて false に
            m_IsInitialized = false;
            return false;
        }
        size_t cacheSize = param.streamCacheSize / strmSoundCount;
        NW_ASSERT_ALIGN(cacheSize, internal::FsFileStream::BUFFER_ALIGN_SIZE);

        void* cache = param.streamCacheBuffer;

        internal::StreamSoundInstanceManager::PriorityList& list =
            m_StreamSoundInstanceManager.GetFreeList();
        for (internal::StreamSoundInstanceManager::PriorityList::Iterator itr = list.GetBeginIter();
             itr != list.GetEndIter();
             ++itr)
        {
            NW_ASSERT_ALIGN(cache, internal::FsFileStream::BUFFER_ALIGN_SIZE);
            itr->SetCacheBuffer(cache, cacheSize);

            cache = ut::AddOffsetToPtr(cache, cacheSize);
        }
    }

    return true;
}

/*---------------------------------------------------------------------------*
  Name:         SetupMram

  Description:  サウンドアーカイブ中のプレイヤー情報にしたがって
                プレイヤーをセットアップする

  Arguments:    arc - サウンドアーカイブ
                buffer - セットアップに使用するメモリ
                size - メモリサイズ

  Returns:      セットアップに成功したらtrue
 *---------------------------------------------------------------------------*/
bool SoundArchivePlayer::SetupMram(
        const SoundArchive* arc,
        void* buffer,
        size_t size,
        size_t userParamSizePerSound )
{
    NW_ASSERT_ALIGN32( buffer );
    if ( reinterpret_cast<u32>(buffer) % MEM_ALIGN_SIZE != 0 )
    {
        return false;
    }

    // 必要以上のメモリは不要・必要以下だとまずい
    NW_ASSERT( size == GetRequiredMemSize( arc, userParamSizePerSound ) );

    const void* endAddr = static_cast<char*>(buffer) + size;

    NW_SND_INIT_LOG("[%s] end(%p) begin(%p)\n", __FUNCTION__, endAddr, buffer);

    void* curAddr = buffer;
    if ( ! SetupSoundPlayer( arc, &curAddr, endAddr ) )
    {
        return false;
    }
    NW_SND_INIT_LOG("[%s] cur(%p) - After SetupSoundPlayer\n", __FUNCTION__, curAddr);

    SoundArchive::SoundArchivePlayerInfo soundArchivePlayerInfo;
    if ( arc->ReadSoundArchivePlayerInfo( &soundArchivePlayerInfo ) )
    {
        if ( ! SetupSequenceSound( soundArchivePlayerInfo.sequenceSoundMax, &curAddr, endAddr ) )
        {
            return false;
        }
        NW_SND_INIT_LOG("[%s] cur(%p) - After SetupSequenceSound\n", __FUNCTION__, curAddr);

        if ( ! SetupWaveSound( soundArchivePlayerInfo.waveSoundMax, &curAddr, endAddr ) )
        {
            return false;
        }
        NW_SND_INIT_LOG("[%s] cur(%p) - After SetupWaveSound\n", __FUNCTION__, curAddr);

        if ( ! SetupStreamSound( soundArchivePlayerInfo.streamSoundMax, &curAddr, endAddr ) )
        {
            return false;
        }
        NW_SND_INIT_LOG("[%s] cur(%p) - After SetupStreamSound\n", __FUNCTION__, curAddr);

        if ( ! SetupSequenceTrack( soundArchivePlayerInfo.sequenceTrackMax, &curAddr, endAddr ) )
        {
            return false;
        }
        NW_SND_INIT_LOG("[%s] cur(%p) - After SetupSequenceTrack\n", __FUNCTION__, curAddr);
    }

    if (userParamSizePerSound > 0 )
    {
        // BasicSound へのユーザーパラメータ用バッファ割り当て
        if ( ! SetupUserParamForBasicSound(soundArchivePlayerInfo,
                    &curAddr, endAddr, userParamSizePerSound) )
        {
            return false;
        }
        NW_SND_INIT_LOG("[%s] cur(%p) - After SetupUserParamForBasicSound\n", __FUNCTION__, curAddr);
    }

#if !defined(NW_RELEASE)
    {
        s32 useSize = static_cast<char*>(curAddr) - static_cast<char*>(buffer);
        s32 requiredSize = static_cast<s32>( GetRequiredMemSize( arc ) );

        NW_ASSERT(useSize == requiredSize ||
                  useSize + static_cast<s32>(DATA_ALIGN_SIZE - MEM_ALIGN_SIZE) == requiredSize);
    }
#endif

    m_pSoundArchive = arc;

    return true;
}

/*---------------------------------------------------------------------------*
  Name:         CreatePlayerHeap

  Description:  プレイヤーヒープの作成

  Arguments:    heap - サウンドヒープ
                size - プレイヤーヒープサイズ

  Returns:      作成したプレイヤーヒープへのポインタ
 *---------------------------------------------------------------------------*/
internal::PlayerHeap* SoundArchivePlayer::CreatePlayerHeap(
    void** allocatedAddr,
    const void* endAddr,
    size_t heapSize
)
{
    NW_ASSERT_ALIGN32( *allocatedAddr );
    if ( reinterpret_cast<u32>(*allocatedAddr) % MEM_ALIGN_SIZE != 0 )
    {
        return NULL;
    }

    // プレイヤーヒープ構造体初期化
    void* estimateEndAddr = ut::RoundUp(
            ut::AddOffsetToPtr( *allocatedAddr, sizeof(internal::PlayerHeap) ), DATA_ALIGN_SIZE );
    if ( ut::ComparePtr( estimateEndAddr, endAddr ) > 0 )
    {
        return NULL;
    }
    void* curAddr = *allocatedAddr;
    *allocatedAddr = estimateEndAddr;
    NW_SND_INIT_LOG("[%s] cur(%p) - in CreatePlayerHeap alloc instance\n", __FUNCTION__, estimateEndAddr);

    internal::PlayerHeap* playerHeap = new ( curAddr ) internal::PlayerHeap();

    // プレイヤーヒープ構築
    // (PlayerHeap インスタンスの後ろに、データロード用のバッファを用意する。
    //  ファイルリード用のバッファは DATA_ALIGN_SIZE 境界にないといけない)
    estimateEndAddr = ut::AddOffsetToPtr( *allocatedAddr, ut::RoundUp( heapSize, MEM_ALIGN_SIZE ) );
    if ( ut::ComparePtr( estimateEndAddr, endAddr ) > 0 )
    {
        return NULL;
    }
    curAddr = *allocatedAddr;
    *allocatedAddr = estimateEndAddr;
    NW_SND_INIT_LOG("[%s] cur(%p) - in CreatePlayerHeap alloc heap\n", __FUNCTION__, estimateEndAddr);

    bool result = playerHeap->Create( curAddr, heapSize );
    if ( ! result )
    {
        return NULL;
    }

    return playerHeap;
}

bool SoundArchivePlayer::SetupSoundPlayer(
    const SoundArchive* arc,
    void** allocatedAddr,
    const void* endAddr
)
{
    NW_ASSERT_ALIGN32( *allocatedAddr );
    if ( reinterpret_cast<u32>(*allocatedAddr) % MEM_ALIGN_SIZE != 0 )
    {
        return false;
    }

    u32 playerCount = arc->GetPlayerCount();
    size_t requireSize = ut::RoundUp( playerCount * sizeof( SoundPlayer ), MEM_ALIGN_SIZE );
    void* estimateEndAddr = ut::AddOffsetToPtr( *allocatedAddr, requireSize );

    if ( ut::ComparePtr( estimateEndAddr, endAddr ) > 0 )
    {
        return false;
    }
    void* curAddr = *allocatedAddr;
    *allocatedAddr = estimateEndAddr;
    NW_SND_INIT_LOG("[%s] cur(%p) - in SetupSoundPlayer\n", __FUNCTION__, curAddr);

    // curAddr 上に SoundPlayer インスタンスを playerCount 個並べる
    m_pSoundPlayers = reinterpret_cast<SoundPlayer*>( curAddr );
    m_SoundPlayerCount = playerCount;

    u8* ptr = static_cast<u8*>( curAddr );
    for ( u32 playerIndex = 0; playerIndex < playerCount;
            ++playerIndex, ptr += sizeof( SoundPlayer ) )
    {
        SoundPlayer* player = new ( ptr ) SoundPlayer();

        SoundArchive::PlayerInfo playerInfo;
        if ( ! arc->ReadPlayerInfo( internal::Util::GetMaskedItemId(
                        playerIndex, internal::ItemType_Player ), &playerInfo ) )
        {
            continue;
        }

        player->SetPlayableSoundCount( playerInfo.playableSoundMax );

        // プレイヤーヒープのセットアップ
        if ( playerInfo.playerHeapSize > 0 )
        {
            for ( int i = 0; i < playerInfo.playableSoundMax ; i++ )
            {
                internal::PlayerHeap* playerHeap = CreatePlayerHeap(
                        allocatedAddr, endAddr, playerInfo.playerHeapSize );

                NW_WARNING( playerHeap != NULL,
                        "failed to create player heap. ( player id = %d )", playerIndex );
                if ( playerHeap == NULL )
                {
                    return false;
                }
                player->detail_AppendPlayerHeap( playerHeap );
            }

            // プレイヤーヒープを使用する場合、
            // 用意されたプレイヤーヒープ数を超える同時再生数は指定できない
            player->detail_SetPlayableSoundLimit( playerInfo.playableSoundMax );
        }
    }

    return true;
}

/*---------------------------------------------------------------------------*
  Name:         SetupSequenceSound

  Description:  シーケンスサウンドのセットアップ

  Arguments:    numSounds - 確保するシーケンスサウンドの数
                heap      - サウンドヒープ

  Returns:      成功したかどうか
 *---------------------------------------------------------------------------*/
bool SoundArchivePlayer::SetupSequenceSound(
    int numSounds,
    void** allocatedAddr,
    const void* endAddr
)
{
    NW_ASSERT_ALIGN32( *allocatedAddr );
    if ( reinterpret_cast<u32>(*allocatedAddr) % MEM_ALIGN_SIZE != 0 )
    {
        return false;
    }

    // SequenceSoundInstanceManager
    {
        size_t requireSize = ut::RoundUp(
                m_SequenceSoundInstanceManager.GetRequiredMemSize(numSounds), MEM_ALIGN_SIZE );

        void* estimateEndAddr = ut::AddOffsetToPtr( *allocatedAddr, requireSize );
        if ( ut::ComparePtr( estimateEndAddr, endAddr ) > 0 )
        {
            return false;
        }

        unsigned long createNum = m_SequenceSoundInstanceManager.Create( *allocatedAddr, requireSize );
        NW_ASSERT( createNum == static_cast<unsigned long>(numSounds) );
        *allocatedAddr = estimateEndAddr;
    }

    // SequenceSoundLoaderManager
    {
        size_t requireSize = ut::RoundUp(
                internal::driver::SequenceSoundLoaderManager::GetRequiredMemSize(numSounds),
                MEM_ALIGN_SIZE );

        void* estimateEndAddr = ut::AddOffsetToPtr( *allocatedAddr, requireSize );
        if ( ut::ComparePtr( estimateEndAddr, endAddr ) > 0 )
        {
            return false;
        }

        u32 createNum = m_SequenceSoundLoaderManager.Create( *allocatedAddr, requireSize );
        NW_ASSERTMSG( static_cast<s32>(createNum) == numSounds,
                "SequenceSoundLoaderManager::Create createNum(%d) but numSounds(%d)",
                createNum, numSounds);
        *allocatedAddr = estimateEndAddr;
    }

    // 各 SequenceSound に SequenceSoundLoaderManager をセット
    internal::SequenceSoundInstanceManager::PriorityList& list =
        m_SequenceSoundInstanceManager.GetFreeList();
    for (internal::SequenceSoundInstanceManager::PriorityList::Iterator itr = list.GetBeginIter();
            itr != list.GetEndIter();
            ++itr)
    {
        itr->SetLoaderManager(m_SequenceSoundLoaderManager);
    }
    internal::driver::SoundThread::GetInstance().RegisterSoundFrameCallback(&m_SequenceSoundLoaderManager);

    return true;
}

/*---------------------------------------------------------------------------*
  Name:         SetupWaveSound

  Description:  ウェーブサウンドのセットアップ

  Arguments:    numSounds - 確保するウェーブサウンドの数
                heap      - サウンドヒープ

  Returns:      成功したかどうか
 *---------------------------------------------------------------------------*/
bool SoundArchivePlayer::SetupWaveSound(
    int numSounds,
    void** allocatedAddr,
    const void* endAddr
)
{
    NW_ASSERT_ALIGN32( *allocatedAddr );
    if ( reinterpret_cast<u32>(*allocatedAddr) % MEM_ALIGN_SIZE != 0 )
    {
        return false;
    }

    // WaveSoundInstanceManager
    {
        size_t requireSize = ut::RoundUp(
                m_WaveSoundInstanceManager.GetRequiredMemSize(numSounds), MEM_ALIGN_SIZE );

        void* estimateEndAddr = ut::AddOffsetToPtr( *allocatedAddr, requireSize );
        if ( ut::ComparePtr( estimateEndAddr, endAddr ) > 0 )
        {
            return false;
        }

        unsigned long createNum = m_WaveSoundInstanceManager.Create( *allocatedAddr, requireSize );
        NW_ASSERT( createNum == static_cast<unsigned long>(numSounds) );
        *allocatedAddr = estimateEndAddr;
    }

    // WaveSoundLoaderManager
    {
        size_t requireSize = ut::RoundUp(
                internal::driver::WaveSoundLoaderManager::GetRequiredMemSize(numSounds),
                MEM_ALIGN_SIZE );

        void* estimateEndAddr = ut::AddOffsetToPtr( *allocatedAddr, requireSize );
        if ( ut::ComparePtr( estimateEndAddr, endAddr ) > 0 )
        {
            return false;
        }

        u32 createNum = m_WaveSoundLoaderManager.Create( *allocatedAddr, requireSize );
        NW_ASSERTMSG( static_cast<s32>(createNum) == numSounds,
                "WaveSoundLoaderManager::Create createNum(%d) but numSounds(%d)",
                createNum, numSounds);
        *allocatedAddr = estimateEndAddr;
    }

    // 各 WaveSound に WaveSoundLoaderManager をセット
    internal::WaveSoundInstanceManager::PriorityList& list =
        m_WaveSoundInstanceManager.GetFreeList();
    for (internal::WaveSoundInstanceManager::PriorityList::Iterator itr = list.GetBeginIter();
            itr != list.GetEndIter();
            ++itr)
    {
        itr->SetLoaderManager(m_WaveSoundLoaderManager);
    }
    internal::driver::SoundThread::GetInstance().RegisterSoundFrameCallback(&m_WaveSoundLoaderManager);

    return true;
}

/*---------------------------------------------------------------------------*
  Name:         SetupStreamSound

  Description:  ストリームサウンドのセットアップ

  Arguments:    numSounds - 確保するストリームサウンドの数
                heap      - サウンドヒープ

  Returns:      成功したかどうか
 *---------------------------------------------------------------------------*/
bool SoundArchivePlayer::SetupStreamSound(
    int numSounds,
    void** allocatedAddr,
    const void* endAddr
)
{
    NW_ASSERT_ALIGN32( *allocatedAddr );
    if ( reinterpret_cast<u32>(*allocatedAddr) % MEM_ALIGN_SIZE != 0 )
    {
        return false;
    }

    // StreamSoundInstanceManager
    {
        size_t requireSize = ut::RoundUp(
            m_StreamSoundInstanceManager.GetRequiredMemSize(numSounds), MEM_ALIGN_SIZE );

        void* estimateEndAddr = ut::AddOffsetToPtr( *allocatedAddr, requireSize );
        if ( ut::ComparePtr( estimateEndAddr, endAddr ) > 0 )
        {
            return false;
        }

        u32 createNum = m_StreamSoundInstanceManager.Create( *allocatedAddr, requireSize );
        NW_ASSERT( static_cast<s32>(createNum) == numSounds );
        *allocatedAddr = estimateEndAddr;
    }

    // StreamSoundLoaderManager
    {
        size_t requireSize = ut::RoundUp(
                internal::driver::StreamSoundLoaderManager::GetRequiredMemSize(numSounds),
                MEM_ALIGN_SIZE );

        void* estimateEndAddr = ut::AddOffsetToPtr( *allocatedAddr, requireSize );
        if ( ut::ComparePtr( estimateEndAddr, endAddr ) > 0 )
        {
            return false;
        }

        u32 createNum = m_StreamSoundLoaderManager.Create( *allocatedAddr, requireSize );
        NW_ASSERTMSG( static_cast<s32>(createNum) == numSounds,
                "StreamSoundLoaderManager::Create createNum(%d) but numSounds(%d)",
                createNum, numSounds);
        *allocatedAddr = estimateEndAddr;
    }

    // 各 StreamSound に StreamSoundLoaderManager をセット
    internal::StreamSoundInstanceManager::PriorityList& list =
        m_StreamSoundInstanceManager.GetFreeList();
    for (internal::StreamSoundInstanceManager::PriorityList::Iterator itr = list.GetBeginIter();
            itr != list.GetEndIter();
            ++itr)
    {
        itr->SetLoaderManager(m_StreamSoundLoaderManager);
    }
    internal::driver::SoundThread::GetInstance().RegisterSoundFrameCallback(&m_StreamSoundLoaderManager);

    return true;
}

/*---------------------------------------------------------------------------*
  Name:         SetupSequenceTrack

  Description:  シーケンスサウンドのセットアップ

  Arguments:    numSounds - 確保するシーケンスサウンドの数
                heap      - サウンドヒープ

  Returns:      成功したかどうか
 *---------------------------------------------------------------------------*/
bool SoundArchivePlayer::SetupSequenceTrack(
    int numTracks,
    void** allocatedAddr,
    const void* endAddr
)
{
    NW_ASSERT_ALIGN32( *allocatedAddr );
    if ( reinterpret_cast<u32>(*allocatedAddr) % MEM_ALIGN_SIZE != 0 )
    {
        return false;
    }

    size_t requireSize = ut::RoundUp(
            numTracks * sizeof( internal::driver::MmlSequenceTrack ), MEM_ALIGN_SIZE );

    void* estimateEndAddr = ut::AddOffsetToPtr( *allocatedAddr, requireSize );
    if ( ut::ComparePtr( estimateEndAddr, endAddr ) > 0 )
    {
        return false;
    }

    unsigned long createNum = m_MmlSequenceTrackAllocator.Create( *allocatedAddr, requireSize );
    NW_ASSERT( createNum == static_cast<unsigned long>(numTracks) );
    *allocatedAddr = estimateEndAddr;

    return true;
}

bool SoundArchivePlayer::SetupUserParamForBasicSound(
    const SoundArchive::SoundArchivePlayerInfo& /* info */,
    void** allocatedAddr,
    const void* endAddr,
    size_t userParamSizePerSound)
{
    NW_ASSERT_ALIGN32( *allocatedAddr );
    if (reinterpret_cast<u32>(*allocatedAddr) % MEM_ALIGN_SIZE != 0)
    {
        return false;
    }

    size_t adjustSize = ut::RoundUp(userParamSizePerSound, 4);

    void* curAddr = *allocatedAddr;
    // WSD 向け
    {
        internal::WaveSoundInstanceManager::PriorityList& list =
            m_WaveSoundInstanceManager.GetFreeList();
        for (internal::WaveSoundInstanceManager::PriorityList::Iterator itr = list.GetBeginIter();
                itr != list.GetEndIter();
                ++itr)
        {
            itr->SetUserParamBuffer(curAddr, adjustSize);

            curAddr = ut::AddOffsetToPtr(curAddr, adjustSize);
        }
    }

    // SEQ 向け
    {
        internal::SequenceSoundInstanceManager::PriorityList& list =
            m_SequenceSoundInstanceManager.GetFreeList();
        for (internal::SequenceSoundInstanceManager::PriorityList::Iterator itr = list.GetBeginIter();
                itr != list.GetEndIter();
                ++itr)
        {
            itr->SetUserParamBuffer(curAddr, adjustSize);

            curAddr = ut::AddOffsetToPtr(curAddr, adjustSize);
        }
    }

    // STRM 向け
    {
        internal::StreamSoundInstanceManager::PriorityList& list =
            m_StreamSoundInstanceManager.GetFreeList();
        for (internal::StreamSoundInstanceManager::PriorityList::Iterator itr = list.GetBeginIter();
                itr != list.GetEndIter();
                ++itr)
        {
            itr->SetUserParamBuffer(curAddr, adjustSize);

            curAddr = ut::AddOffsetToPtr(curAddr, adjustSize);
        }
    }
    m_SoundUserParamSize = adjustSize;

    if ( ut::ComparePtr( curAddr, endAddr ) > 0 )
    {
        return false;
    }

    return true;
}

/*---------------------------------------------------------------------------*
  Name:         SetupStreamBuffer

  Description:  ストリームチャンネルのセットアップ

  Arguments:    heap      - サウンドヒープ

  Returns:      成功したかどうか
 *---------------------------------------------------------------------------*/
bool SoundArchivePlayer::SetupStreamBuffer(
    const SoundArchive* arc,
    void* buffer,
    size_t size
)
{
    NW_ASSERT_ALIGN8( buffer );
    if ( reinterpret_cast<u32>(buffer) % 8 != 0 )
    {
        return false;
    }

    if ( size < GetRequiredStreamBufferSize( arc ) )
    {
        return false;
    }

    int strmChannelCount = 0;

    SoundArchive::SoundArchivePlayerInfo soundArchivePlayerInfo;
    if ( arc->ReadSoundArchivePlayerInfo( &soundArchivePlayerInfo ) )
    {
        strmChannelCount = soundArchivePlayerInfo.streamChannelMax;
    }

    m_StreamBufferPool.Initialize( buffer, size, strmChannelCount );

    return true;
}

/*---------------------------------------------------------------------------*
  Name:         Update

  Description:  プレイヤーのフレーム処理を更新する

  Arguments:    None.

  Returns:      None.
 *---------------------------------------------------------------------------*/
void SoundArchivePlayer::Update()
{
    // 別スレッドのログをまとめて出力
    internal::Util::WarningLogger::GetInstance().SwapBuffer();
    if (m_IsEnableWarningPrint)
    {
        internal::Util::WarningLogger::GetInstance().Print();
    }

    for ( u32 playerIndex = 0;
            playerIndex < m_SoundPlayerCount; ++playerIndex )
    {
        GetSoundPlayer( internal::Util::GetMaskedItemId(
                    playerIndex, internal::ItemType_Player ) ).Update();
    }

    m_SequenceSoundInstanceManager.SortPriorityList();
    m_WaveSoundInstanceManager.SortPriorityList();
    m_StreamSoundInstanceManager.SortPriorityList();

    internal::DriverCommand& cmdmgr = internal::DriverCommand::GetInstance();
    cmdmgr.RecvCommandReply();
    cmdmgr.FlushCommand( false );
}

const SoundArchive& SoundArchivePlayer::GetSoundArchive() const
{
    NW_ASSERTMSG( m_pSoundArchive != NULL, "Setup is not completed." );
    return *m_pSoundArchive;
}

SoundPlayer& SoundArchivePlayer::GetSoundPlayer( SoundArchive::ItemId playerId )
{
    return const_cast<SoundPlayer&>(
        static_cast<const SoundArchivePlayer*>( this )->GetSoundPlayer( playerId ) );
}

SoundPlayer& SoundArchivePlayer::GetSoundPlayer( const char* pStr )
{
    return const_cast<SoundPlayer&>(
        static_cast<const SoundArchivePlayer*>( this )->GetSoundPlayer( pStr ) );
}

const SoundPlayer& SoundArchivePlayer::GetSoundPlayer( SoundArchive::ItemId playerId ) const
{
    u32 itemIndex = internal::Util::GetItemIndex( playerId );
    NW_ASSERT_MAX( itemIndex, m_SoundPlayerCount-1 );
    return m_pSoundPlayers[ itemIndex ];
}

const SoundPlayer& SoundArchivePlayer::GetSoundPlayer( const char* pStr ) const
{
    NW_ASSERT_NOT_NULL( m_pSoundArchive );

    SoundArchive::ItemId playerId = m_pSoundArchive->GetItemId( pStr );
    NW_ASSERT( playerId != SoundArchive::INVALID_ID );
    return GetSoundPlayer( playerId );
}


/*---------------------------------------------------------------------------*
  Name:         AllocSound

  Description:  サウンドを取得する

  Arguments:    manager - サウンドインスタンスマネージャ
                soundId - サウンドID
                priority - プレイヤープライオリティ
                ambientPriority - アンビエントプライオリティ
                ambientArgInfo - アンビエント情報

  Returns:      サウンドへのポインタ。取得できなかったらNULL
 *---------------------------------------------------------------------------*/
template< typename Sound >
Sound* SoundArchivePlayer::AllocSound(
    internal::SoundInstanceManager< Sound >* manager,
    SoundArchive::ItemId soundId,
    int priority,
    int ambientPriority,
    internal::BasicSound::AmbientInfo* ambientArgInfo
)
{
    NW_ASSERT_NOT_NULL( manager );

    // サウンドの確保
    Sound* sound = manager->Alloc( priority, ambientPriority );
    if ( sound == NULL ) return NULL;

    // ID設定
    sound->SetId( soundId );

    // アンビエント設定
    if ( ambientArgInfo != NULL )
    {
        sound->SetAmbientInfo( *ambientArgInfo );
    }

    return sound;
}

const void*
SoundArchivePlayer::detail_GetFileAddress( SoundArchive::FileId fileId ) const
{
    if ( m_pSoundDataManager == NULL ) return NULL;

    return m_pSoundDataManager->detail_GetFileAddress( fileId );
}

SoundStartable::StartResult SoundArchivePlayer::detail_SetupSound(
    SoundHandle* handle,
    u32 soundId,
    bool holdFlag,
    const StartInfo* startInfo
)
{
    return detail_SetupSoundImpl(
        handle,
        soundId,
        NULL,
        NULL,
        holdFlag,
        startInfo
    );
}

/*---------------------------------------------------------------------------*
  Name:         detail_SetupSoundImpl

  Description:  再生の実装関数

  Arguments:    handle  - サウンドハンドル
                soundId - サウンドＩＤ
                ambientArgInfo - 外部パラメータ
                actor - サウンドアクター
                holdFlag - ホールドサウンドフラグ

  Returns:      結果コード
 *---------------------------------------------------------------------------*/
SoundStartable::StartResult SoundArchivePlayer::detail_SetupSoundImpl(
    SoundHandle* handle,
    u32 soundId,
    internal::BasicSound::AmbientInfo* ambientArgInfo,
    SoundActor* actor,
    bool holdFlag,
    const StartInfo* startInfo
)
{
    NW_ASSERT_NOT_NULL( handle );
    NW_ASSERT_NOT_NULL( m_pSoundArchive );

    if ( ! IsAvailable() )
    {
        SoundStartable::StartResult
            result( SoundStartable::StartResult::START_ERR_NOT_AVAILABLE );
        return result;
    }

    // サウンドハンドルを古いサウンドから切断
    if ( handle->IsAttachedSound() ) {
        handle->DetachSound();
    }

    const char* soundLabel = m_pSoundArchive->GetItemLabel(soundId);
    SoundArchive::SoundType soundType = SoundArchive::SOUND_TYPE_INVALID;

    // インゲーム編集のフック処理
    // フック対象でない場合は、処理途中でフック対象にならないように、フックを一時的に無効化する
    // 無効化中に別のスレッドからアクセスされることは考慮していない。
    volatile bool isHookDisabled = false;
    if(IsSoundArchiveFileHooksEnabled() && soundLabel != NULL)
    {
        NW_ASSERT_NOT_NULL(m_pSoundArchive->detail_GetParametersHook());
        NW_ASSERT(m_pSoundArchive->detail_GetParametersHook()->GetIsEnable());

        // 再生要求途中でフック状態が変更されないようにロックする。
        LockSoundArchiveFileHooks();

        if(!m_pSoundArchiveFilesHook->IsTargetItem(soundLabel))
        {
            m_pSoundArchive->detail_GetParametersHook()->SetIsEnable(false);
            m_pSoundArchiveFilesHook->SetIsEnable(false);
            isHookDisabled = true;

            soundType = m_pSoundArchive->GetSoundType( soundId );
        }
        else
        {
            soundType = m_pSoundArchive->detail_GetParametersHook()->GetSoundType( soundLabel );

            // 編集対象かどうかをチェック
            switch (soundType)
            {
            case SoundArchive::SOUND_TYPE_SEQ:
                isHookDisabled = !IsSequenceSoundEdited(soundLabel);
                break;

            case SoundArchive::SOUND_TYPE_STRM:
                isHookDisabled = !IsStreamSoundEdited(soundLabel);
                break;

            case SoundArchive::SOUND_TYPE_WAVE:
                isHookDisabled = !IsWaveSoundEdited(soundLabel);
                break;

            default:
                break;
            }

            if(isHookDisabled)
            {
                m_pSoundArchive->detail_GetParametersHook()->SetIsEnable(false);
                m_pSoundArchiveFilesHook->SetIsEnable(false);
            }
        }
    }
    else
    {
        // HACK : 無効なIDが渡されたときは、ラベルが不正とみなしてます。
        if ( soundId == SoundArchive::INVALID_ID )
        {
            StartResult result( StartResult::START_ERR_INVALID_LABEL_STRING );
            return result;
        }

        soundType = m_pSoundArchive->GetSoundType( soundId );
    }

    // サウンドアーカイブからサウンド情報の取得
    SoundArchive::SoundInfo soundInfo;
    if ( ! m_pSoundArchive->ReadSoundInfo( soundId, &soundInfo ) )
    {
        if(isHookDisabled)
        {
            m_pSoundArchive->detail_GetParametersHook()->SetIsEnable(true);
            m_pSoundArchiveFilesHook->SetIsEnable(true);
        }
        UnlockSoundArchiveFileHooks();
        SoundStartable::StartResult
            result( SoundStartable::StartResult::START_ERR_INVALID_SOUNDID );
        return result;
    }

    // StartInfoの処理
    SoundStartable::StartInfo::StartOffsetType startOffsetType =
        SoundStartable::StartInfo::START_OFFSET_TYPE_MILLISEC;
    int startOffset = 0;
    int delayTime = 0;
    int playerPriority = soundInfo.playerPriority;
    SoundArchive::ItemId playerId = soundInfo.playerId;
    int actorPlayerId = soundInfo.actorPlayerId;
    const SoundStartable::StartInfo::SeqSoundInfo* seqInfo = NULL;
    const SoundStartable::StartInfo::StreamSoundInfo* strmInfo = NULL;
    const SoundArchive::StreamSoundInfo* strmMetaInfo = NULL;
    const SoundStartable::StartInfo::WaveSoundInfo* wsdInfo = NULL;
    VoiceRendererType rendererType = VOICE_RENDERER_SDK;
    if ( startInfo != NULL )
    {
        if ( startInfo->enableFlag & StartInfo::ENABLE_START_OFFSET )
        {
            startOffsetType = startInfo->startOffsetType;
            startOffset = startInfo->startOffset;
        }
        if ( startInfo->enableFlag & StartInfo::ENABLE_PLAYER_PRIORITY )
        {
            playerPriority = startInfo->playerPriority;
        }
        if ( startInfo->enableFlag & StartInfo::ENABLE_PLAYER_ID )
        {
            playerId = startInfo->playerId;
        }
        if ( startInfo->enableFlag & StartInfo::ENABLE_ACTOR_PLAYER_ID )
        {
            actorPlayerId = startInfo->actorPlayerId;
        }
        if ( startInfo->enableFlag & StartInfo::ENABLE_SEQ_SOUND_INFO )
        {
            seqInfo = &startInfo->seqSoundInfo;
        }
        if ( startInfo->enableFlag & StartInfo::ENABLE_STRM_SOUND_INFO )
        {
            strmInfo = &startInfo->streamSoundInfo;
        }
        if ( startInfo->enableFlag & StartInfo::ENABLE_STRM_SOUND_META_INFO )
        {
            strmMetaInfo = &startInfo->streamSoundMetaInfo;
        }
        if ( startInfo->enableFlag & StartInfo::ENABLE_WAVE_SOUND_INFO )
        {
            wsdInfo = &startInfo->waveSoundInfo;
        }
        if ( startInfo->enableFlag & StartInfo::ENABLE_VOICE_RENDERER_TYPE )
        {
            rendererType = static_cast<VoiceRendererType>(startInfo->voiceRendererType);

            if ( rendererType == VOICE_RENDERER_NW && !SoundSystem::detail_IsInitializedNwRenderer() )
            {
                if(isHookDisabled)
                {
                    m_pSoundArchive->detail_GetParametersHook()->SetIsEnable(true);
                    m_pSoundArchiveFilesHook->SetIsEnable(true);
                }
                UnlockSoundArchiveFileHooks();
                return StartResult(StartResult::START_ERR_INVALID_RENDERER_TYPE );
            }
        }
        if ( startInfo->enableFlag & StartInfo::ENABLE_DELAY_TIME )
        {
            delayTime = startInfo->delayTime;
        }
    }

    // プライオリティの計算
    int priority = playerPriority;
    if ( holdFlag ) --priority; // HoldSoundで再生するときは同プライオリティで先着優先

    int ambientPriority = 0;
    if ( ambientArgInfo != NULL )
    {
        ambientPriority = internal::BasicSound::GetAmbientPriority( *ambientArgInfo, soundId );
    }
    int allocPriority = priority + ambientPriority;
    allocPriority = ut::Clamp( allocPriority, internal::BasicSound::PRIORITY_MIN, internal::BasicSound::PRIORITY_MAX );

    // ExternalSoundPlayer
    internal::ExternalSoundPlayer* extPlayer = NULL;
    if ( actor != NULL ) {
        extPlayer = actor->detail_GetActorPlayer( actorPlayerId );
        if ( extPlayer == NULL )
        {
            NW_WARNING(
                false,
                "actorPlayerId(%d) is out of range. (0-%d)",
                actorPlayerId,
                SoundActor::ACTOR_PLAYER_COUNT-1
            );
            if(isHookDisabled)
            {
                m_pSoundArchive->detail_GetParametersHook()->SetIsEnable(true);
                m_pSoundArchiveFilesHook->SetIsEnable(true);
            }
            UnlockSoundArchiveFileHooks();
            return StartResult(StartResult::START_ERR_INVALID_PARAMETER);
        }
    }

    // プレイヤーの空きをチェック
    if ( playerId == SoundArchive::INVALID_ID )
    {
        if(isHookDisabled)
        {
            m_pSoundArchive->detail_GetParametersHook()->SetIsEnable(true);
            m_pSoundArchiveFilesHook->SetIsEnable(true);
        }
        UnlockSoundArchiveFileHooks();
        return StartResult(StartResult::START_ERR_INVALID_PARAMETER);
    }
    SoundPlayer& player = GetSoundPlayer( playerId );
    if ( ! player.detail_CanPlaySound( allocPriority ) ) {
        if(isHookDisabled)
        {
            m_pSoundArchive->detail_GetParametersHook()->SetIsEnable(true);
            m_pSoundArchiveFilesHook->SetIsEnable(true);
        }
        UnlockSoundArchiveFileHooks();
        return StartResult(StartResult::START_ERR_LOW_PRIORITY);
    }
    if ( ( extPlayer != NULL ) && ( ! extPlayer->detail_CanPlaySound( allocPriority ) ) ) {
        if(isHookDisabled)
        {
            m_pSoundArchive->detail_GetParametersHook()->SetIsEnable(true);
            m_pSoundArchiveFilesHook->SetIsEnable(true);
        }
        UnlockSoundArchiveFileHooks();
        return StartResult(StartResult::START_ERR_LOW_PRIORITY);
    }

    // サウンドのアロケート
    internal::BasicSound* sound = NULL;
    internal::SequenceSound* seqSound = NULL;
    internal::StreamSound* strmSound = NULL;
    internal::WaveSound* waveSound = NULL;

    switch ( soundType )
    {
    case SoundArchive::SOUND_TYPE_SEQ:
        seqSound = AllocSound<internal::SequenceSound>(
            &m_SequenceSoundInstanceManager,
            soundId,
            priority,
            ambientPriority,
            ambientArgInfo
        );
        if ( seqSound == NULL )
        {
            NW_WARNING(
                ! internal::Debug_GetWarningFlag( DEBUG_WARNING_NOT_ENOUGH_SEQSOUND ),
                "Failed to start sound (id:0x%08x) for not enough SequenceSound instance.",
                soundId
            );
            if(isHookDisabled)
            {
                m_pSoundArchive->detail_GetParametersHook()->SetIsEnable(true);
                m_pSoundArchiveFilesHook->SetIsEnable(true);
            }
            UnlockSoundArchiveFileHooks();
            SoundStartable::StartResult
                result( SoundStartable::StartResult::START_ERR_NOT_ENOUGH_INSTANCE );
            return result;
        }
        sound = seqSound;
        break;

    case SoundArchive::SOUND_TYPE_STRM:
        strmSound = AllocSound<internal::StreamSound>(
            &m_StreamSoundInstanceManager,
            soundId,
            priority,
            ambientPriority,
            ambientArgInfo
        );
        if ( strmSound == NULL )
        {
            NW_WARNING(
                ! internal::Debug_GetWarningFlag( DEBUG_WARNING_NOT_ENOUGH_STRMSOUND ),
                "Failed to start sound (id:0x%08x) for not enough StreamSound instance.",
                soundId
            );
            if(isHookDisabled)
            {
                m_pSoundArchive->detail_GetParametersHook()->SetIsEnable(true);
                m_pSoundArchiveFilesHook->SetIsEnable(true);
            }
            UnlockSoundArchiveFileHooks();
            SoundStartable::StartResult
                result( SoundStartable::StartResult::START_ERR_NOT_ENOUGH_INSTANCE );
            return result;
        }
        sound = strmSound;
        break;

    case SoundArchive::SOUND_TYPE_WAVE:
        waveSound = AllocSound<internal::WaveSound>(
            &m_WaveSoundInstanceManager,
            soundId,
            priority,
            ambientPriority,
            ambientArgInfo
        );
        if ( waveSound == NULL )
        {
            NW_WARNING(
                ! internal::Debug_GetWarningFlag( DEBUG_WARNING_NOT_ENOUGH_WAVESOUND ),
                "Failed to start sound (id:0x%08x) for not enough WaveSound instance.",
                soundId
            );
            if(isHookDisabled)
            {
                m_pSoundArchive->detail_GetParametersHook()->SetIsEnable(true);
                m_pSoundArchiveFilesHook->SetIsEnable(true);
            }
            UnlockSoundArchiveFileHooks();
            SoundStartable::StartResult
                result( SoundStartable::StartResult::START_ERR_NOT_ENOUGH_INSTANCE );
            return result;
        }
        sound = waveSound;
        break;

    default:
        {
            if(isHookDisabled)
            {
                m_pSoundArchive->detail_GetParametersHook()->SetIsEnable(true);
                m_pSoundArchiveFilesHook->SetIsEnable(true);
            }
            UnlockSoundArchiveFileHooks();
            SoundStartable::StartResult
                result( SoundStartable::StartResult::START_ERR_INVALID_SOUNDID );
            return result;
        }
    }

    // サウンドプレイヤーに登録
    if ( ! player.detail_AppendSound( sound ) )
    {
        sound->Finalize();
        if(isHookDisabled)
        {
            m_pSoundArchive->detail_GetParametersHook()->SetIsEnable(true);
            m_pSoundArchiveFilesHook->SetIsEnable(true);
        }
        UnlockSoundArchiveFileHooks();
        SoundStartable::StartResult
            result( SoundStartable::StartResult::START_ERR_UNKNOWN );
        return result;
    }

    // StartInfoの処理（BasicSoundのAlloc後に行う必要があるもの）
    if (startInfo != NULL)
    {
        if ( startInfo->enableFlag & StartInfo::ENABLE_SOUND_STOP_CALLBACK )
        {
            sound->SetSoundStopCallback( startInfo->soundStopCallback );
        }
    }

    sound->ResetOutputLine();
    sound->SetVoiceRendererType( rendererType );
    sound->SetFrontBypass( soundInfo.isFrontBypass );

    // サウンドのプリペア
    switch ( soundType )
    {
    case SoundArchive::SOUND_TYPE_SEQ:
    {
        NW_ASSERT_NOT_NULL( seqSound );

        // シーケンスサウンド情報の取得
        SoundArchive::SequenceSoundInfo info;

        if ( ! m_pSoundArchive->ReadSequenceSoundInfo( soundId, &info ) ) {
            seqSound->Finalize();
            if(isHookDisabled)
            {
                m_pSoundArchive->detail_GetParametersHook()->SetIsEnable(true);
                m_pSoundArchiveFilesHook->SetIsEnable(true);
            }
            UnlockSoundArchiveFileHooks();
            SoundStartable::StartResult
                result( SoundStartable::StartResult::START_ERR_INVALID_SOUNDID );
            return result;
        }

        // バンクの上書き (必要であれば)
        if ( seqInfo != NULL )
        {
            for ( int i = 0; i < SoundArchive::SEQ_BANK_MAX; i++ )
            {
                SoundArchive::ItemId bankId = seqInfo->bankIds[ i ];
                if ( bankId != SoundArchive::INVALID_ID )
                {
                    info.bankIds[ i ] = seqInfo->bankIds[ i ];
                }
            }
        }

        // シーケンスサウンドの準備
        StartResult result = PrepareSequenceSoundImpl(
            seqSound,
            &soundInfo,
            &info,
            startOffsetType,
            startOffset,
            delayTime,
            seqInfo
        );
        if ( ! result.IsSuccess() )
        {
            seqSound->Finalize();
            if(isHookDisabled)
            {
                m_pSoundArchive->detail_GetParametersHook()->SetIsEnable(true);
                m_pSoundArchiveFilesHook->SetIsEnable(true);
            }
            UnlockSoundArchiveFileHooks();
            return result;
        }
        break;
    }

    case SoundArchive::SOUND_TYPE_STRM:
    {
        NW_ASSERT_NOT_NULL( strmSound );

        // ストリームサウンド情報の取得
        SoundArchive::StreamSoundInfo info;
        if ( strmMetaInfo != NULL )
        {
            info = *strmMetaInfo;
        }
        else
        {
            if ( ! m_pSoundArchive->ReadStreamSoundInfo( soundId, &info ) ) {
                strmSound->Finalize();
                if(isHookDisabled)
                {
                    m_pSoundArchive->detail_GetParametersHook()->SetIsEnable(true);
                    m_pSoundArchiveFilesHook->SetIsEnable(true);
                }
                UnlockSoundArchiveFileHooks();
                SoundStartable::StartResult
                    result( SoundStartable::StartResult::START_ERR_INVALID_SOUNDID );
                return result;
            }
        }

        const SoundArchive::StreamSoundInfo2* pInfo2 = NULL;
        SoundArchive::StreamSoundInfo2 info2;
        if (info.streamFileType == SoundArchive::STREAM_FILE_TYPE_ADTS)
        {
            if ( ! m_pSoundArchive->detail_ReadStreamSoundInfo2( soundId, &info2 ) )
            {
                strmSound->Finalize();
                if(isHookDisabled)
                {
                    m_pSoundArchive->detail_GetParametersHook()->SetIsEnable(true);
                    m_pSoundArchiveFilesHook->SetIsEnable(true);
                }
                UnlockSoundArchiveFileHooks();
                SoundStartable::StartResult
                    result( SoundStartable::StartResult::START_ERR_INVALID_SOUNDID );
                return result;
            }
            pInfo2 = &info2;
        }

        // ストリームサウンドの準備
        StartResult result = PrepareStreamSoundImpl(
            strmSound,
            &soundInfo,
            &info,
            pInfo2,
            startOffsetType,
            startOffset,
            delayTime,
            strmInfo
        );
        if ( ! result.IsSuccess() )
        {
            strmSound->Finalize();
            if(isHookDisabled)
            {
                m_pSoundArchive->detail_GetParametersHook()->SetIsEnable(true);
                m_pSoundArchiveFilesHook->SetIsEnable(true);
            }
            UnlockSoundArchiveFileHooks();
            return result;
        }
        break;
    }

    case SoundArchive::SOUND_TYPE_WAVE:
    {
        NW_ASSERT_NOT_NULL( waveSound );

        // ウェーブサウンド情報の取得
        SoundArchive::WaveSoundInfo info;

        if ( ! m_pSoundArchive->detail_ReadWaveSoundInfo( soundId, &info ) )
        {
            waveSound->Finalize();
            if(isHookDisabled)
            {
                m_pSoundArchive->detail_GetParametersHook()->SetIsEnable(true);
                m_pSoundArchiveFilesHook->SetIsEnable(true);
            }
            UnlockSoundArchiveFileHooks();
            SoundStartable::StartResult
                result( SoundStartable::StartResult::START_ERR_INVALID_SOUNDID );
            return result;
        }

        // ウェーブサウンドの準備
        StartResult result = PrepareWaveSoundImpl(
            waveSound,
            &soundInfo,
            &info,
            startOffsetType,
            startOffset,
            delayTime,
            wsdInfo
        );
        if ( ! result.IsSuccess() )
        {
            waveSound->Finalize();
            if(isHookDisabled)
            {
                m_pSoundArchive->detail_GetParametersHook()->SetIsEnable(true);
                m_pSoundArchiveFilesHook->SetIsEnable(true);
            }
            UnlockSoundArchiveFileHooks();
            return result;
        }
        break;
    }

    default:
        NW_ASSERT( false );
        sound->Finalize();
        {
            if(isHookDisabled)
            {
                m_pSoundArchive->detail_GetParametersHook()->SetIsEnable(true);
                m_pSoundArchiveFilesHook->SetIsEnable(true);
            }
            UnlockSoundArchiveFileHooks();
            return StartResult(StartResult::START_ERR_INVALID_SOUNDID);
        }
    }

    if(isHookDisabled)
    {
        m_pSoundArchive->detail_GetParametersHook()->SetIsEnable(true);
        m_pSoundArchiveFilesHook->SetIsEnable(true);
    }
    UnlockSoundArchiveFileHooks();

    SetCommonSoundParam( sound, &soundInfo );

    // 外部プレイヤーに登録
    if ( extPlayer != NULL )
    {
        if ( ! extPlayer->AppendSound( sound ) )
        {
            sound->Finalize();
            return StartResult(StartResult::START_ERR_UNKNOWN);
        }
    }

    // アクターに登録
    if ( actor != NULL )
    {
        sound->AttachSoundActor( actor );
    }

    // プライオリティの再設定
    if ( holdFlag )
    {
        sound->SetPlayerPriority( playerPriority );
    }

    // サウンドハンドルを確保したサウンドと接続
    handle->detail_AttachSound( sound );

    SoundStartable::StartResult
        result( SoundStartable::StartResult::START_SUCCESS );
    return result;
}

/*---------------------------------------------------------------------------*
  Name:         PrepareSequenceSoundImpl

  Description:  シーケンスを再生

  Arguments:    playerId   - プレイヤー番号
                playerPrio - プレイヤープライオリティ
                info       - シーケンス情報構造体のポインタ
                seqId      - シーケンス番号

  Returns:      成功したかどうか
 *---------------------------------------------------------------------------*/
SoundStartable::StartResult SoundArchivePlayer::PrepareSequenceSoundImpl(
    internal::SequenceSound* sound,
    const SoundArchive::SoundInfo* commonInfo,
    const SoundArchive::SequenceSoundInfo* info,
    SoundStartable::StartInfo::StartOffsetType startOffsetType,
    int startOffset,
    int delayTime,
    const StartInfo::SeqSoundInfo* externalSeqInfo
)
{
    NW_ASSERT_NOT_NULL( info );
    NW_ASSERT_NOT_NULL( m_pSoundArchive );

    const internal::SequenceSoundFile* seqFile = NULL;

    // シーケンスファイルアクセスのフック
    if(m_pSoundArchiveFilesHook != NULL)
    {
        // フックされていないときに余計なコストがかからないように、if 中でラベルを取得する
        const char* soundLabel = m_pSoundArchive->GetItemLabel(sound->GetId());

        if(soundLabel != NULL)
        {
            seqFile = reinterpret_cast<const internal::SequenceSoundFile*>(
                m_pSoundArchiveFilesHook->GetFileAddress(
                    soundLabel,
                    internal::SoundArchiveFilesHook::ITEM_TYPE_SEQUENCE_SOUND,
                    internal::SoundArchiveFilesHook::FILE_TYPE_SEQUENCE_BINARY));
        }
        else
        {
            seqFile = NULL;
        }
    }

    u32 seqOffset = info->startOffset;
    u32 allocTrackBitFlag = info->allocateTrackFlags;

    if(seqFile == NULL)
    {
        seqFile = reinterpret_cast<const internal::SequenceSoundFile*>(
            m_pSoundDataManager->detail_GetFileAddress( commonInfo->fileId ) );

        // シーケンスデータの確定 (外部ファイルかどうか)
        if ( externalSeqInfo != NULL )
        {
            const void* extSeqFile = externalSeqInfo->seqDataAddress;
            if ( extSeqFile != NULL )
            {
                seqFile = reinterpret_cast<const internal::SequenceSoundFile*>( extSeqFile );
            }

            internal::SequenceSoundFileReader seqFileReader( seqFile );
            const char* startLabel = externalSeqInfo->startLocationLabel;
            if ( startLabel != NULL )
            {
                if ( ! seqFileReader.GetOffsetByLabel( startLabel, &seqOffset ) )
                {
                    SoundStartable::StartResult result(
                        SoundStartable::StartResult::START_ERR_INVALID_SEQ_START_LOCATION_LABEL );
                    return result;
                }
                seqOffset = internal::driver::MmlParser::ParseAllocTrack(
                    seqFileReader.GetSequenceData(),
                    seqOffset,
                    &allocTrackBitFlag );
            }
        }
    }

    bool canUsePlayerHeap = sound->GetSoundPlayer()->detail_CanUsePlayerHeap();
    bool isRegisterDataLoadTask = false;
    internal::LoadItemInfo loadSeq;
    internal::LoadItemInfo loadBanks[SoundArchive::SEQ_BANK_MAX];
    internal::LoadItemInfo loadWarcs[SoundArchive::SEQ_BANK_MAX]; // 現状は、バンクの数と同数
    bool isLoadIndividuals[SoundArchive::SEQ_BANK_MAX] = {false};

    // シーケンスデータがない場合は、プレイヤーヒープへのロードフラグを立てる
    loadSeq.itemId = sound->GetId();
    loadSeq.address = seqFile;
    if ( seqFile == NULL )
    {
        if (canUsePlayerHeap == false)
        {
            SoundStartable::StartResult result(
                    SoundStartable::StartResult::START_ERR_NOT_SEQ_LOADED );
            return result;
        }
        isRegisterDataLoadTask = true;
    }

    // バンク、波形アーカイブファイルアクセスのフック
    if(m_pSoundArchiveFilesHook != NULL)
    {
        // フックされていないときに余計なコストがかからないように、if 中でラベルを取得する
        const char* soundLabel = m_pSoundArchive->GetItemLabel(sound->GetId());

        if(soundLabel != NULL)
        {
            for(int i = 0; i < SoundArchive::SEQ_BANK_MAX; ++i)
            {
                // SEQ はフックを無効化していても、バンクのためにフックを一時的に有効化する
                bool isSeqHookDisabled = m_pSoundArchiveFilesHook->GetIsEnable();

                if(!isSeqHookDisabled)
                {
                    NW_ASSERT_NOT_NULL(m_pSoundArchive->detail_GetParametersHook());

                    m_pSoundArchive->detail_GetParametersHook()->SetIsEnable(true);
                    m_pSoundArchiveFilesHook->SetIsEnable(true);
                }

                const void* bankFile = m_pSoundArchiveFilesHook->GetFileAddress(
                    soundLabel,
                    internal::SoundArchiveFilesHook::ITEM_TYPE_SEQUENCE_SOUND,
                    internal::SoundArchiveFilesHook::FILE_TYPE_BANK_BINARY, i);

                const void* warcFile = m_pSoundArchiveFilesHook->GetFileAddress(
                    soundLabel,
                    internal::SoundArchiveFilesHook::ITEM_TYPE_SEQUENCE_SOUND,
                    internal::SoundArchiveFilesHook::FILE_TYPE_WAVE_ARCHIVE_BINARY, i);

                // 一時的に有効化したフックを無効化する
                if(!isSeqHookDisabled)
                {
                    m_pSoundArchive->detail_GetParametersHook()->SetIsEnable(false);
                    m_pSoundArchiveFilesHook->SetIsEnable(false);
                }

                if(bankFile != NULL && warcFile != NULL)
                {
                    loadBanks[i].address = bankFile;
                    loadWarcs[i].address = warcFile;
                    isLoadIndividuals[i] = false;
                }
                else
                {
                    loadBanks[i].address = NULL;
                    loadWarcs[i].address = NULL;
                }
            }
        }
    }

    // バンクデータがあるかチェック
    if ( externalSeqInfo != NULL )
    {
        for ( int i = 0; i < SoundArchive::SEQ_BANK_MAX; i++ )
        {
            loadBanks[i].itemId = externalSeqInfo->bankIds[ i ];
        }
    }
    else
    {
        for ( int i = 0; i < SoundArchive::SEQ_BANK_MAX; i++ )
        {
            loadBanks[i].itemId = info->bankIds[ i ];
        }
    }

    for ( int i = 0; i < SoundArchive::SEQ_BANK_MAX; i++ )
    {
        // フックによりロード済みの場合はスキップ
        if(m_pSoundArchiveFilesHook != NULL && loadBanks[i].address != NULL && loadWarcs[i].address != NULL)
        {
            continue;
        }

        SoundArchive::BankInfo bankInfo;
        bool isReadBankInfo = GetSoundArchive().ReadBankInfo( loadBanks[i].itemId, &bankInfo );

        if ( isReadBankInfo )
        {
            const void* bankFile =
                m_pSoundDataManager->detail_GetFileAddress( bankInfo.fileId );
            loadBanks[i].address = bankFile;

            // バンクデータがない場合は、プレイヤーヒープへのロードフラグを立てる
            if ( bankFile == NULL )
            {
                if (canUsePlayerHeap == false)
                {
                    // バンクデータは無いが再生処理を進める
                    continue;
                }
                else
                {
                    isRegisterDataLoadTask = true;
                    break;
                }
            }
            // バンクデータがある場合は、波形アーカイブデータがあるかチェック
            else
            {
                internal::Util::WaveArchiveLoadStatus status =
                    internal::Util::GetWaveArchiveOfBank(
                        loadWarcs[i],
                        isLoadIndividuals[i],
                        bankFile,
                        *m_pSoundArchive,
                        *m_pSoundDataManager );
                switch ( status )
                {
                case internal::Util::WARC_LOAD_OK:
                case internal::Util::WARC_LOAD_NONEED:
                    // なにもしない (プリペア処理に移る)
                    break;
                case internal::Util::WARC_LOAD_PARTLY:
                    if (canUsePlayerHeap)
                    {
                        // プレイヤーヒープがある場合は、そちらへのロードを試みる
                        // (無い場合は、そのままプリペア処理に移る)
                        isRegisterDataLoadTask = true;
                    }
                    break;
                case internal::Util::WARC_LOAD_NOT_YET:
                    if (canUsePlayerHeap)
                    {
                        // プレイヤーヒープへのロード
                        isRegisterDataLoadTask = true;
                    }
                    else
                    {
                        // 再生失敗
                        SoundStartable::StartResult result(
                                SoundStartable::StartResult::START_ERR_NOT_WARC_LOADED );
                        return result;
                    }
                    break;
                case internal::Util::WARC_LOAD_ERROR:
                    {
                        // 再生失敗
                        SoundStartable::StartResult result(
                            SoundStartable::StartResult::START_ERR_INVALID_BANK_DATA );
                        return result;
                    }
                default:
                    NW_ASSERT( false ); // ここに来ることは無いはず
                    {
                        // 再生失敗
                        SoundStartable::StartResult result(
                            SoundStartable::StartResult::START_ERR_UNKNOWN );
                        return result;
                    }
                }
            }
        }
    }

    // コマンド化のため、トラックが足りないときのケアは無し
    sound->Setup(
            m_pSequenceTrackAllocator,
            allocTrackBitFlag,
            &m_SequenceCallback,
            info->channelPriority,
            info->isReleasePriorityFix,
            m_SequenceUserprocCallback, m_pSequenceUserprocCallbackArg );

    internal::driver::SequenceSoundPlayer::StartOffsetType seqOffsetType;
    switch ( startOffsetType )
    {
    case SoundStartable::StartInfo::START_OFFSET_TYPE_MILLISEC:
        seqOffsetType = internal::driver::SequenceSoundPlayer::START_OFFSET_TYPE_MILLISEC;
        break;
    case SoundStartable::StartInfo::START_OFFSET_TYPE_TICK:
        seqOffsetType = internal::driver::SequenceSoundPlayer::START_OFFSET_TYPE_TICK;
        break;
    case SoundStartable::StartInfo::START_OFFSET_TYPE_SAMPLE:
        seqOffsetType = internal::driver::SequenceSoundPlayer::START_OFFSET_TYPE_TICK;
        startOffset = 0;
        break;
    default:
        seqOffsetType = internal::driver::SequenceSoundPlayer::START_OFFSET_TYPE_TICK;
        startOffset = 0;
        break;
    }

    internal::driver::SequenceSoundPlayer::StartInfo startInfo =
    {
        static_cast<s32>( seqOffset ),
        seqOffsetType,
        startOffset,
        delayTime
    };

    if ( isRegisterDataLoadTask )
    {
        // 必要なデータが欠けている場合は、プレイヤーヒープへのロードタスクを投げる
        // (プリペアやバンクの設定は、タスク完了後のコールバック内で処理される)
        internal::driver::SequenceSoundLoader::LoadInfo loadInfo(
            m_pSoundArchive,
            m_pSoundDataManager,
            &loadSeq,
            loadBanks,
            sound->GetSoundPlayer()
        );
        sound->RegisterDataLoadTask( loadInfo, startInfo );
    }
    else
    {
        internal::SequenceSound::Resource res;
        {
            res.seq = seqFile;
            for ( int i = 0; i < SoundArchive::SEQ_BANK_MAX; i++ )
            {
                res.banks[i] = loadBanks[i].address;
                res.warcs[i] = loadWarcs[i].address;
                res.warcIsIndividuals[i] = isLoadIndividuals[i];
            }
        };

        sound->Prepare( res, startInfo );
    }

    SoundStartable::StartResult startResult(
            SoundStartable::StartResult::START_SUCCESS );
    return startResult;
}

/*---------------------------------------------------------------------------*
  Name:         PrepareStreamSoundImpl

  Description:  ストリームを再生

  Arguments:    playerId   - プレイヤー番号
                playerPrio - プレイヤープライオリティ
                info       - ストリームの情報
                strmId     - ストリーム番号

  Returns:      成功したかどうか
 *---------------------------------------------------------------------------*/
SoundStartable::StartResult SoundArchivePlayer::PrepareStreamSoundImpl(
    internal::StreamSound* sound,
    const SoundArchive::SoundInfo* commonInfo,
    const SoundArchive::StreamSoundInfo* info,
    const SoundArchive::StreamSoundInfo2* info2,
    SoundStartable::StartInfo::StartOffsetType startOffsetType,
    int startOffset,
    int delayTime,
    const StartInfo::StreamSoundInfo* externalStrmInfo
)
{
    NW_ASSERT_NOT_NULL( m_pSoundArchive );

    // プレイヤーのセットアップ
    {
        internal::driver::StreamSoundPlayer::SetupArg arg;

        // 再生失敗時のケア無し
        switch(info->streamFileType){
        case SoundArchive::STREAM_FILE_TYPE_NW_STREAM_BINARY:
            arg.fileType = internal::STREAM_FILE_BFSTM;
            break;
        case SoundArchive::STREAM_FILE_TYPE_ADTS:
            arg.fileType = internal::StreamFileType(1);
            break;
        default:
            SoundStartable::StartResult result(
                SoundStartable::StartResult::START_ERR_UNKNOWN );
            return result;
        }

        for (int i = 0; i < internal::STRM_TRACK_NUM; i++)
        {
            arg.trackInfos.track[i].volume = info->trackInfo[i].volume;
            arg.trackInfos.track[i].pan = info->trackInfo[i].pan;
            arg.trackInfos.track[i].span = info->trackInfo[i].span;
            arg.trackInfos.track[i].flags = info->trackInfo[i].flags;
            arg.trackInfos.track[i].mainSend = info->trackInfo[i].mainSend;
            for (int j = 0; j < AUX_BUS_NUM; j++)
            {
                arg.trackInfos.track[i].fxSend[j] = info->trackInfo[i].fxSend[j];
            }
            arg.trackInfos.track[i].lpfFreq = info->trackInfo[i].lpfFreq;
            arg.trackInfos.track[i].biquadType = info->trackInfo[i].biquadType;
            arg.trackInfos.track[i].biquadValue = info->trackInfo[i].biquadValue;
            arg.trackInfos.track[i].channelCount = info->trackInfo[i].channelCount;
            for (int j = 0; j < WAVE_CHANNEL_MAX; j++)
            {
                arg.trackInfos.track[i].channelIndex[j] =
                    info->trackInfo[i].globalChannelIndex[j];
            }
        }

        arg.pBufferPool = &m_StreamBufferPool;
        arg.allocChannelCount = info->allocateChannelCount;
        arg.allocTrackFlag = info->allocateTrackFlags;

        arg.loopFlag = false;
        arg.loopStart = 0;
        arg.loopEnd = 0xffffffff;
        if (info2 != NULL)
        {
            arg.loopFlag = info2->isLoop;
            if (arg.loopFlag)
            {
                arg.loopStart = info2->loopStartFrame;
                arg.loopEnd = info2->loopEndFrame;
            }
        }

        arg.pitch = info->pitch;
        arg.mainSend = info->mainSend;
        for (int i = 0; i < AUX_BUS_NUM; i++)
        {
            arg.fxSend[i] = info->fxSend[i];
        }

    #if defined(NW_PLATFORM_WIN32) || defined(NW_USE_NINTENDO_SDK) || defined(NW_PLATFORM_ANDROID) || defined(NW_PLATFORM_IOS)
        if (info->streamFileType == SoundArchive::STREAM_FILE_TYPE_ADTS){
            arg.fileType = internal::STREAM_FILE_BFSTM;
        }
    #endif

        sound->Setup(arg);
    }

    // プリペア
    {
        // オフセット
        internal::driver::StreamSoundPlayer::StartOffsetType strmStartOffsetType;
        switch ( startOffsetType )
        {
        case SoundStartable::StartInfo::START_OFFSET_TYPE_MILLISEC:
            strmStartOffsetType = internal::driver::StreamSoundPlayer::START_OFFSET_TYPE_MILLISEC;
            break;
        case SoundStartable::StartInfo::START_OFFSET_TYPE_TICK:
            strmStartOffsetType = internal::driver::StreamSoundPlayer::START_OFFSET_TYPE_SAMPLE;
            startOffset = 0;
            break;
        case SoundStartable::StartInfo::START_OFFSET_TYPE_SAMPLE:
            strmStartOffsetType = internal::driver::StreamSoundPlayer::START_OFFSET_TYPE_SAMPLE;
            break;
        default:
            strmStartOffsetType = internal::driver::StreamSoundPlayer::START_OFFSET_TYPE_SAMPLE;
            startOffset = 0;
            break;
        }

        internal::driver::StreamSoundPlayer::PrepareBaseArg arg;
        {
            // ストリームファイルのフックパラメータの保持
            // TODO : ★フックへのアクセスを遅延させると、Lock() が正しく動作しないので、必要な情報はここでかき集めたいところ・・・。
            if (m_pSoundArchiveFilesHook != NULL)
            {
                arg.fileStreamHookParam.pSoundArchiveFilesHook = m_pSoundArchiveFilesHook;
                arg.fileStreamHookParam.itemLabel = m_pSoundArchive->GetItemLabel(sound->GetId());
            }

            arg.startOffsetType = strmStartOffsetType;
            arg.offset = startOffset;
            arg.delayTime = delayTime;
            arg.regionCallback = externalStrmInfo ? externalStrmInfo->regionCallback : NULL;
            arg.regionCallbackArg = externalStrmInfo ? externalStrmInfo->regionCallbackArg : NULL;
        #if defined(NW_PLATFORM_CAFE)
            arg.fsClient = m_pSoundArchive->detail_GetFsClient();
            arg.fsCommandBufferPool = m_pSoundArchive->detail_GetFsCommandBlockPool();
            arg.fsPriority = m_pSoundArchive->GetFsStreamPriority();
        #endif

            bool externalPathEnable = false;
            if (externalStrmInfo != NULL)
            {
                if (externalStrmInfo->externalPath != NULL)
                {
                    externalPathEnable = true;
                }
            }

            if (externalPathEnable)
            {
                ut::strcpy(arg.filePath, internal::FILE_PATH_MAX,
                        externalStrmInfo->externalPath);
            }
            else if(m_pSoundArchiveFilesHook == NULL || !m_pSoundArchiveFilesHook->GetIsEnable())
            {
                // ファイル情報
                SoundArchive::FileInfo fileInfo;
                if (m_pSoundArchive->detail_ReadFileInfo(commonInfo->fileId, &fileInfo) == false)
                {
                    SoundStartable::StartResult result(
                        SoundStartable::StartResult::START_ERR_INVALID_STRM_FILE_ID );
                    return result;
                }

                if (fileInfo.externalFilePath == NULL)
                {
                    SoundStartable::StartResult result(
                        SoundStartable::StartResult::START_ERR_INVALID_STRM_FILE_PATH );
                    return result;
                }

                char buf[internal::FILE_PATH_MAX];
                m_pSoundArchive->detail_GetExternalFileFullPath(
                        fileInfo.externalFilePath, buf, internal::FILE_PATH_MAX);
                ut::strcpy(arg.filePath, internal::FILE_PATH_MAX, buf);
            }

        #if defined(NW_PLATFORM_WIN32) || defined(NW_USE_NINTENDO_SDK)
            // PC版の時は、代わりのファイルを鳴らすようにするためのパス書き換え
            if (info->streamFileType == SoundArchive::STREAM_FILE_TYPE_ADTS){
                s32 fileNameStartIndex = internal::fnd::String::LastIndexOf(arg.filePath, '/', internal::FILE_PATH_MAX);
                if (fileNameStartIndex != internal::fnd::String::INDEX_NOT_FOUND)
                {
                    fileNameStartIndex++;
                    s32 extensionStartIndex = internal::fnd::String::LastIndexOf(&arg.filePath[fileNameStartIndex], '.', internal::FILE_PATH_MAX - fileNameStartIndex);
                    if (extensionStartIndex != internal::fnd::String::INDEX_NOT_FOUND)
                    {
                        s32 fileNameLength = extensionStartIndex;

                        char fileName[internal::FILE_PATH_MAX];
                        ut::strncpy(fileName, internal::FILE_PATH_MAX, &arg.filePath[fileNameStartIndex], fileNameLength);

                        char resultPathBase[internal::FILE_PATH_MAX];
                        ut::snprintf(resultPathBase, internal::FILE_PATH_MAX, "_forPC/%s.pcm16.bfstm", fileName);

                        size_t resultPathBaseLength = strnlen(resultPathBase, internal::FILE_PATH_MAX);

                        ut::strncpy(&arg.filePath[fileNameStartIndex], internal::FILE_PATH_MAX - fileNameStartIndex, resultPathBase, resultPathBaseLength);
                        //NW_LOG("arg.filePath = %s\n", arg.filePath);
                    }
                }
            }
        #endif
        }

        const void* prefetchFile = NULL;
        if (externalStrmInfo != NULL && externalStrmInfo->prefetchData != NULL)
        {
            prefetchFile = externalStrmInfo->prefetchData;
        }
        else
        {
            if (info->prefetchFileId != SoundArchive::INVALID_ID)
            {
                if (m_pSoundArchiveFilesHook != NULL)
                {
                    const char* soundLabel = m_pSoundArchive->GetItemLabel(sound->GetId());

                    if (soundLabel != NULL)
                    {
                        prefetchFile = m_pSoundArchiveFilesHook->GetFileAddress(
                            soundLabel,
                            internal::SoundArchiveFilesHook::ITEM_TYPE_STREAM_SOUND,
                            internal::SoundArchiveFilesHook::FILE_TYPE_STREAM_PREFETCH_BINARY);
                    }
                }

                if (prefetchFile == NULL)
                {
                    prefetchFile = m_pSoundDataManager->detail_GetFileAddress( info->prefetchFileId );
                }
            }
        }

        // startOffset 使用時はプリフェッチデータを使用しない
        if ( prefetchFile != NULL && arg.offset == 0 )
        {
#if defined(NW_PLATFORM_CAFE)
            // プリフェッチデータとストリームバッファが 512MB 境界を跨いで配置されている場合は再生にプリフェッチを使用できません。
            const void* streamBufferAddress = m_StreamBufferPool.GetBufferAddr();
            if ( !IsValidVoiceOffsetBoundary( prefetchFile, streamBufferAddress ) )
            {
                if ( externalStrmInfo->forcePlayPrefetchFlag )
                {
                    NW_WARNING( false, "Boundary violation occurred. Cannot play prefetchData, so force play with streaming.\n" );
                }
                else
                {
                    SoundStartable::StartResult result(
                        SoundStartable::StartResult::START_ERR_INVALID_VOICE_OFFSET_BOUNDARY );
                    return result;
                }
            }
            else
#endif
            {
                sound->PreparePrefetch( prefetchFile, arg );
            }
        }

        sound->Prepare(arg);
    }

    SoundStartable::StartResult startResult(
            SoundStartable::StartResult::START_SUCCESS );
    return startResult;
}

/*---------------------------------------------------------------------------*
  Name:         PrepareWaveSoundImpl

  Description:  ウェーブサウンドを再生

  Arguments:    playerId   - プレイヤー番号
                bankId     - バンク番号
                playerPrio - プレイヤープライオリティ
                info       - シーケンス情報構造体のポインタ
                seqId      - シーケンス番号

  Returns:      成功したかどうか
 *---------------------------------------------------------------------------*/
SoundStartable::StartResult SoundArchivePlayer::PrepareWaveSoundImpl(
    internal::WaveSound* sound,
    const SoundArchive::SoundInfo* commonInfo,
    const SoundArchive::WaveSoundInfo* info,
    SoundStartable::StartInfo::StartOffsetType startOffsetType,
    int startOffset,
    int delayTime,
    const StartInfo::WaveSoundInfo* externalWsdInfo
)
{
    NW_ASSERT_NOT_NULL( info );
    NW_ASSERT_NOT_NULL( m_pSoundArchive );

    bool canUsePlayerHeap = sound->GetSoundPlayer()->detail_CanUsePlayerHeap();
    bool isRegisterDataLoadTask = false;
    internal::LoadItemInfo loadWsd;


    // データの取得
    const void* wsdFile = NULL;
    const void* waveFile = NULL;

    // ウェーブサウンド、波形アーカイブファイルアクセスのフック
    if(m_pSoundArchiveFilesHook != NULL)
    {
        const char* soundLabel = m_pSoundArchive->GetItemLabel(sound->GetId());

        if(soundLabel != NULL)
        {
            wsdFile = m_pSoundArchiveFilesHook->GetFileAddress(
                soundLabel,
                internal::SoundArchiveFilesHook::ITEM_TYPE_WAVE_SOUND,
                internal::SoundArchiveFilesHook::FILE_TYPE_WAVE_SOUND_BINARY);

            const void* warcFile = m_pSoundArchiveFilesHook->GetFileAddress(
                soundLabel,
                internal::SoundArchiveFilesHook::ITEM_TYPE_WAVE_SOUND,
                internal::SoundArchiveFilesHook::FILE_TYPE_WAVE_ARCHIVE_BINARY);

            if(wsdFile != NULL && warcFile != NULL)
            {
                internal::WaveArchiveFileReader warcReader( warcFile, false );
                waveFile = warcReader.GetWaveFile( info->index );
            }

            // 完全に読み込めなかった場合は、通常動作を試みます。
            if(waveFile == NULL)
            {
                wsdFile = NULL;
            }
        }
    }

    if ( wsdFile == NULL )
    {
        wsdFile = m_pSoundDataManager->detail_GetFileAddress( commonInfo->fileId );
    }

    loadWsd.itemId = sound->GetId();
    loadWsd.address = wsdFile;
    if ( wsdFile == NULL )
    {
        if (canUsePlayerHeap == false)
        {
            SoundStartable::StartResult result(
                    SoundStartable::StartResult::START_ERR_NOT_WSD_LOADED );
            return result;
        }
        isRegisterDataLoadTask = true;
    }
    else if(waveFile == NULL)
    {
        if (externalWsdInfo != NULL)
        {
            waveFile = externalWsdInfo->waveAddress;
        }
        else
        {
            waveFile = internal::Util::GetWaveFileOfWaveSound(
                    wsdFile,
                    info->index,
                    *m_pSoundArchive,
                    *m_pSoundDataManager );
        }
        if ( waveFile == NULL )
        {
            if (canUsePlayerHeap == false)
            {
                SoundStartable::StartResult result(
                        SoundStartable::StartResult::START_ERR_NOT_WARC_LOADED );
                return result;
            }
            isRegisterDataLoadTask = true;
        }
    }

    internal::driver::WaveSoundPlayer::StartOffsetType wsdStartOffsetType;
    switch ( startOffsetType )
    {
    case SoundStartable::StartInfo::START_OFFSET_TYPE_MILLISEC:
        wsdStartOffsetType = internal::driver::WaveSoundPlayer::START_OFFSET_TYPE_MILLISEC;
        break;
    case SoundStartable::StartInfo::START_OFFSET_TYPE_TICK:
        wsdStartOffsetType = internal::driver::WaveSoundPlayer::START_OFFSET_TYPE_SAMPLE;
        startOffset = 0;
        break;
    case SoundStartable::StartInfo::START_OFFSET_TYPE_SAMPLE:
        wsdStartOffsetType = internal::driver::WaveSoundPlayer::START_OFFSET_TYPE_SAMPLE;
        break;
    default:
        wsdStartOffsetType = internal::driver::WaveSoundPlayer::START_OFFSET_TYPE_SAMPLE;
        startOffset = 0;
        break;
    }

    internal::driver::WaveSoundPlayer::StartInfo startInfo =
    {
        static_cast<s32>( info->index ),
        wsdStartOffsetType,
        startOffset,
        delayTime
    };

    if ( isRegisterDataLoadTask )
    {
        // 必要なデータが欠けている場合は、プレイヤーヒープへのロードタスクを投げる
        // (プリペアは、タスク完了後のコールバック内で処理される)
        internal::driver::WaveSoundLoader::LoadInfo loadInfo(
            m_pSoundArchive,
            m_pSoundDataManager,
            &loadWsd,
            sound->GetSoundPlayer()
        );
        sound->RegisterDataLoadTask( loadInfo, startInfo );
    }
    else
    {
        internal::driver::WaveSoundPlayer::PrepareArg arg;
        {
            arg.waveType = WAVE_TYPE_NWWAV;
            arg.wsdFile = wsdFile;
            arg.waveFile = waveFile;
            if (externalWsdInfo != NULL)
            {
                arg.waveType = externalWsdInfo->waveType;
                arg.useContextInfo = externalWsdInfo->useContextInfo;
                if (arg.useContextInfo)
                {
                    arg.contextInfo = externalWsdInfo->contextInfo;
                }
            }
        }
        // 必要なデータが全部揃っている場合は、プリペア処理を行う
        sound->Prepare(startInfo, arg);
    }

    sound->InitializeChannelParam(info->channelPriority, info->isReleasePriorityFix);

    SoundStartable::StartResult startResult(
            SoundStartable::StartResult::START_SUCCESS );
    return startResult;
}

void SoundArchivePlayer::SetCommonSoundParam(
    internal::BasicSound* sound,
    const SoundArchive::SoundInfo* commonInfo
)
{
    NW_ASSERT_NOT_NULL(sound);
    NW_ASSERT_NOT_NULL(commonInfo);

    sound->SetInitialVolume( static_cast<float>( commonInfo->volume ) / 127.0f );
    sound->SetPanMode( commonInfo->panMode );
    sound->SetPanCurve( commonInfo->panCurve );
    sound->SetRemoteFilter( commonInfo->remoteFilter );
}

/*---------------------------------------------------------------------------*/
bool SoundArchivePlayer::IsSoundArchiveFileHooksEnabled() const
{
    return m_pSoundArchiveFilesHook != NULL && m_pSoundArchiveFilesHook->GetIsEnable();
}

/*---------------------------------------------------------------------------*/
void SoundArchivePlayer::LockSoundArchiveFileHooks()
{
    if(IsSoundArchiveFileHooksEnabled())
    {
        m_pSoundArchiveFilesHook->Lock();
    }
}

/*---------------------------------------------------------------------------*/
void SoundArchivePlayer::UnlockSoundArchiveFileHooks()
{
    if(IsSoundArchiveFileHooksEnabled())
    {
        m_pSoundArchiveFilesHook->Unlock();
    }
}

/*---------------------------------------------------------------------------*/
bool SoundArchivePlayer::IsSequenceSoundEdited(const char* soundName) const
{
    if(soundName == NULL)
    {
        return false;
    }

    return m_pSoundArchiveFilesHook->GetFileAddress(
        soundName,
        internal::SoundArchiveFilesHook::ITEM_TYPE_SEQUENCE_SOUND,
        internal::SoundArchiveFilesHook::FILE_TYPE_SEQUENCE_BINARY) != NULL;
}

/*---------------------------------------------------------------------------*/
bool SoundArchivePlayer::IsStreamSoundEdited(const char* soundName) const
{
    if(soundName == NULL)
    {
        return false;
    }

    return true;
}

/*---------------------------------------------------------------------------*/
bool SoundArchivePlayer::IsWaveSoundEdited(const char* soundName) const
{
    if(soundName == NULL)
    {
        return false;
    }

    return m_pSoundArchiveFilesHook->GetFileAddress(
        soundName,
        internal::SoundArchiveFilesHook::ITEM_TYPE_WAVE_SOUND,
        internal::SoundArchiveFilesHook::FILE_TYPE_WAVE_SOUND_BINARY) != NULL;
}

/*---------------------------------------------------------------------------*
  Name:         SetSequenceUserprocCallback

  Description:  シーケンスコマンド'userproc'で呼び出されるコールバックを登録する

  Arguments:    callback - コールバック
                arg - コールバック引数

  Returns:
 *---------------------------------------------------------------------------*/
void SoundArchivePlayer::SetSequenceUserprocCallback( SequenceUserprocCallback callback, void* callbackArg )
{
    m_SequenceUserprocCallback = callback;
    m_pSequenceUserprocCallbackArg = callbackArg;
}

/*---------------------------------------------------------------------------*
  Name:         NoteOn

  Description:  シーケンスサウンドのノートオン処理

  Arguments:

  Returns:      None.
 *---------------------------------------------------------------------------*/
internal::driver::Channel* SoundArchivePlayer::SequenceNoteOnCallback::NoteOn(
    internal::driver::SequenceSoundPlayer* seqPlayer,
    u8 bankIndex,
    const internal::driver::NoteOnInfo& noteOnInfo
)
{
    if ( ! m_pSoundArchivePlayer->IsAvailable() ) return NULL;

    const internal::BankFileReader& bankReader = seqPlayer->GetBankFileReader( bankIndex );
    NW_WARNING(bankReader.IsInitialized(), "invalid bank index(%d) is used in SequenceSound.", bankIndex);

    // 当該ノートに該当する波形アーカイブ ID・波形アーカイブ内インデックス取得
    internal::driver::Bank bank;
    internal::driver::Channel* channel = bank.NoteOn(
        bankReader,
        seqPlayer->GetWaveArchiveFileReader( bankIndex ),
        noteOnInfo
    );

    return channel;
}

void SoundArchivePlayer::SetSequenceSkipIntervalTick( s32 intervalTick )
{
    internal::driver::SequenceSoundPlayer::SetSkipIntervalTick( intervalTick );
}

s32 SoundArchivePlayer::GetSequenceSkipIntervalTick()
{
    return internal::driver::SequenceSoundPlayer::GetSkipIntervalTick();
}

} // namespace nw::snd
} // namespace nw

