﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nw/snd/snd_SoundArchiveFileReader.h>
#include <nw/snd/snd_ItemType.h>

namespace nw {
namespace snd {
namespace internal {

namespace {

// const u32 SIGNATURE_STRING_BLOCK        = NW_UT_MAKE_SIGWORD( 'S', 'T', 'R', 'G' );
// const u32 SIGNATURE_INFO_BLOCK          = NW_UT_MAKE_SIGWORD( 'I', 'N', 'F', 'O' );
// const u32 SIGNATURE_DATA_BLOCK          = NW_UT_MAKE_SIGWORD( 'D', 'A', 'T', 'A' );
const u32 SUPPORTED_FILE_VERSION_SAR         = NW_UT_MAKE_VERSION( 0, 1, 0, 0 );
const u32 CURRENT_FILE_VERSION_SAR           = NW_UT_MAKE_VERSION( 0, 2, 2, 0 );
const u32 STRMSEND_SUPPORTED_VERSION_SAR     = NW_UT_MAKE_VERSION( 0, 2, 1, 0 );
const u32 FILTER_SUPPORTED_VERSION_SAR       = NW_UT_MAKE_VERSION( 0, 2, 1, 0 );
const u32 STRMPREFETCH_SUPPORTED_VERSION_SAR = NW_UT_MAKE_VERSION( 0, 2, 2, 0 );

bool IsValidFileHeaderSar( const void* soundArchiveData )
{
#if defined(NW_PLATFORM_CAFE)
    const ut::BinaryFileHeader& header =
        *reinterpret_cast<const ut::BinaryFileHeader*>( soundArchiveData );
#else
    const BinaryFileHeader& header =
        *reinterpret_cast<const BinaryFileHeader*>( soundArchiveData );
#endif

    // シグニチャ確認
    NW_ASSERTMSG( header.signature == SoundArchiveFileReader::SIGNATURE_FILE,
            "invalid file signature." );
    if ( header.signature != SoundArchiveFileReader::SIGNATURE_FILE )
    {
        return false;
    }

    // バージョン確認
    bool isSupportedVersion = false;
    if ( (SUPPORTED_FILE_VERSION_SAR <= header.version) &&
                                   (header.version <= CURRENT_FILE_VERSION_SAR) )
    {
        isSupportedVersion = true;

    }
    NW_ASSERTMSG( isSupportedVersion,
            "bfsar file is not supported version.\n"
            "please reconvert file using new version tools.\n"
            "(version condition: 0x%08x <= ... <= 0x%08x, but your version[0x%08x])\n",
            SUPPORTED_FILE_VERSION_SAR, CURRENT_FILE_VERSION_SAR, header.version
    );
    return isSupportedVersion;
}

const SoundArchiveFile::StreamTrackInfo*
GetStreamTrackInfo(u32 index, const SoundArchiveFile::StreamTrackInfoTable& table)
{
    return static_cast<const SoundArchiveFile::StreamTrackInfo*>(
            table.table.GetReferedItem(index, ElementType_SoundArchiveFile_StreamSoundTrackInfo));
}

} // anonymous namespace

SoundArchiveFileReader::SoundArchiveFileReader()
: m_pStringBlockBody( NULL ),
  m_pInfoBlockBody( NULL ),
  m_pFileBlockBody( NULL )
{
}

void SoundArchiveFileReader::Initialize( const void* soundArchiveData )
{
    NW_NULL_ASSERT( soundArchiveData );
    if ( ! IsValidFileHeaderSar( soundArchiveData ) ) return;

    const SoundArchiveFile::FileHeader& header =
        *reinterpret_cast<const SoundArchiveFile::FileHeader*>( soundArchiveData );
    m_Header = header;  // 構造体コピー
}

void SoundArchiveFileReader::Finalize()
{
    m_pStringBlockBody = NULL;
    m_pInfoBlockBody = NULL;
    m_pFileBlockBody = NULL;
}

bool SoundArchiveFileReader::IsStreamSendAvailable() const
{
#if defined(NW_PLATFORM_CAFE)
    const ut::BinaryFileHeader& header =
        *reinterpret_cast<const ut::BinaryFileHeader*>( &m_Header );
#else
    const BinaryFileHeader& header =
        *reinterpret_cast<const BinaryFileHeader*>( &m_Header );
#endif
    if (header.version >= STRMSEND_SUPPORTED_VERSION_SAR)
    {
        return true;
    }
    return false;
}

bool SoundArchiveFileReader::IsFilterSupportedVersion() const
{
#if defined(NW_PLATFORM_CAFE)
    const ut::BinaryFileHeader& header =
        *reinterpret_cast<const ut::BinaryFileHeader*>( &m_Header );
#else
    const BinaryFileHeader& header =
        *reinterpret_cast<const BinaryFileHeader*>( &m_Header );
#endif
    if (header.version >= FILTER_SUPPORTED_VERSION_SAR)
    {
        return true;
    }
    return false;
}

bool SoundArchiveFileReader::IsStreamPrefetchAvailable() const
{
    const ut::BinaryFileHeader& header =
        *reinterpret_cast<const ut::BinaryFileHeader*>( &m_Header );
    if (header.version >= STRMPREFETCH_SUPPORTED_VERSION_SAR)
    {
        return true;
    }
    return false;
}

void SoundArchiveFileReader::SetStringBlock( const void* stringBlock )
{
    m_pStringBlockBody =
        &reinterpret_cast<const SoundArchiveFile::StringBlock*>( stringBlock )->body;
}

void SoundArchiveFileReader::SetInfoBlock( const void* infoBlock )
{
    m_pInfoBlockBody =
        &reinterpret_cast<const SoundArchiveFile::InfoBlock*>( infoBlock )->body;
}

u32 SoundArchiveFileReader::GetStringCount() const
{
    NW_NULL_ASSERT( m_pStringBlockBody );
    return m_pStringBlockBody->GetStringCount();
}
const char* SoundArchiveFileReader::GetString( SoundArchive::ItemId stringId ) const
{
    NW_NULL_ASSERT( m_pStringBlockBody );
    return m_pStringBlockBody->GetString( stringId );
}

void SoundArchiveFileReader::DumpTree() const
{
    NW_NULL_ASSERT( m_pStringBlockBody );
    m_pStringBlockBody->DumpTree();
}

SoundArchive::ItemId SoundArchiveFileReader::GetItemId( const char* pStr ) const
{
    if ( m_pStringBlockBody == NULL )
    {
        return SoundArchive::INVALID_ID;
    }
    return m_pStringBlockBody->GetItemId( pStr );
}
const char* SoundArchiveFileReader::GetItemLabel( SoundArchive::ItemId id ) const
{
    NW_NULL_ASSERT( m_pInfoBlockBody );
    if ( m_pStringBlockBody == NULL )
    {
        return NULL;
    }
    SoundArchive::StringId stringId = m_pInfoBlockBody->GetItemStringId( id );
    if ( stringId == SoundArchive::INVALID_ID )
    {
        return NULL;
    }
    return m_pStringBlockBody->GetString( stringId );
}
SoundArchive::FileId SoundArchiveFileReader::GetItemFileId( SoundArchive::ItemId id ) const
{
    NW_NULL_ASSERT( m_pInfoBlockBody );
    return m_pInfoBlockBody->GetItemFileId( id );
}

SoundArchive::FileId SoundArchiveFileReader::GetItemPrefetchFileId( SoundArchive::ItemId id ) const
{
    NW_NULL_ASSERT( m_pInfoBlockBody );
    return m_pInfoBlockBody->GetItemPrefetchFileId( id );
}

u32 SoundArchiveFileReader::GetSoundCount() const
{
    NW_NULL_ASSERT( m_pInfoBlockBody );
    return m_pInfoBlockBody->GetSoundCount();
}

u32 SoundArchiveFileReader::GetBankCount() const
{
    NW_NULL_ASSERT( m_pInfoBlockBody );
    return m_pInfoBlockBody->GetBankCount();
}

u32 SoundArchiveFileReader::GetPlayerCount() const
{
    NW_NULL_ASSERT( m_pInfoBlockBody );
    return m_pInfoBlockBody->GetPlayerCount();
}

u32 SoundArchiveFileReader::GetSoundGroupCount() const
{
    NW_NULL_ASSERT( m_pInfoBlockBody );
    return m_pInfoBlockBody->GetSoundGroupCount();
}

u32 SoundArchiveFileReader::GetGroupCount() const
{
    NW_NULL_ASSERT( m_pInfoBlockBody );
    return m_pInfoBlockBody->GetGroupCount();
}

u32 SoundArchiveFileReader::GetWaveArchiveCount() const
{
    NW_NULL_ASSERT( m_pInfoBlockBody );
    return m_pInfoBlockBody->GetWaveArchiveCount();
}

u32 SoundArchiveFileReader::GetFileCount() const
{
    NW_NULL_ASSERT( m_pInfoBlockBody );
    return m_pInfoBlockBody->GetFileCount();
}

bool SoundArchiveFileReader::ReadSoundInfo(
        SoundArchive::ItemId soundId, SoundArchive::SoundInfo* info ) const
{
    NW_NULL_ASSERT( m_pInfoBlockBody );
    NW_NULL_ASSERT( info );

    const SoundArchiveFile::SoundInfo* data = m_pInfoBlockBody->GetSoundInfo( soundId );
    if ( data == NULL ) return false;

    info->fileId = data->fileId;
    info->playerId = data->playerId;
    info->volume = data->volume;
    info->panMode = data->GetPanMode();
    info->panCurve = data->GetPanCurve();
    info->playerPriority = data->GetPlayerPriority();
    info->actorPlayerId = data->GetActorPlayerId();
    info->remoteFilter = data->remoteFilter;
    info->isFrontBypass = data->IsFrontBypass();

    return true;
}

bool SoundArchiveFileReader::ReadBankInfo(
        SoundArchive::ItemId bankId, SoundArchive::BankInfo* info ) const
{
    NW_NULL_ASSERT( m_pInfoBlockBody );
    NW_NULL_ASSERT( info );

    const SoundArchiveFile::BankInfo* data = m_pInfoBlockBody->GetBankInfo( bankId );
    if ( data == NULL ) return false;

    info->fileId = data->fileId;
    return true;
}

bool SoundArchiveFileReader::ReadPlayerInfo(
        SoundArchive::ItemId playerId, SoundArchive::PlayerInfo* info ) const
{
    NW_NULL_ASSERT( m_pInfoBlockBody );
    NW_NULL_ASSERT( info );

    const SoundArchiveFile::PlayerInfo* data = m_pInfoBlockBody->GetPlayerInfo( playerId );
    if ( data == NULL ) return false;

    info->playableSoundMax = data->playableSoundMax;
    info->playerHeapSize = data->GetPlayerHeapSize();

    return true;
}

bool SoundArchiveFileReader::ReadSoundGroupInfo(
        SoundArchive::ItemId soundGroupId, SoundArchive::SoundGroupInfo* info ) const
{
    NW_NULL_ASSERT( m_pInfoBlockBody );
    NW_NULL_ASSERT( info );

    const SoundArchiveFile::SoundGroupInfo* data = m_pInfoBlockBody->GetSoundGroupInfo( soundGroupId );
    if ( data == NULL ) return false;

    info->startId = data->startId;
    info->endId = data->endId;
    info->fileIdTable = data->GetFileIdTable();
    return true;
}

bool SoundArchiveFileReader::ReadGroupInfo(
        SoundArchive::ItemId groupId, SoundArchive::GroupInfo* info ) const
{
    NW_NULL_ASSERT( m_pInfoBlockBody );
    NW_NULL_ASSERT( info );

    // ファイル ID
    {
        const SoundArchiveFile::GroupInfo* data = m_pInfoBlockBody->GetGroupInfo( groupId );
        if ( data == NULL )
        {
            return false;
        }
        info->fileId = data->fileId;
    }

    // ファイルサイズ
    {
        SoundArchive::FileInfo fileInfo;
        bool result = ReadFileInfo( info->fileId, &fileInfo );
        if ( ! result )
        {
            return false;
        }
        info->groupFileSize = fileInfo.fileSize;
    }

    return true;
}

bool SoundArchiveFileReader::ReadWaveArchiveInfo(
        SoundArchive::ItemId warcId, SoundArchive::WaveArchiveInfo* info ) const
{
    NW_NULL_ASSERT( m_pInfoBlockBody );
    NW_NULL_ASSERT( info );

    const SoundArchiveFile::WaveArchiveInfo* data = m_pInfoBlockBody->GetWaveArchiveInfo( warcId );
    if ( data == NULL ) return false;

    info->fileId = data->fileId;
    info->waveCount = data->GetWaveCount();
    info->isLoadIndividual = data->isLoadIndividual;
    return true;
}

bool SoundArchiveFileReader::ReadFileInfo(
        SoundArchive::FileId id, SoundArchive::FileInfo* info, int index ) const
{
    NW_NULL_ASSERT( m_pInfoBlockBody );
    NW_NULL_ASSERT( info );
    NW_UNUSED_VARIABLE( index );

    const SoundArchiveFile::FileInfo* data = m_pInfoBlockBody->GetFileInfo( id );
    if ( data == NULL ) return false;

    switch ( data->GetFileLocationType() )
    {
        case SoundArchiveFile::FILE_LOCATION_TYPE_INTERNAL:
            {
                const SoundArchiveFile::InternalFileInfo* location = data->GetInternalFileInfo();
                info->fileSize = location->GetFileSize();
                info->offsetFromFileBlockHead = location->GetOffsetFromFileBlockHead();
            }
            break;
        case SoundArchiveFile::FILE_LOCATION_TYPE_EXTERNAL:
            {
                const SoundArchiveFile::ExternalFileInfo* location = data->GetExternalFileInfo();
                info->externalFilePath = location->filePath;
            }
            break;
        case SoundArchiveFile::FILE_LOCATION_TYPE_NONE:
            return false;
    }
    return true;
}

bool SoundArchiveFileReader::ReadSoundArchivePlayerInfo(
        SoundArchive::SoundArchivePlayerInfo* info ) const
{
    NW_NULL_ASSERT( m_pInfoBlockBody );
    NW_NULL_ASSERT( info );

    const SoundArchiveFile::SoundArchivePlayerInfo* data = m_pInfoBlockBody->GetSoundArchivePlayerInfo();
    if ( data == NULL ) return false;

    info->sequenceSoundMax  = data->sequenceSoundMax;
    info->sequenceTrackMax  = data->sequenceTrackMax;
    info->streamSoundMax    = data->streamSoundMax;
    info->streamTrackMax    = data->streamTrackMax;
    info->streamChannelMax  = data->streamChannelMax;
    info->waveSoundMax      = data->waveSoundMax;
    info->waveTrackMax      = data->waveTrackmax;
    if ( IsStreamPrefetchAvailable() )
    {
        info->streamBufferTimes = data->streamBufferTimes;
    }
    else
    {
        info->streamBufferTimes = 1;
    }
    return true;
}

bool SoundArchiveFileReader::ReadSound3DInfo(
        SoundArchive::ItemId soundId, SoundArchive::Sound3DInfo* info ) const
{
    NW_NULL_ASSERT( m_pInfoBlockBody );
    NW_NULL_ASSERT( info );

    const SoundArchiveFile::SoundInfo* soundInfo = m_pInfoBlockBody->GetSoundInfo( soundId );
    if ( soundInfo == NULL )
    {
        return false;
    }

    const SoundArchiveFile::Sound3DInfo* sound3dInfo = soundInfo->GetSound3DInfo();
    if ( sound3dInfo == NULL )
    {
        return false;
    }
    info->flags = sound3dInfo->flags;
    info->decayRatio = sound3dInfo->decayRatio;
    info->decayCurve = sound3dInfo->decayCurve;
    info->dopplerFactor = sound3dInfo->dopplerFactor;
    return true;
}

bool SoundArchiveFileReader::ReadSequenceSoundInfo(
        SoundArchive::ItemId soundId, SoundArchive::SequenceSoundInfo* info ) const
{
    NW_NULL_ASSERT( m_pInfoBlockBody );
    NW_NULL_ASSERT( info );

    const SoundArchiveFile::SoundInfo* soundInfo = m_pInfoBlockBody->GetSoundInfo( soundId );
    if ( soundInfo == NULL )
    {
        return false;
    }
    if ( soundInfo->GetSoundType() != SoundArchive::SOUND_TYPE_SEQ )
    {
        return false;
    }

    const SoundArchiveFile::SequenceSoundInfo& seqSoundInfo = soundInfo->GetSequenceSoundInfo();
    info->startOffset = seqSoundInfo.GetStartOffset();
    info->allocateTrackFlags = seqSoundInfo.allocateTrackFlags;
    info->channelPriority = seqSoundInfo.GetChannelPriority();
    info->isReleasePriorityFix = seqSoundInfo.IsReleasePriorityFix();
    seqSoundInfo.GetBankIds( info->bankIds );
    return true;
}

bool SoundArchiveFileReader::ReadStreamSoundInfo(
        SoundArchive::ItemId soundId, SoundArchive::StreamSoundInfo* info ) const
{
    NW_NULL_ASSERT( m_pInfoBlockBody );
    NW_NULL_ASSERT( info );

    const SoundArchiveFile::SoundInfo* soundInfo = m_pInfoBlockBody->GetSoundInfo( soundId );
    if ( soundInfo == NULL )
    {
        return false;
    }
    if ( soundInfo->GetSoundType() != SoundArchive::SOUND_TYPE_STRM )
    {
        return false;
    }

    const SoundArchiveFile::StreamSoundInfo& strmSoundInfo = soundInfo->GetStreamSoundInfo();
    info->allocateTrackFlags   = strmSoundInfo.allocateTrackFlags;
    info->allocateChannelCount = strmSoundInfo.allocateChannelCount;
    const SoundArchiveFile::StreamSoundExtension* strmSoundExt = strmSoundInfo.GetStreamSoundExtension();
    info->streamFileType = (strmSoundExt == NULL) ? SoundArchive::STREAM_FILE_TYPE_NW_STREAM_BINARY : strmSoundExt->GetStreamFileType();
    if ( IsStreamSendAvailable() )
    {
        info->pitch                            = strmSoundInfo.pitch;
        const SoundArchiveFile::SendValue send = strmSoundInfo.GetSendValue();
        info->mainSend                         = send.mainSend;
        for (u32 i = 0; i < AUX_BUS_NUM; i++)
        {
            info->fxSend[i]                    = send.fxSend[i];
        }
    }
    else
    {
        info->pitch = 1.0f;
        info->mainSend = 127;
        for (u32 i = 0; i < AUX_BUS_NUM; i++)
        {
            info->fxSend[i] = 0;
        }
    }

    if ( IsStreamPrefetchAvailable() )
    {
        info->prefetchFileId = strmSoundInfo.prefetchFileId;
    }
    else
    {
        info->prefetchFileId = SoundArchive::INVALID_ID;
    }

    const SoundArchiveFile::StreamTrackInfoTable* table = strmSoundInfo.GetTrackInfoTable();
    if ( table == NULL )
    {
        return true;
    }

    u32 tableCount = table->GetTrackCount();
    for ( u32 i = 0; i < SoundArchive::STRM_TRACK_NUM && i < tableCount; i++ )
    {
        SoundArchive::StreamTrackInfo& tmp = info->trackInfo[i];
        const SoundArchiveFile::StreamTrackInfo* pTrackInfo = GetStreamTrackInfo(i, *table);
        NW_NULL_ASSERT(pTrackInfo);
        const SoundArchiveFile::StreamTrackInfo& trackInfo = *pTrackInfo;
        tmp.volume = trackInfo.volume;
        tmp.pan = trackInfo.pan;
        tmp.span = trackInfo.span;
        tmp.flags = trackInfo.flags;
        tmp.channelCount = static_cast<u8>(trackInfo.GetTrackChannelCount());
        if ( IsStreamSendAvailable() )
        {
            const SoundArchiveFile::SendValue send = trackInfo.GetSendValue();
            tmp.mainSend                           = send.mainSend;
            for (u32 j = 0; j < AUX_BUS_NUM; j++)
            {
                tmp.fxSend[j]                      = send.fxSend[j];
            }
        }
        else
        {
            tmp.mainSend = 127;
            for (u32 j = 0; j < AUX_BUS_NUM; j++)
            {
                tmp.fxSend[j] = 0;
            }
        }
        if ( IsFilterSupportedVersion() )
        {
            tmp.lpfFreq     = trackInfo.lpfFreq;
            tmp.biquadType  = trackInfo.biquadType;
            tmp.biquadValue = trackInfo.biquadValue;
        }
        else
        {
            tmp.lpfFreq     = 64;
            tmp.biquadType  = 0;
            tmp.biquadValue = 0;
        }

        u32 count = ut::Min(static_cast<u32>(tmp.channelCount), WAVE_CHANNEL_MAX);
        for (u32 j = 0; j < count; j++)
        {
            tmp.globalChannelIndex[j] = trackInfo.GetGlobalChannelIndex(j);
        }
    }
    return true;
}

bool SoundArchiveFileReader::ReadStreamSoundInfo2(
    SoundArchive::ItemId soundId, SoundArchive::StreamSoundInfo2* info ) const
{
    NW_NULL_ASSERT( m_pInfoBlockBody );
    NW_NULL_ASSERT( info );

    const SoundArchiveFile::SoundInfo* soundInfo = m_pInfoBlockBody->GetSoundInfo( soundId );
    if ( soundInfo == NULL )
    {
        return false;
    }
    if ( soundInfo->GetSoundType() != SoundArchive::SOUND_TYPE_STRM )
    {
        return false;
    }

    const SoundArchiveFile::StreamSoundInfo& strmSoundInfo = soundInfo->GetStreamSoundInfo();
    const SoundArchiveFile::StreamSoundExtension* strmSoundExt = strmSoundInfo.GetStreamSoundExtension();

    if( strmSoundExt == NULL )
    {
        return false;
    }

    info->isLoop         = strmSoundExt->IsLoop();
    info->loopStartFrame = strmSoundExt->loopStartFrame;
    info->loopEndFrame   = strmSoundExt->loopEndFrame;

    return true;
}

bool SoundArchiveFileReader::ReadWaveSoundInfo(
        SoundArchive::ItemId soundId, SoundArchive::WaveSoundInfo* info ) const
{
    NW_NULL_ASSERT( m_pInfoBlockBody );
    NW_NULL_ASSERT( info );

    const SoundArchiveFile::SoundInfo* soundInfo = m_pInfoBlockBody->GetSoundInfo( soundId );
    if ( soundInfo == NULL )
    {
        return false;
    }
    if ( soundInfo->GetSoundType() != SoundArchive::SOUND_TYPE_WAVE )
    {
        return false;
    }

    const SoundArchiveFile::WaveSoundInfo& waveSoundInfo = soundInfo->GetWaveSoundInfo();
    info->index                 = waveSoundInfo.index;
    info->allocateTrackCount    = waveSoundInfo.allocateTrackCount;
    info->channelPriority       = waveSoundInfo.GetChannelPriority();
    info->isReleasePriorityFix  = waveSoundInfo.GetIsReleasePriorityFix() > 0 ? true : false;

    return true;
}

#if defined(NW_PLATFORM_CAFE)
const internal::Util::Table<ut::ResU32>*
#else
const internal::Util::Table<u32>*
#endif
SoundArchiveFileReader::GetWaveArchiveIdTable( SoundArchive::ItemId id ) const
{
    NW_NULL_ASSERT( m_pInfoBlockBody );

    switch ( Util::GetItemType( id ) )
    {
        case ItemType_Sound:
            {
                const SoundArchiveFile::SoundInfo* pInfo =
                    m_pInfoBlockBody->GetSoundInfo( id );
                NW_NULL_ASSERT( pInfo );

                if ( pInfo->GetSoundType() == SoundArchive::SOUND_TYPE_WAVE )
                {
                    // ウェーブサウンド ID → ウェーブサウンドセット ID を引く
                    for ( u32 i = 0; i < m_pInfoBlockBody->GetSoundGroupCount(); i++ )
                    {
                        const SoundArchiveFile::SoundGroupInfo* pGroupInfo =
                            m_pInfoBlockBody->GetSoundGroupInfo(
                                    Util::GetMaskedItemId( i, ItemType_SoundGroup ) );
                        NW_NULL_ASSERT( pGroupInfo );

                        if ( id < pGroupInfo->startId || pGroupInfo->endId < id )
                        {
                            continue;
                        }

                        // 該当するウェーブサウンドセットの波形アーカイブテーブルを返す
                        const SoundArchiveFile::WaveSoundGroupInfo*
                            pWaveSoundGroupInfo = pGroupInfo->GetWaveSoundGroupInfo();
                        NW_NULL_ASSERT( pWaveSoundGroupInfo );

                        return pWaveSoundGroupInfo->GetWaveArchiveItemIdTable();
                    }
                }
                return NULL;
            }
        case ItemType_SoundGroup:
            {
                const SoundArchiveFile::SoundGroupInfo* pInfo =
                    m_pInfoBlockBody->GetSoundGroupInfo( id );
                NW_NULL_ASSERT( pInfo );

                const SoundArchiveFile::WaveSoundGroupInfo*
                    pWaveSoundGroupInfo = pInfo->GetWaveSoundGroupInfo();
                NW_NULL_ASSERT( pWaveSoundGroupInfo );

                return pWaveSoundGroupInfo->GetWaveArchiveItemIdTable();
            }
        case ItemType_Bank:
            {
                const SoundArchiveFile::BankInfo* pInfo =
                    m_pInfoBlockBody->GetBankInfo( id );
                NW_NULL_ASSERT( pInfo );

                return pInfo->GetWaveArchiveItemIdTable();
            }
        default:
            return NULL;
    }
}

SoundArchive::SoundType
SoundArchiveFileReader::GetSoundType( SoundArchive::ItemId soundId ) const
{
    NW_NULL_ASSERT( m_pInfoBlockBody );

    const SoundArchiveFile::SoundInfo* soundInfo = m_pInfoBlockBody->GetSoundInfo( soundId );
    if ( soundInfo == NULL )
    {
        return SoundArchive::SOUND_TYPE_INVALID;
    }
    return soundInfo->GetSoundType();
}

u32 SoundArchiveFileReader::GetSoundUserParam( SoundArchive::ItemId soundId ) const
{
    NW_NULL_ASSERT( m_pInfoBlockBody );

    const SoundArchiveFile::SoundInfo* soundInfo = m_pInfoBlockBody->GetSoundInfo( soundId );
    if ( soundInfo == NULL )
    {
        return 0;
    }
    return soundInfo->GetUserParam();
}

bool SoundArchiveFileReader::ReadSoundUserParam(
        SoundArchive::ItemId soundId, int index, u32& value ) const
{
    NW_NULL_ASSERT( m_pInfoBlockBody );

    const SoundArchiveFile::SoundInfo* soundInfo = m_pInfoBlockBody->GetSoundInfo( soundId );
    if ( soundInfo == NULL )
    {
        return false;
    }
    return soundInfo->ReadUserParam( index, value );

}

#if defined(NW_PLATFORM_CAFE)
const Util::Table<ut::ResU32>* SoundArchiveFileReader::GetAttachedGroupTable(
#else
const Util::Table<u32>* SoundArchiveFileReader::GetAttachedGroupTable(
#endif
    SoundArchive::FileId id ) const
{
    NW_NULL_ASSERT( m_pInfoBlockBody );

    const SoundArchiveFile::FileInfo* fileInfo = m_pInfoBlockBody->GetFileInfo( id );
    if ( fileInfo == NULL )
    {
        return NULL;
    }

    const SoundArchiveFile::InternalFileInfo* internalFileInfo =
        fileInfo->GetInternalFileInfo();

    if ( internalFileInfo == NULL )
    {
        return NULL;
    }

    return internalFileInfo->GetAttachedGroupTable();
}


} // namespace nw::snd::internal
} // namespace nw::snd
} // namespace nw
