﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nw/snd/snd_SoundArchive.h>

#include <cstring>      // strlen, etc...
#include <nw/assert.h>
#include <nw/ut.h>
#include <nw/snd/snd_SoundArchiveFileReader.h>
#include <nw/snd/snd_SoundArchiveParametersHook.h>

namespace nw {
namespace snd {

/* ========================================================================
        static variables
   ======================================================================== */

void SoundArchive::StreamSoundInfo::Setup()
{
    // グローバルチャンネルインデックスをチャンネルカウントから自動的に設定
    s8 index = 0;
    u16 bitMask = allocateTrackFlags;
    for ( u32 i = 0; i < STRM_TRACK_NUM; ++i )
    {
        if ( bitMask == 0 )
        {
            break;
        }

        if ( bitMask & 0x1 )
        {
            u32 count = ut::Min(static_cast<u32>(trackInfo[i].channelCount), WAVE_CHANNEL_MAX);
            for ( u32 j = 0; j < count; j++ )
            {
                trackInfo[i].globalChannelIndex[j] = index;
                ++index;
            }
        }
        bitMask >>= 1;
    }
}

/*---------------------------------------------------------------------------*
  Name:         SoundArchive

  Description:  コンストラクタ

  Arguments:    None.

  Returns:      None.
 *---------------------------------------------------------------------------*/
SoundArchive::SoundArchive()
: m_pFileReader( NULL )
, m_pParametersHook( NULL )
#if defined(NW_PLATFORM_CAFE)
, m_pFsClient(NULL)
#endif
{
    m_ExtFileRoot[0] = '/';
    m_ExtFileRoot[1] = '\0';
}

/*---------------------------------------------------------------------------*
  Name:         ~SoundArchive

  Description:  デストラクタ

  Arguments:    None.

  Returns:      None.
 *---------------------------------------------------------------------------*/
SoundArchive::~SoundArchive()
{
}

bool SoundArchive::IsAvailable() const
{
    if ( m_pFileReader == NULL ) return false;
    return true;
}

void SoundArchive::Initialize( internal::SoundArchiveFileReader* fileReader )
{
    NW_NULL_ASSERT( fileReader );
    m_pFileReader = fileReader;
    m_FileBlockOffset = m_pFileReader->GetFileBlockOffset();
}

/*---------------------------------------------------------------------------*
  Name:         Shutdown

  Description:  シャットダウン

  Arguments:    None.

  Returns:      None.
 *---------------------------------------------------------------------------*/
void SoundArchive::Finalize()
{
    if ( m_pFileReader != NULL )
    {
        m_pFileReader->Finalize();
        m_pFileReader = NULL;
    }
    m_ExtFileRoot[0] = '/';
    m_ExtFileRoot[1] = '\0';
}

u32 SoundArchive::GetSoundCount() const
{
    return m_pFileReader->GetSoundCount();
}
u32 SoundArchive::GetPlayerCount() const
{
    return m_pFileReader->GetPlayerCount();
}
u32 SoundArchive::GetSoundGroupCount() const
{
    return m_pFileReader->GetSoundGroupCount();
}
u32 SoundArchive::GetGroupCount() const
{
    return m_pFileReader->GetGroupCount();
}
u32 SoundArchive::GetBankCount() const
{
    return m_pFileReader->GetBankCount();
}
u32 SoundArchive::GetWaveArchiveCount() const
{
    return m_pFileReader->GetWaveArchiveCount();
}
u32 SoundArchive::detail_GetFileCount() const
{
    return m_pFileReader->GetFileCount();
}

const char* SoundArchive::GetItemLabel( ItemId id ) const
{
    const char* result = NULL;

    if ( m_pParametersHook != NULL )
    {
        result = m_pParametersHook->GetItemLabel( id );

        if ( result != NULL )
        {
            return result;
        }
    }

    return m_pFileReader->GetItemLabel( id );
}
SoundArchive::ItemId SoundArchive::GetItemId( const char* pStr ) const
{
    SoundArchive::ItemId result = INVALID_ID;

    if ( m_pParametersHook != NULL )
    {
        result = m_pParametersHook->GetItemId( pStr );

        if ( result != INVALID_ID )
        {
            return result;
        }
    }

    return m_pFileReader->GetItemId( pStr );
}
SoundArchive::FileId SoundArchive::GetItemFileId( ItemId id ) const
{
    return m_pFileReader->GetItemFileId( id );
}
SoundArchive::FileId SoundArchive::GetItemPrefetchFileId( ItemId id ) const
{
    return m_pFileReader->GetItemPrefetchFileId( id );
}

u32 SoundArchive::GetSoundUserParam( ItemId soundId ) const
{
    u32 userParam = m_pFileReader->GetSoundUserParam( soundId );

    if ( m_pParametersHook == NULL )
    {
        return userParam;
    }

    return m_pParametersHook->GetSoundUserParam( soundId, userParam );
}

bool SoundArchive::ReadSoundUserParam( ItemId soundId, int index, u32& value ) const
{
    NW_ASSERT_MINMAX( index, 0, USER_PARAM_INDEX_MAX );
    bool result = m_pFileReader->ReadSoundUserParam( soundId, index, value );

    if ( m_pParametersHook != NULL )
    {
        result |= m_pParametersHook->ReadSoundUserParam( soundId, index, value );
    }

    return result;
}

SoundArchive::SoundType SoundArchive::GetSoundType( ItemId soundId ) const
{
    return m_pFileReader->GetSoundType( soundId );
}
bool SoundArchive::ReadSoundInfo( ItemId soundId, SoundInfo* info ) const
{
    bool result = m_pFileReader->ReadSoundInfo( soundId, info );

    if ( m_pParametersHook != NULL )
    {
        result |= m_pParametersHook->ReadSoundInfo( soundId, info );
    }

    return result;
}
bool SoundArchive::ReadSequenceSoundInfo( ItemId soundId, SequenceSoundInfo* info ) const
{
    bool result = m_pFileReader->ReadSequenceSoundInfo( soundId, info );

    if ( m_pParametersHook != NULL )
    {
        result |= m_pParametersHook->ReadSequenceSoundInfo( soundId, info );
    }

    return result;
}
bool SoundArchive::ReadStreamSoundInfo( ItemId soundId, StreamSoundInfo* info ) const
{
    bool result = m_pFileReader->ReadStreamSoundInfo( soundId, info );

    if ( m_pParametersHook != NULL )
    {
        result |= m_pParametersHook->ReadStreamSoundInfo( soundId, info );
    }

    return result;
}
bool SoundArchive::detail_ReadStreamSoundInfo2( ItemId soundId, StreamSoundInfo2* info ) const
{
    bool result = m_pFileReader->ReadStreamSoundInfo2( soundId, info );

    if ( m_pParametersHook != NULL )
    {
        result |= m_pParametersHook->ReadStreamSoundInfo2( soundId, info );
    }

    return result;
}
bool SoundArchive::detail_ReadWaveSoundInfo( ItemId soundId, WaveSoundInfo* info ) const
{
    bool result = m_pFileReader->ReadWaveSoundInfo( soundId, info );

    if ( m_pParametersHook != NULL )
    {
        result |= m_pParametersHook->ReadWaveSoundInfo( soundId, info );
    }

    return result;
}
bool SoundArchive::ReadPlayerInfo( ItemId playerId, PlayerInfo* info ) const
{
    bool result = m_pFileReader->ReadPlayerInfo( playerId, info );

    if ( m_pParametersHook != NULL )
    {
        result |= m_pParametersHook->ReadPlayerInfo( playerId, info );
    }

    return result;
}
bool SoundArchive::ReadSoundArchivePlayerInfo( SoundArchivePlayerInfo* info ) const
{
    bool result = m_pFileReader->ReadSoundArchivePlayerInfo( info );

    if ( m_pParametersHook != NULL )
    {
        result |= m_pParametersHook->ReadSoundArchivePlayerInfo( info );
    }

    return result;
}
bool SoundArchive::ReadSound3DInfo( ItemId soundId, Sound3DInfo* info ) const
{
    bool result = m_pFileReader->ReadSound3DInfo( soundId, info );

    if ( m_pParametersHook != NULL )
    {
        result |= m_pParametersHook->ReadSound3DInfo( soundId, info );
    }

    return result;
}
bool SoundArchive::ReadBankInfo( ItemId bankId, BankInfo* info ) const
{
    bool result = m_pFileReader->ReadBankInfo( bankId, info );

    if ( m_pParametersHook != NULL )
    {
        result |= m_pParametersHook->ReadBankInfo( bankId, info );
    }

    return result;
}
bool SoundArchive::ReadWaveArchiveInfo( ItemId warcId, WaveArchiveInfo* info ) const
{
    bool result = m_pFileReader->ReadWaveArchiveInfo( warcId, info );

    if ( m_pParametersHook != NULL )
    {
        result |= m_pParametersHook->ReadWaveArchiveInfo( warcId, info );
    }

    return result;
}
bool SoundArchive::detail_ReadSoundGroupInfo( ItemId soundGroupId, SoundGroupInfo* info ) const
{
    bool result = m_pFileReader->ReadSoundGroupInfo( soundGroupId, info );

    if ( m_pParametersHook != NULL )
    {
        result |= m_pParametersHook->ReadSoundGroupInfo( soundGroupId, info );
    }

    return result;
}
bool SoundArchive::ReadGroupInfo( ItemId groupId, GroupInfo* info ) const
{
    bool result = m_pFileReader->ReadGroupInfo( groupId, info );

    if ( m_pParametersHook != NULL )
    {
        result |= m_pParametersHook->ReadGroupInfo( groupId, info );
    }

    return result;
}
#if 0
bool SoundArchive::detail_ReadGroupItemInfo( GroupId groupId, unsigned long index, GroupItemInfo* info ) const
{
    bool result = m_pFileReader->ReadGroupItemInfo( groupId, index, info );

    if (!result || m_pParametersHook == NULL )
    {
        return result;
    }

    m_pParametersHook->ReadGroupItemInfo( groupId, index, info );
    return true;
}
#endif
bool SoundArchive::detail_ReadFileInfo( FileId fileId, FileInfo* info ) const
{
    bool result = m_pFileReader->ReadFileInfo( fileId, info );

    if (!result || m_pParametersHook == NULL )
    {
        return result;
    }

    m_pParametersHook->ReadFileInfo( fileId, info );
    return true;
}
#if 0
bool SoundArchive::detail_ReadFilePos( FileId fileId, unsigned long index, FilePos* info ) const
{
    bool result = m_pFileReader->ReadFilePos( fileId, index, info );

    if (!result || m_pParametersHook == NULL )
    {
        return result;
    }

    m_pParametersHook->ReadFilePos( fileId, index, info );
    return true;
}
#endif

#if defined(NW_PLATFORM_CAFE)
const internal::Util::Table<ut::ResU32>*
#else
const internal::Util::Table<u32>*
#endif
SoundArchive::detail_GetWaveArchiveIdTable( ItemId id ) const
{
    // id にはウェーブサウンドの ID か、バンクの ID しか入らない
    return m_pFileReader->GetWaveArchiveIdTable( id );
}

ut::FileStream* SoundArchive::detail_OpenFileStream(
        FileId fileId, void* buffer, int size,
        void* cacheBuffer, size_t cacheSize ) const
{
    // ファイル情報取得
    FileInfo fileInfo;
    if ( ! detail_ReadFileInfo( fileId, &fileInfo ) )
    {
        return NULL;
    }

    // 外部参照ファイル
    if ( fileInfo.externalFilePath != NULL )
    {
        ut::FileStream* stream =
            OpenExtStreamImpl( buffer, size, fileInfo.externalFilePath, cacheBuffer, cacheSize );
        return stream;
    }

    // ファイルサイズ、またはオフセットが無効値
    if ( fileInfo.fileSize == FileInfo::INVALID_SIZE ||
        fileInfo.offsetFromFileBlockHead == FileInfo::INVALID_OFFSET )
    {
        return NULL;
    }

    // サウンドアーカイブ内ファイル
    ut::FileStream* stream = OpenStream(
        buffer,
        size,
        fileInfo.offsetFromFileBlockHead + m_FileBlockOffset,
        fileInfo.fileSize
    );
    return stream;
}

ut::FileStream* SoundArchive::OpenExtStreamImpl(
    void* buffer,
    int size,
    const char* externalFilePath,
    void* cacheBuffer,
    size_t cacheSize
) const
{
    const char* fullPath;
    char pathBuffer[ FILE_PATH_MAX ];

    if ( externalFilePath[0] == '/' )
    {
        fullPath = externalFilePath;
    }
    else
    {
        std::size_t fileLen = std::strlen( externalFilePath );
        std::size_t dirLen = std::strlen( m_ExtFileRoot );
        if ( fileLen + dirLen >= sizeof( pathBuffer ) / sizeof( pathBuffer[0] ) )
        {
            NW_WARNING( false, "Too long file path \"%s/%s\"",
                    m_ExtFileRoot, externalFilePath );
            return NULL;
        }
        nw::ut::strncpy( pathBuffer, FILE_PATH_MAX, m_ExtFileRoot, dirLen+1 );
        nw::ut::strncat( pathBuffer, FILE_PATH_MAX, externalFilePath, fileLen+1 );
        fullPath = pathBuffer;
    }

    return OpenExtStream( buffer, size, fullPath, cacheBuffer, cacheSize );
}

const char* SoundArchive::detail_GetExternalFileFullPath(
        const char* externalFilePath, char* pathBuffer, size_t bufSize) const
{
    NW_ASSERT_NOT_NULL(externalFilePath);
    if (externalFilePath[0] == '/')
    {
        return externalFilePath;
    }

    std::size_t fileLen = std::strlen( externalFilePath );
    std::size_t dirLen = std::strlen( m_ExtFileRoot );
    if ( fileLen + dirLen >= bufSize )
    {
        NW_WARNING( false, "Too long file path \"%s/%s\"",
                m_ExtFileRoot, externalFilePath );
        return NULL;
    }
    nw::ut::strncpy( pathBuffer, bufSize, m_ExtFileRoot, dirLen+1 );
    nw::ut::strncat( pathBuffer, bufSize, externalFilePath, fileLen+1 );
    return pathBuffer;
}

void SoundArchive::SetExternalFileRoot( const char* extFileRoot )
{
    std::size_t len = std::strlen( extFileRoot );
    std::size_t nullPos = len;

    ut::strncpy( m_ExtFileRoot, FILE_PATH_MAX, extFileRoot, len );

    // 末尾の "/" の有無を吸収
    if ( extFileRoot[ len-1 ] != '/' )
    {
        m_ExtFileRoot[ len ] = '/';
        nullPos++;
    }

    NW_ASSERT( nullPos < FILE_PATH_MAX );
    m_ExtFileRoot[ nullPos ] = '\0';
}

#if defined(NW_PLATFORM_CAFE)
const internal::Util::Table<ut::ResU32>* SoundArchive::detail_GetAttachedGroupTable(
#else
const internal::Util::Table<u32>* SoundArchive::detail_GetAttachedGroupTable(
#endif
    FileId fileId ) const
{
    return m_pFileReader->GetAttachedGroupTable( fileId );
}

#if defined(NW_PLATFORM_CAFE)
size_t SoundArchive::GetRequiredFsCommandBlockPoolMemSize() const
{
    return sizeof(FSCmdBlock) * 2;
}

void SoundArchive::CreateFsCommandBlockBufferPool(void* buffer, size_t bufferSize)
{
    NW_ASSERT_NOT_NULL(buffer);
    m_FsCmdBlockPool.Create(buffer, bufferSize);
}

void SoundArchive::DestroyFsCommandBlockBufferPool()
{
    m_FsCmdBlockPool.Destroy();
}
#endif

} // namespace nw::snd
} // namespace nw

