﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nw/snd/snd_SoundActor.h>
#include <nw/snd/snd_SoundArchivePlayer.h>

namespace nw {
namespace snd {

/*---------------------------------------------------------------------------*
  Name:         SoundActor

  Description:  コンストラクタ

  Arguments:    soundArchivePlayer - サウンドアーカイブプレイヤー

  Returns:      None.
  *---------------------------------------------------------------------------*/
SoundActor::SoundActor()
: m_pSoundArchivePlayer( NULL ),
  m_IsInitialized( false ),
  m_IsFinalized( true )
{
    // 何もしない
}
SoundActor::SoundActor( SoundArchivePlayer& soundArchivePlayer )
: m_pSoundArchivePlayer( NULL ),
  m_IsInitialized( false ),
  m_IsFinalized( true )
{
    Initialize( soundArchivePlayer );
}

/*---------------------------------------------------------------------------*
  Name:         ~SoundActor

  Description:  デストラクタ

  Arguments:    None.

  Returns:      None.
  *---------------------------------------------------------------------------*/
SoundActor::~SoundActor()
{
    Finalize();
}

void SoundActor::Initialize( SoundArchivePlayer& soundArchivePlayer )
{
    if ( m_IsInitialized ) return;

    m_pSoundArchivePlayer = &soundArchivePlayer;

    for ( int i = 0; i < ACTOR_PLAYER_COUNT; i++ )
    {
        const int playableSoundCount = ( i == 0 ) ? INT_MAX : 1;
        m_ActorPlayer[i].SetPlayableSoundCount( playableSoundCount );
    }

    m_IsInitialized = true;
    m_IsFinalized = false;
}
void SoundActor::Finalize()
{
    if ( m_IsFinalized ) return;

    for ( int i = 0; i < ACTOR_PLAYER_COUNT; i++ )
    {
        m_ActorPlayer[i].Finalize( this );
    }

    m_IsFinalized = true;
    m_IsInitialized = false;
    m_pSoundArchivePlayer = NULL;
    m_ActorParam.Reset();
}

/*---------------------------------------------------------------------------*
  Name:         StopAllSound

  Description:  全てのサウンドを停止する

  Arguments:    fadeFrames - フェードアウトフレーム数

  Returns:      None.
 *---------------------------------------------------------------------------*/
void SoundActor::StopAllSound( int fadeFrames )
{
    for ( int i = 0; i < ACTOR_PLAYER_COUNT; i++ )
    {
        m_ActorPlayer[ i ].StopAllSound( fadeFrames );
    }
}

/*---------------------------------------------------------------------------*
  Name:         PauseAllSound

  Description:  全てのサウンドを一時停止または再開する

  Arguments:    flag       - 一時停止か再開か
                fadeFrames - フェードフレーム数

  Returns:      None.
 *---------------------------------------------------------------------------*/
void SoundActor::PauseAllSound( bool flag, int fadeFrames )
{
    for ( int i = 0; i < ACTOR_PLAYER_COUNT; i++ )
    {
        m_ActorPlayer[ i ].PauseAllSound( flag, fadeFrames );
    }
}

/*---------------------------------------------------------------------------*
  Name:         GetPlayingSoundCount

  Description:  現在再生中のサウンド数を取得する

  Arguments:    actorPlayerId - アクタープレイヤーＩＤ

  Returns:      なし
 *---------------------------------------------------------------------------*/
int SoundActor::GetPlayingSoundCount( int actorPlayerId ) const
{
    NW_MINMAXLT_ASSERT( actorPlayerId, 0, ACTOR_PLAYER_COUNT );

    return m_ActorPlayer[ actorPlayerId ].GetPlayingSoundCount();
}

/*---------------------------------------------------------------------------*
  Name:         SetPlayableSoundCount

  Description:  サウンドの最大同時再生数を設定する

  Arguments:    actorPlayerId - アクタープレイヤーＩＤ
                count - サウンドの最大同時再生数

  Returns:      None.
  *---------------------------------------------------------------------------*/
void SoundActor::SetPlayableSoundCount( int actorPlayerId, int count )
{
    NW_MINMAXLT_ASSERT( actorPlayerId, 0, ACTOR_PLAYER_COUNT );

    m_ActorPlayer[ actorPlayerId ].SetPlayableSoundCount( count );
}

/*---------------------------------------------------------------------------*
  Name:         SetPlayableSoundCount

  Description:  サウンドの最大同時再生数を取得する

  Arguments:    actorPlayerId - アクタープレイヤーＩＤ

  Returns:      サウンドの最大同時再生数を返す
  *---------------------------------------------------------------------------*/
int SoundActor::GetPlayableSoundCount( int actorPlayerId ) const
{
    NW_MINMAXLT_ASSERT( actorPlayerId, 0, ACTOR_PLAYER_COUNT );

    return m_ActorPlayer[ actorPlayerId ].GetPlayableSoundCount();
}

/*---------------------------------------------------------------------------*
  Name:         SetupSound [virtual]

  Description:  再生の実装関数

  Arguments:    handle  - サウンドハンドル
                soundId - サウンドＩＤ
                startInfo - サウンド再生パラメータ
                setupArg - セットアップ引数

  Returns:      結果コード
 *---------------------------------------------------------------------------*/
SoundStartable::StartResult SoundActor::SetupSound(
    SoundHandle* handle,
    u32 soundId,
    const SoundStartable::StartInfo* startInfo,
    void* setupArg
)
{
    NW_NULL_ASSERT( setupArg );
    if ( m_IsInitialized == false )
    {
        return SoundStartable::StartResult(SoundStartable::StartResult::START_ERR_ACTOR_NOT_INITIALIZED);
    }

    const SetupInfo* setupInfo = reinterpret_cast<SetupInfo*>(setupArg);

    return m_pSoundArchivePlayer->detail_SetupSoundImpl(
        handle,
        soundId,
        NULL,
        this,
        setupInfo->holdFlag,
        startInfo
    );
}

// AmbientInfoが公開された際には上の関数に統合される予定
// AmbientInfoはStartInfoのメンバで渡される予定
SoundStartable::StartResult SoundActor::detail_SetupSoundWithAmbientInfo(
    SoundHandle* handle,
    u32 soundId,
    const SoundStartable::StartInfo* startInfo,
    internal::BasicSound::AmbientInfo* ambientInfo,
    void* setupArg
)
{
    NW_NULL_ASSERT( setupArg );
    if ( m_IsInitialized == false )
    {
        return SoundStartable::StartResult(SoundStartable::StartResult::START_ERR_ACTOR_NOT_INITIALIZED);
    }

    const SetupInfo* setupInfo = reinterpret_cast<SetupInfo*>(setupArg);

    return m_pSoundArchivePlayer->detail_SetupSoundImpl(
        handle,
        soundId,
        ambientInfo,
        this,
        setupInfo->holdFlag,
        startInfo
    );
}

/*---------------------------------------------------------------------------*
  Name:         detail_SetupSound [override]

  Description:  SoundStartableインターフェイスの実装

  Arguments:    handle  - サウンドハンドル
                soundId - サウンドＩＤ
                holdFlag - ホールドサウンドフラグ
                startInfo - サウンド再生パラメータ

  Returns:      結果コード
 *---------------------------------------------------------------------------*/
SoundStartable::StartResult SoundActor::detail_SetupSound(
    SoundHandle* handle,
    u32 soundId,
    bool holdFlag,
    const SoundStartable::StartInfo* startInfo
)
{
    SetupInfo setupInfo;
    setupInfo.holdFlag = holdFlag;

    return SetupSound(
        handle,
        soundId,
        startInfo,
        &setupInfo
    );
}

/*---------------------------------------------------------------------------*
  Name:         detail_GetItemId [override]

  Description:  ラベル文字列からサウンドＩＤへの変換

  Arguments:    label - ラベル文字列

  Returns:      サウンドＩＤを返す
  *---------------------------------------------------------------------------*/
SoundArchive::ItemId SoundActor::detail_GetItemId( const char* pString )
{
    if ( m_IsInitialized == false )
    {
        return SoundArchive::INVALID_ID;
    }
    return m_pSoundArchivePlayer->detail_GetItemId( pString );
}

} // namespace nw::snd
} // namespace nw

