﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nw/snd/snd_MultiVoice.h>

#include <cstring>              // std::memcpy
#include <nw/snd/snd_Util.h>
#include <nw/snd/snd_Config.h>
#include <nw/snd/snd_MultiVoiceManager.h>
#include <nw/snd/snd_HardwareManager.h>

// #define NW_SND_DUMP_DECODE_DSPADPCM
// ↑↑↑ 有効な場合、「途中再生に必要な DSP ADPCM のデコード」の結果を、
//        CAFE_SAVE_DIR/_cafe_decodeDspAdpcm.raw に書き出します。

#if defined(NW_SND_DUMP_DECODE_DSPADPCM)
static FSClient   s_fsClient;
static FSCmdBlock s_fsCmdBlock;
#endif

#if defined( NW_PLATFORM_WIN32 )
using namespace nw::internal::winext;
#elif defined( NW_USE_NINTENDO_SDK )
// TODO: nn_audio
using namespace nw::internal::winext;
#endif

namespace nw {
namespace snd {
namespace internal {
namespace driver {

namespace
{

u32 DspAdpcmFrameToNibbleAddress( u32 frame )
{
    u32 mod = frame % 14;
    u32 nibbleAddress = ( frame / 14 ) * 16;
    if ( mod != 0 )
    {
        nibbleAddress += ( mod + 2 );
    }

    return nibbleAddress;
}

inline void PanCurveToPanInfo( Util::PanInfo& panInfo, PanCurve curve )
{
    switch ( curve )
    {
    case PAN_CURVE_SQRT:
        panInfo.curve = Util::PAN_CURVE_SQRT;
        break;
    case PAN_CURVE_SQRT_0DB:
        panInfo.curve = Util::PAN_CURVE_SQRT;
        panInfo.centerZeroFlag = true;
        break;
    case PAN_CURVE_SQRT_0DB_CLAMP:
        panInfo.curve = Util::PAN_CURVE_SQRT;
        panInfo.centerZeroFlag = true;
        panInfo.zeroClampFlag = true;
        break;
    case PAN_CURVE_SINCOS:
        panInfo.curve = Util::PAN_CURVE_SINCOS;
        break;
    case PAN_CURVE_SINCOS_0DB:
        panInfo.curve = Util::PAN_CURVE_SINCOS;
        panInfo.centerZeroFlag = true;
        break;
    case PAN_CURVE_SINCOS_0DB_CLAMP:
        panInfo.curve = Util::PAN_CURVE_SINCOS;
        panInfo.centerZeroFlag = true;
        panInfo.zeroClampFlag = true;
        break;
    case PAN_CURVE_LINEAR:
        panInfo.curve = Util::PAN_CURVE_LINEAR;
        break;
    case PAN_CURVE_LINEAR_0DB:
        panInfo.curve = Util::PAN_CURVE_LINEAR;
        panInfo.centerZeroFlag = true;
        break;
    case PAN_CURVE_LINEAR_0DB_CLAMP:
        panInfo.curve = Util::PAN_CURVE_LINEAR;
        panInfo.centerZeroFlag = true;
        panInfo.zeroClampFlag = true;
        break;
    default:
        panInfo.curve = Util::PAN_CURVE_SQRT;
        break;
    }
}

// モノラル・バランス or デュアル用パン計算
inline void CalcPanForMono( f32* left, f32* right, const Util::PanInfo& panInfo )
{
    *left = *right = Util::CalcPanRatio( MultiVoice::PAN_CENTER, panInfo, OUTPUT_MODE_MONO );
}

// ステレオ or サラウンド・バランス用パン計算
inline void CalcBarancePanForStereo(
        f32* left, f32* right, const f32& pan, int channelIndex, const Util::PanInfo& panInfo, OutputMode mode )
{
    if ( channelIndex == 0 )
    {
        *left = Util::CalcPanRatio( pan, panInfo, mode );
        *right = 0.0f;
    }
    else if ( channelIndex == 1 )
    {
        *left = 0.0f;
        *right = Util::CalcPanRatio( MultiVoice::PAN_CENTER - pan, panInfo, mode );
    }
}

// ステレオ or サラウンド用デュアルパン計算
inline void CalcDualPanForStereo(
        f32* left, f32* right, const f32& pan, const Util::PanInfo& panInfo, OutputMode mode )
{
    *left = Util::CalcPanRatio( pan, panInfo, mode );
    *right = Util::CalcPanRatio( MultiVoice::PAN_CENTER - pan, panInfo, mode );
}

// モノラル or ステレオ用サラウンドパン計算
inline void CalcSurroundPanForMono(
        f32* front, f32* rear, const Util::PanInfo& panInfo )
{
    *front = Util::CalcSurroundPanRatio( MultiVoice::SPAN_FRONT, panInfo );
    *rear = Util::CalcSurroundPanRatio( MultiVoice::SPAN_REAR, panInfo );
}

// サラウンド用サラウンドパン計算
inline void CalcSurroundPanForSurround(
        f32* front, f32* rear, const f32& span, const Util::PanInfo& panInfo )
{
    *front = Util::CalcSurroundPanRatio( span, panInfo );
    *rear = Util::CalcSurroundPanRatio( MultiVoice::SPAN_REAR - span, panInfo );
}

} // anonymous namespace


// ------------------------------------------------------------------------
// 定数
const f32 MultiVoice::VOLUME_MIN           = 0.0f;
const f32 MultiVoice::VOLUME_DEFAULT       = 1.0f;
const f32 MultiVoice::VOLUME_MAX           = 2.0f;

const f32 MultiVoice::PAN_LEFT             = -1.0f;
const f32 MultiVoice::PAN_CENTER           = 0.0f;
const f32 MultiVoice::PAN_RIGHT            = 1.0f;

const f32 MultiVoice::SPAN_FRONT           = 0.0f;
const f32 MultiVoice::SPAN_CENTER          = 1.0f;
const f32 MultiVoice::SPAN_REAR            = 2.0f;

const f32 MultiVoice::CUTOFF_FREQ_MIN      = 0.0f;
const f32 MultiVoice::CUTOFF_FREQ_MAX      = 1.0f;

const f32 MultiVoice::BIQUAD_VALUE_MIN     = 0.0f;
const f32 MultiVoice::BIQUAD_VALUE_MAX     = 1.0f;

const f32 MultiVoice::SEND_MIN             = 0.0f;
const f32 MultiVoice::SEND_MAX             = 1.0f;

/* ========================================================================
        inline function
   ======================================================================== */

inline u16 CalcMixVolume( f32 volume )
{
    if ( volume <= 0.f ) return 0UL;

    return static_cast<u16>(
        ut::Min(
            0x0000ffffUL,
            static_cast<unsigned long>( volume * 0x8000 )
        )
    );
}


MultiVoice::MultiVoice()
: m_Callback( NULL ),
  m_IsActive( false ),
  m_IsStart( false ),
  m_IsStarted( false ),
  m_IsPause( false ),
  m_SyncFlag( 0 ),
  m_ChannelCount( 0 )
{
}

MultiVoice::~MultiVoice()
{
    for ( int channelIndex = 0; channelIndex < m_ChannelCount; channelIndex++ )
    {
        m_Voice[ channelIndex ].Free();
    }
}

bool MultiVoice::Alloc(
    int channelCount,
    int priority,
    MultiVoice::VoiceCallback callback,
    void* callbackData,
    VoiceRendererType mode
)
{
    NW_MINMAX_ASSERT( channelCount, 1, WAVE_CHANNEL_MAX );
    channelCount = ut::Clamp( channelCount, 1, static_cast<int>(WAVE_CHANNEL_MAX) );
    m_ChannelCount = 0;

    NW_ASSERT( ! m_IsActive );

    // ボイス取得
    int allocCount = 0;
    for ( int i = 0 ; i < channelCount; i++ )
    {
        if ( ! m_Voice[i].AllocVoice(priority, mode) )
        {
            break;
        }
        allocCount++;
    }
    if ( allocCount != channelCount )
    {
        for ( int i = 0 ; i < allocCount; i++ )
        {
            m_Voice[i].Free();
        }
        return false;
    }

    m_ChannelCount = channelCount;

    // パラメータ初期化
    InitParam( callback, callbackData );
    m_IsActive = true;

    return true;
}


void MultiVoice::Free()
{
    if ( ! m_IsActive ) return;

    for ( int channelIndex = 0; channelIndex < m_ChannelCount; channelIndex++ )
    {
            m_Voice[ channelIndex ].Free();
    }
    m_ChannelCount = 0;

    MultiVoiceManager::GetInstance().FreeVoice( this );

    m_IsActive = false;
}

void MultiVoice::Start()
{
    m_IsStart = true;
    m_IsPause = false;

    m_SyncFlag |= UPDATE_START;
}

void MultiVoice::Stop()
{
    if ( m_IsStarted )
    {
        StopAllSdkVoice();
        m_IsStarted = false;
    }
    m_IsPausing = false;
    m_IsPause = false;
    m_IsStart = false;
}

void MultiVoice::UpdateVoiceStatus()
{
    if ( ! m_IsActive )
    {
        return;
    }

    // 再生停止チェック
    if ( ! m_IsStarted )
    {
        return;
    }

#ifdef NW_SND_CONFIG_ENABLE_VOICE_COMMAND
    for ( int channelIndex = 0; channelIndex < m_ChannelCount; channelIndex++ )
    {
        m_Voice[channelIndex].UpdateVoiceStatus();
    }
#endif

    if ( IsPlayFinished() )
    {
        // 波形停止
        if ( m_Callback != NULL )
        {
            m_Callback( this, CALLBACK_STATUS_FINISH_WAVE, m_pCallbackData );
        }
        m_IsStarted = false;
        m_IsStart = false;
    }
    else
    {
        // ボイスドロップチェック
        bool dropFlag = false;
        for ( int ch = 0 ; ch < m_ChannelCount; ch++ )
        {
            if ( ! m_Voice[ch].IsAvailable() ) {
                dropFlag = true;
                break;
            }
        }
        if ( dropFlag )
        {
            for ( int ch = 0 ; ch < m_ChannelCount; ch++ )
            {
                m_Voice[ch].Free();
            }
            m_ChannelCount = 0;
            m_IsActive = false;

            // TODO: FreeVoiceのタイミングは適切か？
            MultiVoiceManager::GetInstance().FreeVoice( this );
            if ( m_Callback != NULL )
            {
                m_Callback( this, CALLBACK_STATUS_DROP_DSP, m_pCallbackData );
            }

            m_IsStarted = false;
            m_IsStart = false;
        }
    }
}

void MultiVoice::Pause( bool flag )
{
    if ( m_IsPause == flag ) return;

    m_IsPause = flag;
    m_SyncFlag |= UPDATE_PAUSE;
}

void MultiVoice::Calc()
{
    if ( m_IsStart )
    {
        // 再生レート
        if ( m_SyncFlag & UPDATE_SRC )
        {
            CalcSrc( false );
            m_SyncFlag &= ~UPDATE_SRC;
        }

        // ボリューム
        if ( m_SyncFlag & UPDATE_VE )
        {
            CalcVe();
            m_SyncFlag &= ~UPDATE_VE;
        }

        // ミックス
        if ( m_SyncFlag & UPDATE_MIX )
        {
            CalcMix();
            m_SyncFlag &= ~UPDATE_MIX;
        }

        // ローパスフィルタ
        if ( m_SyncFlag & UPDATE_LPF )
        {
            CalcLpf();
            m_SyncFlag &= ~UPDATE_LPF;
        }

        // Biquadフィルタ
        if ( m_SyncFlag & UPDATE_BIQUAD )
        {
            CalcBiquadFilter();
            m_SyncFlag &= ~UPDATE_BIQUAD;
        }

        // リモコンフィルタ
        if ( m_SyncFlag & UPDATE_REMOTE_FILTER )
        {
            CalcRemoteFilter();
            m_SyncFlag &= ~UPDATE_REMOTE_FILTER;
        }
    }
}

void MultiVoice::Update()
{
    if ( ! m_IsActive ) return;

    // ボイスの再生・停止処理をあとでまとめて行うためのフラグ
    enum { NONE, RUN, STOP, PAUSE } runFlag = NONE;

    // 再生開始
    if ( m_SyncFlag & UPDATE_START )
    {
        if ( m_IsStart && !m_IsStarted )
        {
            CalcSrc( true );
            CalcMix();
            CalcVe();   // マスターボリューム反映
            runFlag = RUN;
            m_IsStarted = true;

            m_SyncFlag &= ~UPDATE_START;
            m_SyncFlag &= ~UPDATE_SRC;
            m_SyncFlag &= ~UPDATE_MIX;
        }
    }

    if ( m_IsStarted )
    {
        // 一時停止
        if ( m_SyncFlag & UPDATE_PAUSE )
        {
            if ( m_IsStart )
            {
                if ( m_IsPause )
                {
                    runFlag = PAUSE;
                    m_IsPausing = true;
                }
                else
                {
                    runFlag = RUN;
                    m_IsPausing = false;
                }

                m_SyncFlag &= ~UPDATE_PAUSE;
            }
        }
    }

    for ( int i = 0; i < m_ChannelCount; i++ )
    {
        m_Voice[i].UpdateParam();
    }

    // ボイスの再生・停止処理をまとめて行う
    // 同一フレームで、まだ再生されていないボイスに対して
    // 再生→停止の処理を行うとデポップが発生するため
    switch ( runFlag )
    {
    case RUN:
        RunAllSdkVoice();
        break;
    case STOP:
        StopAllSdkVoice();
        break;
    case PAUSE:
        PauseAllSdkVoice();
    case NONE:
    default:
        break;
    }
}

void MultiVoice::SetSampleFormat( SampleFormat format )
{
    m_Format = format;
    for ( int i = 0; i < m_ChannelCount; i++ )
    {
        m_Voice[i].SetSampleFormat( m_Format );
    }
}

void MultiVoice::SetSampleRate( int sampleRate )
{
    for ( int i = 0; i < m_ChannelCount; i++ )
    {
        m_Voice[i].SetSampleRate( sampleRate );
    }
}

void MultiVoice::SetVolume( f32 volume )
{
    if ( volume < VOLUME_MIN ) volume = VOLUME_MIN;

    if ( volume != m_Volume )
    {
        m_Volume = volume;
        m_SyncFlag |= UPDATE_VE;
    }
}

void MultiVoice::SetPitch( f32 pitch )
{
    if ( pitch != m_Pitch )
    {
        m_Pitch = pitch;
        m_SyncFlag |= UPDATE_SRC;
    }
}

void MultiVoice::SetPanMode( PanMode panMode )
{
    if ( panMode != m_PanMode )
    {
        m_PanMode = panMode;
        m_SyncFlag |= UPDATE_MIX;
    }
}

void MultiVoice::SetPanCurve( PanCurve panCurve )
{
    if ( panCurve != m_PanCurve )
    {
        m_PanCurve = panCurve;
        m_SyncFlag |= UPDATE_MIX;
    }
}


void MultiVoice::SetLpfFreq( f32 lpfFreq )
{
    if ( lpfFreq != m_LpfFreq )
    {
        m_LpfFreq = lpfFreq;
        m_SyncFlag |= UPDATE_LPF;
    }
}

void MultiVoice::SetBiquadFilter( int type, f32 value )
{
    // ここに来るまでに、INHERITの処理は済んでいるはずなので、BIQUAD_FILTER_TYPE_DATA_MINが最小になるはずです。
    NW_MINMAX_ASSERT( type, BIQUAD_FILTER_TYPE_DATA_MIN, BIQUAD_FILTER_TYPE_MAX );
    value = ut::Clamp( value, BIQUAD_VALUE_MIN, BIQUAD_VALUE_MAX );

    bool isUpdate = false;

    if ( type != m_BiquadType )
    {
        m_BiquadType = static_cast<u8>( type );
        isUpdate = true;
    }

    if ( value != m_BiquadValue )
    {
        m_BiquadValue = value;
        isUpdate = true;
    }

    if ( isUpdate )
    {
        m_SyncFlag |= UPDATE_BIQUAD;
    }
}

void MultiVoice::SetPriority( int priority )
{
    m_Priority = priority;
    MultiVoiceManager::GetInstance().ChangeVoicePriority( this );

    for ( int channelIndex = 0; channelIndex < m_ChannelCount; channelIndex++ )
    {
        m_Voice[channelIndex].SetPriority( m_Priority );
    }
}

void MultiVoice::SetOutputLine( u32 lineFlag )
{
    if ( lineFlag != m_OutputLineFlag )
    {
        m_OutputLineFlag = lineFlag;
        m_SyncFlag |= UPDATE_MIX;
    }
}

void MultiVoice::SetOutputParamImpl( const OutputParam& in, OutputParam& out )
{
    // volume
    {
        f32 volume = in.volume;
        if ( volume < VOLUME_MIN )
        {
            volume = VOLUME_MIN;
        }
        if ( volume != out.volume )
        {
            out.volume = volume;
            m_SyncFlag |= UPDATE_MIX;
        }
    }

    // mixMode
    if ( in.mixMode != out.mixMode )
    {
        out.mixMode = in.mixMode;
        m_SyncFlag |= UPDATE_MIX;
    }

    // mixParameter
    for ( int i = 0; i < WAVE_CHANNEL_MAX; i++ )
    {
        for ( int j = 0; j < CHANNEL_INDEX_NUM; j++ )
        {
            if ( in.mixParameter[i].ch[j] != out.mixParameter[i].ch[j] )
            {
                out.mixParameter[i].ch[j] = in.mixParameter[i].ch[j];
                m_SyncFlag |= UPDATE_MIX;
            }
        }
    }

    // pan
    if ( in.pan != out.pan )
    {
        out.pan = in.pan;
        m_SyncFlag |= UPDATE_MIX;
    }

    // span
    if ( in.span != out.span )
    {
        out.span = in.span;
        m_SyncFlag |= UPDATE_MIX;
    }

    // mainSend
    {
        f32 send = in.mainSend;
        if ( send != out.mainSend )
        {
            out.mainSend = send;
            m_SyncFlag |= UPDATE_MIX;
        }
    }

    // fxSend
    for ( int i = 0; i < AUX_BUS_NUM; i++ )
    {
        f32 send = in.fxSend[i];
        if ( send != out.fxSend[i] )
        {
            out.fxSend[i] = send;
            m_SyncFlag |= UPDATE_MIX;
        }
    }
}

void MultiVoice::SetTvParam( const OutputParam& param )
{
    SetOutputParamImpl( param, m_TvParam );
}

void MultiVoice::SetDrcParam( u32 drcIndex, const OutputParam& param )
{
    NW_ASSERT_MAXLT( drcIndex, DRC_OUT_COUNT );
    SetOutputParamImpl( param, m_DrcParam[drcIndex] );
}

void MultiVoice::SetRemoteParam( u32 remoteIndex, const RemoteOutputParam& in )
{
    NW_ASSERT_MAXLT( remoteIndex, REMOTE_OUT_COUNT );
    RemoteOutputParam& out = m_RemoteParam[remoteIndex];

    // volume
    {
        f32 volume = in.volume;
        if ( volume < VOLUME_MIN )
        {
            volume = VOLUME_MIN;
        }
        if ( volume != out.volume )
        {
            out.volume = volume;
            m_SyncFlag |= UPDATE_MIX;
        }
    }

    // mainSend
    {
        f32 send = in.mainSend;
        if ( send != out.mainSend )
        {
            out.mainSend = send;
            m_SyncFlag |= UPDATE_MIX;
        }
    }

    // fxSend
    {
        f32 send = in.fxSend;
        if ( send != out.fxSend )
        {
            out.fxSend = send;
            m_SyncFlag |= UPDATE_MIX;
        }
    }
}




void MultiVoice::InitParam(
    MultiVoice::VoiceCallback callback,
    void* callbackData
)
{
    // Init Flag
    m_SyncFlag  = 0;
    m_IsPause   = false;
    m_IsPausing = false;
    m_IsStart   = false;
    m_IsStarted = false;

    // Init Param
    m_pLastWaveBuffer = NULL;
    m_Callback        = callback;
    m_pCallbackData   = callbackData;
    m_Volume          = VOLUME_DEFAULT;
    m_Pitch           = 1.0f;
    m_PanMode         = PAN_MODE_DUAL;
    m_PanCurve        = PAN_CURVE_SQRT;
    m_LpfFreq         = 1.0f;
    m_BiquadType      = BIQUAD_FILTER_TYPE_NONE;
    m_BiquadValue     = 0.0f;
    m_RemoteFilter    = 0;
    m_IsEnableFrontBypass = false;

    m_TvParam.Initialize();
    for ( int i = 0; i < DRC_OUT_COUNT; i++ )
    {
        m_DrcParam[i].Initialize();
    }
    for ( int i = 0; i < REMOTE_OUT_COUNT; i++ )
    {
        m_RemoteParam[i].Initialize();
    }
}

void MultiVoice::CalcSrc( bool initialUpdate )
{
    NW_UNUSED_VARIABLE( initialUpdate );

    f32 ratio = m_Pitch;

    for ( int channelIndex = 0; channelIndex < m_ChannelCount; channelIndex++ )
    {
        m_Voice[channelIndex].SetPitch( ratio );
    }
}

void MultiVoice::CalcVe()
{
    f32 volume = m_Volume;
    volume *= HardwareManager::GetInstance().GetOutputVolume();

    for ( int channelIndex = 0; channelIndex < m_ChannelCount; channelIndex++ )
    {
        m_Voice[channelIndex].SetVolume( volume );
    }
}

void MultiVoice::CalcMix()
{
    for ( int channelIndex = 0; channelIndex < m_ChannelCount; channelIndex++ )
    {
        Voice& voice = m_Voice[ channelIndex ];

        PreMixVolume preMix;
        // TV
        {
            CalcPreMixVolume( &preMix, m_TvParam, channelIndex, OUTPUT_DEVICE_MAIN );
            OutputMix tvMix;
            CalcTvMix( &tvMix, preMix );
            voice.SetTvMix( tvMix );
        }

        // Remote
        {
            RemoteOutputMix remoteMix;
            CalcRemoteMix( &remoteMix, preMix );
            voice.SetRemoteMix( remoteMix );
        }

        // DRC
        for ( u32 i = 0; i < DRC_OUT_COUNT; i++ )
        {
            OutputMix drcOutMix;
            PreMixVolume preDrcMix;

            CalcPreMixVolume( &preDrcMix, m_DrcParam[i], channelIndex, OUTPUT_DEVICE_DRC );
            CalcDrcMix( i, &drcOutMix, preDrcMix );
            voice.SetDrcMix( i, drcOutMix );
        }
    }
}

void MultiVoice::CalcLpf()
{
    u16 freq = Util::CalcLpfFreq( m_LpfFreq );

    for ( int channelIndex = 0; channelIndex < m_ChannelCount; channelIndex++ )
    {
        Voice& voice = m_Voice[channelIndex];

        if ( freq >= 16000 )
        {
            voice.SetMonoFilter( false );
        }
        else
        {
            voice.SetMonoFilter( true, freq );
        }

    }
}

void MultiVoice::CalcBiquadFilter()
{
    for ( int channelIndex = 0; channelIndex < m_ChannelCount; channelIndex++ )
    {
        Voice& voice = m_Voice[channelIndex];

        const BiquadFilterCallback* cb =
            HardwareManager::GetInstance().GetBiquadFilterCallback( m_BiquadType );
        if ( cb == NULL )
        {
            voice.SetBiquadFilter( false );
        }
        else
        {
            if ( m_BiquadValue <= 0.0f )
            {
                voice.SetBiquadFilter( false );
            }
            else
            {
                BiquadFilterCallback::Coefficients coef;
                cb->GetCoefficients( m_BiquadType, m_BiquadValue, &coef );
                voice.SetBiquadFilter( true, &coef );
            }
        }
    }
}

void MultiVoice::CalcRemoteFilter()
{
    for ( int channelIndex = 0; channelIndex < m_ChannelCount; channelIndex++ )
    {
        Voice& voice = m_Voice[channelIndex];
        if ( m_RemoteFilter > 0 )
        {
            voice.SetRemoteFilter( true, m_RemoteFilter );
        }
        else
        {
            voice.SetRemoteFilter( false );
        }
    }
}

void MultiVoice::CalcPreMixVolume(
        PreMixVolume* mix,
        const OutputParam& param,
        int channelIndex,
        OutputDevice device )
{
    NW_NULL_ASSERT( mix );

    // NW OutputMode に応じたミックスパラメータの計算
    const OutputMode nwMode = HardwareManager::GetInstance().GetOutputMode(device);
    f32 fL = 0.0f; f32 fR = 0.0f; f32 rL = 0.0f; f32 rR = 0.0f;
    f32 fC = 0.0f; f32 lfe = 0.0f; f32 lrMixed = 0.0f;
    if ( param.mixMode == MIX_MODE_PAN )
    {
        f32 left, right, front, rear;
        left = right = front = rear = 0.0f;

        Util::PanInfo panInfo;
        PanCurveToPanInfo( panInfo, m_PanCurve );


        // left, right の計算
        switch ( nwMode )
        {
        case OUTPUT_MODE_MONO:
            CalcPanForMono( &left, &right, panInfo );
            break;
        case OUTPUT_MODE_STEREO:
        case OUTPUT_MODE_SURROUND:
            if ( ( m_ChannelCount > 1 ) && ( m_PanMode == PAN_MODE_BALANCE ) )
            {
                CalcBarancePanForStereo(
                        &left, &right, param.pan, channelIndex, panInfo, nwMode );
            }
            else
            {
                f32 voicePan = param.pan;
                if ( m_ChannelCount == 2 )
                {
                    if ( channelIndex == 0 ) { voicePan -= 1.0f; }
                    if ( channelIndex == 1 ) { voicePan += 1.0f; }
                }
                CalcDualPanForStereo( &left, &right, voicePan, panInfo, nwMode );
            }
            break;
        default:
            NW_ASSERTMSG(false, "[%s] nwMode(%d) is invalid\n", __FUNCTION__, nwMode);
            break;
        }

        // front, rear の計算
        switch ( nwMode )
        {
        case OUTPUT_MODE_MONO:
        case OUTPUT_MODE_STEREO:
            CalcSurroundPanForMono( &front, &rear, panInfo );
            break;
        case OUTPUT_MODE_SURROUND:
            CalcSurroundPanForSurround( &front, &rear, param.span, panInfo );
            break;
        default:
            break;
        }

        fL = front * left;
        fR = front * right;
        rL = rear  * left;
        rR = rear  * right;
        lrMixed = ( left + right ) * 0.5f;
    }
    else
    {
        fL  = param.mixParameter[channelIndex].ch[CHANNEL_INDEX_FRONT_LEFT];
        fR  = param.mixParameter[channelIndex].ch[CHANNEL_INDEX_FRONT_RIGHT];
        rL  = param.mixParameter[channelIndex].ch[CHANNEL_INDEX_REAR_LEFT];
        rR  = param.mixParameter[channelIndex].ch[CHANNEL_INDEX_REAR_RIGHT];
        fC  = param.mixParameter[channelIndex].ch[CHANNEL_INDEX_FRONT_CENTER];
        lfe = param.mixParameter[channelIndex].ch[CHANNEL_INDEX_LFE];
        lrMixed = ( fL + fR + rL + rR ) * 0.25f;
    }
    // NW_LOG("[0] fL(%5.2f) fR(%5.2f) rL(%5.2f) rR(%5.2f)\n", fL, fR, rL, rR);

    // 本体メニュー設定に応じたダウンミックス
    // -----------------------------------------------------
    // Cafe MENU       |            NW mode
    // mode            | MONO   STEREO      SURROUND
    // -----------------------------------------------------
    // MONO       fL   |  M     L+R/2    (fL+rL + fR+rR)/2
    //            fR   |  M     L+R/2    (fL+rL + fR+rR)/2
    //            rL   |  0      0            0
    //            rR   |  0      0            0
    //            C    |  0      0            0
    //            LFE  |  0      0            0
    // -----------------------------------------------------
    // STEREO     fL   |  M      L          fL+rL
    //            fR   |  M      R          fR+rR
    //            rL   |  0      0            0
    //            rR   |  0      0            0
    //            C    |  0      0            0
    //            LFE  |  0      0            0
    // -----------------------------------------------------
    // SURROUND   fL   |  M      L           fL
    //            fR   |  M      R           fR
    //            rL   |  0      0           rL
    //            rR   |  0      0           rR
    //            C    |  0      0            0
    //            LFE  |  0      0            0
    // -----------------------------------------------------
    const OutputMode endUserMode =
        HardwareManager::GetInstance().GetEndUserOutputMode(device);
    switch (endUserMode)
    {
    case OUTPUT_MODE_MONO:
        switch (nwMode)
        {
        // case OUTPUT_MODE_MONO:
        //     // do nothing
        //     break;
        case OUTPUT_MODE_STEREO:
            {
                f32 mono = (fL + fR) * 0.5f;
                fL = fR = mono;
            }
            break;
        case OUTPUT_MODE_SURROUND:
            {
                f32 mono = ((fL + rL) + (fR + rR)) * 0.5f;
                fL = fR = mono;
                rL = rR = 0.0f;
            }
            break;
        default:
            break;
        }
        break;
    case OUTPUT_MODE_STEREO:
        switch (nwMode)
        {
        // case OUTPUT_MODE_MONO:
        //     // do nothing
        //     break;
        // case OUTPUT_MODE_STEREO:
        //     // do nothing
        //     break;
        case OUTPUT_MODE_SURROUND:
            {
                f32 left = fL + rL;
                f32 right = fR + rR;
                fL = left;
                fR = right;
                rL = rR = 0.0f;
            }
            break;
        default:
            break;
        }
        break;
    case OUTPUT_MODE_SURROUND:
        // switch (nwMode)
        // {
        // case OUTPUT_MODE_MONO:
        //     // do nothing
        //     break;
        // case OUTPUT_MODE_STEREO:
        //     // do nothing
        //     break;
        // case OUTPUT_MODE_SURROUND:
        //     // do nothing
        //     break;
        // }
        break;
    default:
        NW_ASSERTMSG(false, "[%s] endUserMode(%d) is invalid\n", __FUNCTION__, endUserMode);
        break;
    }
    // NW_LOG("[1] fL(%5.2f) fR(%5.2f) rL(%5.2f) rR(%5.2f)\n", fL, fR, rL, rR);

    // 値の格納
    {
        mix->volume[CHANNEL_INDEX_FRONT_LEFT]   = fL;
        mix->volume[CHANNEL_INDEX_FRONT_RIGHT]  = fR;
        mix->volume[CHANNEL_INDEX_REAR_LEFT]    = rL;
        mix->volume[CHANNEL_INDEX_REAR_RIGHT]   = rR;
        mix->volume[CHANNEL_INDEX_FRONT_CENTER] = fC;
        mix->volume[CHANNEL_INDEX_LFE]          = lfe;
        mix->lrMixedVolume                     = lrMixed;
    }
}

void MultiVoice::CalcTvMix( OutputMix* mix, const PreMixVolume& preMix )
{
    CalcMixImpl( mix, OUTPUT_DEVICE_INDEX_MAIN, m_TvParam, preMix );
}

void MultiVoice::CalcDrcMix( u32 drcIndex, OutputMix* mix, const PreMixVolume& preMix )
{
    NW_ASSERT_MAXLT( drcIndex, DRC_OUT_COUNT );
    CalcMixImpl( mix, (OUTPUT_DEVICE_INDEX_DRC0 + drcIndex), m_DrcParam[drcIndex], preMix );
}

void MultiVoice::CalcMixImpl(
        OutputMix* mix, u32 outputDeviceIndex, const OutputParam& param, const PreMixVolume& pre )
{
    f32 main = 0.0f;
    f32 aux[AUX_BUS_NUM] = { 0.0f };

    u32 outputDeviceFlag = 0;
    for ( int i = 0; i < OUTPUT_LINE_INDEX_MAX; i++ )
    {
        if ( ( m_OutputLineFlag >> i ) & 1 )
        {
            outputDeviceFlag |= HardwareManager::GetInstance().GetOutputDeviceFlag( i );
        }
    }

    if ( outputDeviceFlag & ( 1 << outputDeviceIndex ) )
    {
        main = ut::Clamp( param.mainSend + 1.0f, 0.0f, 1.0f );
        for ( int i = 0; i < AUX_BUS_NUM; i++ )
        {
            aux[i] = ut::Clamp( param.fxSend[i], 0.0f, 1.0f );
        }
    }

    for ( int i = 0; i < CHANNEL_INDEX_NUM;i ++ )
    {
        f32 channelVolume = pre.volume[i] * param.volume;
        mix->mainBus[i] = main * channelVolume;
        for ( int j = 0; j < AUX_BUS_NUM; j++ )
        {
            mix->auxBus[j][i] = aux[j] * channelVolume;
        }
    }
}

void MultiVoice::CalcRemoteMix( RemoteOutputMix* mix, const PreMixVolume& preMix )
{
    mix->enable = false;

    f32 lrMixed = preMix.lrMixedVolume;

    u32 outputDeviceFlag = 0;
    for ( int i = 0; i < OUTPUT_LINE_INDEX_MAX; i++ )
    {
        if ( ( m_OutputLineFlag >> i ) & 1 )
        {
            outputDeviceFlag |= HardwareManager::GetInstance().GetOutputDeviceFlag( i );
        }
    }

    for ( int i = 0; i < REMOTE_OUT_COUNT; i++ )
    {
        if ( outputDeviceFlag & ( 1 << OUTPUT_DEVICE_INDEX_REMOTE0 << i ) )
        {
            f32 volume = lrMixed * m_RemoteParam[i].volume;
            mix->mainBus[i] = ut::Clamp( m_RemoteParam[i].mainSend + 1.0f, 0.0f, 1.0f ) * volume;
            mix->auxBus[i] = ut::Clamp( m_RemoteParam[i].fxSend, 0.0f, 1.0f ) * volume;
            mix->enable = true;
        }
    }
}

void MultiVoice::RunAllSdkVoice()
{
    for ( int channelIndex = 0; channelIndex < m_ChannelCount; channelIndex++ )
    {
        m_Voice[channelIndex].SetState( VOICE_STATE_PLAY );
    }
}

void MultiVoice::StopAllSdkVoice()
{
    for ( int channelIndex = 0; channelIndex < m_ChannelCount; channelIndex++ )
    {
        m_Voice[channelIndex].SetState( VOICE_STATE_STOP );
    }
}

void MultiVoice::PauseAllSdkVoice()
{
    for ( int channelIndex = 0; channelIndex < m_ChannelCount; channelIndex++ )
    {
        m_Voice[channelIndex].SetState( VOICE_STATE_PAUSE );
    }
}

const Voice& MultiVoice::detail_GetSdkVoice( int channelIndex ) const
{
    NW_ASSERT( channelIndex < WAVE_CHANNEL_MAX );
    return m_Voice[ channelIndex ];
}

u32 MultiVoice::GetCurrentPlayingSample() const
{
    if ( ! IsActive() ) return 0;
    return m_Voice[0].GetPlayPosition();
}


nw::snd::SampleFormat MultiVoice::GetFormat() const
{
    NW_ASSERT( IsActive() );
    return m_Format;
}

bool MultiVoice::IsRun() const
{
    if ( IsActive() )
    {
        return ( m_Voice[0].GetState() == VOICE_STATE_PLAY );
    }
    return false;
}


bool MultiVoice::IsPlayFinished() const
{
    if ( ! IsActive() ) return false;
    if ( m_pLastWaveBuffer == NULL ) return false;
    if ( m_pLastWaveBuffer->status == WaveBuffer::STATUS_WAIT || m_pLastWaveBuffer->status == WaveBuffer::STATUS_PLAY ) return false;

    return true;
}

void MultiVoice::SetFrontBypass( bool isFrontBypass )
{
    for ( int channelIndex = 0; channelIndex < m_ChannelCount; channelIndex++ )
    {
        m_Voice[channelIndex].SetFrontBypass( isFrontBypass );
    }

    m_IsEnableFrontBypass = isFrontBypass;
}

void MultiVoice::SetInterpolationType( u8 interpolationType )
{
    for ( int channelIndex = 0; channelIndex < m_ChannelCount; channelIndex++ )
    {
        m_Voice[channelIndex].SetInterpolationType( interpolationType );
    }
}

void MultiVoice::SetRemoteFilter( u8 filter )
{
    filter = ut::Clamp( filter, REMOTE_FILTER_MIN, REMOTE_FILTER_MAX );

    if ( filter != m_RemoteFilter )
    {
        m_RemoteFilter = filter;
        m_SyncFlag |= UPDATE_REMOTE_FILTER;
    }
}

void MultiVoice::AppendWaveBuffer(
    int channelIndex,
    WaveBuffer* pBuffer,
    bool lastFlag
)
{
    NW_NULL_ASSERT( pBuffer );
    NW_MINMAXLT_ASSERT( channelIndex, 0, WAVE_CHANNEL_MAX );

    m_Voice[channelIndex].AppendWaveBuffer( pBuffer );

#if 0 // DEBUG: ADPCM コンテキストの確認
    if ( pBuffer->pAdpcmContext )
    {
        NW_LOG("MultiVoice[%p]  bufferPS(%04x) ps(%04x) yn1(%6d) yn2(%6d)\n",
                this,
                reinterpret_cast<const u8*>(pBuffer->bufferAddress)[0],
                pBuffer->pAdpcmContext->pred_scale,
                pBuffer->pAdpcmContext->yn1,
                pBuffer->pAdpcmContext->yn2);
    }
    else
    {
        NW_LOG("MultiVoice[%p]  bufferPS(%04x)\n", this,
                reinterpret_cast<const u8*>(pBuffer->bufferAddress)[0] );
    }
#endif

    if ( lastFlag )
    {
        m_pLastWaveBuffer = pBuffer;
    }
}

void MultiVoice::SetAdpcmParam(
        int channelIndex,
        const AdpcmParam& param )
{
    NW_MINMAXLT_ASSERT( channelIndex, 0, WAVE_CHANNEL_MAX );

    m_Voice[channelIndex].SetAdpcmParam( param );
}

u32 MultiVoice::FrameToByte( u32 sample, nw::snd::SampleFormat format )
{
    u32 result = 0;

    switch ( format )
    {
    case nw::snd::SAMPLE_FORMAT_DSP_ADPCM:
        result = DspAdpcmFrameToNibbleAddress( sample ) >> 1;
        break;
    case nw::snd::SAMPLE_FORMAT_PCM_S8:
        result = sample;
        break;
    case nw::snd::SAMPLE_FORMAT_PCM_S16:
        result = ( sample << 1 );
        break;
    default:
        NW_ASSERTMSG( false, "Invalid format\n" );
    }

    return result;
}

void MultiVoice::CalcOffsetAdpcmParam(
    nw::snd::AdpcmContext* context, // in/out
    const nw::snd::AdpcmParam& param,
    u32 offsetSamples,
    const void* dataAddress
)
{
#ifdef NW_SND_DUMP_DECODE_DSPADPCM
    NW_LOG("[%s] ctx:ps(%5d) yn1(%d) yn2(%d) offset:(%d) adr:(%p)\n",
            __FUNCTION__, context->pred_scale, context->yn1, context->yn2,
            offsetSamples, dataAddress);

    FSAddClient(&s_fsClient, FS_RET_ALL_ERROR);
    FSInitCmdBlock(&s_fsCmdBlock);

    FSAStatus status;
    status = FSChangeDir( &s_fsClient, &s_fsCmdBlock, "", FS_RET_ALL_ERROR );
    NW_LOG("[FS] ChangeDir(%d)\n", status);

    FSFileHandle fileHandle;
    status = FSOpenFile(
        &s_fsClient,
        &s_fsCmdBlock,
        "_cafe_decodeDspAdpcm.raw",
        "w",
        &fileHandle,
        FS_RET_ALL_ERROR);
    NW_LOG("[FS] OpenFile(%d)\n", status );

    const u32 DUMP_BUFFER_SIZE = 4 * 1024 + 256;
    s16 dumpBuffer_[ DUMP_BUFFER_SIZE ];
    s16* dumpBuffer = reinterpret_cast<s16*>(ut::RoundUp( dumpBuffer_, 256 ));
    s16 dumpBufferCount = 0;
    NW_LOG("  offsetSamples(%d)\n", offsetSamples );
#endif

    // s32 AXDecodeAdpcmData(
    //     u8* input,
    //     DSPADPCM* info,
    //     s32       samples,
    //     s16*      output);

    u32 currentSample = 0;
    const u8* inputAddress = reinterpret_cast<const u8*>(dataAddress);
    const u8* predScale = reinterpret_cast<const u8*>(dataAddress);
    const u32 DECODE_SAMPLE_COUNT = 14; // 14 サンプル単位でデコードする
    const u32 DECODE_BYTE_SIZE = 8;     // 8 バイト (=14サンプル) 単位でデコードする
    NW_ASSERT( offsetSamples % DECODE_SAMPLE_COUNT == 0 );

    DSPADPCM adpcm = {0};
    for ( int i = 0; i < 16; i++ )
    {
        adpcm.coef[i] = param.coef[i/2][i%2];
    }
    adpcm.ps = context->pred_scale;
    adpcm.yn1 = static_cast<s16>(context->yn1);
    adpcm.yn2 = static_cast<s16>(context->yn2);
    adpcm.num_samples = DECODE_SAMPLE_COUNT;
    adpcm.num_adpcm_nibbles = DECODE_BYTE_SIZE * 2;
    adpcm.sa = 2;
    adpcm.ea = 15;
    adpcm.ca = 2;

    s16 output[DECODE_SAMPLE_COUNT] = {0};
    while ( currentSample < offsetSamples )
    {
        AXDecodeAdpcmData(
            inputAddress,
            &adpcm,
            DECODE_SAMPLE_COUNT,
            output );

    #ifdef NW_SND_DUMP_DECODE_DSPADPCM
        for ( int i = 0; i < DECODE_SAMPLE_COUNT; i++ )
        {
            dumpBuffer[dumpBufferCount] = output[i];
            dumpBufferCount++;
            if ( dumpBufferCount == DUMP_BUFFER_SIZE )
            {
                status = FSWriteFile(
                    &s_fsClient,
                    &s_fsCmdBlock,
                    dumpBuffer,
                    sizeof(s16)*DUMP_BUFFER_SIZE,
                    1,
                    fileHandle,
                    0,
                    FS_RET_ALL_ERROR);
                NW_LOG("[FS] Write(%d)\n", status );
                dumpBufferCount = 0;
            }
        }
    #endif
        currentSample += DECODE_SAMPLE_COUNT;
        predScale =
            reinterpret_cast<const u8*>( ut::AddOffsetToPtr( predScale, DECODE_BYTE_SIZE ) );

        // ps, yn1, yn2 更新
        adpcm.ps = static_cast<u16>(predScale[0]);
        adpcm.yn1 = static_cast<u16>(output[DECODE_SAMPLE_COUNT-1]);
        adpcm.yn2 = static_cast<u16>(output[DECODE_SAMPLE_COUNT-2]);
        adpcm.sa += 16;
        adpcm.ea += 16;
        adpcm.ca += 16;
    }

    // コンテキストの更新
    {
        context->pred_scale = adpcm.ps;
        context->yn1 = output[DECODE_SAMPLE_COUNT-1];
        context->yn2 = output[DECODE_SAMPLE_COUNT-2];
    }


#ifdef NW_SND_DUMP_DECODE_DSPADPCM
    if ( dumpBufferCount > 0 )
    {
        status = FSWriteFile(
                &s_fsClient,
                &s_fsCmdBlock,
                dumpBuffer,
                sizeof(s16)*dumpBufferCount,
                1,
                fileHandle,
                0,
                FS_RET_ALL_ERROR);
        NW_LOG("[FS] Write(%d)\n", status );
    }
    status = FSCloseFile( &s_fsClient, &s_fsCmdBlock, fileHandle, FS_RET_ALL_ERROR );
    NW_LOG("[FS] Close(%d)\n", status);

    status = FSDelClient( &s_fsClient, FS_RET_ALL_ERROR );
    NW_LOG("[FS] DelClient(%d)\n", status );
#endif
}

} // namespace nw::snd::internal::driver
} // namespace nw::snd::internal
} // namespace nw::snd
} // namespace nw
