﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nw/snd/snd_InstancePool.h>
#include <nw/assert.h>
#include <nw/ut/ut_Inlines.h>

namespace nw {
namespace snd {
namespace internal {

/*---------------------------------------------------------------------------*
  Name:         CreateImpl

  Description:  メモリプールの作成

  Arguments:    buffer - バッファの開始アドレス
                size - バッファのサイズ
                objSize - オブジェクトのサイズ

  Returns:      None.
 *---------------------------------------------------------------------------*/
u32 PoolImpl::CreateImpl( void* buffer, size_t size, u32 objSize )
{
    NW_NULL_ASSERT( buffer );

    char* ptr = static_cast<char*>( ut::RoundUp( buffer, 4 ) );
    objSize = ut::RoundUp( objSize, 4 );
    u32 numObjects = ( size - ( ptr - static_cast<char*>(buffer) ) ) / objSize;

    for ( u32 i=0; i < numObjects; i++ )
    {
        PoolImpl *head = reinterpret_cast<PoolImpl*>( ptr );
        head->m_pNext = m_pNext;
        m_pNext = head;
        ptr += objSize;
    }
    m_pBuffer = buffer;
    m_BufferSize = size;
    return numObjects;
}


/*---------------------------------------------------------------------------*
  Name:         DestroyImpl

  Returns:      None.
 *---------------------------------------------------------------------------*/
void PoolImpl::DestroyImpl()
{
    NW_NULL_ASSERT( m_pBuffer );

    void* begin = m_pBuffer;
    void* end = static_cast<void*>( static_cast<char*>( m_pBuffer ) + m_BufferSize );
    PoolImpl* ptr = m_pNext;
    PoolImpl* prev = this;
    while ( ptr != NULL )
    {
        if ( ( begin <= ptr ) && ( ptr < end ) )
        {
            prev->m_pNext = ptr->m_pNext;
        }
        else
        {
            prev = ptr;
        }
        ptr = ptr->m_pNext;
    }
}

/*---------------------------------------------------------------------------*
  Name:         CountImpl

  Description:  空き領域数をカウント

  Arguments:    None.

  Returns:      空き領域数
 *---------------------------------------------------------------------------*/
int PoolImpl::CountImpl() const
{
    int count = 0;
    for ( PoolImpl* ptr = m_pNext; ptr != NULL; ptr = ptr->m_pNext )
    {
        ++count;
    }
    return count;
}

/*---------------------------------------------------------------------------*
  Name:         AllocImpl

  Description:  プールからメモリ領域を確保

  Arguments:    None.

  Returns:      メモリ領域の開始アドレス
 *---------------------------------------------------------------------------*/
void* PoolImpl::AllocImpl()
{
    if ( m_pNext == NULL ) return NULL;
    PoolImpl* head = m_pNext;
    m_pNext = head->m_pNext;
    return head;
}

/*---------------------------------------------------------------------------*
  Name:         FreeImpl

  Description:  メモリ領域をプールに開放

  Arguments:    ptr - メモリ領域の開始アドレス

  Returns:      None.
 *---------------------------------------------------------------------------*/
void PoolImpl::FreeImpl( void* ptr )
{
    PoolImpl* head = reinterpret_cast<PoolImpl*>( ptr );
    head->m_pNext = m_pNext;
    m_pNext = head;
}

} // namespace nw::snd::internal
} // namespace nw::snd
} // namespace nw

