﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nw/snd/snd_FxMultiChChorus.h>
#include <nw/assert.h>

#if defined( NW_PLATFORM_WIN32 )
using namespace nw::internal::winext;
#elif defined( NW_USE_NINTENDO_SDK )
// TODO: nn_audio
using namespace nw::internal::winext;
#endif

#if defined(NW_SND_CONFIG_ENABLE_MULTICHEFT)

namespace nw {
namespace snd {

/* ========================================================================
        constant definition
   ======================================================================== */

const f32 FxMultiChChorus::DELAY_TIME_MIN  = 0.1f;  // 0.1f ms
const f32 FxMultiChChorus::DELAY_TIME_MAX  = 50.0f; // 50.0f ms
const f32 FxMultiChChorus::DEPTH_MIN       = 0.0f;  // 0.0f
const f32 FxMultiChChorus::DEPTH_MAX       = 1.0f;  // 1.0f
const f32 FxMultiChChorus::RATE_MIN        = 0.1f;  // 0.1f
const f32 FxMultiChChorus::RATE_MAX        = 2.0f;  // 2.0f
const f32 FxMultiChChorus::FEEDBACK_MIN    = 0.0f;  // 0.0f
const f32 FxMultiChChorus::FEEDBACK_MAX    = 1.0f;  // 1.0f
const f32 FxMultiChChorus::OUT_GAIN_MIN    = 0.0f;  // 0.0f
const f32 FxMultiChChorus::OUT_GAIN_MAX    = 1.0f;  // 1.0f

/* ========================================================================
        member function
   ======================================================================== */

/*---------------------------------------------------------------------------*
  Name:         FxMultiChChorus

  Description:  コンストラクタ

  Arguments:    None.

  Returns:      None.
 *---------------------------------------------------------------------------*/
FxMultiChChorus::FxMultiChChorus()
: m_IsActive( false ),
  m_ChannelMode( CHANNEL_MODE_6CH ),
  m_SampleRate( SAMPLE_RATE_32000 )
{
    // パラメータ初期化
    const MultiChChorusParam param;
    SetParam( param );
}
/*---------------------------------------------------------------------------*
  Name:         GetRequiredMemSize

  Description:  エフェクトを使用するのに必用となるメモリサイズを取得する

  Arguments:    None.

  Returns:      必要となるメモリサイズ
 *---------------------------------------------------------------------------*/
u32 FxMultiChChorus::GetRequiredMemSize( )
{
    AXFX_MULTI_CH_CHORUS param = m_AxfxParam;

    if ( !m_IsActive )
    {
        // 初期化前にバッファサイズを知るため、一部のパラメータを設定する
        switch ( m_ChannelMode )
        {
        case CHANNEL_MODE_2CH:
            param.mode = AXFX_CHORUS_MODE_2CH;
            break;
        case CHANNEL_MODE_4CH:
            param.mode = AXFX_CHORUS_MODE_4CH;
            break;
        case CHANNEL_MODE_6CH:
            param.mode = AXFX_CHORUS_MODE_6CH;
            break;
        default:
            param.mode = AXFX_CHORUS_MODE_6CH;
        }
        switch ( m_SampleRate )
        {
        case SAMPLE_RATE_32000:
            param.fs = 32000;
            break;
#if defined(NW_SND_CONFIG_ENABLE_SOUND2)
        case SAMPLE_RATE_48000:
            param.fs = 48000;
            break;
#endif
        default:
            param.fs = 32000;
        }
    }

    s32 requiredSize = AXFXMultiChChorusGetMemSize( &param );
    NW_ASSERT( requiredSize >= 0 );
    size_t size = ut::RoundUp(
        sizeof( MEMiHeapHead )
        + sizeof( MEMiFrmHeapHead )
        + requiredSize
        + 32,
        32
    );

    return static_cast<u32>( size );
}

/*---------------------------------------------------------------------------*
  Name:         AssignWorkBuffer

  Description:  エフェクトのワークバッファを割り当てる

  Arguments:    buffer - バッファのアドレス
                size - バッファのサイズ

  Returns:      割り当てに成功したらtrue
 *---------------------------------------------------------------------------*/
bool FxMultiChChorus::AssignWorkBuffer( void* buffer, u32 size )
{
    return m_Impl.CreateHeap( buffer, size );
}

/*---------------------------------------------------------------------------*
  Name:         ReleaseWorkBuffer

  Description:  エフェクトのワークバッファを解放する

  Arguments:    None.

  Returns:      None.
 *---------------------------------------------------------------------------*/
void FxMultiChChorus::ReleaseWorkBuffer()
{
    m_Impl.DestroyHeap();
}

/*---------------------------------------------------------------------------*
  Name:         Initialize

  Description:  エフェクトの開始処理を行う

  Arguments:    None.

  Returns:      成功したらtrue
 *---------------------------------------------------------------------------*/
bool FxMultiChChorus::Initialize()
{
    if ( static_cast<s32>(GetRequiredMemSize()) > m_Impl.GetHeapTotalSize() )
    {
        return false;
    }

    AXFXAlloc alloc;
    AXFXFree free;
    m_Impl.HookAlloc( &alloc, &free );
    AXFX_CHORUS_MODE mode;
    switch ( m_ChannelMode )
    {
    case CHANNEL_MODE_2CH:
        mode = AXFX_CHORUS_MODE_2CH;
        break;
    case CHANNEL_MODE_4CH:
        mode = AXFX_CHORUS_MODE_4CH;
        break;
    case CHANNEL_MODE_6CH:
        mode = AXFX_CHORUS_MODE_6CH;
        break;
    default:
        mode = AXFX_CHORUS_MODE_6CH;
    }

    AXFX_SAMPLE_RATE rate = GetFxSampleRate( m_SampleRate );
    int result = AXFXMultiChChorusInit( &m_AxfxParam, mode, rate );
    u32 allocatedSize = m_Impl.RestoreAlloc( alloc, free );
    m_IsActive = true;

    // AXで実際にAllocateされたメモリと同じかどうか確認
    u32 requiredMemSize = GetRequiredMemSize();
    NW_WARNING(
        ut::RoundUp( sizeof( MEMiHeapHead ) + sizeof( MEMiFrmHeapHead ) + allocatedSize + 32, 32 ) == requiredMemSize,
        "differ between allocated buffer size(%d) and required mem size(%d).",
        ut::RoundUp( sizeof( MEMiHeapHead ) + sizeof( MEMiFrmHeapHead ) + allocatedSize + 32, 32 ),
        requiredMemSize
    );

    return result != 0;
}

/*---------------------------------------------------------------------------*
  Name:         Finalize

  Description:  エフェクトの終了処理を行う

  Arguments:    None.

  Returns:      None.
 *---------------------------------------------------------------------------*/
void FxMultiChChorus::Finalize()
{
    if ( ! m_IsActive ) return;

    m_IsActive = false;

    AXFXAlloc alloc;
    AXFXFree free;
    m_Impl.HookAlloc( &alloc, &free );
    AXFXMultiChChorusShutdown( &m_AxfxParam );
    m_Impl.RestoreAlloc( alloc, free );
}

/*---------------------------------------------------------------------------*
  Name:         SetParam

  Description:  エフェクトのパラメータを変更する

  Arguments:    param - エフェクトパラメータ

  Returns:      None.
 *---------------------------------------------------------------------------*/
bool FxMultiChChorus::SetParam( const MultiChChorusParam& param )
{
    NW_FMINMAX_ASSERT( param.delayTime, DELAY_TIME_MIN, DELAY_TIME_MAX );
    NW_FMINMAX_ASSERT( param.depth,     DEPTH_MIN,      DEPTH_MAX      );
    NW_FMINMAX_ASSERT( param.rate,      RATE_MIN,       RATE_MAX       );
    NW_FMINMAX_ASSERT( param.feedback,  FEEDBACK_MIN,   FEEDBACK_MAX   );
    NW_FMINMAX_ASSERT( param.outGain,   OUT_GAIN_MIN,   OUT_GAIN_MAX   );

    m_Param = param; // struct copy

    m_AxfxParam.delayTime  = ut::Clamp( param.delayTime, DELAY_TIME_MIN, DELAY_TIME_MAX );
    m_AxfxParam.depth      = ut::Clamp( param.depth,     DEPTH_MIN,      DEPTH_MAX      );
    m_AxfxParam.rate       = ut::Clamp( param.rate,      RATE_MIN,       RATE_MAX       );
    m_AxfxParam.feedback   = ut::Clamp( param.feedback,  FEEDBACK_MIN,   FEEDBACK_MAX   );
    m_AxfxParam.outGain    = ut::Clamp( param.outGain,   OUT_GAIN_MIN,   OUT_GAIN_MAX   );

    if ( ! m_IsActive ) return true;

    int result = AXFXMultiChChorusSettingsUpdate( &m_AxfxParam );

    return result != 0;
}

/*---------------------------------------------------------------------------*
  Name:         UpdateBuffer

  Description:  エフェクトコールバック

  Arguments:    param - エフェクトパラメータ

  Returns:      None.
 *---------------------------------------------------------------------------*/
void FxMultiChChorus::UpdateBuffer(
    int numChannels,
    void* buffer[],
    unsigned long bufferSize,
    SampleFormat format,
    f32 sampleRate,
    OutputMode mode
)
{
    if ( ! m_IsActive ) return;

    // DPL2モードの時はバスの解釈が違うので処理しない
    if ( mode == OUTPUT_MODE_DPL2 ) return;

    (void)format;
    (void)sampleRate;

    NW_ASSERT( numChannels >= 2 );
    NW_ASSERT( format == SAMPLE_FORMAT_PCM_S32 );

    AXFX_6CH_BUFFERUPDATE axfxbuf =
    {
        static_cast<s32*>( buffer[0] ),
        static_cast<s32*>( buffer[1] ),
        static_cast<s32*>( buffer[2] ),
        static_cast<s32*>( buffer[3] ),
        static_cast<s32*>( buffer[4] ),
        static_cast<s32*>( buffer[5] )
    };
    AXAUXCBSTRUCT auxCbStruct;

    auxCbStruct.numChs = numChannels;
    auxCbStruct.numSamples = bufferSize / ( numChannels * sizeof(s32) );
    AXFXMultiChChorusCallback( &axfxbuf, &m_AxfxParam, &auxCbStruct );
}

/*---------------------------------------------------------------------------*
  Name:         IsValidChannelNum

  Description:  デバイスに対してチャンネル数が妥当かをチェックする仮想関数です。

  Arguments:    device チェック対象となるデバイスです。

  Returns:      チャンネル数が妥当であれば true を、妥当でなければ false を返します。
 *---------------------------------------------------------------------------*/
bool FxMultiChChorus::IsValidChannelNum( OutputDevice device )
{
    if ( device == OUTPUT_DEVICE_DRC0 || device == OUTPUT_DEVICE_DRC1 )
    {
        if ( m_ChannelMode == CHANNEL_MODE_6CH )
        {
            NW_ASSERTMSG( false, "CHANNEL_MODE_6CH (OUTPUT_DEVICE_DRC) is inValid. Please use lower channel mode." );
            return false;
        }
    }

    return true;
}

} // namespace nw::snd
} // namespace nw

#endif // NW_SND_CONFIG_ENABLE_MULTICHEFT
