﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nw/scfont/scfont_TextureCache.h>

#if defined(NW_PLATFORM_WIN32) || defined(NW_USE_NINTENDO_SDK)
#include <gl/glew.h>
#endif

#include <nw/scfont/scffnd_ScalableFontEngine.h>

namespace {

bool IsPos2(u32 n)
{
    return (((n - 1) & n) == 0);
}

}

namespace nw
{
namespace scfont
{

TextureCache::InitializeArg::InitializeArg()
 : textureCacheWidth(1024)
 , textureCacheHeight(1024)
 , allocator(NULL)
 , fontDatas(NULL)
 , fontDataSizes(NULL)
 , boldWeights(NULL)
 , borderWidths(NULL)
 , fontDataTypes(NULL)
 , fontNum(1)
 , workMemorySize(WORK_MEMORY_SIZE_DEFAULT)
 , noPlotWorkMemorySize(WORK_MEMORY_SIZE_NO_PLOT_DEFAULT)
 , glyphNodeNumMax(GLYPH_NODE_NUM_MAX_DEFALUT)
 , isDrawMultiCore(false)
 , isAutoHint(false)
{
}

TextureCache::TextureCache()
 : mFsState(NULL)
 , mFsStatesNoPlot(NULL)
 , mFontNameBufs(NULL)
 , mTextureCacheBitMap(NULL)
 , mTextureCacheWidth(0)
 , mTextureCacheHeight(0)
 , mLineCurrentPos(0)
 , mFontNum(0)
 , mCurrentFontFace(0)
 , mNoSpaceError(false)
 , mIsDrawMultiCore(false)
 , mFontMetrics(NULL)
{
}

TextureCache::~TextureCache()
{
}

void TextureCache::Initialize(const InitializeArg& arg)
{
    // 引数のチェック
    if ( ! IsPos2(arg.textureCacheWidth))
    {
        NW_ERR("arg.width[%u] is not power of 2.", arg.textureCacheWidth);
        return;
    }
    if (arg.allocator == NULL)
    {
        NW_ERR("arg.allocator must be set.");
        return;
    }
    if (arg.fontDatas == NULL)
    {
        NW_ERR("arg.fontDatas must be set.");
        return;
    }
    if (arg.fontNum == 0 || arg.fontNum > 32)
    {
        NW_ERR("arg.fontNum must be 0 and more, 32 and less.");
        return;
    }
    if (arg.fontDataTypes != NULL && arg.fontDataSizes == NULL)
    {
        for (u32 i = 0; i < arg.fontNum; i++)
        {
            if (arg.fontDataTypes[i] == FONT_DATA_TYPE_BFTTF)
            {
                NW_ERR("arg.fontDataSizes must not be NULL when one of arg.fontDataTypes is FONT_DATA_TYPE_BFTTF.");
                return;
            }
        }
    }

    mFontNum = arg.fontNum;
    mIsDrawMultiCore = arg.isDrawMultiCore;
    mFontNameBufs = static_cast<char*>(arg.allocator->Alloc(FONT_NAME_LEN_MAX * FONT_FACE_NUM_MAX, 4));
    mFontMetrics = static_cast<FontMetrics*>(arg.allocator->Alloc(mFontNum * sizeof(FontMetrics), 4));
    mBoldWeights = static_cast<scffnd::ScFixed*>(arg.allocator->Alloc(mFontNum * sizeof(scffnd::ScFixed)));
    mBorderWidths = static_cast<u8*>(arg.allocator->Alloc(mFontNum * sizeof(u8)));

    mTextureCacheWidth = arg.textureCacheWidth;
    mTextureCacheHeight = arg.textureCacheHeight;
    mLineCurrentPos = 0;
    mCurrentFontFace = UINT_MAX;
    for (u32 k = 0; k < CORE_NUM; k++)
    {
        mCurrentFontFacesNoPlot[k] = UINT_MAX;
    }

    // ステート構造体の確保
    mFsState = static_cast<scffnd::ScalableFontEngine*>(arg.allocator->Alloc(sizeof(scffnd::ScalableFontEngine), 4));
    mFsStatesNoPlot = static_cast<scffnd::ScalableFontEngine*>(arg.allocator->Alloc(sizeof(scffnd::ScalableFontEngine)*GetFsStateNoPlotNum(), 4));

    // ステート構造体の初期化
    mFsState->Initialize(arg.allocator->Alloc(arg.workMemorySize, 4), arg.workMemorySize);
    AssertFsError("Initialize");
    // このステート構造体ではプロットは行わないので、ワークエリアは固定の小さい領域にする。フォントを複数扱う場合は消費されるサイズはその数の倍数必要。
    for (u32 k = 0; k < GetFsStateNoPlotNum(); k++)
    {
        mFsStatesNoPlot[k].Initialize(arg.allocator->Alloc(arg.noPlotWorkMemorySize * mFontNum, 4), arg.noPlotWorkMemorySize * mFontNum);
        AssertFsErrorNoPlot("Initialize", k);
    }

    // フォントの登録
    for (u32 i = 0; i < mFontNum; i++)
    {
        void* fontData;
        if (arg.fontDataTypes != NULL && arg.fontDataTypes[i] == FONT_DATA_TYPE_BFTTF)
        {
            fontData = scffnd::ScalableFontEngineHelper::Decode(arg.fontDatas[i], arg.fontDataSizes[i]);
            NW_ASSERT(fontData != NULL);
        }
        else
        {
            fontData = arg.fontDatas[i];
        }
        if (arg.boldWeights != NULL)
        {
            mBoldWeights[i] = static_cast<scffnd::ScFixed>(static_cast<f32>(1 << 16) * arg.boldWeights[i]);
        }
        else
        {
            mBoldWeights[i] = 0;
        }
        if (arg.borderWidths != NULL)
        {
            mBorderWidths[i] = arg.borderWidths[i];
        }
        else
        {
            mBorderWidths[i] = 0;
        }

        mFsState->LoadFont(NULL, fontData, 0, FONT_NAME_LEN_MAX, GetFontNameBuf(i));
        AssertFsError("LoadFont");
        for (u32 k = 0; k < GetFsStateNoPlotNum(); k++)
        {
            mFsStatesNoPlot[k].LoadFont(NULL, fontData, 0, FONT_NAME_LEN_MAX, GetFontNameBuf(i));
            AssertFsErrorNoPlot("LoadFont", k);
        }

        // フォントのアセントと高さを求めておく
        SetFontFace(i);
        scffnd::ScMetrics metrics;
        mFsState->GetFontMetrics(&metrics);
        AssertFsError("GetFontMetrics");

        mFontMetrics[i].ascent_ratio = static_cast<float>(metrics.os2WinAscent) / metrics.metricsResolution;
        mFontMetrics[i].height_ratio = static_cast<float>(metrics.os2WinAscent + metrics.os2WinDescent) / metrics.metricsResolution;
    }

#if defined(NW_PLATFORM_WIN32) || defined(NW_USE_NINTENDO_SDK)
    mTextureCacheBitMap = static_cast<u8*>(arg.allocator->Alloc(mTextureCacheWidth * mTextureCacheHeight, 4));
    std::memset(mTextureCacheBitMap, 0x0, mTextureCacheWidth * mTextureCacheHeight);
    {
        GLuint textureCacheId;
        glGenTextures( 1, &textureCacheId );
        glBindTexture( GL_TEXTURE_2D_ARRAY, textureCacheId );
        glTexParameteri( GL_TEXTURE_2D_ARRAY, GL_TEXTURE_MIN_FILTER, GL_LINEAR );
        glTexParameteri( GL_TEXTURE_2D_ARRAY, GL_TEXTURE_MAG_FILTER, GL_LINEAR );
        glTexParameteri( GL_TEXTURE_2D_ARRAY, GL_TEXTURE_WRAP_S, GL_CLAMP );
        glTexParameteri( GL_TEXTURE_2D_ARRAY, GL_TEXTURE_WRAP_T, GL_CLAMP );
        glTexParameteri( GL_TEXTURE_2D_ARRAY, GL_TEXTURE_SWIZZLE_R, GL_ONE );
        glTexParameteri( GL_TEXTURE_2D_ARRAY, GL_TEXTURE_SWIZZLE_G, GL_ONE );
        glTexParameteri( GL_TEXTURE_2D_ARRAY, GL_TEXTURE_SWIZZLE_B, GL_ONE );
        glTexImage3D(GL_TEXTURE_2D_ARRAY, 0, GL_ALPHA8, mTextureCacheWidth, mTextureCacheHeight, 1, 0, GL_ALPHA, GL_UNSIGNED_BYTE, mTextureCacheBitMap);
        NW_GL_ASSERT();
        m_TexObj.Set(textureCacheId, mTextureCacheBitMap, font::FONT_SHEET_FORMAT_A8, static_cast<u16>(mTextureCacheWidth), static_cast<u16>(mTextureCacheHeight), 1, true);
    }
#elif defined(NW_PLATFORM_CAFE)
    GX2InitTexture(m_TexObj.GetTexture(), mTextureCacheWidth, mTextureCacheHeight, 1, 1, GX2_SURFACE_FORMAT_TC_R8_UNORM, GX2_SURFACE_DIM_2D_ARRAY);
    m_TexObj.GetTexture()->surface.tileMode = GX2_TILE_MODE_LINEAR_ALIGNED;
    // GX2InitTextureで設定されたアラインメントに従ってメモリ確保。DCZeroRangeを使うために、最低64バイトアラインメントになるようにする
    mTextureCacheBitMap = static_cast<u8*>(arg.allocator->Alloc(mTextureCacheWidth * mTextureCacheHeight, ut::RoundUp(m_TexObj.GetTexture()->surface.alignment, 64)));
    // テクスチャキャッシュを更新するときにキャッシュストアされるので、この時点ではDCZeroRangeでよい
    DCZeroRange(mTextureCacheBitMap, mTextureCacheWidth * mTextureCacheHeight);
    DCFlushRange(mTextureCacheBitMap, mTextureCacheWidth * mTextureCacheHeight);
    GX2InitTexturePtrs(m_TexObj.GetTexture(), mTextureCacheBitMap, 0);
    GX2InitTextureCompSel(m_TexObj.GetTexture(), GX2_GET_COMP_SEL(GX2_COMPONENT_C_1, GX2_COMPONENT_C_1, GX2_COMPONENT_C_1, GX2_COMPONENT_X_R));
    GX2InitTextureRegs(m_TexObj.GetTexture());
    m_TexObj.Set(1, mTextureCacheBitMap, font::FONT_SHEET_FORMAT_A8, mTextureCacheWidth, mTextureCacheHeight, 1, true);
#endif

    mGlyphTreeMap.Initialize(arg.allocator, arg.glyphNodeNumMax);
    if (arg.isAutoHint)
    {
        mFsState->SetAutoHint(true);
    }
}

void TextureCache::Finalize(nw::ut::IAllocator* allocator)
{
    void* work = reinterpret_cast<void*>(mFsState->GetPointerToWorkBuffer());
    void* noPlotWorks[CORE_NUM];
    for (u32 k = 0; k < GetFsStateNoPlotNum(); k++)
    {
        noPlotWorks[k] = reinterpret_cast<void*>(mFsStatesNoPlot[k].GetPointerToWorkBuffer());
    }

    mFsState->Finalize();
    AssertFsError("Finalize");
    for (u32 k = 0; k < GetFsStateNoPlotNum(); k++)
    {
        mFsStatesNoPlot[k].Finalize();
        AssertFsErrorNoPlot("Finalize", k);
    }

    nw::ut::SafeFree( mFsState, allocator );
    nw::ut::SafeFree( mFsStatesNoPlot, allocator );
    nw::ut::SafeFree( work, allocator );
    for (u32 k = 0; k < GetFsStateNoPlotNum(); k++)
    {
        nw::ut::SafeFree( noPlotWorks[k], allocator );
    }
    nw::ut::SafeFree( mTextureCacheBitMap, allocator );
    mGlyphTreeMap.Finalize(allocator);
    nw::ut::SafeFree( mFontNameBufs, allocator );
    nw::ut::SafeFree( mFontMetrics, allocator );
    nw::ut::SafeFree( mBoldWeights, allocator );
    nw::ut::SafeFree( mBorderWidths, allocator );

#if defined(NW_PLATFORM_WIN32) || defined(NW_USE_NINTENDO_SDK)
    if (m_TexObj.GetName() != 0)
    {
        GLuint textureCacheId = m_TexObj.GetName();
        glDeleteTextures(1, &textureCacheId);
    }
#endif
}

bool TextureCache::RegisterGlyph(char16 code, u32 fontSize, u16 fontFace, s32 lockGroup /* = -1 */)
{
    GlyphNode* node = mGlyphTreeMap.Find(code, static_cast<u16>(fontSize), fontFace);

    if (node)
    {
        // 見つかった -> 既に登録されている文字
        node->SetRequested(true);
        if (lockGroup >= 0)
        {
            node->GetLockGroup().SetBitOn(lockGroup);
        }
        // 既に登録されている文字の場合、IsPlottingがtrueのときだけまだ描画できない
        return node->IsPlotting();
    }
    else
    {
        // 見つからない -> まだ登録されていない文字
        node = mGlyphTreeMap.Insert(code, static_cast<u16>(fontSize), fontFace);
        if (node)
        {
            node->SetRequested(true);
            node->SetPlotting(true);
            if (lockGroup >= 0)
            {
                node->GetLockGroup().SetBitOn(lockGroup);
            }
            // 未プロットリストに入れる
            mNeedPlotGlyphList.PushBack(node);
            return true;
        }
        else
        {
            // 平衡二分木のノードに空きがない。
            if (EraseNotInFontGlyphs() > 0) {
                // フォントにない文字のノードを削除し、空きができたら登録を行う。(次は登録できるはず)
                return RegisterGlyph(code, fontSize, fontFace, lockGroup);
            } else {
                // 空きができなかったら、文字の登録を行わずに抜ける。
                NW_LOG("nw::scfont::TextureCache: Node number max over. Please increase InitializeArg::glyphNodeNumMax.\n");
                mNoSpaceError = true;
                return false;
            }
        }
    }
}


u32 TextureCache::RegisterGlyphsWithLength(const char16* codes, u32 codeNum, u32 fontSize, u16 fontFace, s32 lockGroup /* = -1 */)
{
    u32 count = 0;
    for (u32 i = 0; i < codeNum; i++)
    {
        if (codes[i] == 0)
        {
            break;
        }

        count += static_cast<u32>(RegisterGlyph(codes[i], fontSize, fontFace, lockGroup));
    }
    return count;
}

void TextureCache::UpdateTextureCache()
{
    // 結局テクスチャキャッシュ内の文字の配置はUpdateTextureCache内で行わなければならない。
    // 文字の横幅を取る処理が重く、GetGlyphmap() で取ってきてその場で配置したほうがトータルのコストが小さいため。
    GlyphList::Iterator step_it = mNeedPlotGlyphList.GetBeginIter();
    GlyphList::Iterator end_it = mNeedPlotGlyphList.GetEndIter();
    for (;step_it != end_it;)
    {
        GlyphList::Iterator it = step_it;
        ++step_it;
        mNeedPlotGlyphList.Erase(it);

        SetFontFace(it->GetFace());

        // まず、文字がフォントにあるかどうかを調べる。ここで調べているのは、RegisterGlyphでやると時間がかかり、
        // メインスレッドの遅延に繋がるため。UpdateTextureCacheの中であれば、プロットの処理に紛れる
        if ( ! mFsState->CheckGlyphExist(it->GetCode())) {
            // ない場合、削除は行わない。(削除してしまうとその文字のRegisterGlyphが常にtrueになってしまうため。)
            // フォントにない文字フラグを立て、フォントにない文字リストに入れておく。
            it->SetNotInFont(true);
            mNotInFontGlyphList.PushBack(&(*it));
            continue;
        }

        // フォントサイズの設定
        mFsState->SetScale(it->GetSize() << 16, 0, 0, it->GetSize() << 16);
        AssertFsError("SetScale");

        // このフォントのアセントを求める
        int font_ascent = static_cast<int>(mFontMetrics[it->GetFace()].ascent_ratio * it->GetSize() + 0.5f);

        // GRAYMAP8固定にする。そのままA8のテクスチャにコピーできるという利点がある。
        // GRAYMAP8の場合は、ScGlyphMapのメンバbplとwidthは同じ値になる。
        scffnd::ScGlyphMap *map = mFsState->GetGlyphmap(it->GetCode(), scffnd::MAP_GRAYMAP8);
        AssertFsError("GetGlyphmap");

        it->SetLineKind(GlyphNode::CalcLineKind(static_cast<u16>(map->height + font_ascent - map->hiY)));

        // 文字をプロットするラインを決める
        LineInfo* matched_line = NULL;
        for (u32 i = 0; i < mLineCurrentPos; i++)
        {
            if (mLineInfos[i].kind == it->GetLineKind() && (static_cast<u32>(mLineInfos[i].currentX + map->width + GLYPH_PADDING) < mTextureCacheWidth))
            {
                matched_line = &mLineInfos[i];
                break;
            }
        }
        // マッチするラインがなかった場合
        if ( ! matched_line)
        {
            if (mLineCurrentPos >= TEXTURE_CACHE_LINE_NUM_MAX)
            {
                // ラインの最大数を超えた。エラー扱い
                NW_LOG("nw::scfont::TextureCache: line num[%d] exceeds\n", mLineCurrentPos);
                mNeedEraseGlyphList.PushBack(&(*it));
                mFsState->ReleasesGlyph(map);
                AssertFsError("ReleasesGlyph");
                mNoSpaceError = true;
                continue;
            }

            // 新しいラインを作成する
            matched_line = CreateNewLineImpl(static_cast<LineHeight>(it->GetLineKind()));

            if (matched_line == NULL)
            {
                // 新しいラインを作成できなかった。
                // 現在プロットされているものを消して、その上にプロットする必要がある。

                // 消去可能な文字を探す
                GlyphNode* node = NULL;
                node = FindAndReserveEraseGlyph(it->GetLineKind(), map->width);
                if (node)
                {
                    // 元々あったグリフの領域に上書きする
                    GlyphLineList* list = &mLineInfos[node->GetLineNo()].list;
                    it->SetCachePosX(node->GetCachePosX());
                    it->SetCachePosY(node->GetCachePosY());
                    it->SetLineNo(node->GetLineNo());
                    list->Insert(GlyphLineList::GetIteratorFromPointer(node), &(*it));
                }
                else
                {
                    // 入るスペースがないので抜ける
                    NW_LOG("nw::scfont::TextureCache: no erasable glyph. charCode = 0x%x lineHeight = %d\n", it->GetCode(), 1 << it->GetLineKind());
                    mNeedEraseGlyphList.PushBack(&(*it));
                    mFsState->ReleasesGlyph(map);
                    AssertFsError("ReleasesGlyph");
                    mNoSpaceError = true;
                    continue;
                }
            }
        }

        if (matched_line)
        {
            it->SetCachePosX(matched_line->currentX);
            it->SetCachePosY(matched_line->y);
            it->SetLineNo(matched_line->no);
            matched_line->list.PushBack(&(*it));
            matched_line->currentX += map->width + GLYPH_PADDING;
        }

        int offset_from_top = font_ascent - map->hiY;
        int render_origin_y = it->GetCachePosY() + offset_from_top;

        it->SetGlyphWidth(map->width);
        it->SetGlyphHeight(static_cast<u16>(map->height + offset_from_top));
        it->SetAdvanceX(map->idX);
        it->SetLeftOffset(map->loX);

        u32 copy_size = map->width + 2;
#if defined(NW_PLATFORM_CAFE)
        u32 cache_flush_size = map->width + 2 + 32;
#endif

        // 前の文字が描画されている場合があるので、文字の上のスペースをクリアする
        for (s32 l = it->GetCachePosY(); l < render_origin_y; l++) {
            // fontライブラリがy軸反転したテクスチャを要求するため、書き込み先のY軸を反転させる
            u8* line_start = &mTextureCacheBitMap[(mTextureCacheHeight - l - 1) * mTextureCacheWidth + it->GetCachePosX()];
            std::memset(line_start, 0, copy_size);
#if defined(NW_PLATFORM_CAFE)
            // DCFlushRange(line_start, copy_size)と同等の処理
            for (u32 dot = 0; dot < cache_flush_size; dot += 32)
            {
                asm ( "dcbf     %0, %1" : "+g"(line_start), "+g"(dot) );
            }
#endif
        }

        s32 copy_start_y = ut::Max(0, - render_origin_y);
        if (mBorderWidths[it->GetFace()])
        {
            for (s32 l = copy_start_y; l < map->height; l++) {
                // fontライブラリがy軸反転したテクスチャを要求するため、書き込み先のY軸を反転させる
                u8* line_start = &mTextureCacheBitMap[(mTextureCacheHeight - (l + render_origin_y) - 1) * mTextureCacheWidth + it->GetCachePosX()];

                std::memcpy(line_start, &map->bits[l * map->width], map->width);
                // コピーした部分の右の2ピクセルは0でクリアする
                *reinterpret_cast<u16*>(&line_start[map->width]) = 0;
            }

            mFsState->ReleasesGlyph(map);
            mFsState->SetFlags(scffnd::ScalableFontEngine::FLG_OUTLINED_UNFILLED);
            map = mFsState->GetGlyphmap(it->GetCode(), scffnd::MAP_GRAYMAP8);
            AssertFsError("GetGlyphmap");
            mFsState->SetFlags(scffnd::ScalableFontEngine::FLG_OUTLINED_FILLED);

            for (s32 l = copy_start_y; l < map->height; l++) {
                // fontライブラリがy軸反転したテクスチャを要求するため、書き込み先のY軸を反転させる
                u8* line_start = &mTextureCacheBitMap[(mTextureCacheHeight - (l + render_origin_y) - 1) * mTextureCacheWidth + it->GetCachePosX()];

                for (int x = 0; x < map->width; ++x)
                {
                    line_start[x] -= map->bits[l * map->width + x] >> 1;
                }
#if defined(NW_PLATFORM_CAFE)
                // DCFlushRange(line_start, copy_size)と同等の処理
                for (u32 dot = 0; dot < cache_flush_size; dot += 32)
                {
                    asm ( "dcbf     %0, %1" : "+g"(line_start), "+g"(dot) );
                }
#endif
            }
        }
        else
        {
            for (s32 l = copy_start_y; l < map->height; l++) {
                // fontライブラリがy軸反転したテクスチャを要求するため、書き込み先のY軸を反転させる
                u8* line_start = &mTextureCacheBitMap[(mTextureCacheHeight - (l + render_origin_y) - 1) * mTextureCacheWidth + it->GetCachePosX()];

                std::memcpy(line_start, &map->bits[l * map->width], map->width);
                // コピーした部分の右の2ピクセルは0でクリアする
                *reinterpret_cast<u16*>(&line_start[map->width]) = 0;
#if defined(NW_PLATFORM_CAFE)
                // DCFlushRange(line_start, copy_size)と同等の処理
                for (u32 dot = 0; dot < cache_flush_size; dot += 32)
                {
                    asm ( "dcbf     %0, %1" : "+g"(line_start), "+g"(dot) );
                }
#endif
            }
        }
        // heightがそのラインの高さより小さい場合は、最大2ドット0でクリアする
        u32 line_height = 1 << it->GetLineKind();
        for (u32 l = map->height; l < static_cast<u32>(map->height + 2) && l < line_height; l++)
        {
            // fontライブラリがy軸反転したテクスチャを要求するため、書き込み先のY軸を反転させる
            u8* line_start = &mTextureCacheBitMap[(mTextureCacheHeight - (l + render_origin_y) - 1) * mTextureCacheWidth + it->GetCachePosX()];
            std::memset(line_start, 0, copy_size);
#if defined(NW_PLATFORM_CAFE)
            // DCFlushRange(line_start, copy_size)と同等の処理
            for (u32 dot = 0; dot < cache_flush_size; dot += 32)
            {
                asm ( "dcbf     %0, %1" : "+g"(line_start), "+g"(dot) );
            }
#endif
        }

        mFsState->ReleasesGlyph(map);
        AssertFsError("ReleasesGlyph");
    }
}

TextureCache::LineInfo* TextureCache::CreateNewLineImpl(LineHeight lineHeight)
{
    u32 next_line_y;
    if (mLineCurrentPos == 0)
    {
        next_line_y = 0;
    }
    else
    {
        next_line_y = mLineInfos[mLineCurrentPos - 1].y + (1 << mLineInfos[mLineCurrentPos - 1].kind) + GLYPH_PADDING;
    }
    u32 endY = next_line_y + static_cast<u32>(1 << static_cast<u32>(lineHeight));

    if (endY > mTextureCacheHeight)
    {
        // 全てのラインが埋まっていて、新たにラインを作ることができない。
        return NULL;
    }
    else
    {
        // 新しいラインを作成できる
        LineInfo* new_line = &mLineInfos[mLineCurrentPos];
        new_line->currentX = 0;
        new_line->y = static_cast<u16>(next_line_y);
        new_line->kind = static_cast<u8>(lineHeight);
        new_line->no = static_cast<u8>(mLineCurrentPos);

        mLineCurrentPos++;
        return new_line;
    }
}

GlyphNode* TextureCache::FindAndReserveEraseGlyph(u8 lineKind, u16 glyphWidth)
{
    for (u32 l = 0; l < mLineCurrentPos; l++)
    {
        LineInfo* lineInfo = &mLineInfos[l];
        if (lineInfo->kind == lineKind)
        {
            // 種類が一致するラインを見つけた、消去可能な文字がないか探す
            GlyphLineList::Iterator it = lineInfo->list.GetBeginIter();
            GlyphLineList::Iterator begin_it = lineInfo->list.GetBeginIter();
            GlyphLineList::Iterator end_it = lineInfo->list.GetEndIter();
            GlyphLineList::Iterator mostleft_erasable_node;
            u32 erasable_num = 0;
            for (;it != end_it; ++it)
            {
                if (it->IsErasable())
                {
                    u32 erasable_width;
                    if (erasable_num == 0)
                    {
                        mostleft_erasable_node = it;
                        erasable_width = it->GetGlyphWidth();
                    }
                    else
                    {
                        erasable_width = it->GetGlyphWidth() + it->GetCachePosX() - mostleft_erasable_node->GetCachePosX();
                    }
                    s32 mostleft_move_val = 0;
                    if (mostleft_erasable_node != begin_it)
                    {
                        // mostleft_erasable_nodeが最初のノードでなければ、左側の空きスペースを使うことができる
                        GlyphLineList::Iterator mostleft_prev_it = mostleft_erasable_node;
                        do {
                            --mostleft_prev_it;
                        } while (mostleft_prev_it->IsErased() && begin_it != mostleft_prev_it);

                        mostleft_move_val = mostleft_erasable_node->GetCachePosX() - (mostleft_prev_it->GetCachePosX() + mostleft_prev_it->GetGlyphWidth() + GLYPH_PADDING);
                        if (mostleft_move_val > 0)
                        {
                            erasable_width += mostleft_move_val;
                        }
                    }
                    if (erasable_width >= glyphWidth)
                    {
                        // 消去可能な幅が必要な文字幅を超えた
                        // もう一度消去されることがないように消去フラグを立て、消去予約済みリストに入れる
                        GlyphLineList::Iterator erasable_it = mostleft_erasable_node;
                        GlyphLineList::Iterator erasable_end_it = it;
                        ++erasable_end_it;
                        for (; erasable_it != erasable_end_it; ++erasable_it)
                        {
                            if ( ! erasable_it->IsErased())
                            {
                                erasable_it->SetErased(true);
                                mNeedEraseGlyphList.PushBack(&(*erasable_it));
                            }
                        }
                        if (mostleft_move_val > 0)
                        {
                            // mostleft_erasable_nodeの左側の空きスペースを使っている場合はCachePosXを動かす
                            mostleft_erasable_node->SetCachePosX(static_cast<u16>(mostleft_erasable_node->GetCachePosX() - mostleft_move_val));
                        }
                        return &(*mostleft_erasable_node);
                    }
                    erasable_num += 1;
                }
                else
                {
                    erasable_num = 0;
                }
            }
        }
    }
    return NULL;
}

void TextureCache::CompleteTextureCache()
{
#if defined(NW_PLATFORM_WIN32) || defined(NW_USE_NINTENDO_SDK)
    glBindTexture(GL_TEXTURE_2D_ARRAY, m_TexObj.GetName());
    glTexImage3D(GL_TEXTURE_2D_ARRAY, 0, GL_ALPHA8, mTextureCacheWidth, mTextureCacheHeight, 1, 0, GL_ALPHA, GL_UNSIGNED_BYTE, mTextureCacheBitMap);
    NW_GL_ASSERT();
#endif

    // 消去予約済みリストに入っているノードを破棄する
    // 消去予約済みリストに入るのは、新しい文字を表示する必要があって削除する場合と、
    // プロットすることができなかったために削除する場合の二通りある。
    // 前者はIsErasedがtrueになり、後者はfalseになるので区別できる。後者はラインの
    // リストから削除する必要がない。
    GlyphList::Iterator step_it = mNeedEraseGlyphList.GetBeginIter();
    GlyphList::Iterator end_it = mNeedEraseGlyphList.GetEndIter();
    for (;step_it != end_it;)
    {
        GlyphList::Iterator it = step_it;
        ++step_it;
        mNeedEraseGlyphList.Erase(it);
        if (it->IsErased())
        {
            GlyphLineList& list = mLineInfos[it->GetLineNo()].list;
            list.Erase(&(*it));
            // グリフが消えたことによりラインの右側が空く可能性があるので、currentXを設定する
            if (list.GetSize() > 0)
            {
                mLineInfos[it->GetLineNo()].currentX = list.GetBack().GetCachePosX() + list.GetBack().GetGlyphWidth() + GLYPH_PADDING;
            }
            else
            {
                mLineInfos[it->GetLineNo()].currentX = 0;
            }
        }
        mGlyphTreeMap.Erase(it->GetCode(), it->GetSize(), it->GetFace());
    }

    mGlyphTreeMap.UpdateFlagsForCompleteTextureCache();
}

void TextureCache::ClearLockAllGlyphs(s32 lockGroup)
{
    mGlyphTreeMap.ClearLockGroup(1 << lockGroup);
}

void TextureCache::ClearLockGlyphsWithLength(const char16* codes, u32 codeLength, u32 fontSize, u16 fontFace, s32 lockGroup)
{
    u32 groupLockMask = 1 << lockGroup;

    for (u32 i = 0; i < codeLength; i++)
    {
        if (codes[i] == 0)
        {
            break;
        }

        GlyphNode* node = mGlyphTreeMap.Find(codes[i], static_cast<u16>(fontSize), fontFace);
        if (node)
        {
            node->GetLockGroup().SetMaskOff(groupLockMask);
        }
    }
}

void TextureCache::ResetTextureCache()
{
#if defined(NW_PLATFORM_WIN32) || defined(NW_USE_NINTENDO_SDK)
    std::memset(mTextureCacheBitMap, 0x0, mTextureCacheWidth * mTextureCacheHeight);
#elif defined(NW_PLATFORM_CAFE)
    // テクスチャキャッシュを更新するときにキャッシュストアされるので、この時点ではDCZeroRangeでよい
    DCZeroRange(mTextureCacheBitMap, mTextureCacheWidth * mTextureCacheHeight);
#endif
    mNeedPlotGlyphList.Clear();
    mNeedEraseGlyphList.Clear();
    mNotInFontGlyphList.Clear();

    // ラインをリセット
    for (u32 i = 0; i < mLineCurrentPos; i++)
    {
        mLineInfos[i].list.Clear();
    }
    mLineCurrentPos = 0;

    // 平衡二分木をリセット
    mGlyphTreeMap.Reset();
}

bool TextureCache::IsGlyphExistInFont(char16 code, u16 fontFace)
{
    u32 coreId = GetCoreId();
    SetFontFaceNoPlot(fontFace, coreId);
    return mFsStatesNoPlot[coreId].CheckGlyphExist(code);
}

const TextureCache::FontMetrics& TextureCache::GetFontMetrics(u16 fontFace)
{
    NW_ASSERT(fontFace < mFontNum);
    return mFontMetrics[fontFace];
}

int TextureCache::CalcCharWidth(char16 code, u32 fontSize, u16 fontFace)
{
    s16 i_dx, i_dy;
    scffnd::ScFixed dx, dy;
    u32 coreId = GetCoreId();
    SetFontFaceNoPlot(fontFace, coreId);
    mFsStatesNoPlot[coreId].SetScale(fontSize << 16, 0, 0, fontSize << 16);
    AssertFsErrorNoPlot("SetScale", coreId);
    if (mFsStatesNoPlot[coreId].GetAdvance(code, scffnd::MAP_GRAYMAP8, &i_dx, &i_dy, &dx, &dy) == scffnd::SUCCESS)
    {
        return i_dx;
    }
    else
    {
        return 0;
    }
}

int TextureCache::CalcKerning(char16 c0, char16 c1, u32 fontSize, u16 fontFace)
{
    scffnd::ScFixed dx, dy;
    u32 coreId = GetCoreId();
    SetFontFaceNoPlot(fontFace, coreId);
    mFsStatesNoPlot[coreId].SetScale(fontSize << 16, 0, 0, fontSize << 16);
    AssertFsErrorNoPlot("SetScale", coreId);
    if (mFsStatesNoPlot[coreId].GetKerning(c0, c1, &dx, &dy) == scffnd::SUCCESS)
    {
        return (dx >> 16);
    }
    else
    {
        return 0;
    }
}

void TextureCache::SetFontFace(u32 fontFace)
{
    if (fontFace != mCurrentFontFace)
    {
        // 今から使用するフォントの設定
        // 指定する文字のエンコーディングを設定
        mFsState->SetFont(GetFontNameBuf(fontFace));
        AssertFsError("SetFont");

        // ボールドのウェイトの設定
        mFsState->SetBoldWeight(mBoldWeights[fontFace]);
        AssertFsError("SetBoldWeight");

        // 縁取りの設定
        if (mBorderWidths[fontFace])
        {
            mFsState->SetOutlineWidth(mBorderWidths[fontFace]);
            mFsState->SetFlags(scffnd::ScalableFontEngine::FLG_OUTLINED_FILLED);
        }
        else
        {
            mFsState->SetFlags(scffnd::ScalableFontEngine::FLG_NO_EFFECT);
        }
        AssertFsError("SetFlags");

        mCurrentFontFace = fontFace;
    }
}

void TextureCache::SetFontFaceNoPlot(u32 fontFace, u32 coreId)
{
    if (fontFace != mCurrentFontFacesNoPlot[coreId])
    {
        // 今から使用するフォントの設定
        // 指定する文字のエンコーディングを設定
        mFsStatesNoPlot[coreId].SetFont(GetFontNameBuf(fontFace));
        AssertFsErrorNoPlot("SetFont", coreId);

        // ボールドのウェイトの設定
        mFsStatesNoPlot[coreId].SetBoldWeight(mBoldWeights[fontFace]);
        AssertFsErrorNoPlot("SetBoldWeight", coreId);

        // 縁取りの設定
        if (mBorderWidths[fontFace])
        {
            mFsStatesNoPlot[coreId].SetOutlineWidth(mBorderWidths[fontFace]);
            mFsStatesNoPlot[coreId].SetFlags(scffnd::ScalableFontEngine::FLG_OUTLINED_FILLED);
        }
        else
        {
            mFsStatesNoPlot[coreId].SetFlags(scffnd::ScalableFontEngine::FLG_NO_EFFECT);
        }
        AssertFsErrorNoPlot("SetFlags", coreId);

        mCurrentFontFacesNoPlot[coreId] = fontFace;
    }
}

void TextureCache::AssertFsError(const char* api_name)
{
    NW_UNUSED_VARIABLE(api_name);
    NW_ASSERTMSG(mFsState->GetError() == scffnd::NO_ERR, "%s error: %d", api_name, mFsState->GetError());
}

void TextureCache::AssertFsErrorNoPlot(const char* api_name, u32 coreId)
{
    NW_UNUSED_VARIABLE(api_name);
    NW_ASSERTMSG(mFsStatesNoPlot[coreId].GetError() == scffnd::NO_ERR, "%s error: %d", api_name, mFsStatesNoPlot[coreId].GetError());
}

u32 TextureCache::EraseNotInFontGlyphs()
{
    u32 count = 0;
    GlyphList::Iterator step_it = mNotInFontGlyphList.GetBeginIter();
    GlyphList::Iterator end_it = mNotInFontGlyphList.GetEndIter();
    for (;step_it != end_it;)
    {
        GlyphList::Iterator it = step_it;
        ++step_it;
        // 現在使われている文字については削除しない
        if ( ! it->IsRequestedOrKeeped()) {
            mNotInFontGlyphList.Erase(it);
            mGlyphTreeMap.Erase(it->GetCode(), it->GetSize(), it->GetFace());
            count++;
        }
    }
    return count;
}


} // namespace nw::scfont
} // namespace nw
