﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nw/lyt/lyt_TextureContainer.h>

#include <nw/lyt/lyt_Layout.h>
#include <nw/lyt/lyt_Util.h>
#include <nw/ut/ut_String.h>

#if defined(NW_PLATFORM_WIN32) || defined(NW_USE_NINTENDO_SDK)
#include <gl/glew.h>
#endif

namespace nw
{
namespace lyt
{

//----------------------------------------
TextureRefLink::TextureRefLink()
{
    m_Name[0] = '\0';
}

//----------------------------------------
TextureRefLink::~TextureRefLink()
{
#if defined(NW_PLATFORM_WIN32) || defined(NW_USE_NINTENDO_SDK)
    GLuint texName = m_TexInfo.GetTextureObject();
    if (texName != m_TexInfo.INVALID)
    {
        glDeleteTextures(1, &texName);
    }
#endif
}

//----------------------------------------
void
TextureRefLink::SetName(
    const char* name)
{
    NW_ASSERT_NOT_NULL(name);

    NW_ASSERT(std::strlen(name) < TexImageNameMax);
    ut::strcpy(m_Name, sizeof(m_Name), name);
}

//----------------------------------------
TextureContainer::~TextureContainer()
{
    this->Finalize();
}

//----------------------------------------
void
TextureContainer::Finalize()
{
    while (!this->empty())
    {
        TextureRefLink* pLink = &this->front();
        this->erase(pLink);
        Layout::DeleteObj(pLink);
    }
}

//----------------------------------------
TextureInfo*
TextureContainer::RegistTexture(const char* name)
{
    NW_ASSERT(name);

    TextureRefLink* pLink = Layout::NewObj<TextureRefLink>();
    if (pLink == NULL)
    {
        return NULL;
    }

    pLink->SetName(name);

    this->push_back(pLink);

    return &pLink->GetTextureInfo();
}

//----------------------------------------
void
TextureContainer::UnregistTexture(TextureInfo* info)
{
    NW_ASSERT_NOT_NULL(info);

    for (Iterator iter = this->GetBeginIter(); iter != this->GetEndIter(); ++iter)
    {
        if (&iter->GetTextureInfo() == info)
        {
            this->erase(iter);
            Layout::DeleteObj(&(*iter));
            break;
        }
    }
}

//----------------------------------------
const TextureInfo*
TextureContainer::FindTextureByName(const char* name) const
{
    ConstIterator iter_end = this->GetEndIter();
    for (ConstIterator iter = this->GetBeginIter(); iter != iter_end; ++iter)
    {
        if (internal::EqualsTexImageName(name, iter->GetResourceName()))
        {
            return &iter->GetTextureInfo();
        }
    }

    return NULL;
}

} // namespace nw::lyt
} // namespace nw
