﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nw/lyt/lyt_Common.h>
#include <nw/lyt/lyt_DrawInfo.h>
#include <nw/lyt/lyt_GraphicsResource.h>
#include <nw/lyt/lyt_Material.h>
#include <nw/lyt/lyt_Layout.h>
#include <nw/lyt/lyt_Resources.h>
#include <nw/lyt/lyt_Animation.h>
#include <nw/lyt/lyt_ResourceAccessor.h>
#include <nw/lyt/lyt_Util.h>
#include <nw/lyt/lyt_MaterialHelper.h>

#include <nw/gfnd/gfnd_ShaderHelper.h>
namespace shhelp = nw::gfnd::shader_helper;

#include <nw/dev/dev_Profile.h>

namespace nw
{
namespace lyt
{

//----------------------------------------
Material::Material()
{
    Init();
}

//----------------------------------------
Material::Material(
    const res::Material* pBaseRes,
    const res::Material* pOverrideRes,
    const BuildArgSet& buildArgSet
)
{
    Init();
    // 上書き設定の解決
    const res::Material* pRes = NULL;
    const res::Material* pColorRes = NULL;
    const BuildResSet* buildResSet = NULL;
    if (pOverrideRes)
    {
        // 上書きのリソースがある
        if (internal::TestBit(buildArgSet.overrideMaterialUsageFlag, MATERIALOVERRIDEUSAGEFLAG_INTERPOLATECOLOR_ENABLED))
        {
            // カラーの部分上書き
            pRes = pBaseRes;
            pColorRes = pOverrideRes;
            buildResSet = buildArgSet.pCurrentBuildResSet;
        }
        else if (buildArgSet.overrideUsageFlag != 0)
        {
            // 一つでもTextBoxの部分上書きのフラグが立っていたら、マテリアルは上書きしない
            pRes = pBaseRes;
            pColorRes = pBaseRes;
            buildResSet = buildArgSet.pCurrentBuildResSet;
        }
        else
        {
            // 完全に上書き
            pRes = pOverrideRes;
            pColorRes = pOverrideRes;
            buildResSet = buildArgSet.pOverrideBuildResSet;
        }
    }
    else
    {
        // 上書きがない
        pRes = pBaseRes;
        pColorRes = pBaseRes;
        buildResSet = buildArgSet.pCurrentBuildResSet;
    }

    SetName(pRes->name);

    // constant color
    for (int i = 0; i < MatColorMax; ++i)
    {
        m_Colors[i] = pColorRes->colors[i];
    }

    u32 resOffs = sizeof(res::Material);

    const res::TexMap *const pResTexMap = internal::ConvertOffsToPtr<res::TexMap>(pRes, resOffs);
    resOffs += sizeof(res::TexMap) * pRes->resNum.GetTexMapNum();

    const res::TexSRT *const pResTexSRTs = internal::ConvertOffsToPtr<res::TexSRT>(pRes, resOffs);
    resOffs += sizeof(res::TexSRT) * pRes->resNum.GetTexSRTNum();

    const res::TexCoordGen *const pResTexCoordGens = internal::ConvertOffsToPtr<res::TexCoordGen>(pRes, resOffs);
    resOffs += sizeof(res::TexCoordGen) * pRes->resNum.GetTexCoordGenNum();

    const res::TevStage *const pResTevStages = internal::ConvertOffsToPtr<res::TevStage>(pRes, resOffs);
    resOffs += sizeof(res::TevStage) * pRes->resNum.GetTevStageNum();

    const res::AlphaCompare *const pResAlphaCompare = internal::ConvertOffsToPtr<res::AlphaCompare>(pRes, resOffs);
    resOffs += pRes->resNum.HasAlphaCompare()? sizeof(res::AlphaCompare) : 0;

    const res::BlendMode *const pResBlendMode = internal::ConvertOffsToPtr<res::BlendMode>(pRes, resOffs);
    resOffs += pRes->resNum.HasBlendMode()? sizeof(res::BlendMode) : 0;

    const res::BlendMode *const pResBlendModeAlpha = internal::ConvertOffsToPtr<res::BlendMode>(pRes, resOffs);
    resOffs += pRes->resNum.IsSeparateBlendMode()? sizeof(res::BlendMode) : 0;

    const res::IndirectParameter *const pResIndirectParameter = internal::ConvertOffsToPtr<res::IndirectParameter>(pRes, resOffs);
    resOffs += pRes->resNum.HasIndirectParameter()? sizeof(res::IndirectParameter) : 0;

    const res::ProjectionTexGenParamaters *const pResProjectionTexGenParameter = internal::ConvertOffsToPtr<res::ProjectionTexGenParamaters>(pRes, resOffs);
    resOffs += sizeof(res::ProjectionTexGenParamaters) * pRes->resNum.GetProjectionTexGenNum();

    const res::FontShadowParameter *const pFontShadowParameter = internal::ConvertOffsToPtr<res::FontShadowParameter>(pRes, resOffs);
    resOffs += pRes->resNum.HasFontShadowParameter()? sizeof(res::FontShadowParameter) : 0;

    const u8 texMapNum = ut::Min(pRes->resNum.GetTexMapNum(), u8(TexMapMax));
    const u8 texSRTNum = ut::Min(pRes->resNum.GetTexSRTNum(), u8(TexMapMax));
    const u8 texCoordGenNum = ut::Min(pRes->resNum.GetTexCoordGenNum(), u8(TexMapMax));
    const u8 tevStageNum = ut::Min(pRes->resNum.GetTevStageNum(), u8(TevStageMax));
    const u8 projectionTexGenNum = ut::Min(pRes->resNum.GetProjectionTexGenNum(), u8(TevStageMax));
    const bool allocAlpComp = pRes->resNum.HasAlphaCompare();
    const bool allocBlendMode = pRes->resNum.HasBlendMode();
    const bool allocBlendModeAlpha = pRes->resNum.IsSeparateBlendMode();
    const bool allocIndirectParameter = pRes->resNum.HasIndirectParameter();
    const bool allocFontShadowParameter = pRes->resNum.HasFontShadowParameter();
    u8 blendModeNum;
    if (allocBlendModeAlpha) {
        blendModeNum = 2;
    } else {
        if (allocBlendMode) {
            blendModeNum = 1;
        } else {
            blendModeNum = 0;
        }
    }

    SetThresholdingAlphaInterpolation(pRes->resNum.IsThresholdingAlphaInterpolation());

    // TODO
    ReserveMem(
        texMapNum,
        texSRTNum,
        texCoordGenNum,
        tevStageNum,
        allocAlpComp,
        blendModeNum,
        allocIndirectParameter,
        projectionTexGenNum,
        allocFontShadowParameter
        );

    this->SetTextureOnly(pRes->resNum.IsTextureOnly());

    if (m_pMem)
    {
        // texmap
        for (u32 i = 0; i < m_MemCap.texMap; ++i)
        {
            new (GetTexMapAry() + i) TexMap();
        }

        SetTexMapNum(texMapNum);
        if (texMapNum > 0)
        {
            NW_ASSERTMSG(buildResSet->pTextureList, "buildResSet->pTextureList must not be NULL for Material[%s] Layout[%s]", GetName(), buildResSet->pLayout->GetName());

            const res::Texture *const pResTextures = internal::ConvertOffsToPtr<res::Texture>(buildResSet->pTextureList, sizeof(*buildResSet->pTextureList));
            TexMap *const texMaps = GetTexMapAry();

            u8 di = 0;
            for (u8 si = 0; si < texMapNum; ++si)
            {
                int texIdx = pResTexMap[si].texIdx;
                NW_ASSERTMSG(texIdx < buildResSet->pTextureList->texNum,
                    "out of bouds: texIdx[%d] < buildResSet->pTextureList->texNum[%d] for Material[%s] Layout[%s]",
                    texIdx, buildResSet->pTextureList->texNum, GetName(), buildResSet->pLayout->GetName());
                const char* fileName = internal::ConvertOffsToPtr<char>(pResTextures, pResTextures[texIdx].nameStrOffset);

                const TextureInfo* texInfo = buildResSet->pResAccessor->GetTexture(fileName);
                texMaps[di].Set(texInfo);
                texMaps[di].SetWrapMode(pResTexMap[si].GetWarpModeS(), pResTexMap[si].GetWarpModeT());
                texMaps[di].SetFilter(pResTexMap[si].GetMinFilter(), pResTexMap[si].GetMagFilter());
                ++di;
            }
        }

        // テクスチャSRT
        res::TexSRT* texSRTs = GetTexSRTAry();
        for (u32 i = 0; i < m_MemCap.texSRT; ++i)
        {
            new(static_cast<void*>(&texSRTs[i])) res::TexSRT();
        }
        SetTexSRTNum(texSRTNum);
        for (int i = 0; i < texSRTNum; ++i)
        {
            texSRTs[i].translate = pResTexSRTs[i].translate;
            texSRTs[i].rotate    = pResTexSRTs[i].rotate;
            texSRTs[i].scale     = pResTexSRTs[i].scale;
        }

        // テクスチャ座標生成
        res::TexCoordGen* texCoordGens = GetTexCoordGenAry();
        for (u32 i = 0; i < m_MemCap.texCoordGen; ++i)
        {
            new(static_cast<void*>(&texCoordGens[i])) res::TexCoordGen();
        }
        SetTexCoordGenNum(texCoordGenNum);
        u32 numProjectionTexGen = 0;
        for (u32 i = 0; i < m_MemNum.texCoordGen; ++i)
        {
            texCoordGens[i] = pResTexCoordGens[i];

            res::ProjectionTexGenParamaters *const projectionTexGen = GetProjectionTexGenAry();
            if(texCoordGens[i].IsProjection())
            {
                NW_ASSERTMSG(numProjectionTexGen < projectionTexGenNum,
                    "out of bouds: numProjectionTexGen[%d] < projectionTexGenNum[%d] for Material[%s] Layout[%s]",
                    numProjectionTexGen, projectionTexGenNum, GetName(), buildResSet->pLayout->GetName());
                texCoordGens[i].SetProjectionTexGenParamaters(&projectionTexGen[numProjectionTexGen]);
                numProjectionTexGen++;
            }else{
                texCoordGens[i].SetProjectionTexGenParamaters(NULL);
            }
        }

        if (tevStageNum > 0)
        {
            res::TevStage *const tevStages = GetTevStageAry();
            for (u32 i = 0; i < m_MemCap.tevStage; ++i)
            {
                new(static_cast<void*>(&tevStages[i])) res::TevStage();
            }
            SetTevStageNum(tevStageNum);
            for (int i = 0; i < tevStageNum; ++i)
            {
                tevStages[i] = pResTevStages[i];
            }
        }

        if (allocAlpComp)
        {
            NW_ASSERTMSG(IsAlphaCompareCap(), "IsAlphaCompareCap() must be true for Material[%s] Layout[%s]",
                GetName(), buildResSet->pLayout->GetName());
            *GetAlphaComparePtr() = *pResAlphaCompare;
        }

        if (allocBlendMode)
        {
            NW_ASSERTMSG(IsBlendModeCap(), "IsBlendModeCap() must be true for Material[%s] Layout[%s]",
                GetName(), buildResSet->pLayout->GetName());
            *GetBlendModePtr() = *pResBlendMode;
        }

        if (allocBlendModeAlpha)
        {
            NW_ASSERTMSG(IsBlendModeAlphaCap(), "IsBlendModeAlphaCap() must be true for Material[%s] Layout[%s]",
                GetName(), buildResSet->pLayout->GetName());
            *GetBlendModeAlphaPtr() = *pResBlendModeAlpha;
        }

        if (allocIndirectParameter)
        {
            NW_ASSERTMSG(IsIndirectParameterCap(), "IsIndirectParameterCap() must be true for Material[%s] Layout[%s]",
                GetName(), buildResSet->pLayout->GetName());
            *GetIndirectParameterPtr() = *pResIndirectParameter;
        }

        if (projectionTexGenNum > 0)
        {
            res::ProjectionTexGenParamaters *const projectionTexGen = GetProjectionTexGenAry();
            for (u32 i = 0; i < m_MemCap.projectionTexGen; ++i)
            {
                new(static_cast<void*>(&projectionTexGen[i])) res::ProjectionTexGenParamaters();
            }
            SetProjectionTexGenNum(projectionTexGenNum);
            for (int i = 0; i < projectionTexGenNum; ++i)
            {
                projectionTexGen[i] = pResProjectionTexGenParameter[i];
            }
        }

        if (allocFontShadowParameter)
        {
            NW_ASSERTMSG(IsFontShadowParameterCap(), "IsFontShadowParameterCap() must be true for Material[%s] Layout[%s]",
                GetName(), buildResSet->pLayout->GetName());
            *GetFontShadowParameterPtr() = *pFontShadowParameter;
        }
    }

    //----------------------------------------
    // シェーダの選択とインダイレクトテクスチャの設定。
    bool indirectEnabled = false;
    for (int idx = 0; idx < this->GetTevStageNum(); ++idx)
    {
        TevMode tevMode = this->GetTevStage(idx).GetCombineRgb();
        if (tevMode == TEVMODE_INDIRECT ||
            tevMode == TEVMODE_BLEND_INDIRECT ||
            tevMode == TEVMODE_EACH_INDIRECT)
        {
            indirectEnabled = true;
            break;
        }
    }

    char shaderNameBuf[6];
    shaderNameBuf[0] = '\0';
    m_ShaderInfo = NULL;
    if (m_MemNum.texMap == 0)
    {
        m_ShaderId = NULL_TEXTURE_SHADER;
    }
    else if (m_MemNum.texMap == 1)
    {
        m_ShaderId = SINGLE_TEXTURE_SHADER;
    }
    else if (indirectEnabled)
    {
        switch (m_MemNum.texMap)
        {
        case 2:
            m_ShaderId = DOUBLE_INDIRECT_TEXTURE_SHADER;
            break;
        case 3:
            GetShaderName(shaderNameBuf);
            break;
        default:
            break;
        }

        TexMap *const texMaps = GetTexMapAry();
        TevMode lastMode = TEVMODE_REPLACE;
        for (int i = 0; i < this->GetTevStageNum(); ++i)
        {
            int changeCompSel = 0;
            int textureIndex = i + 1;
            const res::TevStage& tevStage = this->GetTevStage(i);
            TevMode tevMode = tevStage.GetCombineRgb();
            if (tevMode == TEVMODE_INDIRECT ||
                tevMode == TEVMODE_BLEND_INDIRECT ||
                tevMode == TEVMODE_EACH_INDIRECT ||
                lastMode == TEVMODE_BLEND_INDIRECT)
            {
                if (textureIndex < GetTexMapNum())
                {
                    if (texMaps[textureIndex].GetFormat() == TEXFORMAT_LA4 ||
                        texMaps[textureIndex].GetFormat() == TEXFORMAT_LA8 ||
                        texMaps[textureIndex].GetFormat() == TEXFORMAT_BC5)
                    {
                        changeCompSel = 1;
                    }
                    else if (texMaps[textureIndex].GetFormat() == TEXFORMAT_RGB8 ||
                        texMaps[textureIndex].GetFormat() == TEXFORMAT_RGB565 ||
                        texMaps[textureIndex].GetFormat() == TEXFORMAT_BC1 ||
                        texMaps[textureIndex].GetFormat() == TEXFORMAT_BC1_SRGB)
                    {
                        changeCompSel = 2;
                    }
                }
            }

            switch (changeCompSel)
            {
            case 1:
                {
#if defined(NW_PLATFORM_CAFE)
                    GX2Texture* texture = const_cast<GX2Texture*>(texMaps[textureIndex].GetTexture());
                    GX2InitTextureCompSel(texture, GX2_GET_COMP_SEL(GX2_COMPONENT_X_R, GX2_COMPONENT_Y_G, GX2_COMPONENT_C_0, GX2_COMPONENT_C_1));
                    GX2InitTextureRegs(texture);
#elif defined(NW_PLATFORM_WIN32) || defined(NW_USE_NINTENDO_SDK)
                    glBindTexture(GL_TEXTURE_2D, texMaps[textureIndex].GetTextureObject());
                    static const GLint swizzle[4] = { GL_RED, GL_ALPHA, GL_ZERO, GL_ONE };
                    glTexParameteriv(GL_TEXTURE_2D, GL_TEXTURE_SWIZZLE_RGBA, swizzle);
#endif
                }
                break;
            case 2:
                {
#if defined(NW_PLATFORM_CAFE)
                    GX2Texture* texture = const_cast<GX2Texture*>(texMaps[textureIndex].GetTexture());
                    GX2InitTextureCompSel(texture, GX2_GET_COMP_SEL(GX2_COMPONENT_X_R, GX2_COMPONENT_Y_G, GX2_COMPONENT_C_0, GX2_COMPONENT_Z_B));
                    GX2InitTextureRegs(texture);
#elif defined(NW_PLATFORM_WIN32) || defined(NW_USE_NINTENDO_SDK)
                    glBindTexture(GL_TEXTURE_2D, texMaps[textureIndex].GetTextureObject());
                    static const GLint swizzle[4] = { GL_RED, GL_GREEN, GL_ZERO, GL_BLUE };
                    glTexParameteriv(GL_TEXTURE_2D, GL_TEXTURE_SWIZZLE_RGBA, swizzle);
#endif
                }
                break;
            default:
                break;
            }

            lastMode = tevStage.GetCombineRgb();
        }
    }
    else
    {
        if (m_MemNum.texMap > 2)
        {
            GetShaderName(shaderNameBuf);
        }
        else
        {
            const res::TevStage& tevStage = this->GetTevStage(0);
            switch (tevStage.GetCombineRgb())
            {
            case 0: // REPLACE モードです。
                m_ShaderId = DOUBLE_TEXTURE1_SHADER;
                break;
            case 1: // MODULATE モードです。
                m_ShaderId = DOUBLE_TEXTURE2_SHADER;
                break;
            case 2: // ADD モードです。
                m_ShaderId = DOUBLE_TEXTURE3_SHADER;
                break;
            case 3: // EXCLUSION モードです。
                m_ShaderId = DOUBLE_TEXTURE5_SHADER;
                break;
            case 4: // REPLACE モードです。
                m_ShaderId = DOUBLE_TEXTURE1_SHADER;
                break;
            case 5: // SUBTRACT モードです。
                m_ShaderId = DOUBLE_TEXTURE4_SHADER;
                break;
            case 6: // COLOR_DODGE モードです。
                m_ShaderId = DOUBLE_TEXTURE6_SHADER;
                break;
            case 7: // COLOR_BURN モードです。
                m_ShaderId = DOUBLE_TEXTURE7_SHADER;
                break;
            case 8: // OVERLAY モードです。
                m_ShaderId = DOUBLE_TEXTURE8_SHADER;
                break;
            default:
                m_ShaderId = DOUBLE_TEXTURE1_SHADER;
                break;
            }
        }
    }

#if defined(NW_PLATFORM_CAFE)
    // shaderNameBufが設定されていればアーカイブシェーダを使用する
    if (shaderNameBuf[0] != '\0') {
        m_ShaderInfo = buildResSet->pLayout->GetArchiveShader(shaderNameBuf);
    }
#endif
}

//----------------------------------------
Material::Material(const Material& material)
 : m_pMem(NULL)
 , m_ShaderInfo(material.m_ShaderInfo)
 , m_ShaderId(material.m_ShaderId)
 , m_Flag(material.m_Flag)
{
    SetName(material.GetName());

    // constant color
    for (int i = 0; i < MatColorMax; ++i)
    {
        m_Colors[i] = material.m_Colors[i];
    }

    InitMatMemNums(&m_MemCap);
    InitMatMemNums(&m_MemNum);

    ReserveMem(
        material.m_MemCap.texMap,
        material.m_MemCap.texSRT,
        material.m_MemCap.texCoordGen,
        material.m_MemCap.tevStage,
        material.m_MemCap.alpComp > 0,
        material.m_MemCap.blendMode,
        material.m_MemCap.indirectParameter > 0,
        material.m_MemCap.projectionTexGen,
        material.m_MemCap.fontShadowParameter
        );

    // TextureOnlyフラグはm_Flagのコピーでコピーされる

    if (m_pMem)
    {
        SetTexMapNum(material.m_MemNum.texMap);
        SetTexSRTNum(material.m_MemNum.texSRT);
        SetTexCoordGenNum(material.m_MemNum.texCoordGen);
        SetTevStageNum(material.m_MemNum.tevStage);
        SetProjectionTexGenNum(material.m_MemNum.projectionTexGen);
        SetFontShadowParameterNum(material.m_MemNum.fontShadowParameter);

        u32 copySize = sizeof(TexMap) * m_MemCap.texMap +
            sizeof(res::TexSRT) * m_MemCap.texSRT +
            sizeof(res::TexCoordGen) * m_MemCap.texCoordGen +
            sizeof(res::TevStage) * m_MemCap.tevStage +
            sizeof(res::AlphaCompare) * m_MemCap.alpComp +
            sizeof(res::BlendMode) * m_MemCap.blendMode +
            sizeof(res::IndirectParameter) * m_MemCap.indirectParameter +
            sizeof(res::ProjectionTexGenParamaters) * m_MemCap.projectionTexGen +
            sizeof(res::FontShadowParameter) * m_MemCap.fontShadowParameter;

        ut::MemCpy(m_pMem, material.m_pMem, copySize);
    }
}

//----------------------------------------
void
Material::Init()
{
    const u32 DefaultBlackColor = 0x00000000;
    const u32 DefaultWhiteColor = 0xffffffff;
    m_Colors[INTERPOLATECOLOR_BLACK] = ut::Color8(DefaultBlackColor);
    m_Colors[INTERPOLATECOLOR_WHITE] = ut::Color8(DefaultWhiteColor);

    InitMatMemNums(&m_MemCap);
    InitMatMemNums(&m_MemNum);

    m_Name = NULL;
    m_Flag = 0;
    m_pMem = 0;
    m_ShaderInfo = NULL;
    m_ShaderId = DEFAULT_SHADER;
}

//----------------------------------------
Material::~Material()
{
#if defined(NW_PLATFORM_WIN32) || defined(NW_USE_NINTENDO_SDK)
    // テクスチャ
    for (u32 i = 0; i < m_MemCap.texMap; ++i)
    {
        GetTexMapAry()[i].~TexMap();
    }
#endif

    if (m_pMem)
    {
        Layout::FreeMemory(m_pMem);
        m_pMem = 0;
    }
}

//----------------------------------------
void
Material::InitMatMemNums(internal::MatMemNums* ptr)
{
    ptr->texMap = 0;
    ptr->texSRT = 0;
    ptr->texCoordGen = 0;
    ptr->tevStage = 0;
    ptr->alpComp = 0;
    ptr->blendMode = 0;
    ptr->indirectParameter = 0;
    ptr->projectionTexGen = 0;
    ptr->fontShadowParameter = 0;
}

//----------------------------------------
void
Material::GetShaderName(char* shader_name_buf)
{
    s32 id[2] =
    {
        static_cast<s32>(this->GetTevStage(0).GetCombineRgb()) + 1,
        static_cast<s32>(this->GetTevStage(1).GetCombineRgb()) + 1
    };
    // レイアウトエディタとランタイムのブレンド番号を合わせる
    for (s32 i = 0; i < 2; i++)
    {
        switch (id[i])
        {
            case 6: id[i] = 4; break;
            case 4: id[i] = 5; break;
            case 7: id[i] = 6; break;
            case 8: id[i] = 7; break;
            case 9: id[i] = 8; break;
            case 10: id[i] = 9; break;
            case 11: id[i] = 10; break;
            case 12: id[i] = 11; break;
        }
    }

    // 数値を"xx_xx"の形式の名前に変換し、shader_name_bufに設定
    NW_ASSERT(1 <= id[0] && id[0] <= 99 && 1 <= id[1] && id[1] <= 99);
    char* p = shader_name_buf;
    s32 num;
    num = id[0] / 10;
    if (num != 0)
    {
        *p = static_cast<char>(num + '0');
        p++;
        id[0] -= num * 10;
    }
    *p = static_cast<char>(id[0] + '0');
    p++;
    *p = '_';
    p++;
    num = id[1] / 10;
    if (num != 0)
    {
        *p = static_cast<char>(num + '0');
        p++;
        id[1] -= num * 10;
    }
    *p = static_cast<char>(id[1] + '0');
    p++;
    *p = '\0';
}

//----------------------------------------
bool
Material::ReserveMem(
    u8 texMapNum,
    u8 texSRTNum,
    u8 texCoordGenNum,
    u8 tevStageNum,
    bool allocAlpComp,
    u8 blendModeNum,
    bool allocIndirectParameter,
    u8 projectionTexGenNum,
    bool fontShadowParameter
)
{
    bool result = true;

    NW_ASSERTMSG(texMapNum <= TexMapMax, "out of bounds: texMapNum[%d] <= TexMapMax for Material[%s]", texMapNum, GetName());
    NW_ASSERTMSG(texSRTNum <= TexMapMax, "out of bounds: texSRTNum[%d] <= TexMapMax for Material[%s]", texSRTNum, GetName());
    NW_ASSERTMSG(texCoordGenNum <= TexMapMax, "out of bounds: texCoordGenNum[%d] <= TexMapMax for Material[%s]", texCoordGenNum, GetName());
    NW_ASSERTMSG(tevStageNum <= TevStageMax, "out of bounds: tevStageNum[%d] <= TevStageMax for Material[%s]", tevStageNum, GetName());
    NW_ASSERTMSG(projectionTexGenNum <= TevStageMax, "out of bounds: projectionTexGenNum[%d] <= TevStageMax for Material[%s]", projectionTexGenNum, GetName());

    const u32 alpCompNum = static_cast<u32>(allocAlpComp ? 1: 0);
    const u32 indirectParameterNum = static_cast<u32>(allocIndirectParameter ? 1: 0);
    const u32 fontShadowParameterNum = static_cast<u32>(fontShadowParameter ? 1 : 0);

    // いずれかが要求しているサイズのほうが大きい場合
    if ( m_MemCap.texMap < texMapNum
      || m_MemCap.texSRT < texSRTNum
      || m_MemCap.texCoordGen < texCoordGenNum
      || m_MemCap.tevStage < tevStageNum
      || m_MemCap.alpComp < alpCompNum
      || m_MemCap.blendMode < blendModeNum
      || m_MemCap.indirectParameter < indirectParameterNum
      || m_MemCap.projectionTexGen < projectionTexGenNum
      || m_MemCap.fontShadowParameter < fontShadowParameterNum
    )
    {
        if (m_pMem)
        {
            Layout::FreeMemory(m_pMem);
            m_pMem = 0;

            InitMatMemNums(&m_MemCap);
            InitMatMemNums(&m_MemNum);
        }

        m_pMem = Layout::AllocMemory(
            sizeof(TexMap) * texMapNum +
            sizeof(res::TexSRT) * texSRTNum +
            sizeof(res::TexCoordGen) * texCoordGenNum +
            sizeof(res::AlphaCompare) * alpCompNum +
            sizeof(res::BlendMode) * blendModeNum +
            sizeof(res::IndirectParameter) * indirectParameterNum +
            sizeof(res::TevStage) * tevStageNum +
            sizeof(res::ProjectionTexGenParamaters) * projectionTexGenNum +
            sizeof(res::FontShadowParameter) * fontShadowParameterNum
            );

        if (m_pMem)
        {
            m_MemCap.texMap = texMapNum;
            m_MemCap.texSRT = texSRTNum;
            m_MemCap.texCoordGen = texCoordGenNum;
            m_MemCap.tevStage = tevStageNum;
            m_MemCap.alpComp = alpCompNum;
            m_MemCap.blendMode = blendModeNum;
            m_MemCap.indirectParameter = indirectParameterNum;
            m_MemCap.projectionTexGen = projectionTexGenNum;
            m_MemCap.fontShadowParameter = fontShadowParameterNum;

            m_MemNum.texSRT = m_MemCap.texSRT;
            InitTexSRT(GetTexSRTAry(), m_MemNum.texSRT);

            m_MemNum.alpComp = m_MemCap.alpComp;
            if (m_MemNum.alpComp)
            {
                *GetAlphaComparePtr() = res::AlphaCompare();
            }

            m_MemNum.blendMode = m_MemCap.blendMode;
            if (m_MemNum.blendMode > 0)
            {
                *GetBlendModePtr() = res::BlendMode();
                if (m_MemNum.blendMode == 2)
                {
                    *GetBlendModeAlphaPtr() = res::BlendMode();
                }
            }

            m_MemNum.indirectParameter = m_MemCap.indirectParameter;
            if (m_MemNum.indirectParameter)
            {
                *GetIndirectParameterPtr() = res::IndirectParameter();
            }

            m_MemNum.fontShadowParameter = m_MemCap.fontShadowParameter;
            if (m_MemNum.fontShadowParameter)
            {
                new (GetFontShadowParameterPtr()) res::FontShadowParameter();
            }
        }
        else
        {
            result = false;
        }
    }

    return result;
}

//----------------------------------------
void
Material::BindAnimation(AnimTransform* pAnimTrans)
{
    pAnimTrans->BindMaterial(this);
}

//----------------------------------------
void
Material::UnbindAnimation(AnimTransform* pAnimTrans)
{
    pAnimTrans->UnbindMaterial(this);
}

//----------------------------------------
u32
Material::GetProjectionTexGenParamatersIdxFromTexCoordGenIdx(u32 texCoordGenIdx) const
{
    u32 texMapNum = ut::Min<u32>(ut::Min<u32>(GetTexMapNum(), GetTexCoordGenNum()), TexMapMax);
    int projTexIndex = 0;

    for (u32 i = 0; i < texMapNum; ++i)
    {
        const res::TexCoordGen& texCoordGen = this->GetTexCoordGen(i);
        if (texCoordGen.GetTexGenSrc() == TEXGENSRC_ORTHO_PROJECTION || texCoordGen.GetTexGenSrc() == TEXGENSRC_PANE_BASE_ORTHO_PROJECTION)
        {
            if (i == texCoordGenIdx) {
                return projTexIndex;
            }
            projTexIndex++;
        }
    }

    NW_ERR("texCoordGenIdx[%d] must be index of projection maped TEXGENSRC", texCoordGenIdx);
    return 0;
}

//----------------------------------------
void
Material::CalcTextureMtx(math::MTX23* pTexMtx, const res::TexSRT& texSRT, const TexMap& texMap)
{
    NW_ASSERT_NOT_NULL(pTexMtx);
    NW_UNUSED_VARIABLE(texMap)

    math::MTX23& texMtx = *pTexMtx;

    const math::VEC2 center(0.5f, 0.5f);

    f32 sinR, cosR;
    math::SinCosDeg(&sinR, &cosR, texSRT.rotate);

    f32 a0, a1;

    a0 = cosR * texSRT.scale.x;
    a1 = -sinR * texSRT.scale.y;

    // SRT の順
    texMtx.m[0][0] = a0;
    texMtx.m[0][1] = a1;
    texMtx.m[0][2] = texSRT.translate.x + center.x + a0 * (-center.x) + a1 * (-center.y);

    a0 = sinR * texSRT.scale.x;
    a1 = cosR * texSRT.scale.y;
    texMtx.m[1][0] = a0;
    texMtx.m[1][1] = a1;
    texMtx.m[1][2] = texSRT.translate.y + center.y + a0 * (-center.x) + a1 * (-center.y);

    // math::MTX23 m(
    //      1,   0,  0,
    //      0, - 1,  1);
    // math::MTX23Mult(&texMtx, &m, &texMtx);
    texMtx.f._10 *= -1.f;
    texMtx.f._11 *= -1.f;
    texMtx.f._12 = texMtx.f._12 * -1.f + 1.0f;
}

//----------------------------------------
void
Material::CalcIndirectMtx(math::MTX23* pIndirectMtx, const f32 rotate, const res::Vec2& scale)
{
    NW_ASSERT_NOT_NULL(pIndirectMtx);

    math::MTX23& mtx = *pIndirectMtx;

    const math::VEC2 center(0.5f, 0.5f);

    f32 sinR, cosR;
    math::SinCosDeg(&sinR, &cosR, rotate);

    f32 a0, a1;

    // SRT の順
    a0 = cosR * scale.x * 2.0f;
    a1 = -sinR * scale.y * 2.0f;
    mtx.m[0][0] = a0;
    mtx.m[0][1] = a1;
    mtx.m[0][2] = a0 * (-center.x) + a1 * (-center.y);

    a0 = sinR * scale.x * 2.0f;
    a1 = cosR * scale.y * 2.0f;
    mtx.m[1][0] = a0;
    mtx.m[1][1] = a1;
    mtx.m[1][2] = a0 * (-center.x) + a1 * (-center.y);
}

//----------------------------------------
void
Material::SetupGraphics(
    DrawInfo& drawInfo,
    u8 alpha,
    ShaderVariation shaderVariation,
    bool bInitFrameTransform,
    const math::MTX34& paneGlbMtx,
    const nw::lyt::Size* paneSize
)
{
    NW_PROFILE("nw::lyt::Material::SetupGraphics");

    if (GetTextureOnly())
    {
        SetupSubmaterialOf_TextureMatrix(drawInfo, paneGlbMtx, paneSize);
        SetupSubmaterialOf_Texture(drawInfo);
    }
    else
    {
        drawInfo.SetAllowableTextureQuantity(ut::Min(m_MemNum.texMap, u16(TexMapMax)));

        const f32 scale = 1.0f / 255.0f; // 頂点カラーを[0, 1]に正規化するためのスケールです。
#if defined(NW_PLATFORM_CAFE)
        // m_ShaderInfoがNULLなら、共通シェーダを用いる
        if (m_ShaderInfo == NULL)
        {
#endif
            drawInfo.SetShader(GetShaderIndex(GetShaderId(), shaderVariation));
#if defined(NW_PLATFORM_CAFE)
        }
        else
        {
            drawInfo.SetArchiveShader(m_ShaderInfo, static_cast<s32>(shaderVariation));
        }
#endif
        switch (shaderVariation)
        {
#if defined(NW_PLATFORM_WIN32) || defined(NW_USE_NINTENDO_SDK)
        case WITHOUT_VERTEX_COLOR_SHADER_VARIATION:
            {
                static const ut::Color8 colors[VERTEXCOLOR_MAX] = {
                    ut::Color8(255, 255, 255, 255),
                    ut::Color8(255, 255, 255, 255),
                    ut::Color8(255, 255, 255, 255),
                    ut::Color8(255, 255, 255, 255)
                };
                internal::SetupVertexColor(drawInfo, colors);
                // 頂点カラーへ積算するカラーを設定します。
                shhelp::SetVertexUniformReg(
                    drawInfo.GetUniformId(UNIFORM_uColor),
                    scale * 1.0f,
                    scale * 1.0f,
                    scale * 1.0f,
                    scale * scale * alpha);
            }
            break;
#else
        case WITHOUT_VERTEX_COLOR_SHADER_VARIATION:
            break;
#endif
        default:
            {
                // 頂点カラーへ積算するカラーを設定します。
                shhelp::SetVertexUniformReg(
                    drawInfo.GetUniformId(UNIFORM_uColor),
                    scale * 1.0f,
                    scale * 1.0f,
                    scale * 1.0f,
                    scale * scale * alpha);
            }
            break;
        }
        drawInfo.LoadProjectionMtx();

        if (bInitFrameTransform)
        {
            gfnd::ShaderUniformId loc = drawInfo.GetUniformId(UNIFORM_uFrameSpec);
            shhelp::SetVertexUniformReg<gfnd::ShaderInt>(loc, TEXTUREFLIP_NONE);
            NW_GL_ASSERT();
        }

        SetupSubmaterialOf_TextureMatrix(drawInfo, paneGlbMtx, paneSize);
        SetupSubmaterialOf_Texture(drawInfo);
        SetupSubmaterialOf_Tev(drawInfo);
        SetupSubmaterialOf_Blender();
    }
}

//----------------------------------------
void
Material::SetupSubmaterialOf_TextureMatrix(DrawInfo& drawInfo, const math::MTX34& paneGlbMtx, const nw::lyt::Size* paneSize)
{
    NW_PROFILE("nw::lyt::Material::SetupSubmaterialOf_TextureMatrix");

    // テクスチャ行列
    if (m_MemNum.texSRT > 0)
    {
        const u32 num = ut::Min(
            ut::Min(ut::Min(u32(m_MemNum.texSRT), u32(this->GetTexMapNum())), u32(TexMapMax)),
            drawInfo.GetAllowableTextureQuantity());
        for (u32 i = 0; i < num; ++i)
        {
            const res::TexSRT& texSRT = this->GetTexSRT(i);
            const TexMap& texMap = this->GetTexMap(i);

            math::MTX23 texMtx;
            CalcTextureMtx(&texMtx, texSRT, texMap);

            LoadTexCoordMatrix(drawInfo, i, texMtx.m);
        }
    }
    NW_GL_ASSERT();

    if (m_MemNum.texCoordGen > 0)
    {
        u32 i = 0;
        const u32 num = ut::Min(u32(m_MemNum.texCoordGen), u32(TexMapMax));
        for (; i < num; ++i)
        {
            const res::TexCoordGen& texCoordGen = this->GetTexCoordGen(i);
            drawInfo.SetTexCoordSrc(static_cast<int>(i), texCoordGen.GetTexGenSrc());
        }
        for (; i < TexMapMax; ++i)
        {
            drawInfo.SetTexCoordSrc(static_cast<int>(i), -1);
        }
    }
    else
    {
        drawInfo.SetTexCoordSrc(0, -1);
    }

    if (GetTexMapCap())
    {
        gfnd::ShaderInt generatingTexCoord[TexMapMax];

        int texMapNum = ut::Min<int>(
            ut::Min<int>(GetTexMapNum(), GetTexCoordGenNum()),
            TexMapMax);
        int projTexIndex = 0;
        for (int i = 0; i < texMapNum; ++i)
        {
            const res::TexCoordGen& texCoordGen = this->GetTexCoordGen(i);

            generatingTexCoord[i] =
                (texCoordGen.GetTexGenSrc() == TEXGENSRC_ORTHO_PROJECTION || texCoordGen.GetTexGenSrc() == TEXGENSRC_PANE_BASE_ORTHO_PROJECTION) ? 1 :
            #ifdef PERSPECTIVE_PROJECTION_MAP_ENABLED
                (texCoordGen.GetTexGenSrc() == TEXGENSRC_PERS_PROJECTION) ? 2 : 0;
            #else
                0;
            #endif

            const TexMap& texMap = GetTexMap(i);

            if (generatingTexCoord[i] == 1)
            {
                NW_ASSERTMSG(projTexIndex < GetProjectionTexGenNum(),
                    "out of bouds: projTexIndex[%d] < GetProjectionTexGenNum[%d] for Material[%s]",
                    projTexIndex, GetProjectionTexGenNum(), GetName());
                const res::ProjectionTexGenParamaters& texGen = GetProjectionTexGenAry()[projTexIndex];
                ++projTexIndex;

                const bool isFittingLayoutSize = (texGen.flag & (1 << TEXPROJECTIONFLAG_FITTINGLAYOUTSIZE)) != 0;
                const bool isFittingPaneSizeEnabled = (texGen.flag & (1 << TEXPROJECTIONFLAG_FITTINGPANESIZEENABLED)) != 0;
                const bool isAdjustProjectionSREnabled = (texGen.flag & (1 << TEXPROJECTIONFLAG_ADJUSTPROJECTIONSRENABLED)) != 0;

                float baseWidth;
                float baseHeight;
                if (isFittingLayoutSize)
                {
                    const Size& layoutSize = drawInfo.GetLayout()->GetLayoutSize();
                    baseWidth = layoutSize.width;
                    baseHeight = layoutSize.height;
                }
                else if (isFittingPaneSizeEnabled)
                {
                    NW_ASSERT(paneSize != NULL);
                    baseWidth = paneSize->width;
                    baseHeight = paneSize->height;
                }
                else
                {
                    baseWidth = texMap.GetWidth();
                    baseHeight = texMap.GetHeight();
                }
                float halfWidth = baseWidth * 0.5f;
                float halfHeight = baseHeight * 0.5f;

                math::MTX44 ortho;
                if (texCoordGen.GetTexGenSrc() == TEXGENSRC_PANE_BASE_ORTHO_PROJECTION)
                {
                    if (isAdjustProjectionSREnabled)
                    {
                        math::MTX34TextureProjectionOrtho(
                            reinterpret_cast<math::MTX34*>(&ortho),
                            -halfWidth,
                            halfWidth,
                            halfHeight,
                            -halfHeight,
                            0.5f / texGen.scale.x,
                            0.5f / texGen.scale.y,
                            0.5f - texGen.translate.x / texGen.scale.x / baseWidth,
                            0.5f + texGen.translate.y / texGen.scale.y / baseHeight);
                        math::MTX34 inv;
                        inv.SetInverse(paneGlbMtx);
                        math::MTX34Mult(reinterpret_cast<math::MTX34*>(&ortho), reinterpret_cast<math::MTX34*>(&ortho), &inv);
                    }
                    else
                    {
                        math::MTX34TextureProjectionOrtho(
                            reinterpret_cast<math::MTX34*>(&ortho),
                            -halfWidth,
                            halfWidth,
                            halfHeight,
                            -halfHeight,
                            0.5f / texGen.scale.x,
                            0.5f / texGen.scale.y,
                            0.5f - (texGen.translate.x + paneGlbMtx._03) / texGen.scale.x / baseWidth,
                            0.5f + (texGen.translate.y + paneGlbMtx._13) / texGen.scale.y / baseHeight);
                    }
                }
                else
                {
                    math::MTX34TextureProjectionOrtho(
                        reinterpret_cast<math::MTX34*>(&ortho),
                        -halfWidth,
                        halfWidth,
                        halfHeight,
                        -halfHeight,
                        0.5f / texGen.scale.x,
                        0.5f / texGen.scale.y,
                        0.5f - texGen.translate.x / texGen.scale.x / baseWidth,
                        0.5f + texGen.translate.y / texGen.scale.y / baseHeight);
                }

                MTX34Mult(
                    reinterpret_cast<math::MTX34*>(&ortho),
                    *reinterpret_cast<math::MTX34*>(&ortho),
                    drawInfo.GetModelViewMtx());

                ortho.m[3][0] = 0.0f; ortho.m[3][1] = 0.0f; ortho.m[3][2] = 0.0f; ortho.m[3][3] = 1.0f;
                gfnd::ShaderUniformId id = drawInfo.GetUniformId(UNIFORM_uVertexTexCoord0 + i);
                shhelp::SetVertexUniformReg(id, ortho);
            }
            #ifdef PERSPECTIVE_PROJECTION_MAP_ENABLED
            else if (generatingTexCoord[i] == 2)
            {
                NW_ASSERTMSG(projTexIndex < GetProjectionTexGenNum(),
                    "out of bouds: projTexIndex[%d] < GetProjectionTexGenNum[%d] for Material[%s]",
                    projTexIndex, GetProjectionTexGenNum(), GetName());
                const res::ProjectionTexGenParamaters& texGen = GetProjectionTexGenAry()[projTexIndex];
                ++projTexIndex;

                math::MTX44 pers;
                math::MTX34TextureProjectionPerspective(
                    reinterpret_cast<math::MTX34*>(&pers),
                    NW_MATH_DEG_TO_RAD(texGen.fovy),
                    texMap.GetWidth() / texMap.GetHeight(),
                    -texGen.scale.x / texMap.GetWidth(),
                    -texGen.scale.y / texMap.GetHeight(),
                    texGen.translate.x / texGen.scale.x / texMap.GetWidth(),
                    texGen.translate.y / texGen.scale.y / texMap.GetHeight());
                pers.m[3][0] = 0.0f; pers.m[3][1] = 0.0f; pers.m[3][2] = -1.0f; pers.m[3][3] = 0.0f;

                math::MTX44 view44;
                math::MTX34* view = reinterpret_cast<math::MTX34*>(&view44);
                MTX34RotXYZDeg(view, texGen.rotate.x, texGen.rotate.y, texGen.rotate.z);
                MTX34Mult(view, *view, drawInfo.GetModelViewMtx());
                view44.m[3][0] = 0.0f; view44.m[3][1] = 0.0f; view44.m[3][2] = 0.0f; view44.m[3][3] = 1.0f;

                MTX44Mult(&pers, pers, view44);
                gfnd::ShaderUniformId id = drawInfo.GetUniformId(UNIFORM_uVertexTexCoord0 + i);
                shhelp::SetVertexUniformReg(id, pers);
            }
            #endif
        }
        for (int i = texMapNum; i < TexMapMax; ++i) { generatingTexCoord[i] = 0; }

        shhelp::SetVertexUniformReg<gfnd::ShaderInt>(
            drawInfo.GetUniformId(UNIFORM_uGeneratingTexCoord),
            generatingTexCoord[0],
            generatingTexCoord[1],
            generatingTexCoord[2]);
        NW_GL_ASSERT();
    }
}

//----------------------------------------
void
Material::SetupSubmaterialOf_Texture(const DrawInfo& drawInfo)
{
    NW_PROFILE("nw::lyt::Material::SetupSubmaterialOf_Texture");

    // テクスチャ
    if (m_MemNum.texMap > 0)
    {
#if defined(NW_PLATFORM_WIN32) || defined(NW_USE_NINTENDO_SDK)
        glEnable( GL_TEXTURE_2D );
#endif

        const u32 num = ut::Min(
            ut::Min(u32(m_MemNum.texMap), u32(TexMapMax)),
            drawInfo.GetAllowableTextureQuantity());
        for (u32 i = 0; i < num; ++i)
        {
            const TexMap& texMap = this->GetTexMap(i);

            NW_ASSERTMSG(texMap.GetTextureObject() != TextureInfo::INVALID, "texMap.GetTextureObject() must not be invalid for Material[%s]", GetName());

            u32 textureId = drawInfo.GetUniformId(UNIFORM_uTexture0 + i);
#if defined(NW_PLATFORM_WIN32) || defined(NW_USE_NINTENDO_SDK)
            glActiveTexture(GL_TEXTURE0 + i);
            glBindSampler( i, texMap.GetSampler() );
            glBindTexture(GL_TEXTURE_2D, texMap.GetTextureObject());
            shhelp::SetPixelUniformReg<gfnd::ShaderInt>(textureId, i);

#elif defined(NW_PLATFORM_CAFE)
            GX2SetPixelTexture(texMap.GetTexture(), textureId);
            GX2SetPixelSampler(&texMap.GetSampler(), textureId);
#endif

            if (i == 0)
            {
                gfnd::ShaderUniformId loc = drawInfo.GetUniformId(UNIFORM_uRcpTexSize0);
                shhelp::SetVertexUniformReg(loc, 1.0f / texMap.GetWidth(), 1.0f / texMap.GetHeight(), 0.0f, 0.0f);
            }
            NW_GL_ASSERT();
        }
    }
}

//----------------------------------------
void
Material::SetupSubmaterialOf_Tev(DrawInfo& drawInfo)
{
    NW_PROFILE("nw::lyt::Material::SetupSubmaterialOf_Tev");
    GraphicsResource& graphicsResource = *drawInfo.GetGraphicsResource();

    // 白黒補間の値を設定（現状では、TEVの設定に関係なく白黒補間は常に有効）
    if (m_MemNum.texMap == 0) {
        ut::FloatColor white(this->GetColor(INTERPOLATECOLOR_WHITE));
        gfnd::ShaderUniformId loc = drawInfo.GetUniformId(UNIFORM_uInterpolateWidth);
        shhelp::SetPixelUniformReg(loc, white);
#if defined(NW_PLATFORM_WIN32) || defined(NW_USE_NINTENDO_SDK)
        NW_UNUSED_VARIABLE(graphicsResource);

        loc = drawInfo.GetUniformId(UNIFORM_uInterpolateOffset);
        shhelp::SetPixelUniformReg(loc, 0.f, 0.f, 0.f, 0.f);
#endif
    } else {
        ut::FloatColor white(this->GetColor(INTERPOLATECOLOR_WHITE));
        ut::FloatColor black(this->GetColor(INTERPOLATECOLOR_BLACK));
        const gfnd::ShaderUniformId locWidth = drawInfo.GetUniformId(UNIFORM_uInterpolateWidth);
        const gfnd::ShaderUniformId locOffset = drawInfo.GetUniformId(UNIFORM_uInterpolateOffset);
        if (GetThresholdingAlphaInterpolation())
        {
            const f32 a = 1.0f / (white.a - black.a + 0.0001f); // 傾きaを求めます。発散を防ぐために微小な値を足しています。
            ut::FloatColor width;
            width.SetColor(white.r - black.r, white.g - black.g, white.b - black.b);
            width.SetAlpha(a);
            shhelp::SetPixelUniformReg(locWidth, width);
            ut::FloatColor offset;
            offset.SetColor(black.r, black.g, black.b);
            offset.SetAlpha(-black.a * a);
            shhelp::SetPixelUniformReg(locOffset, offset);
        }
        else
        {
            shhelp::SetPixelUniformReg(locWidth, white - black);
            shhelp::SetPixelUniformReg(locOffset, black);
        }
    }

    //  TEV
    if (m_MemNum.tevStage > 0)
    {
        gfnd::ShaderUniformId textureModeLoc = drawInfo.GetUniformId(UNIFORM_uTextureMode);
        if (m_MemNum.texMap >= 2) {
            if (static_cast<s32>(textureModeLoc) != -1)
            {
                int textureMode = 0;
                // 仮に、TEV設定のステージ1のカラーステージのコンバインモードを見る
                // デフォルトは二枚目を一枚目に上書きするモード
                static const u32 tevmodeTable[TEVMODE_MAX] =
                {
                    2, // REPLACE モードです。
                    3, // MODULATE モードです。
                    4, // ADD モードです。
                    6, // EXCLUSION モードです。
                    2, // REPLACE モードです。
                    5, // SUBTRACT モードです。
                    7, // COLOR_DODGE モードです。
                    8, // COLOR_BURN モードです。
                    9, // OVERLAY モードです。
                    14, // INDIRECT モードです。
                    15, // BLEND_INDIRECT モードです。
                    16, // EACH_INDIRECT モードです。
                };
                for (int i = 0; i < this->GetTevStageNum(); ++i)
                {
                    const int TEXTURE_MODE_BITS = 8;
                    const res::TevStage& tevStage = this->GetTevStage(i);
                    textureMode |= tevmodeTable[tevStage.GetCombineRgb()] << (i * TEXTURE_MODE_BITS);
                    if (tevStage.GetCombineAlpha() == TEVMODE_MODULATE) {
                        // TEV設定のステージ1のアルファステージのコンバインモードがTEVMODE_MODULATE
                        // ならアルファは最小値を取る、それ以外なら最大値を取る
                        textureMode |= 1 << (i * TEXTURE_MODE_BITS + 7);
                    }
                }

                shhelp::SetPixelUniformReg<gfnd::ShaderInt>(textureModeLoc, textureMode);
            }
            if (IsIndirectParameterCap())
            {
                const res::IndirectParameter& param = GetIndirectParameter();

                math::MTX23 mtx;
                CalcIndirectMtx(&mtx, param.rotate, param.scale);

                gfnd::ShaderUniformId xzId = drawInfo.GetUniformId(UNIFORM_uIndirectMtx0);
                gfnd::ShaderUniformId ywId = drawInfo.GetUniformId(UNIFORM_uIndirectMtx1);
                if (static_cast<s32>(xzId) != -1 && static_cast<s32>(ywId) != -1)
                {
                    shhelp::SetPixelUniformReg(xzId, mtx.m[0][0], mtx.m[0][1], 0.0f, mtx.m[0][2]);
                    shhelp::SetPixelUniformReg(ywId, mtx.m[1][0], mtx.m[1][1], 0.0f, mtx.m[1][2]);
                }
            }
        } else {

#if defined(NW_PLATFORM_CAFE)
            ShaderSlot& shaderSlot = graphicsResource.GetShaderSlot();
            if (shaderSlot.IsMultiTextureEnabled(shaderSlot.GetConnection(m_ShaderId)))
            {
                shhelp::SetPixelUniformReg<gfnd::ShaderInt>(textureModeLoc, m_MemNum.texMap);
            }
#else
            shhelp::SetPixelUniformReg<gfnd::ShaderInt>(textureModeLoc, m_MemNum.texMap);
#endif
        }
    }
    else
    {
        // 以前の設定がそのまま使えるか調べる。
        drawInfo.SetTexEnvAuto(true, m_MemNum.texMap);
        gfnd::ShaderUniformId textureModeLoc = drawInfo.GetUniformId(UNIFORM_uTextureMode);
        switch (m_MemNum.texMap)
        {
        case 0:
#if defined(NW_PLATFORM_CAFE)
            {
                ShaderSlot& shaderSlot = graphicsResource.GetShaderSlot();
                if (shaderSlot.IsMultiTextureEnabled(shaderSlot.GetConnection(m_ShaderId)))
                {
                    shhelp::SetPixelUniformReg<gfnd::ShaderInt>(textureModeLoc, 0);
                }
            }
#else
            shhelp::SetPixelUniformReg<gfnd::ShaderInt>(textureModeLoc, 0);
#endif
            break;
        case 1:
#if defined(NW_PLATFORM_CAFE)
            {
                ShaderSlot& shaderSlot = graphicsResource.GetShaderSlot();
                if (shaderSlot.IsMultiTextureEnabled(shaderSlot.GetConnection(m_ShaderId)))
                {
                    shhelp::SetPixelUniformReg<gfnd::ShaderInt>(textureModeLoc, 1);
                }
            }
#else
            shhelp::SetPixelUniformReg<gfnd::ShaderInt>(textureModeLoc, 1);
#endif
            break;
        default:
            // TEVの設定がなくてマルチテクスチャの場合は、二枚目を一枚目に上書きするモードで
            // アルファは最大値を取る設定にする
            shhelp::SetPixelUniformReg<gfnd::ShaderInt>(textureModeLoc, 2);
            break;
        }
    }
}

//----------------------------------------
void
Material::SetupSubmaterialOf_Blender()
{
    NW_PROFILE("nw::lyt::Material::SetupSubmaterialOf_Blender");

    if (IsBlendModeCap())
    {
        const res::BlendMode& blendMode = this->GetBlendMode();

        if (blendMode.GetLogicOp() == LOGICOP_DISABLE)
        {
            if (blendMode.GetBlendOp() == BLENDOP_DISABLE)
            {
                #if defined(NW_PLATFORM_WIN32) || defined(NW_USE_NINTENDO_SDK)
                glDisable(GL_COLOR_LOGIC_OP);
                glDisable(GL_BLEND);
                NW_GL_ASSERT();

                #elif defined(NW_PLATFORM_CAFE)
                GX2SetColorControl(
                    GX2_LOGIC_OP_COPY,
                    0x00,
                    GX2_DISABLE,
                    GX2_ENABLE);
                #endif
            }
            else if (IsBlendModeAlphaCap())
            {
                // カラーとアルファのブレンドを別々に設定する
                const res::BlendMode& blendModeAlpha = this->GetBlendModeAlpha();
                #if defined(NW_PLATFORM_WIN32) || defined(NW_USE_NINTENDO_SDK)
                glDisable(GL_COLOR_LOGIC_OP);
                glEnable(GL_BLEND);
                glBlendFuncSeparate(
                    ToBlendFunction(blendMode.GetSrcFactor()),
                    ToBlendFunction(blendMode.GetDstFactor()),
                    ToBlendFunction(blendModeAlpha.GetSrcFactor()),
                    ToBlendFunction(blendModeAlpha.GetDstFactor()));
                glBlendEquationSeparate(ToBlendCombine(blendMode.GetBlendOp()), ToBlendCombine(blendModeAlpha.GetBlendOp()));
                NW_GL_ASSERT();

                #elif defined(NW_PLATFORM_CAFE)
                GX2SetColorControl(
                    GX2_LOGIC_OP_COPY,
                    0x01,
                    GX2_DISABLE,
                    GX2_ENABLE);
                GX2SetBlendControl(
                    GX2_RENDER_TARGET_0,
                    ToBlendFunction(blendMode.GetSrcFactor()),
                    ToBlendFunction(blendMode.GetDstFactor()),
                    ToBlendCombine(blendMode.GetBlendOp()),
                    GX2_TRUE,
                    ToBlendFunction(blendModeAlpha.GetSrcFactor()),
                    ToBlendFunction(blendModeAlpha.GetDstFactor()),
                    ToBlendCombine(blendModeAlpha.GetBlendOp()));
                #endif
            }
            else
            {
                // カラーとアルファのブレンドを一緒に設定する
                #if defined(NW_PLATFORM_WIN32) || defined(NW_USE_NINTENDO_SDK)
                glDisable(GL_COLOR_LOGIC_OP);
                glEnable(GL_BLEND);
                glBlendFunc(
                    ToBlendFunction(blendMode.GetSrcFactor()),
                    ToBlendFunction(blendMode.GetDstFactor()));
                glBlendEquation(ToBlendCombine(blendMode.GetBlendOp()));
                NW_GL_ASSERT();

                #elif defined(NW_PLATFORM_CAFE)
                GX2SetColorControl(
                    GX2_LOGIC_OP_COPY,
                    0x01,
                    GX2_DISABLE,
                    GX2_ENABLE);
                GX2SetBlendControl(
                    GX2_RENDER_TARGET_0,
                    ToBlendFunction(blendMode.GetSrcFactor()),
                    ToBlendFunction(blendMode.GetDstFactor()),
                    ToBlendCombine(blendMode.GetBlendOp()),
                    GX2_FALSE,
                    GX2_BLEND_SRC_ALPHA,
                    GX2_BLEND_ONE_MINUS_SRC_ALPHA,
                    GX2_BLEND_COMBINE_ADD);
                #endif
            }
        }
        else
        {
            #if defined(NW_PLATFORM_WIN32) || defined(NW_USE_NINTENDO_SDK)
            // COLOR_LOGIC_OP が Enable だと、BLEND は無効扱いになる。
            glEnable(GL_COLOR_LOGIC_OP);
            glLogicOp(ToLogicOp(blendMode.GetLogicOp()));
            NW_GL_ASSERT();

            #elif defined(NW_PLATFORM_CAFE)
            GX2SetColorControl(
                ToLogicOp(blendMode.GetLogicOp()),
                0x00,
                GX2_DISABLE,
                GX2_ENABLE);
            #endif
        }
    }
    else
    {
        #if defined(NW_PLATFORM_WIN32) || defined(NW_USE_NINTENDO_SDK)
        glDisable(GL_COLOR_LOGIC_OP); // 論理演算はしない
        glEnable(GL_BLEND);
        glBlendFunc(GL_SRC_ALPHA, GL_ONE_MINUS_SRC_ALPHA); // ソースアルファでブレンド
        glBlendEquation(GL_FUNC_ADD);
        NW_GL_ASSERT();

        #elif defined(NW_PLATFORM_CAFE)
        GX2SetColorControl(
            GX2_LOGIC_OP_COPY,
            0x01,
            GX2_DISABLE,
            GX2_ENABLE);
        GX2SetBlendControl(
            GX2_RENDER_TARGET_0,
            GX2_BLEND_SRC_ALPHA,
            GX2_BLEND_ONE_MINUS_SRC_ALPHA,
            GX2_BLEND_COMBINE_ADD,
            GX2_FALSE,
            GX2_BLEND_SRC_ALPHA,
            GX2_BLEND_ONE_MINUS_SRC_ALPHA,
            GX2_BLEND_COMBINE_ADD);
        #endif
    }

    if (IsAlphaCompareCap())
    {
        const res::AlphaCompare& alphaCompare = this->GetAlphaCompare();

        #if defined(NW_PLATFORM_WIN32) || defined(NW_USE_NINTENDO_SDK)
        static const GLenum functions[ALPHATEST_MAX] =
        {
            GL_NEVER,
            GL_LESS,
            GL_LEQUAL,
            GL_EQUAL,
            GL_NOTEQUAL,
            GL_GEQUAL,
            GL_GREATER,
            GL_ALWAYS,
        };
        glEnable(GL_ALPHA_TEST);
        glAlphaFunc(functions[alphaCompare.GetFunc()], alphaCompare.GetRef());

        #elif defined(NW_PLATFORM_CAFE)
        static const GX2CompareFunction functions[ALPHATEST_MAX] =
        {
            GX2_COMPARE_NEVER,
            GX2_COMPARE_LESS,
            GX2_COMPARE_LEQUAL,
            GX2_COMPARE_EQUAL,
            GX2_COMPARE_NOTEQUAL,
            GX2_COMPARE_GEQUAL,
            GX2_COMPARE_GREATER,
            GX2_COMPARE_ALWAYS,
        };
        GX2SetAlphaTest(GX2_TRUE, functions[alphaCompare.GetFunc()], alphaCompare.GetRef());
        #endif
    }
    else
    {
        #if defined(NW_PLATFORM_WIN32) || defined(NW_USE_NINTENDO_SDK)
        glDisable(GL_ALPHA_TEST);

        #elif defined(NW_PLATFORM_CAFE)
        GX2SetAlphaTest(GX2_FALSE, GX2_COMPARE_NEVER, 0.0f);
        #endif
    }
}

} // namespace nw::lyt
} // namespace nw
