﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nw/lyt/lyt_ArcExtractor.h>
#include <nw/ut/ut_String.h>
#include <nw/ut/res/ut_ResTypes.h>

namespace
{
    //------------------------------------------------------------------------------

    NW_INLINE u32 calcHash32( const char* str, u32 key )
    {
        u32 ret = 0;
        for( s32 i = 0; str[ i ] != '\0'; ++i )
        {
            ret = ret * key + str[ i ];
        }
        return ret;
    }

    //------------------------------------------------------------------------------

    template<typename TValue>
    NW_INLINE
    TValue EndianToHost(nw::lyt::EndianTypes endian, const TValue& value)
    {
#if defined(NW_PLATFORM_CAFE)
        return (endian == nw::lyt::LITTLE_ENDIAN)
            ? nw::ut::EndianSwap::BSwap(value)
            : value;
#else
        return (endian == nw::lyt::BIG_ENDIAN)
            ? nw::ut::EndianSwap::BSwap(value)
            : value;
#endif
    }

    //------------------------------------------------------------------------------

#if defined(NW_PLATFORM_WIN32) || defined(NW_USE_NINTENDO_SDK)
    NW_INLINE s32 binarySearch_( u32 hash, const nw::lyt::ArcExtractor::FATEntry* buffer, s32 start, s32 end, nw::lyt::EndianTypes endian_type )
#else
    NW_INLINE s32 binarySearch_( u32 hash, const nw::lyt::ArcExtractor::FATEntry* buffer, s32 start, s32 end )
#endif
    {
        // 中点を取る.
        s32 middle;
        for(;;)
        {
            middle = ( start + end ) / 2;
#if defined(NW_PLATFORM_WIN32) || defined(NW_USE_NINTENDO_SDK)
            u32 buf_hash =EndianToHost( endian_type, buffer[ middle ].hash );
#else
            u32 buf_hash = buffer[ middle ].hash;
#endif
            if ( buf_hash == hash )
            {
                return middle;
            }
            else if ( buf_hash < hash )
            {
                if ( start == middle )
                {
                    return -1;
                }
                start = middle;
            }
            else
            {
                if ( end == middle )
                {
                    return -1;
                }
                end = middle;
            }
        }
    }

    //------------------------------------------------------------------------------

    /**
     *  エンディアンマーク(Byte Order Mark)からエンディアンを判定します.
     *
     *  @param[in]  mark    エンディアンマーク(BOM). 0xff 0xfeか0xfe 0xff以外を渡した場合、エラー.
     *  @return             判定結果.
     */
    NW_INLINE nw::lyt::EndianTypes MarkToEndian(u16 mark)
    {
        u8* mark8 = reinterpret_cast<u8*>( &mark );
        if ( ( *mark8 == 0xff ) && ( *(mark8 + 1 ) == 0xfe ) )
        {
            return nw::lyt::LITTLE_ENDIAN;
        }
        else if ( ( *mark8 == 0xfe ) && ( *(mark8 + 1 ) == 0xff ) )
        {
            return nw::lyt::BIG_ENDIAN;
        }
        else
        {
            NW_ERR( "Undefined endian mark(0x%02x 0x%02x).", *((u8*)&mark),  *((u8*)&mark + 1) );
            return nw::lyt::LITTLE_ENDIAN;
        }
    }

}

namespace nw
{
namespace lyt
{

#if defined(NW_PLATFORM_WIN32) || defined(NW_USE_NINTENDO_SDK)
#define SHARC_ENDIAN_TO_HOST(val) (EndianToHost(mEndianType, (val)))
#else
#define SHARC_ENDIAN_TO_HOST(val) (val)
#endif

//------------------------------------------------------------------------------

ArcExtractor::ArcExtractor(const void* archive)
: mArchiveBlockHeader( NULL )
, mFATBlockHeader( NULL )
, mFNTBlock( NULL )
, mDataBlock( NULL )
#if defined(NW_PLATFORM_WIN32) || defined(NW_PLATFORM_CTR) || defined(NW_USE_NINTENDO_SDK)
, mEndianType(LITTLE_ENDIAN)
#else
, mEndianType(BIG_ENDIAN)
#endif
{
    PrepareArchive(archive);
}

//------------------------------------------------------------------------------

ArcExtractor::ArcExtractor()
: mArchiveBlockHeader( NULL )
, mFATBlockHeader( NULL )
, mFNTBlock( NULL )
, mDataBlock( NULL )
#if defined(NW_PLATFORM_WIN32) || defined(NW_PLATFORM_CTR) || defined(NW_USE_NINTENDO_SDK)
, mEndianType(LITTLE_ENDIAN)
#else
, mEndianType(BIG_ENDIAN)
#endif
{
}

//------------------------------------------------------------------------------

ArcExtractor::~ArcExtractor()
{
}

//------------------------------------------------------------------------------

bool ArcExtractor::PrepareArchive( const void *archive )
{
    if ( !archive )
    {
        NW_ERR( "archive must not be NULL." );
        return false;
    }
    const u8* archive8 = reinterpret_cast<const u8*>( archive );
    mArchiveBlockHeader = reinterpret_cast<const ArchiveBlockHeader*>( archive8 );
    if ( std::strncmp( mArchiveBlockHeader->signature, "SARC", 4 ) != 0 )
    {
        NW_ERR( "Invalid ArchiveBlockHeader" );
        return false;
    }

    mEndianType = MarkToEndian( mArchiveBlockHeader->byte_order );

#if !(defined(NW_PLATFORM_WIN32) || defined(NW_USE_NINTENDO_SDK))
    // Win以外であればエンディアンが異なるのは許容できない
    if( mEndianType != HOST_ENDIAN )
    {
        NW_ERR( "Invalid Endian Type" );
        return false;
    }
#endif

    if ( SHARC_ENDIAN_TO_HOST(mArchiveBlockHeader->version) != cArchiveVersion )
    {
        NW_ERR( "unmatching version ( expect: %x, actual: %x )", cArchiveVersion, mArchiveBlockHeader->version );
        return false;
    }
    if( SHARC_ENDIAN_TO_HOST(mArchiveBlockHeader->header_size) != sizeof( ArchiveBlockHeader ) )
    {
        NW_ERR( "Invalid ArchiveBlockHeader" );
        return false;
    }
    mFATBlockHeader = reinterpret_cast<const FATBlockHeader*>( archive8 + SHARC_ENDIAN_TO_HOST(mArchiveBlockHeader->header_size) );
    if ( std::strncmp( mFATBlockHeader->signature, "SFAT", 4 ) != 0 )
    {
        NW_ERR( "Invalid FATBlockHeader" );
        return false;
    }
    if( SHARC_ENDIAN_TO_HOST(mFATBlockHeader->header_size) != sizeof( FATBlockHeader ) )
    {
        NW_ERR( "Invalid FATBlockHeader" );
        return false;
    }
    if( SHARC_ENDIAN_TO_HOST(mFATBlockHeader->file_num) > cArchiveEntryMax)
    {
        NW_ERR( "Invalid FATBlockHeader" );
        return false;
    }

    mFATEntrys.SetArray(
        SHARC_ENDIAN_TO_HOST(mFATBlockHeader->file_num),
        const_cast<FATEntry*>(
            reinterpret_cast<const FATEntry*>(archive8 + SHARC_ENDIAN_TO_HOST(mArchiveBlockHeader->header_size) +
                                       SHARC_ENDIAN_TO_HOST(mFATBlockHeader->header_size ) ) )
         );
    const FNTBlockHeader* fnt_header = reinterpret_cast<const FNTBlockHeader*>( archive8 +
                                                      SHARC_ENDIAN_TO_HOST(mArchiveBlockHeader->header_size) +
                                                      SHARC_ENDIAN_TO_HOST(mFATBlockHeader->header_size) +
                                                      SHARC_ENDIAN_TO_HOST(mFATBlockHeader->file_num ) * sizeof( FATEntry ) );
    if ( std::strncmp( fnt_header->signature, "SFNT", 4 ) != 0 )
    {
        NW_ERR( "Invalid FNTBlockHeader" );
        return false;
    }
    if( SHARC_ENDIAN_TO_HOST(fnt_header->header_size) != sizeof(FNTBlockHeader) )
    {
        NW_ERR( "Invalid FNTBlockHeader" );
        return false;
    }
    mFNTBlock = reinterpret_cast<const char*>( reinterpret_cast<const u8*>( fnt_header ) + SHARC_ENDIAN_TO_HOST(fnt_header->header_size) );
    if( SHARC_ENDIAN_TO_HOST(static_cast<s32>(mArchiveBlockHeader->data_block_offset)) < ut::GetOffsetFromPtr(mArchiveBlockHeader, mFNTBlock) )
    {
        NW_ERR( "Invalid data block offset" );
        return false;
    }
    mDataBlock = archive8 + SHARC_ENDIAN_TO_HOST(mArchiveBlockHeader->data_block_offset);
    return true;
}

//------------------------------------------------------------------------------

void* ArcExtractor::GetFileFast( s32 entry_id, ArcFileInfo* file_info )
{
    if( entry_id < 0 || entry_id >= mFATEntrys.size() )
    {
        return NULL;
    }
    u32 start_offset = SHARC_ENDIAN_TO_HOST(mFATEntrys[entry_id].data_start_offset);

    if ( file_info )
    {
        u32 end_offset = SHARC_ENDIAN_TO_HOST(mFATEntrys[entry_id].data_end_offset);
        if( start_offset > end_offset )
        {
            return NULL;
        }
        u32 length = end_offset - start_offset;
        SetArcFileInfo( file_info, start_offset, length );
    }
    return ut::AddOffsetToPtr(const_cast<u8*>(mDataBlock), start_offset);
}

//------------------------------------------------------------------------------

s32 ArcExtractor::ConvertPathToEntryID( const char* file_path )
{
    // ハッシュ計算.
    u32 hash = calcHash32( file_path, SHARC_ENDIAN_TO_HOST( mFATBlockHeader->hash_key ) );

    // テーブルを引く.
    s32 start = 0;
    s32 end = mFATEntrys.size();
    // 二分探索.
#if defined(NW_PLATFORM_WIN32) || defined(NW_USE_NINTENDO_SDK)
    s32 id = binarySearch_( hash, mFATEntrys.GetArrayPtr(), start, end, mEndianType );
#else
    s32 id = binarySearch_( hash, mFATEntrys.GetArrayPtr(), start, end );
#endif
    if ( id == -1 )
    {
        return -1;
    }

    // 名前引き.
    u32 offset = SHARC_ENDIAN_TO_HOST(mFATEntrys[id].name_offset);
    if ( offset != 0 )
    {
        id = id - ( ( offset >> 24 ) - 1 );
        while( id < end )
        {
            const FATEntry* entry = mFATEntrys.Get(id);
            if ( SHARC_ENDIAN_TO_HOST(entry->hash) != hash )
            {
                return -1;
            }
            else {
                u32 name_offset = SHARC_ENDIAN_TO_HOST(entry->name_offset);
                if( ut::AddOffsetToPtr( mFNTBlock, name_offset & 0x00ffffff ) > mDataBlock )
                {
                    NW_WARNING( false, "Invalid data start offset" );
                    return -1;
                }
                //if ( file_path.isEqual( mFNTBlock + ( name_offset & 0x00ffffff ) * cFileNameTableAlign ) )
                if ( std::strcmp(file_path, mFNTBlock + ( name_offset & 0x00ffffff ) * cFileNameTableAlign ) == 0 )
                {
                    return id;
                }
            }
            ++id;
        }
    }

    return id;
}

//------------------------------------------------------------------------------

u32 ArcExtractor::ReadEntry( u32* handle, ArcEntry entry[], u32 num )
{
    // sharcでは、HandleBufferにu32の数値を入れて、現在位置を記憶します.
    u32 count = 0;

    while( *handle + count < SHARC_ENDIAN_TO_HOST(mFATBlockHeader->file_num) && count < num )
    {
        u32 id = *handle + count;
        // オーバーフローを監視するアサート
        NW_ASSERT( id >= (*handle) );
        u32 offset = SHARC_ENDIAN_TO_HOST(mFATEntrys[id].name_offset);
        if ( offset == 0 )
        {
            // 名前は記録されていないのでハッシュ値を出力.
            ut::snprintf(
                entry[ count ].name,
                ArcEntry::MAX_NAME_LENGTH,
                ArcEntry::MAX_NAME_LENGTH - 1,
                "%08x",
                SHARC_ENDIAN_TO_HOST(mFATEntrys[id].hash));
        }
        else
        {
            // 記録されている名前を出力.
            if( ut::AddOffsetToPtr( mFNTBlock, offset & 0x00ffffff ) > mDataBlock )
            {
                NW_WARNING( false, "Invalid data start offset" );
                entry[ count ].name[0] = '\0';
            }
            else
            {
                ut::strcpy( entry[ count ].name, ArcEntry::MAX_NAME_LENGTH, mFNTBlock + ( offset & 0x00ffffff ) * cFileNameTableAlign );
            }
        }
        count++;
    }
    *handle += count;
    return count;
}

} // namespace lyt
} // namespace nw

