﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nw/lyt/lyt_Layout.h>
#include <nw/lyt/lyt_ControlCreator.h>
#include <nw/ctrl/ctrl_DragButton.h>

namespace nw
{
namespace ctrl
{

DragButton::DragButton()
 : AnimButton()
 , mRootPane(NULL)
 , mDragStartPos(0.f, 0.f)
 , mDragStartTransrate(0.f, 0.f)
 , m_IsMoveHorizontal(true)
 , m_IsMoveVertical(true)
{
    SetDragMode(DRAG_MODE_ON_DOWN);
}

void DragButton::Build(const lyt::ControlSrc& controlSrc, lyt::Layout* layout)
{
    AnimButton::Build(controlSrc, layout);

    // Releaseアニメの取得
    m_CancelAnimator = layout->CreateGroupAnimatorAuto(controlSrc.FindFunctionalAnimName("Release"), false);
    NW_ASSERTMSG(m_CancelAnimator, "cannot create CancelAnimator[%s] for Layout[%s]", controlSrc.FindFunctionalAnimName("Release"), layout->GetName());

    mRootPane = layout->GetRootPane();
}

void DragButton::InitDragPosition(const math::VEC2& pos)
{
    mDragStartPos = pos;
    mDragStartTransrate.Set(mRootPane->GetTranslate().x, mRootPane->GetTranslate().y);
}

void DragButton::UpdateDragPosition(const math::VEC2* pos)
{
    if (pos) {
        math::VEC3 newPos;
        if (m_IsMoveHorizontal) {
            newPos.x = mDragStartTransrate.x + (pos->x - mDragStartPos.x);
        } else {
            newPos.x = mDragStartTransrate.x;
        }
        if (m_IsMoveVertical) {
            newPos.y = mDragStartTransrate.y + (pos->y - mDragStartPos.y);
        } else {
            newPos.y = mDragStartTransrate.y;
        }
        newPos.z = mRootPane->GetTranslate().z;
        mRootPane->SetTranslate(newPos);
    }
}

bool DragButton::ProcessOff()
{
    // STATE_DOWN及びSTATE_DOWN_IDLE中のoffアクションを無視するようにします。
    // また、STATE_CANCEL中のoffアクションは待機するようにします。
    switch (m_State) {
    case STATE_OFF_IDLE:
        return true;
    case STATE_ON:
        ChangeState(STATE_OFF);
        StartOff();
        return true;
    case STATE_OFF:
        return true;
    case STATE_ON_IDLE:
        ChangeState(STATE_OFF);
        StartOff();
        return true;
    case STATE_DOWN:
        return true;
    case STATE_DOWN_IDLE:
        return true;
    case STATE_CANCEL:
        return false;
    };

    return true;
}

bool DragButton::ProcessCancel()
{
    // Cancelアクションを受け付けるようにします。
    switch (m_State) {
    case STATE_OFF_IDLE:
        return true;
    case STATE_ON:
        return true;
    case STATE_OFF:
        return true;
    case STATE_ON_IDLE:
        return true;
    case STATE_DOWN:
        return false;
    case STATE_DOWN_IDLE:
        ChangeState(STATE_CANCEL);
        StartCancel();
        return true;
    case STATE_CANCEL:
        return false;
    };

    return true;
}

void DragButton::FinishCancel()
{
    // STATE_CANCELの後は、デフォルトのSTATE_OFF_IDLEではなくSTATE_ON_IDLEに移行します。
    ChangeState(STATE_ON_IDLE);
}

} // namespace nw::ctrl
} // namespace nw
