﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nw/ctrl/ctrl_AnimButton.h>
#include <nw/lyt/lyt_Animator.h>
#include <nw/lyt/lyt_ControlCreator.h>
#include <nw/lyt/lyt_Layout.h>
#include <nw/ut/ut_Inlines.h>

namespace nw
{
namespace ctrl
{

AnimButton::AnimButton()
 : m_StateChangeCallback(NULL)
 , m_StateChangeCallbackParam(NULL)
 , m_OnOffAnimator(NULL)
 , m_DownAnimator(NULL)
 , m_CancelAnimator(NULL)
 , m_DisableAnimator(NULL)
 , m_HitPane(NULL)
 , m_HitBoxBottomLeft(0.f, 0.f)
 , m_HitBoxTopRight(0.f, 0.f)
 , m_Tag(0)
 , m_Name(NULL)
{
}

void AnimButton::Build(const lyt::ControlSrc& controlSrc, lyt::Layout* layout)
{
    // 番をOnOffアニメとDownアニメを取得する。
    // また、アニメはすべてグループアニメであり、グループが一つ以上割り当てられている必要がある。
    m_OnOffAnimator = layout->CreateGroupAnimatorAuto(controlSrc.FindFunctionalAnimName("OnOff"), true);
    NW_ASSERTMSG(m_OnOffAnimator, "cannot create OnOffAnimator[%s] for Layout[%s]", controlSrc.FindFunctionalAnimName("OnOff"), layout->GetName());
    m_OnOffAnimator->StopAtMin();
    m_DownAnimator = layout->CreateGroupAnimatorAuto(controlSrc.FindFunctionalAnimName("Down"), false);
    NW_ASSERTMSG(m_DownAnimator, "cannot create DownAnimator[%s] for Layout[%s]", controlSrc.FindFunctionalAnimName("Down"), layout->GetName());
    // Disableアニメがあれば作成
    {
        const char* anim_name = controlSrc.FindFunctionalAnimName("Disable");
        if (anim_name && anim_name[0]) {
            m_DisableAnimator = layout->CreateGroupAnimatorAuto(anim_name, false);
        }
    }

    // 当たりのペインを取得
    m_HitPane = layout->GetRootPane()->FindPaneByName(controlSrc.FindFunctionalPaneName("Hit"));
    NW_ASSERTMSG(m_HitPane, "cannot get HitPane[%s] for Layout[%s]", controlSrc.FindFunctionalPaneName("Hit"), layout->GetName());
    // ボタンの名前は、レイアウトが部品であれば部品ペイン名（ルートペインとなっている）、
    // 部品でなければレイアウト名とする
    if (layout->IsPartsLayout()) {
        m_Name = layout->GetRootPane()->GetName();
    } else {
        m_Name = layout->GetName();
    }
}

void AnimButton::UpdateHitBox()
{
    if (m_HitPane == NULL) {
        return;
    }

    // グローバル行列からスケールを取得して大きさを計算する。
    f32 half_width = ut::Abs(m_HitPane->GetSize().width * m_HitPane->GetGlobalMtx()._00) / 2.f;
    f32 half_height = ut::Abs(m_HitPane->GetSize().height * m_HitPane->GetGlobalMtx()._11) / 2.f;
    math::VEC2 center;

    center.x = m_HitPane->GetGlobalMtx()._03;
    center.y = m_HitPane->GetGlobalMtx()._13;

    switch(m_HitPane->GetBasePositionH()) {
    case nw::lyt::HORIZONTALPOSITION_LEFT:
        center.x += half_width;
        break;
    case nw::lyt::HORIZONTALPOSITION_RIGHT:
        center.x -= half_width;
        break;
    }

    switch(m_HitPane->GetBasePositionV()) {
    case nw::lyt::VERTICALPOSITION_TOP:
        center.y -= half_height;
        break;
    case nw::lyt::VERTICALPOSITION_BOTTOM:
        center.y += half_height;
        break;
    }

    m_HitBoxBottomLeft.Set(center.x - half_width, center.y - half_height);
    m_HitBoxTopRight.Set(center.x + half_width, center.y + half_height);
}

bool AnimButton::HitTest(const math::VEC2& vec2) const
{
    if (m_HitPane) {
        return (m_HitBoxBottomLeft.x <= vec2.x && vec2.x <= m_HitBoxTopRight.x &&
            m_HitBoxBottomLeft.y <= vec2.y && vec2.y <= m_HitBoxTopRight.y);
    } else {
        return false;
    }
}

void AnimButton::InitDragPosition(const math::VEC2& /* pos */)
{
}

void AnimButton::UpdateDragPosition(const math::VEC2* /* pos */)
{
}

void AnimButton::SetStateChangeCallback(StateChangeCallback callback, void* param)
{
    m_StateChangeCallback = callback;
    m_StateChangeCallbackParam = param;
}

void AnimButton::PlayDisableAnim(bool is_disable)
{
    if (m_DisableAnimator) {
        m_DisableAnimator->SetEnable(true);
        m_DisableAnimator->PlayFromCurrent(lyt::Animator::PLAYTYPE_ONESHOT, is_disable ? 1.f : - 1.f);
    }
}

void AnimButton::SetAllAnimatorDisable()
{
    if (m_OnOffAnimator) {
        m_OnOffAnimator->SetEnable(false);
    }
    if (m_DownAnimator) {
        m_DownAnimator->SetEnable(false);
    }
    if (m_CancelAnimator) {
        m_CancelAnimator->SetEnable(false);
    }
    if (m_DisableAnimator) {
        m_DisableAnimator->SetEnable(false);
    }
}

void AnimButton::ForceOff()
{
    ButtonBase::ForceOff();
    if (m_OnOffAnimator) {
        EnableAnim(m_OnOffAnimator);
        m_OnOffAnimator->StopAtMin();
    }
}

void AnimButton::ForceOn()
{
    ButtonBase::ForceOn();
    if (m_OnOffAnimator) {
        EnableAnim(m_OnOffAnimator);
        m_OnOffAnimator->StopAtMax();
    }
}

void AnimButton::ForceDown()
{
    ButtonBase::ForceDown();
    if (m_DownAnimator) {
        EnableAnim(m_DownAnimator);
        m_DownAnimator->StopAtMax();
    }
}

void AnimButton::StartOn()
{
    if (m_OnOffAnimator) {
        EnableAnim(m_OnOffAnimator);
        m_OnOffAnimator->PlayFromCurrent(lyt::Animator::PLAYTYPE_ONESHOT, 1.f);
    }
}

void AnimButton::StartOff()
{
    if (m_OnOffAnimator) {
        EnableAnim(m_OnOffAnimator);
        m_OnOffAnimator->PlayFromCurrent(lyt::Animator::PLAYTYPE_ONESHOT, -1.f);
    }
}

void AnimButton::StartDown()
{
    if (m_DownAnimator) {
        EnableAnim(m_DownAnimator);
        m_DownAnimator->Play(lyt::Animator::PLAYTYPE_ONESHOT, 1.f);
    }
}

void AnimButton::StartCancel()
{
    if (m_CancelAnimator) {
        EnableAnim(m_CancelAnimator);
        m_CancelAnimator->Play(lyt::Animator::PLAYTYPE_ONESHOT, 1.f);
    }
}

bool AnimButton::UpdateOn()
{
    if (m_OnOffAnimator) {
        return m_OnOffAnimator->IsEndTrigger();
    } else {
        return true;
    }
}

bool AnimButton::UpdateOff()
{
    if (m_OnOffAnimator) {
        return m_OnOffAnimator->IsEndTrigger();
    } else {
        return true;
    }
}

bool AnimButton::UpdateDown()
{
    if (m_DownAnimator) {
        return m_DownAnimator->IsEndTrigger();
    } else {
        return true;
    }
}

bool AnimButton::UpdateCancel()
{
    if (m_CancelAnimator) {
        return m_CancelAnimator->IsEndTrigger();
    } else {
        return true;
    }
}

bool AnimButton::ProcessCancel()
{
    // AnimButtonは、デフォルトではCancelイベントを受け付けても
    // 何もしないようにしておく(Cancelイベントに反応するボタンの
    // 方が少ないため)
    return true;
}

void AnimButton::ChangeState(State state)
{
    if (m_State != state) {
        if (m_StateChangeCallback) {
            m_StateChangeCallback(this, m_State, state, m_StateChangeCallbackParam);
        }
        m_State = state;
    }
}

void AnimButton::EnableAnim(lyt::Animator* animator)
{
    if (m_OnOffAnimator) {
        m_OnOffAnimator->SetEnable(animator == m_OnOffAnimator);
    }
    if (m_DownAnimator) {
        m_DownAnimator->SetEnable(animator == m_DownAnimator);
    }
    if (m_CancelAnimator) {
        m_CancelAnimator->SetEnable(animator == m_CancelAnimator);
    }
}

} // namespace nw::ctrl
} // namespace nw
