﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#ifndef NW_UT_TYPETRAITS_H_
#define NW_UT_TYPETRAITS_H_

namespace nw
{
namespace ut
{

/*---------------------------------------------------------------------------*
  internal
 *---------------------------------------------------------------------------*/
namespace internal
{

template <typename TType, TType Value>
struct integral_constant
{
    static const TType value = Value;
    typedef TType value_type;
};

typedef integral_constant<bool, true>  TrueType;
typedef integral_constant<bool, false> FalseType;

template<bool Cond, typename Then, typename Else>
struct IfCond;

template<typename Then, typename Else>
struct IfCond<true, Then, Else>
{
    typedef Then type;
};

template<typename Then, typename Else>
struct IfCond<false, Then, Else>
{
    typedef Else type;
};

struct True  { char a;    };
struct False { True  a[2]; };

}

/*---------------------------------------------------------------------------*
  declare_type_remove_template
 *---------------------------------------------------------------------------*/
#define NW_UT_DECLARE_TYPE_REMOVE_TEMPLATE_BASE_(name, ttype)  \
template<typename ttype>                                       \
struct name                                                    \
{                                                              \
    typedef ttype type;                                        \
};

#define NW_UT_DECLARE_TYPE_REMOVE_TEMPLATE_(name, ttype, modifier) \
template<typename ttype>                                           \
struct name<modifier>                                              \
{                                                                  \
    typedef TType type;                                            \
};

#define NW_UT_DECLARE_TYPE_REMOVE_TEMPLATE2_(name, ttype, modifier, ...) \
template<typename __VA_ARGS__>                                           \
struct name<modifier>                                                    \
{                                                                        \
    typedef TType type;                                                  \
};

/*---------------------------------------------------------------------------*
  remove_bounds
 *---------------------------------------------------------------------------*/
NW_UT_DECLARE_TYPE_REMOVE_TEMPLATE_BASE_(remove_bounds, TType)
NW_UT_DECLARE_TYPE_REMOVE_TEMPLATE_(remove_bounds, TType, TType[])
NW_UT_DECLARE_TYPE_REMOVE_TEMPLATE2_(remove_bounds, TType, TType[Size], TType, int Size)

/*---------------------------------------------------------------------------*
  remove_const
 *---------------------------------------------------------------------------*/
NW_UT_DECLARE_TYPE_REMOVE_TEMPLATE_BASE_(remove_const, TType)
NW_UT_DECLARE_TYPE_REMOVE_TEMPLATE_(remove_const, TType, const TType)

/*---------------------------------------------------------------------------*
  remove_reference
 *---------------------------------------------------------------------------*/
NW_UT_DECLARE_TYPE_REMOVE_TEMPLATE_BASE_(remove_reference, TType)
NW_UT_DECLARE_TYPE_REMOVE_TEMPLATE_(remove_reference, TType, TType&)

/*---------------------------------------------------------------------------*
  remove_pointer
 *---------------------------------------------------------------------------*/
NW_UT_DECLARE_TYPE_REMOVE_TEMPLATE_BASE_(remove_pointer, TType)
NW_UT_DECLARE_TYPE_REMOVE_TEMPLATE_(remove_pointer, TType, TType*)
NW_UT_DECLARE_TYPE_REMOVE_TEMPLATE_(remove_pointer, TType, TType* const)

/*---------------------------------------------------------------------------*
  Is_*
 *---------------------------------------------------------------------------*/
#define NW_TRAITS_SPEC0(Spec, Value) \
    template <> \
    struct Spec : public internal::integral_constant<bool, Value> {};

#define NW_TRAITS_SPEC1(Spec, Value) \
    template <typename Type> \
    struct Spec : public internal::integral_constant<bool, Value> {};

#define NW_TRAITS_SPEC2(Spec, Value) \
    template <typename Type, typename ClassName> \
    struct Spec : public internal::integral_constant<bool, Value> {};

#define NW_TRAITS_SPEC(Order, Traits, SpecialType, Value)            \
    NW_TRAITS_SPEC##Order(Traits<SpecialType>,                Value) \
    NW_TRAITS_SPEC##Order(Traits<SpecialType const>,          Value) \
    NW_TRAITS_SPEC##Order(Traits<SpecialType volatile>,       Value) \
    NW_TRAITS_SPEC##Order(Traits<SpecialType const volatile>, Value)


//---------------------------------------------------------------------------
//! @brief テンプレート中で、ポインタからポインタを除いた型を取得します。
//---------------------------------------------------------------------------
template < typename T >
struct PeelPointer
{
    template< typename U >  struct In                       {   typedef U   Type;   };
    template< typename U >  struct In<U *>                  {   typedef U   Type;   };
    template< typename U >  struct In<const U *>            {   typedef U   Type;   };
    template< typename U >  struct In<volatile U *>         {   typedef U   Type;   };
    template< typename U >  struct In<const volatile U *>   {   typedef U   Type;   };

    typedef typename In<T>::Type        Type;
};

//---------------------------------------------------------------------------
//! @brief テンプレート中で、参照から参照を除いた型を取得する。
//---------------------------------------------------------------------------
template < typename T >
struct PeelReference
{
    template< typename U >  struct In                       {   typedef U   Type;   };
    template< typename U >  struct In<U &>                  {   typedef U   Type;   };
    template< typename U >  struct In<const U &>            {   typedef U   Type;   };
    template< typename U >  struct In<volatile U &>         {   typedef U   Type;   };
    template< typename U >  struct In<const volatile U &>   {   typedef U   Type;   };

    typedef typename In<T>::Type        Type;
};

//---------------------------------------------------------------------------
//! @brief テンプレート中で、型が配列かどうかを判定します。
//---------------------------------------------------------------------------
template <typename TType>
struct IsArray : public internal::FalseType {};

//! @briefprivate
template <typename TType, int Size>
struct IsArray<TType[Size]> : public internal::TrueType {};

//! @briefprivate
template <typename TType>
struct IsArray<TType[]> : public internal::TrueType {};

//---------------------------------------------------------------------------
//! @brief テンプレート中で、型がポインタかどうかを判定します。
//---------------------------------------------------------------------------
template <typename Type>
struct IsPointer : public internal::FalseType {};
NW_TRAITS_SPEC(1, IsPointer, Type*, true)


//---------------------------------------------------------------------------
//! @brief テンプレート中で、型が参照かどうかを判定します。
//---------------------------------------------------------------------------
template <typename Type>
struct IsReference : public internal::FalseType {};

//! @briefprivate
template <typename Type>
struct IsReference<Type&> : public internal::TrueType {};

//---------------------------------------------------------------------------
//!  @brief テンプレート中で、二つの型が等しいかどうかを判定します。
//!
//!  @details
//!  IsSame を NW_STATIC_ASSERT などのマクロで使用する際には、 IsSame 全体を
//!  括弧で括らないと "," がマクロ引数と解釈され構文エラーとなりますのでご注意下さい。
//---------------------------------------------------------------------------
template <typename, typename>
struct IsSame : public internal::FalseType {};

//! @briefprivate
template <typename Type>
struct IsSame<Type, Type> : public internal::TrueType {};

//---------------------------------------------------------------------------
//!  @brief テンプレート中で、型がクラスかどうかを判定します。
//---------------------------------------------------------------------------
template <class T>
struct IsClass
{
    template <class C> static internal::True test(int C::*);
    template <class C> static internal::False test(...);
    static const bool value = (sizeof(test<T>(0)) == sizeof(internal::True));
};

//---------------------------------------------------------------------------
//!  @brief テンプレート中で、条件によってい処理を切り替えます。
//---------------------------------------------------------------------------
template<typename Cond, typename Then, typename Else>
struct If_ : public internal::IfCond<Cond::value, Then, Else> {};

} // namespace ut
} // namespace nw

#endif // NW_UT_TYPETRAITS_H_
