﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#ifndef NW_UT_MEMORYRANGE_H_
#define NW_UT_MEMORYRANGE_H_

#include <nw/types.h>

namespace nw
{
namespace ut
{

//---------------------------------------------------------------------------
//! @brief 範囲を表すクラスです。
//---------------------------------------------------------------------------
template<typename TElement, typename TSize = size_t>
class Range
{
public:
    typedef TElement* iterator;
    typedef const TElement* const_iterator;
    typedef TElement& reference;
    typedef const TElement& const_reference;

    //! @brief デフォルトコンストラクタです。
    Range() : m_Begin(NULL), m_End(NULL) {}

    //! @brief コピーコンストラクタです。
    //! @param[in] src コピー元の Range です。
    Range(const Range& src)
    : m_Begin(src.m_Begin), m_End(src.m_End) {}

    //! @brief コンストラクタです。
    //!
    //! @param[in] begin 範囲の開始位置です。
    //! @param[in] end 範囲の終了位置です。
    template<typename TType>
    Range(TType begin, TType end)
    : m_Begin(begin), m_End(end)
    {
        NW_ASSERT(begin <= end);
    }

    //! @brief 範囲の開始位置を取得します。
    //! @return 範囲の開始位置です。
    iterator Begin() { return m_Begin; }
    //! @brief 範囲の開始位置を取得します。
    //! @return 範囲の開始位置です。
    const iterator Begin() const { return m_Begin; }
    //! @brief 範囲の終了位置を取得します。
    //! @return 範囲の終了位置です。
    iterator End() { return m_End; }
    //! @brief 範囲の終了位置を取得します。
    //! @return 範囲の終了位置です。
    const iterator End() const { return m_End; }

    //! @brief 範囲が空の場合 true を返します。
    //! @return 範囲が空の場合 true を返します。
    bool Empty() const { return m_Begin == m_End; }

    //! @brief 範囲の要素数を返します。
    //! @return 範囲の要素数です。
    TSize Size() const { return static_cast<TSize>(std::distance(m_Begin, m_End)); }

private:
    iterator m_Begin;
    iterator m_End;
};

//! @brief Range を作成します。
//! @param[in] begin 範囲の開始位置です。
//! @param[in] end 範囲の終了位置です。
//! @return 作成した Range です。
template<typename TElement>
NW_INLINE Range<TElement>
MakeRange(TElement* begin, TElement* end)
{
    return Range<TElement>(begin, end);
}

//! @brief 要素の先頭と数を指定して Range を作成します。
//! @param[in] begin 要素の先頭です。
//! @param[in] size 要素の数です。
//! @return 作成した Range です。
template<typename TElement, typename TSize>
NW_INLINE Range<TElement, TSize>
MakeRange(TElement* begin, TSize size)
{
    return Range<TElement, TSize>(begin, begin + size);
}

//! @brief 配列から Range を作成します。
//! @param[in] array 配列です。
//! @return 作成した Range です。
template<typename TElement, size_t Size>
NW_INLINE Range<TElement>
MakeRange(TElement (&array)[Size])
{
    return Range<TElement, size_t>(array, array + Size);
}

//----------------------------------------
//! @brief メモリの範囲を表すクラスです。
typedef Range<u8, size_t> MemoryRange;

//! @brief MemoryRange を作成します。
//! @param[in] address メモリのアドレスです。
//! @param[in] size メモリの範囲の大きさです。
//! @return 作成した MemoryRange です。
NW_INLINE MemoryRange
MakeMemoryRange(void* address, size_t size)
{
    return MemoryRange(static_cast<u8*>(address), static_cast<u8*>(address) + size);
}

//! @brief MemoryRange のメモリのアドレスを取得します。
//! @param[in] range 取得元となる MemoryRange の参照です。
//! @return 取得したメモリのアドレスです。
template<typename TType>
NW_INLINE TType
GetAddress(MemoryRange& range)
{
    return reinterpret_cast<TType>(range.Begin());
}

//! @brief MemoryRange のメモリのアドレスを取得します。
//! @param[in] range 取得元となる MemoryRange の const 参照です。
//! @return 取得したメモリのアドレスです。
template<typename TType>
NW_INLINE const TType
GetAddress(const MemoryRange& range)
{
    return reinterpret_cast<const TType>(range.Begin());
}

} // namespace ut
} // namespace nw

#endif // NW_UT_MEMORYRANGE_H_
