﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#ifndef NW_UT_MEMORY_FILE_STREAM_H_
#define NW_UT_MEMORY_FILE_STREAM_H_

#include <nw/ut/ut_FileStream.h>

namespace nw   {
namespace ut   {

//---------------------------------------------------------------------------
//! @brief     メモリ上のデータをファイルとして利用するファイルストリームです。
//---------------------------------------------------------------------------
class MemoryFileStream : public FileStream
{
//------ publicメンバ --------------------------
public:
    NW_UT_RUNTIME_TYPEINFO( FileStream )   // ダウンキャスト用の実行時型情報を埋め込みます

    //---------------------------------------------------------------------------
    //! @brief       コンストラクタです。
    //---------------------------------------------------------------------------
    MemoryFileStream();

    //---------------------------------------------------------------------------
    //! @brief       コンストラクタです。
    //!
    //! @param[in]   fileStream  MemoryFileStream のポインタです。
    //---------------------------------------------------------------------------
    /* ctor */ explicit MemoryFileStream( MemoryFileStream* fileStream );

    //---------------------------------------------------------------------------
    //! @brief       ストリームをオープンします。
    //!
    //! @param[in]   addr       開始アドレスです。
    //! @param[in]   size       サイズです。
    //! @return      オープンに成功した場合は true を返します。
    //---------------------------------------------------------------------------
    bool Open( void* addr, u32 size );

    //---------------------------------------------------------------------------
    //! @brief       ストリームをクローズします。
    //---------------------------------------------------------------------------
    virtual void        Close();

    //---------------------------------------------------------------------------
    //! @brief       ストリームからデータを読み込みます(同期処理)
    //!
    //! @param[in]   buf         読み込み先バッファへのポインタ
    //! @param[in]   length      読み込むデータ長
    //!
    //! @return      実際に読み込まれたデータサイズを返します。
    //---------------------------------------------------------------------------
    virtual s32         Read( void* buf, u32 length );

    //---------------------------------------------------------------------------
    //! @brief       ストリームにデータを書き込みます。(同期版)
    //!
    //! @param[in]   buf         書き込み元バッファ
    //! @param[in]   length      書き込みデータのバイト数
    //!
    //! @return      実際に書き込まれたバイト数を返します。
    //---------------------------------------------------------------------------
    virtual s32         Write( const void* buf, u32 length );

    //---------------------------------------------------------------------------
    //! @brief       非同期処理の可否を取得します。
    //!
    //! @return      非同期処理が可能な場合、 true を返します。
    //---------------------------------------------------------------------------
    virtual bool        CanAsync() const { return false; }

    //---------------------------------------------------------------------------
    //! @brief       Read 処理の可否を取得します。
    //!
    //! @return      Read が可能な場合、 true を返します。
    //---------------------------------------------------------------------------
    virtual bool        CanRead() const { return m_Available; }

    //---------------------------------------------------------------------------
    //! @brief       Write 処理の可否を取得します。
    //!
    //! @return      Write が可能な場合、 true を返します。
    //---------------------------------------------------------------------------
    virtual bool        CanWrite() const { return m_Available; }

    //---------------------------------------------------------------------------
    //! @brief       ストリームがオープンできているかを取得します。
    //!
    //! @return      ストリームがオープンできている場合には true を返します。
    //---------------------------------------------------------------------------
    virtual bool        IsAvailable() const { return m_Available; }

    //---------------------------------------------------------------------------
    //! @brief       メモリサイズを取得します。
    //!
    //! @return      メモリサイズを返します。
    //---------------------------------------------------------------------------
    virtual u32         GetSize() const;

    //---------------------------------------------------------------------------
    //! @brief        メモリポインタを移動します。
    //!
    //! @param[in]    offset  ファイルポインタ移動値を指定します。
    //! @param[in]    origin  ファイルポインタ移動の基準点を指定します。
    //!                       FILE_STREAM_SEEK_BEGIN      - ファイルの先頭を基準
    //!                       FILE_STREAM_SEEK_CURRENT    - 現在の読み込み位置を基準
    //!                       FILE_STREAM_SEEK_END        - ファイルの終端を基準(offset <= 0で指定されるはずです)
    //! @return       移動に成功した場合は true を返します。
    //---------------------------------------------------------------------------
    virtual bool        Seek( s32 offset, u32 origin );

    //---------------------------------------------------------------------------
    //! @brief        メモリポインタをランダムに移動できるかどうかの属性を取得します。
    //!
    //! @return       Seek が可能な場合、 true を返します。
    //---------------------------------------------------------------------------
    virtual bool        CanSeek() const { return true; }

    //---------------------------------------------------------------------------
    //! @brief        メモリの Read をキャンセルできるかどうかの属性を取得します。
    //!
    //! @return       キャンセルが可能な場合、 true を返します。
    //---------------------------------------------------------------------------
    virtual bool        CanCancel() const { return false; }

    //---------------------------------------------------------------------------
    //! @brief        現在のメモリ上のポインタの位置を取得します。
    //!
    //! @return       ポインタの位置を返します。
    //---------------------------------------------------------------------------
    virtual u32         Tell()  const { return m_CurPos; }


private:
    bool m_Available;
    u32 m_StartPos;
    u32 m_CurPos;
    u32 m_Size;
};

}   /* namespace ut   */
}   /* namespace nw   */

/* NW_UT_MEMORY_FILE_STREAM_H_ */
#endif
