﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#ifndef NW_UT_FOREACH_H_
#define NW_UT_FOREACH_H_

#include <nw/types.h>
#include <nw/ut/ut_Range.h>
#include <utility>

namespace nw
{
namespace ut
{
namespace internal
{

struct static_any_base
{
    operator bool() const
    {
        return false;
    }
};

template <typename Type>
struct static_any : public static_any_base
{
    explicit static_any(const Type& item) : m_Item(item) {}
    mutable Type m_Item;
};

typedef const static_any_base& static_any_t;


template <typename Type>
NW_INLINE Type& static_any_cast(static_any_t value)
{
    return static_cast<const static_any<Type>&>(value).m_Item;
}

//----------------------------------------
// typeof
//----------------------------------------
template <typename Type>
struct contain_type
{
    typedef Type type;
};

template <typename Type>
NW_INLINE contain_type<Type>* encode_type(Type&)
{
    return 0;
}

template <typename Type>
NW_INLINE contain_type<const Type>* encode_type(const Type&)
{
    return 0;
}

#define NW_FOREACH_TYPEOF(COL) \
    (true ? 0 : nw::ut::internal::encode_type(COL))


//----------------------------------------
// contain
//----------------------------------------
template <typename Type>
NW_INLINE static_any<Type> contain(const Type& col)
{
    return static_cast<static_any<Type> >(col);
}

template <typename Type>
NW_INLINE static_any<Type*> contain(Type& col)
{
    return static_cast<static_any<Type*> >(&col);
}

template <typename Type, int Size>
NW_INLINE static_any<Type*> contain(Type (&col)[Size])
{
    return static_cast<static_any<Type*> >(col);
}

//----------------------------------------
// Array
//----------------------------------------
template <typename Type, int Size>
NW_INLINE static_any<Type*> begin(static_any_t cur, contain_type<Type[Size]>*)
{
    return static_cast<static_any<Type*> >(static_any_cast<Type*>(cur));
}

template <typename Type, int Size>
NW_INLINE static_any<Type*> end(static_any_t cur, contain_type<Type[Size]>*)
{
    return static_cast<static_any<Type*> >(static_any_cast<Type*>(cur) + Size);
}

template <typename Type, int Size>
NW_INLINE void next(static_any_t cur, contain_type<Type[Size]>*)
{
    ++static_any_cast<Type*>(cur);
}

template <typename Type, int Size>
NW_INLINE Type& extract(static_any_t cur, contain_type<Type[Size]>*)
{
    return *static_any_cast<Type*>(cur);
}


template <typename Type, int Size>
NW_INLINE bool done(static_any_t cur, static_any_t end, contain_type<Type[Size]>*)
{
    return static_any_cast<Type*>(cur) == static_any_cast<Type*>(end);
}

//----------------------------------------
// Container
//----------------------------------------
template <typename Type>
NW_INLINE static_any<typename Type::iterator> begin(static_any_t cur, contain_type<Type>*)
{
    return static_cast<static_any<typename Type::iterator> >(static_any_cast<Type*>(cur)->begin());
}

template <typename Type>
NW_INLINE static_any<typename Type::const_iterator> begin(static_any_t cur, contain_type<const Type>*)
{
    return static_cast<static_any<typename Type::const_iterator> >(static_any_cast<Type>(cur).begin());
}

template <typename Type>
NW_INLINE static_any<typename Type::iterator> end(static_any_t cur, contain_type<Type>*)
{
    return static_cast<static_any<typename Type::iterator> >(static_any_cast<Type*>(cur)->end());
}

template <typename Type>
NW_INLINE static_any<typename Type::const_iterator> end(static_any_t cur, contain_type<const Type>*)
{
    return static_cast<static_any<typename Type::cosnt_iterator> >(static_any_cast<Type>(cur).end());
}

template <typename Type>
NW_INLINE void next(static_any_t cur, contain_type<Type>*)
{
    ++static_any_cast<typename Type::iterator>(cur);
}

template <typename Type>
NW_INLINE typename Type::reference extract(static_any_t cur, contain_type<Type>*)
{
    return *static_any_cast<typename Type::iterator>(cur);
}

template <typename Type>
NW_INLINE bool done(static_any_t cur, static_any_t end, contain_type<Type>*)
{
    typedef typename Type::iterator Iter;
    return static_any_cast<Iter>(cur) == static_any_cast<Iter>(end);
}

//----------------------------------------
// pair
//----------------------------------------
template <typename Type>
NW_INLINE static_any<Type> begin(static_any_t cur, contain_type<std::pair<Type, Type> >*)
{
    return static_cast<static_any<Type> >(static_any_cast<std::pair<Type, Type> >(cur).first);
}

template <typename Type>
NW_INLINE static_any<Type> end(static_any_t cur, contain_type<std::pair<Type, Type> >*)
{
    return static_cast<static_any<Type> >(static_any_cast<std::pair<Type, Type> >(cur).second);
}

template <typename Type>
NW_INLINE void next(static_any_t cur, contain_type<std::pair<Type, Type> >*)
{
    ++static_any_cast<Type>(cur);
}

template <typename Type>
NW_INLINE typename Type::reference extract(static_any_t cur, contain_type<std::pair<Type, Type> >*)
{
    return *static_any_cast<Type>(cur);
}

template <typename Type>
NW_INLINE bool done(static_any_t cur, static_any_t end, contain_type<std::pair<Type, Type> >*)
{
    return static_any_cast<Type>(cur) == static_any_cast<Type>(end);
}

//----------------------------------------
// pair
//----------------------------------------
template <typename Type>
NW_INLINE static_any<Type*> begin(static_any_t cur, contain_type<std::pair<Type*, Type*> >*)
{
    return static_cast<static_any<Type*> >(static_any_cast<std::pair<Type*, Type*> >(cur).first);
}

template <typename Type>
NW_INLINE static_any<Type*> end(static_any_t cur, contain_type<std::pair<Type*, Type*> >*)
{
    return static_cast<static_any<Type*> >(static_any_cast<std::pair<Type*, Type*> >(cur).second);
}

template <typename Type>
NW_INLINE void next(static_any_t cur, contain_type<std::pair<Type*, Type*> >*)
{
    ++static_any_cast<Type*>(cur);
}

template <typename Type>
NW_INLINE Type& extract(static_any_t cur, contain_type<std::pair<Type*, Type*> >*)
{
    return *static_any_cast<Type*>(cur);
}

template <typename Type>
NW_INLINE bool done(static_any_t cur, static_any_t end, contain_type<std::pair<Type*, Type*> >*)
{
    return static_any_cast<Type*>(cur) == static_any_cast<Type*>(end);
}

//----------------------------------------
// Range
//----------------------------------------
template <typename Type>
NW_INLINE static_any<typename Range<Type>::iterator> begin(static_any_t cur, contain_type<Range<Type> >*)
{
    return static_cast<static_any<typename Range<Type>::iterator> >(static_any_cast<Range<Type>*>(cur)->Begin());
}

template <typename Type>
NW_INLINE static_any<typename Range<Type>::const_iterator> begin(static_any_t cur, contain_type<const Range<Type> >*)
{
    return static_cast<static_any<typename Range<Type>::const_iterator> >(static_any_cast<Range<Type> >(cur).Begin());
}

template <typename Type>
NW_INLINE static_any<typename Range<Type>::iterator> end(static_any_t cur, contain_type<Range<Type> >*)
{
    return static_cast<static_any<typename Range<Type>::iterator> >(static_any_cast<Range<Type>*>(cur)->End());
}

template <typename Type>
NW_INLINE static_any<typename Range<Type>::const_iterator> end(static_any_t cur, contain_type<const Range<Type> >*)
{
    return static_cast<static_any<typename Range<Type>::const_iterator> >(static_any_cast<Range<Type> >(cur).End());
}

template <typename Type>
NW_INLINE void next(static_any_t cur, contain_type<Range<Type> >*)
{
    ++static_any_cast<typename Range<Type>::iterator>(cur);
}

template <typename Type>
NW_INLINE typename Range<Type>::reference extract(static_any_t cur, contain_type<Range<Type> >*)
{
    return *static_any_cast<typename Range<Type>::iterator>(cur);
}

template <typename Type>
NW_INLINE bool done(static_any_t cur, static_any_t end, contain_type<Range<Type> >*)
{
    typedef typename Range<Type>::iterator Iter;
    return static_any_cast<Iter>(cur) == static_any_cast<Iter>(end);
}

} // namespace internal
} // namespace ut
} // namespace nw

#define NW_FOREACH_CONTAIN(COL) \
    nw::ut::internal::contain(COL)

#define NW_FOREACH_BEGIN(COL) \
    nw::ut::internal::begin(_contain, NW_FOREACH_TYPEOF(COL))

#define NW_FOREACH_END(COL) \
    nw::ut::internal::end(_contain, NW_FOREACH_TYPEOF(COL))

#define NW_FOREACH_DONE(COL) \
    nw::ut::internal::done(_cur, _end, NW_FOREACH_TYPEOF(COL))

#define NW_FOREACH_NEXT(COL) \
    nw::ut::internal::next(_cur, NW_FOREACH_TYPEOF(COL))

#define NW_FOREACH_EXTRACT(COL) \
    nw::ut::internal::extract(_cur, NW_FOREACH_TYPEOF(COL))

//! foreach のマクロです。
#define NW_FOREACH(VAR, COL) \
    if       (nw::ut::internal::static_any_t _contain = NW_FOREACH_CONTAIN(COL)) {} \
    else if  (nw::ut::internal::static_any_t _cur     = NW_FOREACH_BEGIN(COL))   {} \
    else if  (nw::ut::internal::static_any_t _end     = NW_FOREACH_END(COL))     {} \
    else for (bool _continue = true;                                                \
              _continue && !NW_FOREACH_DONE(COL);                                   \
              _continue ? NW_FOREACH_NEXT(COL) : (void)0)                           \
             if      ((_continue = false) == true) {}                               \
             else for (VAR = NW_FOREACH_EXTRACT(COL); !_continue; _continue = true)

#endif // NW_UT_FOREACH_H_
