﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#ifndef NW_UT_FILE_STREAM_H_
#define NW_UT_FILE_STREAM_H_

#include <nw/ut/ut_IoStream.h>

namespace nw   {
namespace ut   {

enum
{
    FILE_STREAM_SEEK_BEGIN,
    FILE_STREAM_SEEK_CURRENT,
    FILE_STREAM_SEEK_END
};

//---------------------------------------------------------------------------
//! @brief     ファイルストリームクラスです。
//---------------------------------------------------------------------------
class FileStream : public IoStream
{
//------ publicメンバ --------------------------
public:
    NW_UT_RUNTIME_TYPEINFO( IoStream )   // ダウンキャスト用の実行時型情報を埋め込みます

    //---------------------------------------------------------------------------
    //! @brief        ファイルサイズを取得します。
    //---------------------------------------------------------------------------
    virtual u32         GetSize() const         = 0;

    //---------------------------------------------------------------------------
    //! @brief        ファイルポインタを移動します。
    //!
    //! @param[in]    offset  ファイルポインタ移動値を指定します。
    //! @param[in]    origin  ファイルポインタ移動の基準点を指定します。
    //!                       FILE_STREAM_SEEK_BEGIN      - ファイルの先頭を基準
    //!                       FILE_STREAM_SEEK_CURRENT    - 現在の読み込み位置を基準
    //!                       FILE_STREAM_SEEK_END        - ファイルの終端を基準(offset <= 0で指定されるはずです)
    //!
    //! @return       移動に成功した場合は true を返します。
    //---------------------------------------------------------------------------
    virtual bool        Seek( s32 offset, u32 origin );

    //---------------------------------------------------------------------------
    //! @brief        非同期処理をキャンセルします。(同期関数)
    //---------------------------------------------------------------------------
    virtual void        Cancel();

    //---------------------------------------------------------------------------
    //! @brief        非同期処理をキャンセルします。(非同期関数)
    //!
    //! @param[in]    callback  処理完了時のコールバック関数です。
    //! @param[out]   arg       コールバックで返されるユーザパラメータです。
    //!
    //! @return      コマンドが正常に発行された場合には真を返します。
    //!              発行されなかった場合には偽を返します。
    //---------------------------------------------------------------------------
    virtual bool        CancelAsync( IoStreamCallback callback, void* arg );

    //---------------------------------------------------------------------------
    //! @brief        ファイルポインタをランダムに移動できるかどうかの属性を取得します。
    //! @return       移動が可能であれば true を返します。
    //---------------------------------------------------------------------------
    virtual bool        CanSeek  () const         = 0;

    //---------------------------------------------------------------------------
    //! @brief        ファイルのリードをキャンセルできるかどうかの属性を取得します。
    //! @return       キャンセル可能であれば true を返します。
    //---------------------------------------------------------------------------
    virtual bool        CanCancel() const         = 0;

    //---------------------------------------------------------------------------
    //! @brief        ファイルポインタの位置を取得します。
    //! @return       ファイルポインタの位置です。
    //---------------------------------------------------------------------------
    virtual u32         Tell()  const                   = 0;

    //---------------------------------------------------------------------------
    //! @brief        ファイルポインタがファイル末尾にあるかどうかを取得します。
    //! @return       ファイルポインタがファイル末尾にあれば true を返します。
    //---------------------------------------------------------------------------
    bool                IsEof() const { return Tell() >= GetSize();  }

//------ protectedメンバ --------------------------
protected:
    //---  コンストラクタ/デストラクタ -----------
    /* ctor */          FileStream() : IoStream() {}

    //---------------------------------------------------------------------------
    //! @brief ファイル読み込み位置管理用クラス
    //---------------------------------------------------------------------------
    class FilePosition
    {
    public:
        //---------------------------------------------------------------------------
        //! @brief        デフォルトコンストラクタです。
        //---------------------------------------------------------------------------
        /* ctor */          FilePosition() : m_FileSize(0), m_Position(0) {}

        //---------------------------------------------------------------------------
        //! @brief        ファイルサイズを設定します。
        //!
        //! @param[in]    fileSize ファイルサイズです。
        //---------------------------------------------------------------------------
        void                SetFileSize( u32 fileSize ) { m_FileSize = fileSize; }

        //---------------------------------------------------------------------------
        //! @brief        ファイルサイズを取得します。
        //!
        //! @return       ファイルサイズです。
        //---------------------------------------------------------------------------
        u32                 GetFileSize() const         { return m_FileSize; }

        //---------------------------------------------------------------------------
        //! @brief        ファイルポインタの移動。
        //!                 ファイルの終端を越えた場合には、ファイルポインタは終端位置を示します。
        //!
        //! @param[in]    offset   ファイルポインタ移動オフセット
        //!
        //! @return       移動後のファイル位置を返します。
        //---------------------------------------------------------------------------
        u32                 Skip( s32 offset );

        //---------------------------------------------------------------------------
        //! @brief        データ追加によるファイルポインタの移動。
        //!               ファイルの終端を越えた場合には、その分ファイルサイズを増加します。
        //!
        //! @param[in]    offset  ファイルポインタ移動オフセット
        //!
        //! @return       移動後のファイル位置を返します。
        //---------------------------------------------------------------------------
        u32                 Append( s32 offset );

        //---------------------------------------------------------------------------
        //! @brief        ファイルポインタを移動。
        //!
        //! @param[in]    offset  ファイルポインタ移動値を指定します。
        //! @param[in]    origin  ファイルポインタ移動の基準点を指定します。
        //!                       FILE_STREAM_SEEK_BEGIN      - ファイルの先頭を基準
        //!                       FILE_STREAM_SEEK_CURRENT    - 現在の読み込み位置を基準
        //!                       FILE_STREAM_SEEK_END        - ファイルの終端を基準(offset <= 0で指定されるはずです)
        //---------------------------------------------------------------------------
        void                Seek( s32 offset, u32 origin );

        //---------------------------------------------------------------------------
        //! @brief        ファイルポインタの現在位置を取得します。
        //!
        //! @return       ファイルの先頭からのオフセットです。
        //---------------------------------------------------------------------------
        u32                 Tell() const { return m_Position; }
    private:
        u32                 m_FileSize;      //!< ファイルサイズ
        u32                 m_Position;      //!< 読み込み位置
    };
};


}   /* namespace ut     */
}   /* namespace nw   */


/* NW_UT_FILE_STREAM_H_ */
#endif
