﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#ifndef NW_UT_BITSET_H_
#define NW_UT_BITSET_H_

#include <nw/types.h>
#include <nw/ut/ut_Inlines.h>

namespace nw {
namespace ut {

//---------------------------------------------------------------------------
//! @brief        固定長の簡易ビットセットです。
//!
//! @details      固定長のビットセットを表すクラスです。
//!               std::bitset を使用した際に、stream のスタティックイニシャライザが
//!               実行されて都合が悪いといったケースでの利用を想定しています。
//---------------------------------------------------------------------------
template <size_t MaxSize>
class BitSet
{
public:
    //---------------------------------------------------------------------------
    //! @brief        コンストラクタです。
    //---------------------------------------------------------------------------
    /* ctor */ BitSet()
    {
        this->Reset();
    }

    //---------------------------------------------------------------------------
    //! @brief        すべてのビットを 0 でリセットします。
    //!
    //! @return       *this を返します。
    //---------------------------------------------------------------------------
    BitSet<MaxSize>& Reset()
    {
        for ( int i = 0; i < ( sizeof(m_Flags) / sizeof(u32) ); ++i )
        {
            m_Flags[ i ] = 0;
        }

        return *this;
    }


    //---------------------------------------------------------------------------
    //! @brief        ビットフラグを 0 に設定します。
    //!
    //! @param[in]    index   ビットのインデックスです。
    //!
    //! @return       *this を返します。
    //---------------------------------------------------------------------------
    BitSet<MaxSize>& Reset( int index )
    {
        return this->Set( index, 0 );
    }

    //---------------------------------------------------------------------------
    //! @brief        ビットフラグを設定します。
    //!
    //! @param[in]    index   ビットのインデックスです。
    //! @param[in]    value   設定する値です。
    //!
    //! @return       *this を返します。
    //---------------------------------------------------------------------------
    BitSet<MaxSize>& Set( int index, int value = 1 )
    {
        NW_ASSERT_MINMAXLT(index, 0, MaxSize);

        if ( value )
        {
            m_Flags[ index / 32 ] |= 0x1 << (index % 32);
        }
        else
        {
            m_Flags[ index / 32 ] &= ~(0x1 << (index % 32));
        }

        return *this;
    }

    //---------------------------------------------------------------------------
    //! @brief        ビットフラグを取得します。
    //!
    //! @param[in]    index   ビットのインデクスです。
    //!
    //! @return       ビットが 0 ならば false, 1 ならば true を返します。
    //---------------------------------------------------------------------------
    bool Test( int index ) const
    {
        NW_ASSERT_MINMAXLT(index, 0, MaxSize);

        return (m_Flags[ index / 32 ] & (1 << (index % 32))) != 0;
    }

    //---------------------------------------------------------------------------
    //! @brief        保持できるビットのサイズを取得します。
    //!
    //! @return       保持できるビット数です。
    //---------------------------------------------------------------------------
    int Size() const
    {
        return MaxSize;
    }

    //---------------------------------------------------------------------------
    //! @brief        立っているビットの数を取得します。
    //!
    //! @return       立っているビットの数を取得します。
    //---------------------------------------------------------------------------
    int Count() const
    {
        // TODO: 結果をキャッシュしておく。

        int count = 0;

        for ( int i = 0; i < sizeof(m_Flags) / sizeof(u32); ++i )
        {
            u32 value = m_Flags[ i ];

            while ( value )
            {
                if (value & 0x1) { ++count; }
                value >>= 1;
            }
        }

        return count;
    }

    //---------------------------------------------------------------------------
    //! @brief        1 つでも 1 のビットがあるかどうかを取得します。
    //!
    //! @return       1であるビットが存在すれば true、存在しなければ false を返します。
    //---------------------------------------------------------------------------
    bool Any() const
    {
        for ( int i = 0; i < sizeof(m_Flags) / sizeof(u32); ++i )
        {
            if ( m_Flags[ i ] ) { return true; }
        }

        return false;
    }

    //---------------------------------------------------------------------------
    //! @brief        1 つも 1 のビットがないかどうかを取得します。
    //!
    //! @return       1 であるビットが存在しない場合は true を返します。
    //---------------------------------------------------------------------------
    bool None() const
    {
        return ! this->Any();
    }

    //---------------------------------------------------------------------------
    //! @brief        二つのビットセットの比較演算子です。
    //!
    //! @param[in]    bitSet  比較対象のビットセットです。
    //!
    //! @return       すべてのビットが等しければ true を返します。
    //---------------------------------------------------------------------------
    bool operator ==(const BitSet<MaxSize>& bitSet)
    {
        for ( int i = 0; i < ( sizeof(m_Flags) / sizeof(u32) ); ++i )
        {
            if (this->m_Flags[ i ] != bitSet.m_Flags[ i ]) { return false; }
        }

        return true;
    }

    //---------------------------------------------------------------------------
    //! @brief        二つのビットセットの否定の比較演算子です。
    //!
    //! @param[in]    bitSet  比較対象のビットセットです。
    //!
    //! @return       すべてのビットが等しければ true を返します。
    //---------------------------------------------------------------------------
    bool operator !=(const BitSet<MaxSize>& bitSet)
    {
        return ! (*this == bitSet);
    }

private:
    u32 m_Flags[ NW_UT_DIV_UP(MaxSize, 32) ];
};



} /* namespace ut */
} /* namespace nw */

#endif //  NW_UT_BITSET_H_
