﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#ifndef NW_SND_SPY_SPY_SESSION_H_
#define NW_SND_SPY_SPY_SESSION_H_

#include <nw/snd/spy/sndspy_Config.h>
#ifdef NW_SND_SPY_ENABLE

#include <nw/snd/spy/fnd/os/sndspyfnd_CriticalSection.h>
#include <nw/snd/spy/fnd/os/sndspyfnd_Thread.h>
#include <nw/snd/spy/fnd/hio/sndspyfnd_HioChannel.h>
#include <nw/snd/spy/sndspy_SpyPacketReader.h>
#include <nw/snd/spy/sndspy_SpyDataInfo.h>

namespace nw {
namespace snd {
namespace spy {
namespace internal {

struct PacketHeader;

//! @brief Spy セッションを制御します。
//!        すべて同期関数です。
class SpySession
{
public: // 型
    //! @brief ポートの型です。
    typedef nw::snd::spy::internal::fnd::HioChannel::PortType PortType;

    //! @brief 状態変更コールバック関数です。
    //! @param[in]  param  任意のパラメータを指定します。
    typedef void (*StateChangedCallback)(void* param);

    //! @brief メッセージハンドラです。
    class IMessageHandler
    {
    public:
        virtual void OnInitializeSession() = 0;
        virtual void OnFinalizeSession() = 0;

        virtual void OnSelectDataID(u32 numFlags, u32 flags[]) = 0;
        virtual void OnSetOutputDirPath(const char* outputDir) = 0;
        virtual void OnDataRead(u32 fileNo) = 0;

        //! データタイプを問い合わせます。
        virtual const SpyDataInfo* OnQueryDataInfo() = 0;
    };

private: // 型
    //! @brief セッションの状態です。
    enum State
    {
        STATE_NOT_INITIALIZED,
        STATE_INITIALIZED,
        STATE_OPENING,
        STATE_OPENED,
        STATE_DISCONNECTING,
        STATE_CONNECTING,
        STATE_PREPARING,
        STATE_PREPARED
    };

public: // コンストラクタ
    SpySession();

public: // メソッド
    void Initialize(void* buffer, u32 bufferLength);
    void Finalize();

    static size_t GetRequiredMemorySize();

    void SetStateChangedCallback(StateChangedCallback callback, void* param)
    {
        m_StateChangedCallback = callback;
        m_StateChangedCallbackParam = param;
    }

    bool Open(PortType syncPort, PortType dataPort);
    void Close();

    bool Connect();
    void Disconnect();

    bool IsInitialized() const { return m_State > STATE_NOT_INITIALIZED; }
    bool IsOpened() const { return m_State > STATE_OPENING; }
    bool IsConnecting() const { return m_State == STATE_CONNECTING; }
    bool IsConnected() const { return m_State >= STATE_PREPARING; }
    bool IsPreparing() const { return m_State == STATE_PREPARING; }
    bool IsPrepared() const { return m_State >= STATE_PREPARED; }

    //! @brief 同期チャンネルのパケットを受信して処理します。
    //!        通信のレスポンスを上げるには、できるだけ頻繁に呼び出してください。
    bool ProcessSyncPacket(IMessageHandler* msgHandler);

    u32 SendData(const void* buffer, u32 length);

private: // メソッド
    void SetState(State value);

    bool OpenSyncChannel(PortType port);
    void CloseSyncChannel();

    void* GetPacketBodyBuffer();
    u32 GetPacketBodyBufferSize() { return m_SyncChannelBufferLength - sizeof(PacketHeader); }

    bool OnInitialize(IMessageHandler* msgHandler);
    bool OnFinalize(IMessageHandler* msgHandler);
    bool OnSelectDataID(IMessageHandler* msgHandler);
    bool OnSetOutputDir(IMessageHandler* msgHandler);
    bool OnNotifyDataRead(IMessageHandler* msgHandler);
    bool OnPing();
    bool OnQueryDataInfo(IMessageHandler* msgHandler);

    static const char* StateToString(State value);

private: // メンバ変数
    volatile State       m_State;
    StateChangedCallback m_StateChangedCallback;
    void*                m_StateChangedCallbackParam;

    nw::snd::spy::internal::fnd::CriticalSection m_StateLock;

    nw::snd::spy::internal::fnd::HioChannel m_SyncChannel;
    nw::snd::spy::internal::fnd::HioChannel m_DataChannel;

    internal::SpyPacketReader m_SyncPacketReader;

    void* m_SyncChannelBuffer;          //!< SyncChannel のパケット送受信に利用します。
    u32   m_SyncChannelBufferLength;
};

} // namespace nw::snd::spy::internal
} // namespace nw::snd::spy
} // namespace nw::snd
} // namespace nw

#endif // NW_SND_SPY_ENABLE

#endif // NW_SND_SPY_SPY_SESSION_H_
