﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#ifndef NW_SND_SPY_SPY_DATA_INFO_H_
#define NW_SND_SPY_SPY_DATA_INFO_H_

#include <nw/snd/spy/sndspy_Config.h>
#ifdef NW_SND_SPY_ENABLE

#include <nw/snd/spy/fnd/binary/sndspyfnd_Binary.h>
#include <nw/snd/spy/sndspy_SpyDataID.h>

namespace nw {
namespace snd {
namespace spy {

class SpyController;
class SpyModule;

//! @brief SoundSpyに送信するデータのタイプ情報です。
//!
//! @li SoundSpyに対して、アプリケーションが対応しているデータを通知します。
//! @li セッション中に割り当てられるDataIDを保持します。
//! @li SoundSpyがデータを要求しているのかの情報を提供します。
//!
//! データタイプごとにインスタンスを1つ用意します。
//! SpyController::RegisterDataInfo() で登録します。
//!
class SpyDataInfo
{
public:
    SpyDataInfo()
        : m_DataID(DATA_ID_INVALID)
    {
    }

    //! @brief SoundSpy に、対応するデータのタイプ名を提供します。
    //!
    //! ASCII 文字列で文字列長は 128 以下でなければなりません。
    //!
    virtual const char* GetDataName() const = 0;

    //! @brief SoundSpy に、対応するデータのタイプバージョンを提供します。
    //!
    //! バージョン番号(0, 0, 0, 0)は不正です。
    //!
    virtual u32 GetDataVersion() const = 0;

    //! @brief セッション中に有効なDataIDの値を得ます。
    SpyDataID GetDataID() const
    {
        return static_cast<SpyDataID>(m_DataID);
    }

private:
    //! @brief セッション中に有効なDataIDを割り当てます。
    void SetDataID(SpyDataID dataId)
    {
        m_DataID = static_cast<u16>(dataId);
    }

private:
     u16 m_DataID;

     friend class SpyModule;
};

//!
//! @brief SpyDataInfoの簡単な実装を提供するクラスです。
//!
class SpySimpleDataInfo : public SpyDataInfo
{
public:
    //! コンストラクタです。
    //!
    //! @param[in] dataName  データ名です。
    //! @param[in] dataVersion データのバージョンです。
    //!
    SpySimpleDataInfo(const char* dataName, const u32 dataVersion)
        : m_DataName(dataName)
        , m_DataVersion(dataVersion)
    {
    }

    virtual const char* GetDataName() const
    {
        return m_DataName;
    }

    virtual u32 GetDataVersion() const
    {
        return m_DataVersion;
    }

private:
    const char* m_DataName;
    u32 m_DataVersion;
};

} // namespace spy
} // namespace snd
} // namespace nw

#endif // NW_SND_SPY_ENABLE

#endif // NW_SND_SPY_SPY_DATA_INFO_H_
