﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#ifndef NW_SND_SOUND_THREAD_H_
#define NW_SND_SOUND_THREAD_H_

#include <nw/types.h>

#if defined( NW_PLATFORM_CAFE )
  #include <cafe/os.h>
#elif defined( NW_PLATFORM_WIN32 )
  #include <winext/cafe/os.h>
#elif defined( NW_USE_NINTENDO_SDK )
  #include <nn/os/os_Tick.h>
#endif

#include <nw/ut.h>
#include <nw/snd/snd_Config.h>
#include <nw/snd/snd_HardwareManager.h>
#include <nw/snd/snd_ThreadStack.h>
#include <nw/snd/snd_Global.h>
#include <nw/snd/snd_ProfileReader.h>

namespace nw {
namespace snd {
namespace internal {

class NwVoiceRenderer;

namespace driver {

class SoundThread : public nw::ut::ThreadHandler
{
    /* ------------------------------------------------------------------------
            typename definition
       ------------------------------------------------------------------------ */
public:
    class SoundFrameCallback
    {
      public:
        ut::LinkListNode m_Link;

        virtual ~SoundFrameCallback() {}
        virtual void OnBeginSoundFrame() {}
        virtual void OnEndSoundFrame() {}
    };

    class PlayerCallback
    {
      public:
        ut::LinkListNode m_Link;

        virtual ~PlayerCallback() {}
        virtual void OnUpdateFrameSoundThread() {}
        virtual void OnShutdownSoundThread() {}
    };

    /* ------------------------------------------------------------------------
            constant definition
       ------------------------------------------------------------------------ */
private:
    static const int THREAD_MESSAGE_BUFSIZE = 32;

    // 上位 4ビットがタイプ
    // 下位28ビットがデータ
    enum Message
    {
        MESSAGE_HW_CALLBACK  = 0x10000000,
        MESSAGE_SHUTDOWN     = 0x20000000,
        MESSAGE_FORCE_WAKEUP = 0x30000000
    };

    /* ------------------------------------------------------------------------
            class member
       ------------------------------------------------------------------------ */
public:
    static SoundThread& GetInstance();

    bool CreateSoundThread(
            s32 priority,
            void* stackBase,
            u32 stackSize,
            u32 affinityMask,
            bool enableGetTick );
    void Destroy();
    bool IsCreated() const { return m_CreateFlag != 0; }

    void Initialize();
    void Finalize();

    void detail_SetNwVoiceRenderer( NwVoiceRenderer* renderer ) { m_pNwVoiceRenderer = renderer; }
    NwVoiceRenderer* detail_GetNwVoiceRenderer() { return m_pNwVoiceRenderer; }

    void Pause( bool pauseFlag ) { m_PauseFlag = pauseFlag; }

    u32 GetFrameCounter() const { return m_AudioFrameCounter; }
    u32 GetAxCallbackCounter() const { return m_AxCallbackCounter; }

    // SoundMaker の波形出力処理で使用するため、public である必要があります。
    void FrameProcess();

    void RegisterSoundFrameUserCallback( SoundFrameUserCallback callback, uptr arg )
    {
        m_UserCallbackArg = arg;
#if defined( NW_PLATFORM_CAFE )
        OSCoherencyBarrier();
#endif
        m_UserCallback = callback;
    }
    void ClearSoundFrameUserCallback()
    {
        m_UserCallback = NULL;
#if defined( NW_PLATFORM_CAFE )
        OSCoherencyBarrier();
#endif
        m_UserCallbackArg = static_cast<uptr>(NULL);
    }

    void RegisterSoundFrameCallback( SoundFrameCallback* callback );
    void UnregisterSoundFrameCallback( SoundFrameCallback* callback );

    void RegisterPlayerCallback( PlayerCallback* callback );
    void UnregisterPlayerCallback( PlayerCallback* callback );

    void Lock() { m_CriticalSection.Enter(); }
    void Unlock() { m_CriticalSection.Leave(); }

    // プロファイル
    int GetDspCycles() const
    {
        return m_DspCycles;
    }
    void ClearDspCycles()
    {
        m_DspCycles = 0;
    }

    void RegisterProfileReader( ProfileReader& profileReader )
    {
        m_ProfileReaderList.PushBack(&profileReader);
    }

    void UnregisterProfileReader( ProfileReader& profileReader )
    {
        m_ProfileReaderList.Erase(&profileReader);
    }

    void ForceWakeup();

private:
#if defined( NW_PLATFORM_WIN32 )
    typedef nw::internal::winext::AXUserCallback AXUserCallback;
    typedef nw::internal::winext::OSTick OSTick;
    typedef nw::internal::winext::AXPROFILE AXPROFILE;
#elif defined( NW_PLATFORM_ANDROID ) || defined( NW_PLATFORM_IOS )
    typedef nw::internal::winext::AXUserCallback AXUserCallback;
    typedef nw::internal::winext::OSTick OSTick;
    typedef nw::internal::winext::AXPROFILE AXPROFILE;
#elif defined( NW_USE_NINTENDO_SDK )
    // TODO: nn_audio
    typedef nw::internal::winext::AXUserCallback AXUserCallback;
    typedef nn::os::Tick OSTick;
    typedef nw::internal::winext::AXPROFILE AXPROFILE;
#endif

    typedef
        ut::LinkList< SoundFrameCallback, offsetof(SoundFrameCallback,m_Link)>
        SoundFrameCallbackList;
    typedef
        ut::LinkList< PlayerCallback, offsetof(PlayerCallback,m_Link)>
        PlayerCallbackList;

    SoundThread();
    virtual ~SoundThread() {}

    bool PrepareForCreate( bool enableGetTick );

    virtual void ThreadHandlerProc();

    static void HwCallbackFunc();
    void HwCallbackProc();

    // #ifndef NW_SND_CONFIG_ENABLE_VOICE_COMMAND
    nw::ut::Thread          m_Thread;
    ut::MessageQueue m_BlockingQueue;
    ut::MessageQueue::BufferType m_MsgBuffer[ THREAD_MESSAGE_BUFSIZE ];
    // #endif

    OSTick                  m_SoundThreadSumTick;   // サウンドスレッド累積処理時間
    int                     m_SoundThreadCount;     // サウンドスレッド累積処理回数
    int                     m_DspCycles;            // 累積 DSP サイクル

    u32 m_AxCallbackCounter;
    u32 m_AudioFrameCounter;

    mutable ut::CriticalSection m_CriticalSection;

#ifndef NW_SND_CONFIG_ENABLE_APPFRAMECALLBACK
    AXUserCallback          m_AxUserCallback;
#endif
    SoundFrameCallbackList  m_SoundFrameCallbackList;
    PlayerCallbackList      m_PlayerCallbackList;

    // 最適化で m_UserCallbackArgと設定順が逆にならないように volatile をつける
    volatile SoundFrameUserCallback  m_UserCallback;
    volatile uptr                    m_UserCallbackArg;

    s32                     m_SoundThreadAffinityMask;
    bool                    m_CreateFlag;
    bool                    m_PauseFlag;
    bool                    m_IsEnableGetTick;

    NwVoiceRenderer* m_pNwVoiceRenderer;

    void RecordProfile( const AXPROFILE& src, nw::ut::Tick beginTick,  nw::ut::Tick endTick, u32 nwVoiceCount );

    bool m_LastProfileAvailable;
    nw::ut::Tick m_LastProfileBegin;
    nw::ut::Tick m_LastProfileEnd;
    AXPROFILE m_ProfileUpdateBuffer[3][internal::MAX_PROFILE_COUNT];
    const AXPROFILE* m_pLastProfile;
    int m_CurrentProfileBuffer;
    ProfileReaderList m_ProfileReaderList;
};

class SoundThreadLock
{
public:
    SoundThreadLock()
    {
        SoundThread::GetInstance().Lock();
    }

    ~SoundThreadLock()
    {
        SoundThread::GetInstance().Unlock();
    }

private:
    NW_DISALLOW_COPY_AND_ASSIGN( SoundThreadLock );
};

} // namespace nw::snd::internal::driver
} // namespace nw::snd::internal
} // namespace nw::snd
} // namespace nw


#endif /* NW_SND_SOUND_THREAD_H_ */

